#include "../binary_c.h"
#ifdef NUCSYN

#include "nucsyn_yield_macros.h"
/************************************************************/
/* local functions */
static Abundance Pure_function Total_yield(const Isotope i,
                                           struct stardata_t * RESTRICT const stardata);
#ifdef NUCSYN_ID_SOURCES_GCE
static Abundance Pure_function Total_yield_from_source(const Yield_source s,
                                                       const Isotope i,
                                                       const struct stardata_t * RESTRICT const stardata);
#endif// NUCSYN_ID_SOURCES_GCE
/************************************************************/

void nucsyn_binary_yield(struct stardata_t * RESTRICT const stardata,
                         const Boolean final)
{
    /*
     * Function to log chemical yields from the binary system
     * often as a function of source type.
     *
     * Note: this function does not add up the yields, it only
     * outputs them. When the timestep is zero, it does nothing.
     * This function relies on the binary_c timestep being short
     * enough to resolve yield timesteps (this is done in the 
     * timestepping code automatically).
     */
#ifdef NUCSYN_YIELDS
    double apparent_time = Yields_time(stardata->model.time);

    Dprint("at apparent_time %g\n",apparent_time);
    
    double m; /* (usually?) total mass - reused in many places */
    Star_number MAYBE_UNUSED k;
#ifdef TESTYIELDS
    double deltat=0.0, testmultiplier=1.0;
#endif
    
    // counters
    Isotope i;
    Yield_source MAYBE_UNUSED j;

#ifdef SAVE_YIELDS
    /*
     * First timestep setup
     */
    if(unlikely(Is_zero(stardata->model.time)))
    {
        memset(stardata->model.oldyields,0,ISOTOPE_MEMSIZE);
#ifdef NUCSYN_ID_SOURCES
        memset(stardata->model.oldyieldsources,
               0,
               NUMBER_OF_STARS*ISOTOPE_MEMSIZE*NUMBER_OF_SOURCES);
#endif
    }
#endif

    
    /*
     * When running with log times, we require a start time 
     * and cannot output until then.
     */
    if(stardata->preferences->yields_logtimes == TRUE &&
       unlikely(stardata->model.time < stardata->preferences->yields_startlogtime))
    {
        return;
    }
    
#ifdef NUCSYN_GCE
    /* GCE populations every timestep */
    //if(final==TRUE) nucsyn_ensemble_log(stardata,final);
#endif

    if(unlikely(
#ifndef NUCSYN_LOG_YIELDS_EVERY_TIMESTEP
        /* We only want to output the yields if "final" is set to TRUE */
        final == FALSE ||
#endif // NUCSYN_LOG_YIELDS_EVERY_TIMESTEP
        
        /*
         * Prevent yield output in old BSE algorithm 
         * and when the timestep is zero: yields will be saved
         * for the next positive timestep.
         */
        stardata->model.dt < TINY
          ))
    {
        return;
    }

#ifdef NUCSYN_SPARSE_YIELDS
    /*
     * only output yields every NUCSYN_YIELDS_DT Myr : means there
     * are MANY fewer lines of data than if you output every timestep.
     * < 1Myr resolution is NOT usually required for GCE!
     */
    Boolean output = timestep_fixed_trigger(stardata,FIXED_TIMESTEP_YIELDS);
    
    if(final==FALSE && output==FALSE)
    {
        /*
         * No output required
         */
        Dprint("Timestep FIXED_TIMESTEP_YIELDS failed to trigger at t=%g\n", 
               stardata->model.time);
        return; 
    }
    else
    {
        /*
         * We should output in either log or linear time
         */
        apparent_time = stardata->preferences->yields_logtimes ?
            log10(stardata->model.time) : stardata->model.time;
        if(unlikely(fabs(apparent_time)<1e-15)) apparent_time = 0.0;
        Dprint("updated apparent time to %g\n",apparent_time);
        
#ifdef TESTYIELDS
        /*
         * Our test yield rate is constant, 1.0 Msun/y
         * for the whole population.The mass ejected in 
         * any one Yields_timestep is thus proportional
         * to the (linear) timestep.
         */
          testmultiplier = Yields_timestep_linear;
#endif // TESTYIELDS
    }
#endif // NUCSYN_SPARSE_YIELDS

#if defined NUCSYN_IGNORE_ZERO_BINARY_DX_YIELD && !defined TESTYIELDS

    /*
     * Test if the summed yields are > 0, if not, do nothing
     */
    Boolean yields_are_zero = TRUE;
    Ordered_isotope_loop(i)
    {
        if(Total_yield(i,stardata)>TINY)
        {
            yields_are_zero = FALSE;
            break;
        }
    }

    if(yields_are_zero==TRUE)
    {
        /*
         * no yield : just return
         */
        Dprint("yields are zero (star0 H = %g, He = %g) : return\n",
               stardata->star[0].Xyield[XH1],
               stardata->star[0].Xyield[XHe4]
            );
        return;
    }

#endif // NUCSYN_IGNORE_ZERO_BINARY_DX_YIELD && !defined TESTYIELDS

#if defined NUCSYN_YIELD_COMPRESSION && !defined TESTYIELDS
    /*
     * Check if yield integrals have changed. 
     * If not, don't do anything! 
     * (except the first timestep)
     */
    if(stardata->model.time > TINY)
    {
        Boolean changed = FALSE;
        Ordered_isotope_loop(i)
        {
            if(!Fequal(Total_yield(i,stardata),stardata->model.oldyields[i]))
            {
                changed=TRUE;
                break;
            }
        }
        if(changed==FALSE)
        {
            return;
        }
    }
#endif // NUCSYN_YIELD_COMPRESSION && !defined TESTYIELDS
    
#if defined NUCSYN_XTOT_CHECKS && !defined TESTYIELDS

    /* Check that the mass fractions add to 1 */
    printf("Mass fraction sums ");
    Starloop(k)
    {
        if(stardata->star[k].stellar_type!=MASSLESS_REMNANT)
        {
            double m=0.0;
            Isotope_loop(i)
            {
                prefetch(stardata->star[k].Xenv + Isotope_loop_sign);
                m += stardata->star[k].Xenv[i];
            }
            if(m>TINY)
            {
                printf(" %d->%12.12g ",k,m);

                if(Abs_less_than(1.0-m,MAXPCERR))
                {
                    fprintf(stderr,"Warning : star %d mass fraction error of %g %% > MAXPCERR = %g",k,fabs(1.0-m),MAXPCERR);
                    fprintf(stderr,"\nAbund array : abund (running total)\n");
                    m=0.0;
                    Ordered_isotope_loop(i)
                    {
                        m += stardata->star[k].Xenv[i];
                        fprintf(stderr," %12.12g (%12.12g) ",stardata->star[k].Xenv[i],m);
                    }
                    fprintf(stderr," \n");
                    Exit_binary_c(BINARY_C_OUT_OF_RANGE,"Abundance is out of range (see previous errors)\n");
                }
                else if(fabs(1.0-m)>0.001)
                {
                    printf("Mass fraction error (%g %%): rescaling ",100*fabs(1.0-m));
                    /* 2% error in mass fraction sum - rescale them all! */
                    Ordered_isotope_loop(i)
                    {
                        stardata->star[k].Xenv[i]/=m;
                        printf(" X%d=%g ",i,stardata->star[k].Xenv[i]);
                    }
                }
            }
            else
            {
                printf(" error m=0 ");
            }
        }
        else
        {
            printf(" %d->MASSLESS ",k);
        }
    }

    printf("\n");

#endif/* NUCSYN_XTOT_CHECKS */
    m = 0.0;

#if defined NUCSYN_ID_SOURCES_GCE
    Sourceloop(j)
    {
#ifndef TESTYIELDS
        /*
         * Only output if the yield of one isotope is > 0
         */
        Ordered_isotope_loop(i)
        {
            
            if(Is_not_zero(Total_yield_from_source(j,i,stardata))) 
                goto source_loop_breakout;
        }
        continue;
    source_loop_breakout:
#endif // ! TESTYIELDS
#endif // NUCSYN_ID_SOURCES_GCE
        
#ifdef NUCSYN_LOG_BINARY_DX_YIELDS
        Printf("D");
#endif

#ifdef NUCSYN_LOG_BINARY_X_YIELDS
#ifdef NUCSYN_ID_SOURCES_GCE
        Printf("XYIELDbin%d__ %0.4e ",j,apparent_time);
#else
        Printf("XYIELDbin__ %0.4e ",apparent_time);
#endif // NUCSYN_ID_SOURCES_GCE

#else
        if(unlikely(final==TRUE))
        {
#ifdef NUCSYN_ID_SOURCES_GCE
            Printf("XYIELDbin%d__ %0.4e ",j,apparent_time);
#else
            Printf("XYIELDbin__ %0.4e ",apparent_time);
#endif
        }
#endif

#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
        int zerocount=0;
#endif//COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT


        /*
         * Loop to output yield of each isotope
         */
        for(i=0;i<
#ifdef NUCSYN_GCE
                GCE_ISOTOPES
#else
                ISOTOPE_ARRAY_SIZE
#endif // NUCSYN_GCE
                ;i++)
        {
#ifdef NUCSYN_CHECK_LESS_THAN_ZERO_YIELDS
            if(Total_yield(i,stardata)<0.0)
            {
                fprintf(stderr,
                        "Yield of isotope %d is < 0 (=%g) <stellar types %d %d> <surface 1=%g 2=%g> <yields 1=%g 2=%g>\n",
                        i,
                        Total_yield(i,stardata),
                        stardata->star[0].stellar_type,
                        stardata->star[1].stellar_type,
                        stardata->star[0].Xenv[i],
                        stardata->star[1].Xenv[i],
                        stardata->star[0].Xyield[i],
                        stardata->star[1].Xyield[i]
                    );

                fprintf(stderr,"intpol=%d dt=%g\n",
                        stardata->model.intpol,
                        stardata->model.dt);
                printf("M acc 1 %g 2 %g\n",
                       stardata->star[0].dmacc,
                       stardata->star[1].dmacc);

                printf("Elem     Xsurf1      Xacc1     Xsurf2      Xacc2\n");
                int x;
                for(x=0;x<=10;x++)
                {
                    printf("%4d %10.4g %10.4g %10.4g %10.4g\n",
                           x,
                           stardata->star[0].Xenv[x],
                           stardata->star[0].Xacc[x],
                           stardata->star[1].Xenv[x],
                           stardata->star[1].Xacc[x]);
                }
                Exit_binary_c(BINARY_C_OUT_OF_RANGE,"Total_yield(%d) is <0.0",i);
            }
#endif /* NUCSYN_CHECK_LESS_THAN_ZERO_YIELDS */

#ifdef NUCSYN_LOG_BINARY_X_YIELDS       

#ifdef NUCSYN_ID_SOURCES_GCE
            {
                double dy = Total_yield_from_source(j,i,stardata);

#ifdef TESTYIELDS
                if(i==TESTYIELDISOTOPE)
                {
                    dy *= testmultiplier;
                }
#endif
                if(Yield_is_zero(dy))
                {
#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                    zerocount++;
#else
                    Printf("0 ");
#endif
                }
                else
                {
#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                    if(zerocount!=0)
                    {
                        Dump_zeros(zerocount);
                        zerocount=0;
                    }
#endif //COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                    Yield_print(dy);
                }

#ifdef BINARY_YIELD_LT_ZERO_CHECK
                /*
                 * The binary yield MAY go < 0 if both the timestep is negative at some
                 * point (e.g. during RLOF) and the star changes its stellar type
                 * when going backward. There is nothing that can be done except reduce 
                 * the timestep so the error is smaller.
                 */
                if(dy < 0.0)
                {
                    Exit_binary_c(BINARY_C_OUT_OF_RANGE,
                                  "binary yield < 0 error for source %d : %g (X1=%g X2=%g)\n",
                                  j,dy,stardata->star[0].Xenv[i],stardata->star[1].Xenv[i]);
                }
#endif
            }
#else

            if(Yield_is_zero(Total_yield(i,stardata)))
            {
#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                zerocount++;
#else
                Printf("0 ");
#endif
            }
            else
            {
#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                if(zerocount!=0)
                {
                    Dump_zeros(zerocount);
                    zerocount=0;
                }
#endif //COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                Yield_print(Total_yield(i,stardata));
            }
#endif

#else
            if(unlikely(final==TRUE))
            {

#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                if(zerocount!=0)
                {
                    Dump_zeros(zerozcount);
                    zerocount=0;
                }
#endif //COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT

                Yield_print(Total_yield(i,stardata));
            }
#endif // NUCSYN_LOG_BINARY_X_YIELDS
            m += Total_yield(i,stardata);
        }

#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
        if(zerocount!=0)
        {
            Dump_zeros(zerocount);
        }
#endif //COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT

#ifdef NUCSYN_GCE
#ifdef NUCSYN_GCE_ELSE
        /*
         * Output the sum of all the other isotopes so we can 
         * count the metallicity properly
         */
        double Xother=0.0;
        for(i=GCE_ISOTOPES;i<ISOTOPE_ARRAY_SIZE;i++)
        {
#ifdef NUCSYN_ID_SOURCES_GCE
            Xother += Total_yield_from_source(j,i,stardata);
#else 
            Xother += Total_yield(i,stardata);
            //printf("Add %g of isotope %d to else, Xother now %g\n",Total_yield(i,stardata),i,Xelse);
#endif // NUCSYN_ID_SOURCES     
        }
        Yield_print(Xother);
#endif // NUCSYN_GCE_ELSE
#endif // NUCSYN_GCE

#if defined NUCSYN_ID_SOURCES && defined NUCSYN_ID_SOURCES_OLD_METHOD
        /* output yields by source */
        Printf("\n");
        double ytot = 0.0;
        Starloop(k)
        {
            ytot += nucsyn_totalX(stardata->star[k].Xyield,
                                  0,
                                  ISOTOPE_ARRAY_SIZE);
        }

        if(ytot > TINY)
        {
            Sourceloop(j)
            {
                double srctot = 0.0;
                Starloop(k)
                {
                    srctot += nucsyn_totalX(stardata->star[k].Xsource[j],
                                            0,
                                            ISOTOPE_ARRAY_SIZE);
                }
                printf("\nSource %d %s mass %g frac %g %%\n",
                       j,
                       Nucsyn_source_string(j),
                       Max(0.0,srctot),
                       100.0*Max(0.0,srctot)/ytot);
                if(srctot>TINY)
                {
                    Ordered_isotope_loop(i)
                    {
                        double yy = 0.0;
                        Starloop(k)
                        {
                            yy += stardata->star[k].Xsource[j][i];
                        }
                        printf(YIELD_FORMAT,yy);
                    }
                }
                printf("\n");
            }
        }
#endif //NUCSYN_ID_SOURCES_OLD_METHOD && NUCSYN_ID_SOURCES

#ifdef NUCSYN_ID_SOURCES_GCE
        Printf("\n");
    } // end of loop over source types
#endif // NUCSYN_ID_SOURCES_GCE

    /*
     * Binary mp yields
     */
#if (defined NUCSYN_LOG_BINARY_X_YIELDS && !defined NUCSYN_ID_SOURCES_GCE)
    Printf("\n");
#else
    if(unlikely(final==TRUE))
    {
        Printf("\n");
    }
#endif // NUCSYN_LOG_BINARY_X_YIELDS

#ifdef NUCSYN_LOG_BINARY_MPYIELDS
    Printf("MPYIELDbin__ %0.4e ",apparent_time);
    m = 0.0;
    Ordered_isotope_loop(i)
    {
        Starloop(k)
        {
            /*
             * Yield p_m
             */

            double mass_ejected = stardata->star[k].pms_mass - stardata->star[k].mass;
            double d = 
                stardata->star[k].Xyield[i] - 
                stardata->common.XZAMS[i] * mass_ejected * stardata->model.probability;
            m += d;
            stardata->star[k].mpyield[i] = d;
        }
        /* remember that to get the p(i) you must divide by the zams mass */
        /* though to compare to lynnette don't divide */
        Yield_print(TOTAL_MPYIELD/TOTAL_PMS_MASS);
    }
    Printf("\n");
#endif//NUCSYN_LOG_BINARY_MPYIELDS

#endif // NUCSYN_YIELDS

#ifdef SAVE_YIELDS
    Isotope_loop(i)
    {
        stardata->model.oldyields[i] = 0.0;
        Starloop(k)
        {
            prefetch(stardata->star[k].Xyield + Isotope_loop_sign);
            stardata->model.oldyields[i] += stardata->star[k].Xyield[i];
        }

#ifdef NUCSYN_ID_SOURCES
        Sourceloop(j)
        {
            Starloop(k)
            {
                stardata->model.oldyieldsources[k][j][i] = 
                    stardata->star[k].Xsource[j][i];
            }
        }
#endif //NUCSYN_ID_SOURCES
    }
#endif//SAVE_YIELDS

#ifdef NANCHECKS
    Starloop(k)
    {
        nancheck("nucsyn_binary_yield",
                 stardata->star[k].Xyield,
                 ISOTOPE_ARRAY_SIZE);
    }
#endif//NANCHECKS 

#ifdef LOG_FINAL_TPAGB_ABUNDS
    Starloop(k)
    {
        if(stardata->star[k].final_tpagbX !=NULL)
        {
            printf("FINALTPAGB%d ",k);
            Ordered_isotope_loop(i)
            {
                printf("%g ",stardata->star[k].final_tpagbX[i]);
            }
            printf("\n");
            printf("FINAL C12=%g N14=%g\n",
                   stardata->star[k].final_tpagbX[XC12],
                   stardata->star[k].final_tpagbX[XN14]);
        }
    }
#endif    //LOG_FINAL_TPAGB_ABUNDS

    return;

}

#ifdef NUCSYN_ID_SOURCES_GCE
static Abundance Pure_function Total_yield_from_source(
    const Yield_source s,
    const Isotope i,
    const struct stardata_t * RESTRICT const stardata
    )
{
    /*
     * Return the total yield of isotope i from source s,
     * or the yield since the previous call
     */
    Abundance totalyield = 0.0;
#ifdef NUCSYN_ID_SOURCES
    Star_number k;
    Starloop(k)
    {   
#ifdef TESTYIELDS
        if(i == TESTYIELDISOTOPE && k==0)
        {
            totalyield += TESTYIELDX;
            continue;
        }
#else
        totalyield += stardata->star[k].Xsource[s][i];
#endif

#ifdef NUCSYN_LOG_BINARY_DX_YIELDS
        /* 
         * Require differential yield
         */
        totalyield -= stardata->model.oldyieldsources[k][s][i];
#endif
    }
#endif
    return totalyield;
}
#endif // NUCSYN_ID_SOURCES_GCE

static Abundance Pure_function Total_yield(const Isotope i,
                                           struct stardata_t * RESTRICT const stardata)
{
    /*
     * Return the total yield of isotope i,
     * or the yield since the previous call
     */
    Abundance totalyield = 0.0;
    Star_number k;


    Starloop(k)
    {
#ifdef TESTYIELDS
        if(i==TESTYIELDISOTOPE && k==0)
        {
            totalyield += TESTYIELDX;
            continue;
        }
#endif //TESTYIELDS
        totalyield += stardata->star[k].Xyield[i];
    }

#ifdef NUCSYN_LOG_BINARY_DX_YIELDS
    /* 
     * Require differential yield
     */    
    totalyield -= stardata->model.oldyields[i];
#endif
    return totalyield;
}





#endif // NUCSYN
