#include "../binary_c.h"


/*
 * Event handling function for novae: when 
 * stardata->preferences->individual_novae is TRUE this 
 * function is called to update the stars and orbit
 * when a nova happens.
 *
 * When you choose individual_novae to be on, nova
 * derivatives (in star->derivative and stardata->model.derivative)
 * are zero on input to this function. We set them here so the
 * various logging functions (which follow this function call) know what
 * the approximate nova rates would be. 
 *
 * Novae are treated as instanteous events here, much like
 * a supernova would be.
 */

Event_handler_function nova_event_handler(void * const eventp,
                                          struct stardata_t * const stardata,
                                          void * data)
{
    //struct binary_c_event_t * const event = eventp;

    struct binary_c_nova_event_t * const nova_data =
        (struct binary_c_nova_event_t *) data;

    struct star_t * const donor = nova_data->donor;
    struct star_t * const exploder = nova_data->accretor;
    
    Dprint("caught nova event at %20.12g : exploder = %d with H-layer mass %g, donor = %d : f = %g\n",
           stardata->model.time,
           exploder->starnum,
           exploder->dm_novaH,
           donor->starnum,
           nova_data->f
        );

    /*
     * Calculate mass lost: note that dm_lost is positive,
     * while dm_loss_rate is negative.
     */
    const double dm_lost = (1.0 - nova_data->f) * exploder->dm_novaH; 
    const double dm_loss_rate = - dm_lost / stardata->model.dt; 

    /*
     * Re-accretion of a fraction
     */
    const double dm_gain = exploder->nova_beta * dm_lost; 
    const double dm_gain_rate = +dm_gain / stardata->model.dt;
    
    /*
     * Simulate a nova in the derivative 
     * (although this is only really useful for logging) 
     */
    exploder->derivative[DERIVATIVE_STELLAR_MASS_NOVA] = dm_loss_rate;
    donor->derivative[DERIVATIVE_STELLAR_MASS_NOVA] = dm_gain_rate;
    
    /*
     * Remove material from the white dwarf:
     * no hydrogen is left on the surface
     *
     * Here I just remove dm_lost from dm_novaH... but you 
     * could argue this material is mixed in and becomes part
     * of the white dwarf after the nova. 
     */
    exploder->dm_novaH = 0.0;
    exploder->dmacc = 0.0;
    exploder->mass -= dm_lost;

#ifdef NUCSYN
    /*
     * calculate nova yield abundances
     */
    double Xnova[ISOTOPE_ARRAY_SIZE];
    nucsyn_set_nova_abunds(stardata,
                           exploder,
                           donor->Xenv,//THIS IS WRONG
                           Xnova);

    /*
     * yield material in nova explosion
     */
    nucsyn_calc_yields(stardata,
                       exploder,
                       dm_lost,
                       Xnova,
                       0.0,
                       NULL,
                       exploder->starnum,
                       YIELD_NOT_FINAL,
                       NUCSYN_SOURCE_NOVAE);
#endif
    
    /*
     * Accrete some on the companion?
     */
    if(Is_not_zero(dm_gain))
    {
      
        /*
         * Put material on companion star: we put it 
         * in the accretion layer, let it get mixed
         * on the next timestep
         */
        donor->mass += dm_gain;
        donor->dmacc += dm_gain;

#ifdef NUCSYN
        /*
         * Put accreted material into the star's accretion layer
         */
        nucsyn_dilute_shell(
            donor->dmacc,
            donor->Xacc,
            dm_gain,
            Xnova
            );
        
        /*
         * Perhaps mix the star's accretion layer into its envelope
         */
        if(nucsyn_thermohaline_unstable(stardata,donor))
        {
            nucsyn_thermohaline_mix_star(stardata,
                                         donor);
        }                                

        /*
         * "negative-yield" accreted material
         */
        nucsyn_calc_yields(stardata,
                           donor,
                           0.0,
                           NULL,
                           dm_gain,
                           Xnova,
                           exploder->starnum,
                           YIELD_NOT_FINAL,
                           NUCSYN_SOURCE_NOVAE);
#endif // NUCSYN
    }
    
    /*
     * update stellar and orbital angular momenta
     * caused by mass loss from the exploder
     */
    double dJstar,dJorb;
    nova_angular_momentum_changes(
        stardata,
        exploder,
        -dm_lost,
        &dJstar,
        &dJorb
        );
    exploder->angular_momentum += dJstar;
    stardata->common.orbit.angular_momentum += dJorb;
    exploder->derivative[DERIVATIVE_STELLAR_ANGMOM_NOVA] += dJstar/stardata->model.dt;
    stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_NOVA] += dJorb/stardata->model.dt;
       
    /*
     * update stellar and orbital angular momenta
     * caused by mass gained onto the donor
     */
    nova_angular_momentum_changes(
        stardata,
        donor,
        dm_gain,
        &dJstar,
        &dJorb
        );
    donor->angular_momentum += dJstar;
    stardata->common.orbit.angular_momentum += dJorb;
    stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_NOVA] += dJorb/stardata->model.dt;
    donor->derivative[DERIVATIVE_STELLAR_ANGMOM_NOVA] += dJstar/stardata->model.dt;
    
    /*
     * update orbit
     */
    update_orbital_variables(stardata,
                             &stardata->common.orbit,
                             &stardata->star[0],
                             &stardata->star[1]);

    
    return NULL;
}

