#pragma once
#ifndef MEMORY_FUNCTION_MACROS_H
#define MEMORY_FUNCTION_MACROS_H

#ifdef __HAVE_MALLOC_H__
#include <malloc.h>
#endif // __HAVE_MALLOC_H__
#include <stdlib.h>
#include <string.h>

#include "../binary_c_code_options.h"
#include "../binary_c_structures.h"
#include "../breakpoints/breakpoints_prototypes.h"
#include "../binary_c_exit_prototypes.h"

#ifdef CODESTATS
static void increment_codestat(const int n,
                               const char *file,
                               const int line);

static void MAYBE_UNUSED increment_codestat(const int n,
                                            const char *file,
                                            const int line)
{
    codestats.counters[n]++;
    struct codestats_entry_t *array = codestats.entries[n];
    int i;
    const int vb = 0;
    if(vb)printf("CODESTAT search n=%d file=%s line=%d\n",n,file,line);

    if(array!=NULL)
    {
        /* search for hash entry */
        for(i=0;i<codestats.nentries[n];i++)
        {
            if(vb)printf("CODESTAT CF args %d %s to entry %d %s\n",
                         line,
                         file,
                         codestats.entries[n][i].line,
                         codestats.entries[n][i].file);
            if(line == codestats.entries[n][i].line &&
               Strings_equal(codestats.entries[n][i].file,file))
            {
                /*
                 * match
                 */
                codestats.entries[n][i].count++;
                if(vb)printf("CODESTAT FOUND at %d,%d -> new count %lu\n",
                             n,
                             i,
                             codestats.entries[n][i].count);
                return;
            }
        }
    }
    /* not found */

    if(vb)printf("CODESTAT : NOT FOUND\n");
    /* allocate new entry */
    i = codestats.nentries[n];
    codestats.entries[n] = realloc(codestats.entries[n],
                                   (i+1)*sizeof(struct codestats_entry_t));

    codestats.entries[n][i].file = malloc(sizeof(char)*(strlen(file)+2));
    strlcpy(codestats.entries[n][i].file,
            file,
            strlen(file)+1);
    codestats.entries[n][i].line = line;
    codestats.entries[n][i].count = 1;
    
    /* up counter */
    codestats.nentries[n]++;
}
#endif


/*
 * Memory function macros for binary_c
 *
 * We have custom Malloc, Calloc and Realloc to replace
 * the system calls (or use them, whatever you like).
 *
 * There are also macros for allocating and clearing pieces
 * of memory, such as GSL vectors and functions.
 */

/* 
 * "safe" freeing of memory via the Safe_free macro,
 * which enforces a NULL in the pointer after a call 
 * to free. 
 * Note tha the PTR must actually be a pointer (i.e. an lvalue), 
 * not an expression.
 */
#define Safe_free(PTR)                                                  \
    if(likely((PTR)!=NULL))                                             \
    {                                                                   \
        free(PTR);                                                      \
        (PTR)=NULL;                                                     \
    };

/*
 * As Safe_free but without the NULL setting: this should be considered
 * unsafe, but is still better than just freeing the pointer without
 * a NULL check.
 */
#define Unsafe_free(PTR)                        \
    if(likely((PTR)!=NULL))                     \
    {                                           \
        free(PTR);                              \
    }

/*
 * Check if a pointer has been allocated correctly.
 */
#ifdef ALLOC_CHECKS
#define _Alloc_check(P)                                  \
    if(unlikely((P)==NULL))                              \
    {                                                    \
        fprintf(stderr,"Memory allocation failed.\n");   \
        Backtrace;                                       \
        Exit_binary_c_no_stardata(                       \
            BINARY_C_ALLOC_FAILED,                                \
            "Memory allocation failed.\n"                \
            );                                           \
    }
#else
#define _Alloc_check(P) /* _Alloc_check(P) do nothing */
#endif



/*
 * Macro to choose the size of an array so it is forced
 * to be memory-aligned
 */
#if defined ALIGNSIZE && defined ALIGN_ARRAYS
#define Align(N) ( ALIGNSIZE * ( (int) (((int)(N))/((int)ALIGNSIZE)) + 1  ))
#else
#define Align(N) (N)
#endif

/*
 * Push malloc, calloc and realloc through macros to force alignment if
 * ALIGNSIZE is defined (e.g. by the configure script)
 */

#if defined __GNUC__ &&  defined ALIGNSIZE && !defined __NATIVE_MALLOC__

/* 
 * define macros to replace malloc, calloc and realloc
 */
#define Malloc(S)                               \
    ({                                          \
        Increment_codestat(CODESTAT_MALLOC);    \
        (aligned_malloc((S),(ALIGNSIZE)));      \
    })

#define Calloc(N,S) \
    ({                                          \
        Increment_codestat(CODESTAT_CALLOC);    \
        (aligned_calloc((N),(S),(ALIGNSIZE)));  \
    })

#define Realloc(P,S) \
    ({                                          \
        Increment_codestat(CODESTAT_REALLOC);   \
        (aligned_realloc((P),(S),(ALIGNSIZE))); \
    })

/* you could use the system realloc, but it's just as slow */
//#define Realloc(P,S) realloc((P),(S))

/* only in gcc > 4.7 or recent clang */
#if __GNUC_PREREQ(4,7) || defined __clang__
#define Builtin_aligned(P,A) __builtin_assume_aligned((P),(A))
#define Assume_aligned(P) __builtin_assume_aligned((P),(ALIGNSIZE))
#else
#define Builtin_aligned(P,A) (P)
#define Assume_aligned(P)  (P)
#endif



static inline void * Alloc_size_first Malloc_like aligned_malloc(size_t size,   
                                                                 size_t alignment);

static inline void * Alloc_size_first Malloc_like  aligned_malloc(size_t size,   
                                                                  size_t alignment)
{
    if(unlikely(size==0)) return NULL;
    void *p = NULL;
    if(0)fprintf(stderr, "size %ld at %p\n",  
                 (long int)size,p);
    int r = posix_memalign(&p,alignment,size); 
    _Alloc_check(p);
    return r==0 ? (void *) Builtin_aligned(p,alignment) : NULL;
}

static inline void * Alloc_size_product Malloc_like aligned_calloc(size_t nelem,  
                                                                   size_t elsize, 
                                                                   size_t alignment);
static inline void * Alloc_size_product Malloc_like aligned_calloc(size_t nelem,  
                                                                   size_t elsize, 
                                                                   size_t alignment)
{
    void *p = aligned_malloc(nelem*elsize,alignment);
    _Alloc_check(p);
    if(p) memset(p, 0, nelem*elsize);
    return p;
}

static inline void * Alloc_size_second aligned_realloc(void *ptr,         
                                                       size_t size,     
                                                       size_t alignment);
static inline void * Alloc_size_second aligned_realloc(void *ptr,       
                                                       size_t size,     
                                                       size_t alignment)
{
#ifdef HAVE_MALLOC_USABLE_SIZE
    if(unlikely(size==0)) return NULL;
    size_t usable = Malloc_usable_size(ptr);
    void *p = NULL;
    p = posix_memalign(&p,alignment,size)==0 ?          
        (void *) Builtin_aligned(p,alignment) :         
        NULL;                                           
    _Alloc_check(p);                            
    
    if(p!=NULL && ptr!=NULL)
    {
        memcpy(p,ptr,usable<size?usable:size);
        free(ptr);
    }
    return p;
#else
    // no Malloc_usable_size : use system realloc
    void * p = realloc(ptr,size);
    _Alloc_check(p);
    return p;
#endif //HAVE_MALLOC_USABLE_SIZE
}

/* for variables (not mallocd) use Aligned */
#define Aligned __attribute__ ((aligned (ALIGNSIZE)))

#include "memory_alignment_checks.h"

/*
 * Check that memcpy is aligned if CHECK_MEMCPY_ALIGNMENT
 * is defined.
 */
#ifdef CHECK_MEMCPY_ALIGNMENT
#define memcpy(dest,src,n)                              \
    check_aligned_memcpy_heap_source(dest,src,n)
#endif // CHECK_MEMCPY_ALIGNMENT



#else

/*
 * For one of many reasons, use standard malloc, etc.
 * (This may be faster!) 
 */
static inline void * Alloc_size_first Malloc_like _binary_c_malloc(size_t size); 
static inline void * Alloc_size_product Malloc_like _binary_c_calloc(size_t nmemb,      
                                                                     size_t size);
static inline void * Alloc_size_second _binary_c_realloc(void *ptr,        
                                                                     size_t size);

static inline void * Alloc_size_first Malloc_like _binary_c_malloc(size_t size) 
{
    void * p = malloc(size);
    _Alloc_check(p);
    return p;
}
static inline void * Alloc_size_product Malloc_like _binary_c_calloc(size_t nmemb,size_t size) 
{
    void * p = calloc(nmemb,size);
    _Alloc_check(p);
    return p;
}

static inline void * Alloc_size_second _binary_c_realloc(void *ptr, 
                                                         size_t size) 
{
#ifdef HAVE_MALLOC_USABLE_SIZE
    void *p;
    if(ptr)
    {
        p = realloc(ptr,size);
    }
    else
    {
        p = malloc(size);
    }
    _Alloc_check(p);
    return p;
#else
    void * p = realloc(ptr,size);
    _Alloc_check(p);
    return p;    
#endif
}


/* standard C malloc */
#define Malloc(S)                               \
    ({                                          \
        Increment_codestat(CODESTAT_MALLOC);    \
        (_binary_c_malloc(S));                  \
    })
#define Calloc(N,S)                             \
    ({                                          \
        Increment_codestat(CODESTAT_CALLOC);    \
        (_binary_c_calloc((N),(S)));            \
    })
#define Realloc(P,S)                            \
    ({                                          \
        Increment_codestat(CODESTAT_REALLOC);   \
        (_binary_c_realloc((P),(S)));           \
    })
#define Assume_aligned(P) (P)
#define Aligned

#endif //__GNUC__ && ALIGNSIZE && !__NATIVE_MALLOC__ 

/*
 * Alloc_or_clear makes sure pointer P points
 * to a zeroed piece of memory of size S.
 * If P is NULL, this is done with Calloc.
 * If P is non-NULL, the memory at P is assumed to 
 * be of the correct size, and is then memset to zero.

 * There are three variants:
 * 
 * Clear_stack_location is for an array on the stack,
 * so simply clears it.
 *
 * Alloc_or_clear_pointer is for an array ALLOC'd on the heap
 *
 */

#define Clear_stack_location(P,S)               \
    memset((P),0,(S)/sizeof(*(P)));

#define Clear_heap_location(P,S)                \
    memset((P),0,(S));

#define Alloc_or_clear_pointer(P,S)                     \
    if((P)!=NULL)                                       \
    {                                                   \
        Clear_heap_location((P),(S));                   \
    }                                                   \
    else                                                \
    {                                                   \
        (P)=Calloc(1,(S));                              \
        _Alloc_check(P);                                \
    }



#define Alloc_or_clear(P,S) Clear_stack_location((P),(S))

#ifdef USE_GSL

#include <gsl/gsl_vector.h>
#include <gsl/gsl_multiroots.h>
/*
 * Wrappers for GSL allocation functions
 * which also perform binary_c's standard alloc checks
 */

#define New_GSL_vector(N) _New_GSL_vector((N))

static inline void *_New_GSL_vector(const int n);
static inline void *_New_GSL_vector(const int n)
{
    gsl_vector * v = gsl_vector_alloc(n);
    _Alloc_check(v);
    return v;
}
#define Safe_free_GSL_vector(PTR)                                       \
    if(likely((PTR)!=NULL))                                             \
    {                                                                   \
        gsl_vector_free(PTR);                                           \
        (PTR)=NULL;                                                     \
    };

#define GSL_multiroot_fsolver_ALLOC(T,N) _GSL_multiroot_fsolver_alloc((T),(N))

static inline void *_GSL_multiroot_fsolver_alloc(const gsl_multiroot_fsolver_type * T, size_t n);
static inline void *_GSL_multiroot_fsolver_alloc(const gsl_multiroot_fsolver_type * T, size_t n)
{
    gsl_multiroot_fsolver * s = gsl_multiroot_fsolver_alloc (T, n);
    _Alloc_check(s);
    return s;
}

#define Safe_free_GSL_multiroot_fsolver(PTR)    \
    if(likely((PTR)!=NULL))                     \
    {                                           \
        gsl_multiroot_fsolver_free(PTR);        \
        (PTR)=NULL;                             \
    };

#endif // USE_GSL

/*
 * Override standard memcpy with Skywind's fast_memcpy?
 */
#ifdef USE_SKYWIND_FAST_MEMCPY
#undef memcpy
#define memcpy memcpy_fast
#ifdef HAVE_AVX
#include "FastMemcpy_Avx.h"
#else
#include "FastMemcpy.h"
#endif
#endif

/*
 * Override standard memcpy/memmove with Apex' functions?
 */
#ifdef USE_APEX_MEMMOVE
#undef memmove 
#undef memcpy 
#define memmove apex_memmove
#define memcpy apex_memcpy
#endif


/*
 * Other memcpys
 */
#ifdef USE_MEMCPY_DDR4
#undef memcpy
#define memcpy fastMemcpy_ddr4
#endif

#ifdef USE_ASMLIB
#undef memcpy
#define memcpy A_memcpy
#endif

/*
 * Override memcpy with CODESTATS version
 */
#if defined CODESTATS && !defined memcpy
#define memcpy(D,S,N) codestats_memcpy((D),(S),(N),__FILE__,__LINE__)
#endif // CODESTATS && !memcpy

/*
 * Malloc with memcpy, where TYPE is 
 * a variable or struct type, as would be 
 * passed to sizeof(), and FROM is its 
 * memory location.
 */
#define Malloc_from(FROM,TYPE)                          \
    memcpy(Malloc(sizeof(FROM)),(FROM),sizeof(FROM))



#endif // MEMORY_FUNCTION_MACROS_H
