#include "../binary_c.h"
#include "memory_alignment_checks.h"

struct stardata_t * Returns_nonnull Nonnull_all_arguments copy_stardata(
    const struct stardata_t  * RESTRICT Aligned const from,
    struct stardata_t  * RESTRICT Aligned const to,
    const unsigned int copy_previous 
    )
{
    /*
     * Copy stardata in "from" to "to", where from != to.
     *
     * copy_previous:
     *   COPY_STARDATA_PREVIOUS_NONE  :  do nothing, previous stack is left NULL
     *   COPY_STARDATA_PREVIOUS_POINTERS  :  copy history pointers
     *   COPY_STARDATA_PREVIOUS_COPY  :  make copy of history data into new stardatas
     *
     * If to == from, exit with an error (using "from" as the reference stardata).
     *
     * Returns a pointer to "to".
     *
     * Neither from nor to should ever be NULL.
     */
    if(to == from)
    {
        struct stardata_t * stardata = (struct stardata_t *) from;
        Exit_binary_c(BINARY_C_POINTER_FAILURE,
                      "to == from in copy_stardata : this should not happen.");
    }
    else
    {
        if(likely(copy_previous == COPY_STARDATA_PREVIOUS_NONE))
        {
            memcpy(to,
                   from,
                   sizeof(struct stardata_t));

            /*
             * Don't copy history: force this to be NULL
             */
            to->previous_stardatas = NULL;
            to->previous_stardata = NULL;
            to->n_previous_stardatas = 0;
        }
        else if(copy_previous == COPY_STARDATA_MAINTAIN_FROM_STACK_POINTERS)
        {
            /*
             * Preserve existing previous_stardata and stardata_stack
             * from "from". This is just a straight memcpy.
             */
            memcpy(to,
                   from,
                   sizeof(struct stardata_t));
        }
        else if(copy_previous == COPY_STARDATA_MAINTAIN_TO_STACK_POINTERS)
        {
            /*
             * Preserve existing previous_stardata and stardata_stack
             * from "to"
             */
            const int np = to->n_previous_stardatas;
            const int ns = to->n_stardata_stack;
            struct stardata_t ** const pstack = to->previous_stardatas;
            struct stardata_t ** const sstack = to->stardata_stack;
            memcpy(to,
                   from,
                   sizeof(struct stardata_t));
            to->previous_stardatas = pstack;
            if(pstack != NULL) to->previous_stardata = pstack[0];
            to->stardata_stack = sstack;
            to->n_previous_stardatas = np;
            to->n_stardata_stack = ns;
        }
        else if(copy_previous == COPY_STARDATA_PREVIOUS_COPY)
        {            
            /*
             * Copy the data onto a new stack
             */
            memcpy(to,
                   from,
                   sizeof(struct stardata_t));

            /*
             * Make the new stack
             */
            to->previous_stardatas =
                Malloc(sizeof(struct stardata_t*) *
                       Max(1,from->n_previous_stardatas));
            
            if(from->n_previous_stardatas == 0)
            {
                /*
                 * We require a previous stardata but don't have one, 
                 * so just copy stardata
                 */
                to->n_previous_stardatas = 1;
                to->previous_stardatas[0] = New_stardata_from(to);
            }
            else
            {
                /*
                 * Copy the from stack's data to to the to stack 
                 */
                int i;
                for(i=0;i<from->n_previous_stardatas;i++)
                {
                    to->previous_stardatas[i] = New_stardata;
                    memcpy(to->previous_stardatas[i],
                           from->previous_stardatas[i],
                           sizeof(struct stardata_t));
                }
                to->n_previous_stardatas = from->n_previous_stardatas;
            }
            to->previous_stardata = to->previous_stardatas[0];
        }
        else
        {
            struct stardata_t * stardata = (struct stardata_t *) from;
            Exit_binary_c(BINARY_C_ALGORITHM_OUT_OF_RANGE,
                          "Unknown copy_stardata algorithm");
        }
    }
    return to;
}
