#include "../binary_c.h"

/*
 * Random number generator. 
 *
 * Returns a double-precision number between 0.0 and 1.0.
 *
 * We used to use the Numerical Recipes routine, but
 * this is not freely distributable, so now we use 
 * the GNU drand48_r routine. This takes an initial 
 * seed so is reproducible when the seed is given,
 * otherwise the seed is generated by the current time.
 *
 * If random_seed is NULL, use stardata's random seed.
 */

#include <stdlib.h>

#ifdef BINARY_C_USE_LOCAL_RAND48
#include "binary_c_drandr.h"
#endif // BINARY_C_USE_LOCAL_RAND48


double random_number_buffer(const struct stardata_t * const stardata,
                            Random_buffer * const buffer)
{
    /*
     * Given a buffer, return a random number
     */
    double r;
#ifdef USE_DRAND48_R
    drand48_r(buffer,&r);
#endif // USE_DRAND48_R
#ifdef USE_MERSENNE_TWISTER
    r = genrand64_real1(buffer);
#endif // USE_MERSENNE_TWISTER
    return r;
}


void set_random_buffer(Random_seed seed,
                       Random_buffer * buffer)
{
#ifdef USE_DRAND48_R
    srand48_r(seed,
              buffer);
#endif // USE_DRAND48_R
#ifdef USE_MERSENNE_TWISTER
    init_genrand64(seed,
                   buffer);
#endif // USE_MERSENNE_TWISTER
}


#ifdef USE_DRAND48_R
#define __LOGSTUB "d48r"
#endif // USE_DRAND48_R
#ifdef USE_MERSENNE_TWISTER
#define __LOGSTUB "Mers"
#endif // USE_MERSENNE_TWISTER

double random_number(struct stardata_t * const stardata,
                     Random_seed * random_seed)
{
    double r; /* return value */
    
    /*
     * Use stardata's random seed if none is given
     */
    if(random_seed == NULL)
    {
        random_seed = &stardata->common.random_seed;
    }

    /*
     * If we haven't yet, set up the buffer
     */
    if(unlikely(stardata->common.random_buffer.set == FALSE))
    {
        Dprint(
            "Set drand buffer from random_seed = %ld ",*random_seed);      
        stardata->common.random_buffer.set = TRUE;
#ifdef RANDOM_NUMBER_LOG
        Append_logstring(LOG_RANDOM,
                         "Randbuf=%ld ",*random_seed);
#endif // RANDOM_NUMBER_LOG
        
        stardata->common.random_buffer.count=0;
        set_random_buffer(*random_seed,
                          &stardata->common.random_buffer.buffer);
        
        /* 
         * perhaps skip some numbers
         */
        if(stardata->preferences &&
           stardata->preferences->random_skip > 0)
        {
            int i;
            for(i=0;i<stardata->preferences->random_skip;i++)
            {
                random_number_buffer(stardata,
                                     &stardata->common.random_buffer.buffer);
            }
            stardata->common.random_buffer.count += stardata->preferences->random_skip;
        }
    }
    
    /*
     * set the random number in r 
     */
    r = random_number_buffer(stardata,
                             &stardata->common.random_buffer.buffer);
#ifdef RANDOM_NUMBER_LOG
    Append_logstring(LOG_RANDOM,"%s(%ld)=%g ",
                     __LOGSTUB,
                     stardata->common.random_buffer.count,
                     r);
#endif // RANDOM_NUMBER_LOG

    Dprint("YYY RAND %g (seed %ld, count %ld)\n",
           r,
           (long int)*random_seed,
           stardata->common.random_buffer.count);

    stardata->common.random_buffer.count++;
    return r;
}

