#pragma once
#ifndef MATHS_PROTOTYPES_H
#define MATHS_PROTOTYPES_H

#include "../binary_c_code_options.h"
#include "../binary_c_parameters.h"
#include "../binary_c_macros.h"

#ifdef USE_GSL
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_permutation.h>
#endif // USE_GSL

#ifdef BINARY_C_USE_LOCAL_RAND48
#include "binary_c_drandr.h"
#endif // BINARY_C_USE_LOCAL_RAND48

/* Mathematical functions, generally stolen from numerical recipes */
#ifndef USE_NATIVE_ISNAN
int my_isnan(double val);
#endif // USE_NATIVE_ISNAN

/** Random number generator function **/
double random_number(struct stardata_t * const stardata,
                    Random_seed * random_seed);

void set_random_buffer(Random_seed seed,
                       Random_buffer * buffer);
double random_number_buffer(const struct stardata_t * const stardata,
                            Random_buffer *buffer);


double qsimp(double (*func)(double), double a, double b);
double trapzd(double (*func)(double),double a,double b,int n);

double qsimp_array(double (*func)(double,double*,double*), double a, double b,
		   double *xarray, double *yarray);


double trapzd_array(double (*func)(double,double*,double*),
		   double a,
		   double b,
		   int n,
		   double *xarray,
		    double *yarray);

double kaps_rentrop_GSL(const double h, // timestep
		    double * RESTRICT y, // number densities
		    double * RESTRICT y2, // dummy array
		    const unsigned int n, // number of isotopes
		    double ** RESTRICT J, // Jacobian (modified)
		    const double * RESTRICT sigmav, // reaction rates
		    double *N, // number densities of all isotopes
		    void (*jacobian)(double **J,const double *N,const double * RESTRICT sigmav),
		    void (*derivatives)(const double *y,double *dydt, const double * RESTRICT sigmav,const double *N),
		    double * RESTRICT f1,double * RESTRICT f2,
		    double * RESTRICT f3,double * RESTRICT f4,
		    int * RESTRICT indx
		    );

void SolveCubic2(const double  b,
                 const double  c,
                 const double  d,
                 unsigned int    *solutions, /* number of REAL solutions */
                 double * RESTRICT x,
                 double * RESTRICT y);

#ifdef NANCHECKS
void _nancheck(struct stardata_t * RESTRICT const stardata,
               const char * RESTRICT const label,
               const double * RESTRICT const x,
               const unsigned int s);
#endif


void kaps_rentrop_LU_decomp(double  ** RESTRICT a,
                            const int n,
                            gsl_matrix * m,
                            gsl_permutation * p
    ); 
void kaps_rentrop_LU_backsub(double  ** RESTRICT a,
                             double  * RESTRICT b,
                             const int n,
                             gsl_matrix  * m,
                             gsl_permutation * p
    );



void interpolate_cubic_spline(const double * RESTRICT table, // (pointer to) the data table
			      const int n, // the number of parameters (i.e. dimensions)
			      const int d, // the number of data items
			      const int l, // the number of lines of data
			      const double * RESTRICT x, // the values of the parameters
			      double * RESTRICT r,  // the result of the interpolation
			      const unsigned int table_id, // table identifier
			      const Boolean cache_hint // tells us to use the cache, or not
    );


struct probability_distribution_t * new_pdist(struct stardata_t * RESTRICT const stardata,
                                              void *,
                                              double xmin,
                                              double xmax,
                                              int nlines,
                                              int maxcount);

double select_from_pdist(struct stardata_t * const stardata,
                         struct probability_distribution_t * const p);


void free_pdist(struct probability_distribution_t * RESTRICT const p);

double Constant_function fastPow(const double a, const double b) ;

double brent_valist(double (*func)(double,va_list args),
                    const int itmax,
                    const double xmin,
                    const double xmax,
                    const double tol,
                    va_list args
    );

double brent(double (*func)(double,va_list args),
             const double xmin,
             const double xmax,
             const double tol,
             ...
    );




double generic_bisect(int * RESTRICT error,
                      const int monochecks,
                      int n,
                      brent_function func,
                      double guess,
                      double min,
                      double max,
                      const double tol,
                      const int itmax,
                      const Boolean uselog,
                      const double alpha,
                      ...);
void timestamp (void);

double Li(const int s,
          const double z) Constant_function;
double polyalgorithm(const int s,
                     const double z) Constant_function;



Boolean inverse3( const double theMatrix [/*Y=*/3] [/*X=*/3],
                  double theOutput [/*Y=*/3] [/*X=*/3] );

double GSL_integrator(const double lower,
                      const double upper,
                      const double tolerance,
                      double * const error,
                      const int integrator,
                      double func(double x,void *params), 
                      ...);
void setup_GSL_handlers(struct stardata_t * RESTRICT const stardata);


void check_nans_are_signalled(void);

#ifdef USE_MERSENNE_TWISTER
#include "mersenne_twister_prototypes.h"
#endif // USE_MERSENNE_TWISTER

double max_from_list(int n,...);


Constant_function double sin2(double x);
Constant_function double cos2(double x);

double generic_minimizer(int * status,
                         const int n,
                         const Boolean logscale,
                         const double accuracy,
                         double *minimumx,
                         const brent_function func,
                         const double min,
                         const double max,
                         ...);


void apply_derivative(struct stardata_t * const stardata,
                      double * const value, /* value to which we add the derivative */
                      double * const d, /* array of derivatives */
                      const double dt, /* timestep */
                      const Derivative nd /* derivative number */);


void test_integrators(struct stardata_t * stardata) No_return;

Constant_function double fermi(const double x,
                               const double a, 
                               const double b,
                               const double c,
                               const double d);
Constant_function double dfermi(const double x,
                                const double a, 
                                const double b,
                                const double c,
                                const double d);

double Pure_function bin_data(const double x,
                              const double bin_width);

#endif // MATHS_PROTOTYPES
