#include "binary_c.h"
/*
 * The binary_c stellar population nucleosynthesis framework.
 *
 * Contact: r.izzard@surrey.ac.uk or rob.izzard@gmail.com
 *
 * http://personal.ph.surrey.ac.uk/~ri0005/binary_c.html
 * https://gitlab.eps.surrey.ac.uk/ri0005/binary_c
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-announce
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-devel
 * https://twitter.com/binary_c_code
 * https://www.facebook.com/groups/149489915089142/
 *
 * Please see the files README, LICENCE and CHANGES,
 * and the doc/ directory for documentation.
 *
 *
 * main2 : functions used in binary_c which 
 * don't really go elsewhere... often subserviant to 
 * the main() function, or system calls.
 *
 */

#ifdef RANDOM_SYSTEMS
void set_random_system(struct stardata_t * const RESTRICT stardata)
{
    
/* choose M1,M2,period,metallicity, eccentricity to be random */
/*
#define RAN ((double)random_number_buffer(stardata,                   \
                                          &common->random_systems_buffer))
*/
#define RAN ((double)random_number(stardata,NULL))
    
/* choose a random number between A and B */
#define RANNUM(A,B) (((A)+RAN*((B)-(A))))
    
/* choose a log-distributed random number between A and B */
#define LOGRANNUM(A,B) (exp(RANNUM(log(A),log(B))))

/* random interger between A and B */
#define RANINT(A,B) ((int)(RANNUM((A),(B))+0.5))

#define MMIN 0.1
#define MMAX 100.0
#define PMIN 0.5
#define PMAX 1e4

    struct common_t * common=&(stardata->common);
    struct star_t * s1=&(stardata->star[0]);
    struct star_t * s2=&(stardata->star[1]);
    struct preferences_t * p = stardata->preferences;

    /*
     * Seed for generating the random numbers to make the 
     * random systems
     */
    if(common->random_systems_seed == 0)
    {
        common->random_systems_seed = random_seed();
        set_random_buffer(common->random_systems_seed,
                          &common->random_buffer.buffer);
    }
        
    s1->mass = LOGRANNUM(MMIN,MMAX);
    s2->mass = LOGRANNUM(MMIN,s1->mass);
    common->orbit.period = common->zams_period = LOGRANNUM(PMIN,PMAX);
    common->orbit.separation = calculate_orbital_separation(stardata);
    common->zams_eccentricity = common->orbit.eccentricity = RAN;
    common->metallicity = RANNUM(1e-4,0.03);

    long int new_random_seed = random_seed();
    common->random_seed = new_random_seed;
    p->cmd_line_random_seed = new_random_seed;
    common->init_random_seed = new_random_seed;
    common->random_buffer.set = FALSE;
    
    if(stardata->preferences->random_systems >= 2)
    {
        /*
         * If random_systems >= 2, randomize some other variables
         */
        p->alpha_ce = RAN;
        p->lambda_ce = 0.1 + 2.0 * RAN;
        p->lambda_ionisation = RAN;
        p->lambda_enthalpy = RAN;
        p->comenv_merger_spin_method = RANINT(0,2);
        p->comenv_ejection_spin_method = RANINT(0,1);
        p->comenv_post_eccentricity = RAN;
        p->nelemans_gamma = RANNUM(0.1,2.0);

        p->pre_main_sequence = RANINT(0,1);
        s1->vrot0 = RANNUM(0.0,500.0);
        s2->vrot0 = RANNUM(0.0,500.0);
        stardata->model.max_evolution_time = LOGRANNUM(10.0, 100000.0);

        p->wind_angular_momentum_loss = RANINT(0,3);
        p->wind_djorb_fac = RANNUM(0.0,10.0);
        p->lw = RANNUM(0.0, 10.0);
        p->gravitational_radiation_model = RANINT(0,4);
        p->magnetic_braking_factor = RANNUM(0,100.0);
        p->tidal_strength_factor = RANNUM(0.0,1.0);
        p->E2_prescription = RANINT(0,1);
        p->RLOF_method = 0;//RANINT(0,3);
        p->RLOF_interpolation_method = 0;//RANINT(0,1);
        p->RLOF_mdot_factor = RANNUM(0.0,10.0);
        p->rlperi = RANINT(0,1); 
        p->RLOF_angular_momentum_transfer_model = RANINT(0,1);
        p->nonconservative_angmom_gamma =
            RAN < 0.5 ? RANINT(-2,-1) : RANNUM(0.0,10.0);
        p->merger_angular_momentum_factor = RANNUM(0.0,1.0);
        p->hachisu_disk_wind = RANNUM(0,1);
        p->hachisu_qcrit =
            RAN < 0.5 ? -1 : RANNUM(0.5,10.0);
        p->WD_accretion_rate_novae_upper_limit_hydrogen_donor = LOGRANNUM(1e-10,1e-5);
        p->WD_accretion_rate_novae_upper_limit_helium_donor = LOGRANNUM(1e-10,1e-5);
        p->WD_accretion_rate_novae_upper_limit_other_donor = LOGRANNUM(1e-10,1e-5);

        p->WD_accretion_rate_new_giant_envelope_lower_limit_hydrogen_donor = RANNUM(
            p->WD_accretion_rate_novae_upper_limit_hydrogen_donor,
            p->WD_accretion_rate_novae_upper_limit_hydrogen_donor * 100.0);
        p->WD_accretion_rate_new_giant_envelope_lower_limit_helium_donor = RANNUM(
            p->WD_accretion_rate_novae_upper_limit_helium_donor,
            p->WD_accretion_rate_novae_upper_limit_helium_donor * 100.0);
        p->WD_accretion_rate_new_giant_envelope_lower_limit_other_donor = RANNUM(
            p->WD_accretion_rate_novae_upper_limit_other_donor,
            p->WD_accretion_rate_novae_upper_limit_other_donor * 100.0);

        p->comenv_prescription = RANINT(0,2);
        
    }
    

    
#ifdef RANDOM_SYSTEMS_SHOW_ARGS 
#ifdef RANDOM_SYSTEMS_SHOW_ARGS_AND_START_TIME
    time_t t = time(NULL);
    char timestring[100];
    if(strftime(timestring,sizeof(timestring),"At %F %T :   ",localtime(&t)))
    {
        fprintf(RANDOM_SYSTEMS_SHOW_ARGS_STREAM,
                "%s",
                timestring);
    }
#endif//RANDOM_SYSTEMS_SHOW_ARGS_AND_START_TIME

    fprintf(RANDOM_SYSTEMS_SHOW_ARGS_STREAM,
            "--M_1 %30.20e --M_2 %30.20e --orbital_period %30.20e --eccentricity %30.20e --metallicity %30.20e --random_seed %ld ",
            s1->mass,
            s2->mass,
            common->zams_period,
            common->zams_eccentricity,
            common->metallicity,
            new_random_seed
        );
    if(stardata->preferences->random_systems >= 2)
    {
        fprintf(RANDOM_SYSTEMS_SHOW_ARGS_STREAM,"--alpha_ce %30.20e --lambda_ce %30.20e --lambda_ionisation %30.20e --lambda_enthalpy %30.20e --comenv_merger_spin_method %d --comenv_ejection_spin_method %d --comenv_post_eccentricity %30.20e --nelemans_gamma %30.20e ",
               p->alpha_ce,
               p->lambda_ce,
               p->lambda_ionisation,
               p->lambda_enthalpy,
               p->comenv_merger_spin_method,
               p->comenv_ejection_spin_method,
               p->comenv_post_eccentricity,
               p->nelemans_gamma);

        fprintf(RANDOM_SYSTEMS_SHOW_ARGS_STREAM,"--pre_main_sequence %d --vrot1 %30.20e --vrot2 %30.20e --max_evolution_time %30.20e ",
               p->pre_main_sequence, 
               s1->vrot0,
               s2->vrot0,
               stardata->model.max_evolution_time);

        fprintf(RANDOM_SYSTEMS_SHOW_ARGS_STREAM,"--wind_angular_momentum_loss %d --wind_djorb_fac %30.20e --lw %30.20e --gravitational_radiation_model %d --magnetic_braking_factor %30.20e --tidal_strength_factor %30.20e --E2_prescription %d --RLOF_method %d --RLOF_interpolation_method %d --RLOF_mdot_factor %30.20e --rlperi %d --RLOF_angular_momentum_transfer_model %d --nonconservative_angmom_gamma %30.20e --merger_angular_momentum_factor %30.20e --hachisu_disk_wind %d --hachisu_qcrit %20.30e --WD_accretion_rate_novae_upper_limit_hydrogen_donor %30.20e --WD_accretion_rate_new_giant_envelope_lower_limit_hydrogen_donor %30.20e --WD_accretion_rate_novae_upper_limit_helium_donor %30.20e --WD_accretion_rate_new_giant_envelope_lower_limit_helium_donor %30.20e --WD_accretion_rate_novae_upper_limit_other_donor %30.20e --WD_accretion_rate_new_giant_envelope_lower_limit_other_donor %30.20e --comenv_prescription %d ",
                p->wind_angular_momentum_loss,
                p->wind_djorb_fac,
                p->lw,
                p->gravitational_radiation_model,
                p->magnetic_braking_factor,
                p->tidal_strength_factor,
                p->E2_prescription,
                p->RLOF_method,
                p->RLOF_interpolation_method,
                p->RLOF_mdot_factor,
                p->rlperi,
                p->RLOF_angular_momentum_transfer_model,
                p->nonconservative_angmom_gamma, 
                p->merger_angular_momentum_factor,
                p->hachisu_disk_wind,
                p->hachisu_qcrit,
                p->WD_accretion_rate_novae_upper_limit_hydrogen_donor,
                p->WD_accretion_rate_new_giant_envelope_lower_limit_hydrogen_donor,
                p->WD_accretion_rate_novae_upper_limit_helium_donor,
                p->WD_accretion_rate_new_giant_envelope_lower_limit_helium_donor,
                p->WD_accretion_rate_novae_upper_limit_other_donor,
                p->WD_accretion_rate_new_giant_envelope_lower_limit_other_donor,
                p->comenv_prescription);
    }
    fprintf(RANDOM_SYSTEMS_SHOW_ARGS_STREAM,"\n");
    fflush(NULL);
#endif
    
    /* restore random_seed after choosing the random parameters */
    common->random_buffer.set = FALSE;
    common->random_seed = new_random_seed;
    common->init_random_seed = new_random_seed;
}
#endif //RANDOM_SYSTEMS


void fool_cpu(void)
{
#if (defined __INTEL_COMPILER && FOOL_INTEL_COMPILER)
    /* Intel's compiler is crippled on AMD64, so un-cripple it */
    extern  int __intel_cpu_indicator;
    extern  void __intel_cpu_indicator_init(void)  ;
    
    //__intel_cpu_indicator_init(); // not required
    Dprint("INTEL says cpu = %d\n",__intel_cpu_indicator);

    /* fool intel's compiler into thinking we have sse(2) even
       thought we really do have sse(2)! */
    
    /*
     * 0x0001=1 MMX and CMOV not supported
     * 0x0010=16 MMX and CMOV supported
     * 0x0080=128 SSE supported by CPU and operating
     * 0x0200=512 SSE2 (family = * 0x0F)
     * 0x0400=1024 SSE2 (family != * 0x0F or not Intel)
     * 0x0800=2048 SSE3
     * 0x1000=4096 SSE3B (supplementary SSE3)
     */
    //__intel_cpu_indicator = 1024; 
    __intel_cpu_indicator = FAKE_INTEL_CPU_INDICATOR; 

#endif // FOOL_INTEL_COMPILER && __INTEL_COMPILER
}



