#include "binary_c_main.h"


/*
 * The binary_c stellar population nucleosynthesis framework.
 *
 * Contact: r.izzard@surrey.ac.uk or rob.izzard@gmail.com
 *
 * http://personal.ph.surrey.ac.uk/~ri0005/binary_c.html
 * https://gitlab.eps.surrey.ac.uk/ri0005/binary_c
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-announce
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-devel
 * https://twitter.com/binary_c_code
 * https://www.facebook.com/groups/149489915089142/
 *
 * Please see the files README, LICENCE and CHANGES,
 * and the doc/ directory for documentation.
 *
 *
 *
 * main function for the binary_c code
 *
 * The code is in the src/... directories which are
 * (roughly) thematically arranged.
 * 
 * For help and other documention, please see the doc/
 * directory.
 */

int main (int argc,
          char **  argv)
{
    /*
     * This is binary_c's entry point when called on the 
     * command line.  
     *
     * In main() we set up signals and floating-point
     * options, make a stardata, then call binary_c main
     * with the stardata.
     */
#if !defined SEGFAULTS
    char * altstack = setup_segfaults();
#endif
    
    Set_FPU_precision;
    //Set_FPU_policy;
    Set_stack_size;
    
    struct stardata_t * stardata Aligned = NULL;
    
    const int ret = binary_c_main(argc,argv,&stardata);
    
#if !defined SEGFAULTS
    Safe_free(altstack);
#endif
    Output_codestats;
    
    return ret;
}

static int binary_c_main(int argc,
                         char ** argv,
                         struct stardata_t ** RESTRICT s)
{
    ticks start_tick = getticks();
    struct stardata_t * stardata Aligned = *s = NULL;
    struct preferences_t * preferences = NULL;
    struct store_t * store = NULL;
    
    /*
     * binary_c_main is where you end up if you call binary_c
     * from the command line.
     * 
     * Most of what is in here is setup then a loop to run
     * binary stellar evolution, possibly in batchmode.
     *
     * First, do the main memory allocations and build the store
     */
    main_allocations(&stardata,
                     NULL,
                     &preferences,
                     &store);
    *s = stardata;
    
#ifdef USE_GSL
    /* set up GSL */
    setup_GSL_handlers(stardata);
#endif//USE_GSL
    
    /* welcome message */
    Dprint ("Welcome to the Binary Star Evolution program binary_c/nucsyn version %s; written by Robert Izzard %d-%d \n",
	    BINARY_C_VERSION,
	    BINARY_C_START_YEAR,
	    BINARY_C_END_YEAR);

//#define TEST_OPACITY

#ifndef TEST_OPACITY
    /* examine command line arguments */
    set_up_variables(argc,argv,NULL,stardata); 
#endif

#ifdef MAKE_BSE_TABLES
    make_BSE_tables(stardata);
#endif

    /* save argv and argc */
    stardata->common.argv = argv;
    stardata->common.argc = argc;


#ifdef TEST_OPACITY

    struct opacity_t op;
    
    if(! argc == 5)
    {
         printf("You must run binary_c with four arguments (X,Z,T,rho) to test the opacity table\n");
    }
    op.H = strtod(argv[1],NULL);
    op.Z = strtod(argv[2],NULL);
    op.temperature = strtod(argv[3],NULL);
    op.density = strtod(argv[4],NULL);

    double kap_pac = opacity_paczynski(stardata,&op,OPACITY_ALGORITHM_PACZYNSKI);
    double kap_fop = opacity_ferguson_opal(stardata,&op,OPACITY_ALGORITHM_FERGUSON_OPAL);

    printf("%g %g %g %g %g %g\n",
           op.H,
           op.Z,
           op.temperature,
           op.density,
           kap_pac,
           kap_fop);

    fflush(stdout);
    Exit_binary_c(0,"Exit after testing opacity table");

#endif

    
    /* 
     * set up repeat-run loop
     *
     * binary_c has a parameter preferences->repeat which you can
     * use to run the current system as many times as you like. 
     */
    int repeat;
    Boolean repeat_reset = (stardata->preferences->repeat>1);

    Dprint("Repeat loop: repeat_reset = %d stardata->preferences=%p\n",
	   repeat_reset,stardata->preferences);

    for(repeat=0;repeat<stardata->preferences->repeat;repeat++)
    {
        /* 
         * if we're repeating we need to reset and reload vars
         */
        if(repeat_reset && repeat)
        {
            set_default_preferences(preferences); 
            set_up_variables(argc,argv,NULL,stardata);
#ifdef LOG_REPEAT_NUMBER
            stardata->model.repeat_number = repeat;
#endif //LOG_REPEAT_NUMBER
            stardata->model.model_number = 0;
        }
        check_nans_are_signalled();

#ifdef DISC_TESTING
        tbdisc_testing(stardata);
#endif

#ifdef RANDOM_SYSTEMS
        /* perhaps randomize input parameters */
        if(stardata->preferences->random_systems)
        {
            set_random_system(stardata);
        }
#endif //RANDOM_SYSTEMS

#ifdef BATCHMODE
        Dprint("Entering batch processing (batchmode=%d)\n",
               stardata->preferences->batchmode);
#else
        Dprint("Calling evolve_system(stardata)\n");
#endif //BATCHMODE

#ifdef BATCHMODE
        if(Batchmode_is_on(stardata->preferences->batchmode))
        {
            /* 
             * Call the batchmode loop handler to evolve many stars
             */ 
            Call_batchmode_loop;
        }
        else
        {
#endif // BATCHMODE
            /* 
             * Call the evolve_system function to evolve the stars
             */
            evolve_system(stardata);
#ifdef BATCHMODE
        }
#endif // BATCHMODE
    }
     
    Dprint ("End of binary star evolution program.\n");

    /* show timers */
    Show_timers;

    /* free all allocated memory */
    free_memory(s,TRUE,TRUE,TRUE,TRUE);

    return BINARY_C_NORMAL_EXIT;
}

#ifdef CODESTATS
static void output_codestats(void)
{
    Boolean same = FALSE;
    unsigned int n,i;
    fflush(NULL);
    for(n=0;n<NUMBER_OF_CODESTATS;n++)
    {
        fprintf(stderr,
                "%25s : %ld -> %d\n",
                Codestat_string(n),
                codestats.counters[n],
                codestats.nentries[n]
            );
        for(i=0;i<codestats.nentries[n];i++)
        {
            if(same == FALSE)
            {
                printf("  %25s called %lu time%s in %s line(s) %d",
                       Codestat_string(n),
                       codestats.entries[n][i].count,
                       codestats.entries[n][i].count==1 ? "" : "s",
                       codestats.entries[n][i].file,
                       codestats.entries[n][i].line);
            }
            else
            {
                printf(",%d",
                       codestats.entries[n][i].line);
            }
            
            if(i+1<codestats.nentries[n] &&
               codestats.entries[n][i].count ==
               codestats.entries[n][i+1].count &&
               Strings_equal(codestats.entries[n][i].file,
                             codestats.entries[n][i+1].file))
            {
                /* same filenames */
                same = TRUE;
            }
            else
            {
                printf("\n");
                same = FALSE;
            }
            Safe_free(codestats.entries[n][i].file);
        }
        Safe_free(codestats.entries[n]);
    }
    fflush(NULL);
}
#endif//CODESTATS
