#include "../binary_c.h"

/*
 * binary_c output to standard (BSE-format) logfile 
 *
 * If this file is called with fp==NULL, reset the log_count
 */

#define LOGSTRINGLENGTH 2048
#define STELLAR_TYPE_STRING_LENGTH 5
#define NANCHECKL(A)                                                    \
    if(isnan(A)!=0)                                                     \
    {                                                                   \
        Backtrace;                                                      \
        Exit_binary_c(BINARY_C_EXIT_NAN,"NaN in logfile at %s\n",Stringof(A));   \
    }

#define LOG_APPEND(S)                                                   \
    {                                                                   \
        char tmpchar[LOGSTRINGLENGTH];                                  \
        snprintf(tmpchar,LOGSTRINGLENGTH,"%s %s",logstring,(S));        \
        strlcpy(logstring,tmpchar,LOGSTRINGLENGTH);                     \
    }

void output_to_logfile(FILE * const fp,
                       const double time,
                       const double m1,
                       const double m2,
                       const int stellar_type0,
                       const Boolean hybrid1,
                       const int stellar_type1,
                       const Boolean hybrid2,
                       const double sep,
                       const double ecc,
                       const double r1_rol, /* r1/roche_radius */
                       const double r2_rol, /* r2/roche_radius */
                       const char type[],
                       struct stardata_t * RESTRICT const stardata
    )
{
    Dprint("Output to logfile\n");
    if(fp==NULL) return; // do nothing if no log file is open
    
#ifdef FILE_LOG
    char logstring[LOGSTRINGLENGTH];
#ifdef CGI_EXTENSIONS
#define CGISTRINGLENGTH 1024
    char cgistring[CGISTRINGLENGTH];
    char cgistring_star[NUMBER_OF_STARS][CGISTRINGLENGTH];
#ifdef NUCSYN
    double *X;
#endif /* NUCSYN */
    unsigned int j;
#endif /* CGI_EXTENSIONS */

    // stellar type strings
    char c1[STELLAR_TYPE_STRING_LENGTH], c2[STELLAR_TYPE_STRING_LENGTH]; 
        
#ifdef NANCHECKS
    NANCHECKL(time);
    NANCHECKL(m1);
    NANCHECKL(m2);
    //NANCHECKL(sep);
    NANCHECKL(ecc);
#endif    
    snprintf(c1,STELLAR_TYPE_STRING_LENGTH," %2d%s",stellar_type0,hybrid1?"*":" ");
    snprintf(c2,STELLAR_TYPE_STRING_LENGTH," %2d%s",stellar_type1,hybrid2?"*":" ");
    
#if (DEBUG==1)
    char teststring[STRING_LENGTH];
    snprintf(teststring,STRING_LENGTH,
             "%11.4lf  %9.3lf  %9.3lf  %i  %i  %13.3lf %6.2lf  %.3lf %.3lf  %s\n",
             time,m1,m2,stellar_type0,stellar_type1,sep,ecc,Min(999.0,r1_rol),Min(999.0,r2_rol),type);
    Dprint("FP output to log file \"%s\"\n",teststring);
#endif /*DEBUG*/  

#ifdef DETAILED_COMPACT_LOG
    printf("DETLOG %s\n",type);
#endif

    if(fp==NULL) return;
#ifdef LOG_TO_STDOUT
    fp=stdout;
#endif // LOG_TO_STDOUT
#ifdef LOG_TO_STDERR
    fp=stderr;
#endif // LOG_TO_STDERR

    /* save previous logging type */
#define prevtype (stardata->common.file_log_prevtype)
#define prev_rlof_exit (stardata->common.file_log_prev_rlof_exit)
#define cached (stardata->common.file_log_cached)
#define n_rlof (stardata->common.file_log_n_rlof)
#define prevstring (stardata->common.file_log_prevstring)
    
    if(prevstring==NULL)
    {
        prevstring = Calloc(sizeof(char)*LOGSTRINGLENGTH,1);
    }

    Boolean skip = FALSE;

    if(strncmp(type,"INITIAL",7)==0)
    {
        prev_rlof_exit = 0;
        n_rlof = 0;
        skip = FALSE;
        cached = FALSE;
    }

    snprintf(logstring,
             LOGSTRINGLENGTH,
             "%11.4lf%9.3lf%9.3lf%3s%3s%13.5g%6.2lf%8.3lf%8.3lf  %s",
             time,
             m1,
             m2,
             c1,
             c2,
             (NEITHER_STAR_MASSLESS ? sep : -1.0),
             (NEITHER_STAR_MASSLESS ? ecc : -1.0),
             (NEITHER_STAR_MASSLESS ? Min(999.0,r1_rol) : 0.0),
             (NEITHER_STAR_MASSLESS ? Min(999.0,r2_rol) : 0.0),
             type);
    
#ifdef CIRCUMBINARY_DISK_DERMINE
    {
        char cb_log[LOGSTRINGLENGTH];
        snprintf(cb_log,
                 LOGSTRINGLENGTH,
                 "%3.3e ",
                 stardata->common.m_circumbinary_disk);
        LOG_APPEND(cb_log);
    }
#endif

#ifdef STELLAR_TYPE_LOG
    _printf("LOG %11.4lf%9.3lf%9.3lf%3i%3i%13.3lf%6.2lf%8.3lf%8.3lf  %s\n",
            time,m1,m2,s1,s2,sep,ecc,Min(999.0,r1_rol),Min(999.0,r2_rol),type);
#endif

  
            
#ifdef CGI_EXTENSIONS
    /* 
     * extra output for CGI script:
     */
        
    /* effective temperatures */
        

    /* chemistry for each star */
    /* X,Y,C,N,O,Fe,Ba as mass fractions for each star */
    Starloop(j)
    {
#ifdef NUCSYN
        X = nucsyn_observed_surface_abundances(&(stardata->star[j]));
        snprintf(cgistring_star[j],
                 CGISTRINGLENGTH,
                 "%3.3e %3.3e %3.3e %3.3e %3.3e %3.3e %3.3e",
                 X[XH1],X[XHe4],X[XC12],X[XN14],X[XO16],X[XFe56],
                 nucsyn_elemental_abundance("Ba",X,stardata,stardata->store));
#else
        cgistring_star[j][0] = '\0';
#endif //NUCSYN
    }

    snprintf(cgistring,
             CGISTRINGLENGTH,
             "%3.3e %3.3e %s %s",
             Teff(0),
             Teff(1),
             cgistring_star[0],
             cgistring_star[1]
        );

    LOG_APPEND(cgistring);

#endif /* CGI_EXTENSIONS */

    /* look for rapid END_RCHE,BEG_RCHE : pretend it didn't happen */
    if((strncmp(type,"BEG_RCHE",8)==0)&&
       (strncmp(prevtype,"END_RCHE",8)==0)&&
       (stardata->model.time - prev_rlof_exit < 0.01))
    {
        skip = TRUE;
        n_rlof++;
    }
    /* for END_RCHE, always skip (is saved in prevstring) */
    else if(strncmp(type,"END_RCHE",8)==0)
    {
        cached = TRUE;
        skip = TRUE;
    }
    /* anything but rapid BEG_RCHE, END_RCHE pair, or END_RCHE,
     * flush any cache */
    else if(cached==TRUE)
    {
        if((strncmp(prevtype,"BEG_RCHE",8)==0)
           && (n_rlof>0))
        {
            prevstring[72]='O';
            prevstring[73]='s';
            prevstring[74]='c';
            fprintf(fp,"%s %d\n",prevstring,n_rlof);
        }
        else
        {
            fprintf(fp,"%s\n",prevstring);
        }
        cached = FALSE;
        skip = FALSE;
        n_rlof = 0;
        prev_rlof_exit = 0.0;
    }

    /* save previous exit time for END_RCHE */
    if(strncmp(type,"END_RCHE",8)==0)
    {
        prev_rlof_exit = stardata->model.time;
    }

#ifndef CGI_EXTENSIONS
    strlcpy(prevtype,type,20);
    strlcpy(prevstring,logstring,LOGSTRINGLENGTH);
#endif

    /* output (perhaps!) */
    if((fp!=NULL)&&(skip==FALSE))
    {
        fprintf(fp,"%s\n",logstring);
    }
    
#ifdef FILE_LOG_ECHO
    /* echo to stdout */
#ifdef FILE_LOG_COLOURS
    printf("%s%s%s\n",YELLOW,logstring,COLOUR_RESET);
#else
    printf("%s%s%s\n",YELLOW,logstring,COLOUR_RESET);
#endif//FILE_LOG_COLOURS
#endif //FILE_LOG_ECHO

    if(fp != NULL && strncmp(type,"MAX_TIME",8)==0)
    {
        fprintf(fp,"Probability : %.10g\n",
                stardata->model.probability); 
    }

#ifdef FLUSH_LOG
    fflush(fp); // slow : but useful!
#ifdef FILE_LOG_ECHO
    fflush(stdout);
#endif//FILE_LOG_ECHO
#endif//FLUSH_LOG

#endif/*FILE_LOG*/
}
