#include "../binary_c.h"

#ifdef GAIAHRD


/*
 * Calculate the data for the Gaia-like HRD
 *
 * if do_log is TRUE, Teff, L, primary and is_single are set.
 * 
 * if do_log is FALSE, they are not set
 */

void gaia_log(struct stardata_t * const stardata,
              double * const Teff,
              double * const L,
              Star_number * primary,
              Boolean * const do_log,
              Boolean * const is_single)
{
#define Sum(X) (M2_is_star ? (stardata->star[0].X + stardata->star[1].X) : stardata->star[0].X)
    /*
     * M2min = 0.01 Msun
     *
     * Below this, only the primary is used
     */
    const double min_M2 = 0.01;  // Msun
    const Boolean M2_is_star = stardata->star[1].mass > min_M2;
        
    /*
     * Total luminosity and minimum observable limit
     */
    const double Lgaia = Sum(luminosity);
    const double min_L = 0.5e-10; // Lsun
    const double Lbin = stardata->preferences->gaia_L_binwidth; // bin width of log10(luminosity)

    /*
     * Only log if L > min_L : otherwise nothing is seen
     */
    if(Lgaia > min_L)
    {
        /*
         * Mean effective temperature
         */
        const double weighted_Teff =
            M2_is_star == TRUE
            ?
            ((stardata->star[0].luminosity * Teff(0) +
              stardata->star[1].luminosity * Teff(1))
             / Sum(luminosity))
            :
            Teff(0);

        const double min_Teff = 1500; // K
            
        if(weighted_Teff > min_Teff)
        {
            const double Teffbin = stardata->preferences->gaia_Teff_binwidth; // bin width of log10(Teff)
            
            /*
             * Identify the primary star as the brightest
             */
            const Star_number k =
                M2_is_star == TRUE
                ?
                (stardata->star[0].luminosity > stardata->star[1].luminosity ? 0 : 1)
                :
                0;

            /*
             * Skip black holes, neutron stars, massless remnants
             */
            if(stardata->star[k].stellar_type < NEUTRON_STAR)
            {
                /*
                 * Radial velocity K1 of the primary, assuming sideways on
                 */
                const double K1 = 1e-5 * radial_velocity_K(stardata,PI/2.0,k+1); // km/s
                const double min_K1 = 5.0; // km/s
            
                /*
                 * Hence is this a detected single or binary?
                 */
                * is_single =
                    Boolean_(
                        M2_is_star == FALSE ||
                        System_is_single ||
                        K1 < min_K1
                        );
                * Teff = bin_data(log10(weighted_Teff),Teffbin);
                * L = bin_data(log10(Lgaia),Lbin);
                * primary = k;
                * do_log = TRUE;
                /*
                if(0)printf("GAIA Teff = (%g * %g + %g * %g) / %g = %g, L = %g, primary = %d, do_log = %s\n",
                       stardata->star[0].luminosity,
                       Teff(0),
                       stardata->star[1].luminosity,
                       Teff(1),
                       Sum(luminosity),
                       log10(weighted_Teff),
                       log10(Lgaia),
                       *primary,
                       Yesno(*do_log));
                */
            }
            else
            {
                * do_log = FALSE;
            }
        }
        else
        {
            * do_log = FALSE;
        }
    }
    else
    {
        * do_log = FALSE;
    }
}

#endif // GAIAHRD
