
#include "../binary_c.h"

/*
 * Do stellar evolution to acquire stellar parameters
 * (M, R, L, Mc, etc.) at time model->time
 *
 * "codetype" is:
 *
 * DETACHED_SYSTEM_CALL : this is during detached evolution, not RLOF
 * RLOF_SYSTEM_CALL  : called later in the main evolution loop, this is during RLOF
 *
 * Return value:
 * SYSTEM_IS_BROKEN_APART
 * STOP_LOOPING_RESULT
 * KEEP_LOOPING_RESULT
 *
 */
static void set_derivatives(struct star_t * const old,
                            struct star_t * const new,
                            const double dt);

int stellar_evolution(struct stardata_t * RESTRICT const stardata,
                      const int codetype)
{
    struct common_t *common MAYBE_UNUSED = &stardata->common;
    struct model_t *model = &stardata->model;
    double age,m0,mc_1tp,mt,mc,lum,rm,rc,mcCO,mcGB,mcmaxMS;
    age=m0=mc_1tp=mt=mc=lum=rm=rc=mcCO=mcGB=mcmaxMS=0.0;
    Star_number k;
    Stellar_type stellar_type;  
    int retval=KEEP_LOOPING_RESULT;


    Dprint("Stellar evolution evolving starloop");
    Evolving_STARLOOP(k)
    { 
        SETstar(k);

        Dprint("acquire stellar parameters for star %d\n",k); 
	    
        age = model->time - star->epoch;

        Dprint("stellarev top star %d: Set age = time=%g - epoch=%g = %g\n",
               star->starnum,
               model->time,
               star->epoch,
               age);

#ifdef ACQUIRE_TEST_FOR_NEGATIVE_AGES
        /* allow small numerical problems? */
        if(age>-1e-3 && age<0.0)
        {
            age=0.0;
        }
        else if(age < -1e-2)
        {
            Exit_binary_c(stardata,
                          BINARY_C_ACQUIRE_NEGATIVE_AGE,
                          "(acquire_stellar_parameters) Age is negative oops : time=%g epoch=%g -> age=%g dtm=%g\n",
                          model->time,
                          star->epoch,age,
                          model->dtm);
        }
#endif // ACQUIRE_TEST_FOR_NEGATIVE_AGES

#ifdef ACQUIRE_FIX_NEGATIVE_AGES
        /* fudge to force age to be position */
        age=Max(0.0,age);
#endif//ACQUIRE_FIX_NEGATIVE_AGES
        
#ifdef WD_KICKS___DEPRECATED
        /* Kick WD at the end of the AGB */
        Dprint("WD kick check %d %d\n",star->kick_WD,star->SN_type);
        if(star->kick_WD==TRUE && model->sgl==FALSE)
        { 
            star->kick_WD=FALSE;
            star->already_kicked_WD=TRUE;
            int it,nkicks;
#ifdef NUCSYN
            if(stardata->preferences->wd_kick_when==WD_KICK_AT_EVERY_PULSE)
            {
                nkicks = ((int)(star->num_thermal_pulses+0.01))-star->prev_kick_pulse_number;
            }                   
            else
#endif //NUCSYN
            {
                nkicks = 1;
            }
            star->SN_type = SN_WDKICK;
            for(it=0;it<nkicks;it++)
            {
                stardata->preferences->sn_kick_distribution[SN_WDKICK] = 
                    KICK_VELOCITY_FIXED;
                monte_carlo_kick(star,
                                 star,
                                 star->mass,
                                 star->mass,
                                 stardata->star[Other_star(k)].mass,
                                 &common->orbit.eccentricity,
                                 &common->orbit.separation,
                                 &common->orbit.angular_momentum,
                                 stardata,
                                 stardata);
            }
            star->SN_type = SN_NONE;

#ifdef NUCSYN
            /* save pulse number of previous kick */
            star->prev_kick_pulse_number=((int)(star->num_thermal_pulses+0.01));
#endif//NUCSYN

            if(Is_not_zero(stardata->preferences->sn_kick_dispersion[SN_WDKICK]))
            {
                Append_logstring(LOG_SN_AND_KICK,"WDKICK %3.1f km/s",stardata->preferences->sn_kick_dispersion[SN_WDKICK]);
            }

            /* if broken binary, return */
            if(common->orbit.eccentricity > 1.0)
            {
                retval = SYSTEM_IS_BROKEN_APART;
                continue;
            }
        }
#endif//WD_KICKS

        Dprint("age=%12.12e from time=%12.12e epoch=%12.12e phase_start_mass=%g mass(t)=%g\n",age,model->time,star->epoch,star->phase_start_mass,star->mass);

        stellar_type = star->stellar_type;   
        m0 = star->phase_start_mass;
        mt = star->mass;
        mc = star->core_mass;
        mcCO = star->CO_core_mass;
        mcGB = star->GB_core_mass;
        mcmaxMS = star->max_MS_core_mass;

        if(codetype==DETACHED_SYSTEM_CALL)
        {
            /* non-RLOF */
            Dprint("Early call intpol=%d\n",model->intpol);
            star->aj0 = age;

            if(model->intpol==0)
            {
                star->mcxx = mc;
                Dprint("Early call: set mass_at_RLOF_start=m0=%g intpol=%d dtm=%g\n",m0,model->intpol,model->dtm);
            } 
            else if(model->intpol>0)
            {
                mc = star->mcxx;
            }
            star->mass_at_RLOF_start = m0;
        }

        if(mt>MAXIMUM_STELLAR_MASS)
        {
            /*
             * Masses over 100Msun should probably not be trusted in the
             * evolution formulae. MAXIMUM_STELLAR_MASS is usually
             * set to 100.
             */
            strip_supermassive_star(stardata,star,mt);
            star->mass = mt = MAXIMUM_STELLAR_MASS;
            retval = STOP_LOOPING_RESULT;
            continue;
        }

        if(stardata->tmpstore->stellar_evolution_newstar == NULL)
        {
            stardata->tmpstore->stellar_evolution_newstar = New_star_from(star);
        }
        else
        {
            Copy_star(star,
                       stardata->tmpstore->stellar_evolution_newstar);
        }
        
        struct star_t * newstar = stardata->tmpstore->stellar_evolution_newstar;
        
#ifdef BSE
        Dprint("Call stellar_timescales m0=%12.12e mt=%12.12e\n",m0,mt);

        double GB[GB_ARRAY_SIZE];
        double lums[LUMS_ARRAY_SIZE];
        double tscls[TSCLS_ARRAY_SIZE];
        double tm,tn;
        
        stellar_timescales(stellar_type,
                           m0,
                           mt,
                           &tm,
                           &tn,
                           tscls,
                           lums,
                           GB,
                           stardata,
                           star);
            
        Dprint("call stellar_structure age=%g m0=%g m00=%g mt=%g tm=%g tn=%g st=%d mc=%g\n"
               ,age,m0,star->mass_at_RLOF_start,
               mt,tm,tn,stellar_type,mc);


      
        stellar_structure_longcall(&m0,
                                   &mc_1tp,
                                   &age,
                                   &mt,
                                   &tm,
                                   &tn,
                                   tscls,
                                   lums,
                                   GB,
                                   &rm,
                                   &lum,
                                   &stellar_type,
                                   &mc,
                                   &mcCO,
                                   &mcGB,
                                   &mcmaxMS,
                                   &rc,
                                   &star->menv,
                                   &star->renv,
                                   &star->k2,
                                   &star->time_first_pulse,
                                   &star->num_thermal_pulses,
                                   &star->time_prev_pulse,
                                   &star->prev_tagb,
                                   &star->menv_1tp,
                                   &star->mc_1tp,
                                   &star->core_mass_no_3dup,
                                   &star->interpulse_period,
                                   &star->time_next_pulse,
                                   &star->lambda_3dup,
                                   &star->num_thermal_pulses_since_mcmin,
                                   &star->spiky_luminosity,
                                   &star->SN_type,
                                   TRUE,
                                   newstar,
                                   stardata,
                                   STELLAR_STRUCTURE_CALLER_stellar_evolution);

#else
        stellar_structure(newstar,
                          stardata,
                          TRUE,
                          STELLAR_STRUCTURE_CALLER_stellar_evolution);
#endif //BSE

        Dprint("post call\n");

        /*
         * Save derivatives
         */
        if(model->intermediate_step == FALSE)
        {
            set_derivatives(star,newstar,model->dtm * 1e6);
        }

        /*
         * Todo : replace the use of rdot with derivative[]
         */
        if(codetype==DETACHED_SYSTEM_CALL)
        {
            /*
             * If detached, set radius derivative for
             * Roche-lobe overflow calculations
             */
            if(Is_not_zero(model->dtm))
            {
                double dr = newstar->radius - star->radius;
                newstar->rdot = fabs(dr)/model->dtm;
                newstar->drdt = dr;
            }
            else
            {
                newstar->rdot = 0.0;
                newstar->drdt = 0.0;
            }
        }
      
        Dprint("ret  stellar_structure m0=%12.12g age=%12.12g mt=%12.12g tm=%30.22g tn=%30.22g rm=%12.12g lum=%12.12g kw=%d mc=%12.12g rc=%12.12g mcGB=%g mcCO=%g SN=%d\n",
               m0,
               age,
               mt,
#ifdef BSE
               tm,
               tn,
#else
               0.0, // main sequence lifetime
               0.0, // nuclear burning lifetime
#endif
               rm,
               lum,
               stellar_type,
               mc,
               rc,
               mcGB,
               mcCO,
               star->SN_type);

        Dprint("Check for stellar type change k=%d, kw=%d, kstar=%d, sgl=%d\n",k,stellar_type,star->stellar_type,model->sgl);

        if(stellar_type != star->stellar_type)
        {
            newstar->mass = mt;
            newstar->phase_start_core_mass = newstar->core_mass;

            if(codetype==DETACHED_SYSTEM_CALL)
                newstar->stellar_timestep = 0.01;

            if(stellar_type==MASSLESS_REMNANT)
            {
                /* Accretion induced collapse can get you here */
                Dprint("SN with no remnant (codetype=%d)\n",codetype);
                newstar->stellar_type = stellar_type;
                retval = SYSTEM_IS_BROKEN_APART;
                continue;
            }
            newstar->phase_start_mass = fabs(m0);
            newstar->mass_at_RLOF_start = newstar->phase_start_mass; // required for helium nova prevention
            newstar->epoch = model->time - age;
            
            Dprint("Set epoch (2) = %g from %g - %g (newstar->age = %g)\n",
                   newstar->epoch,
                   model->time,
                   age,
                   newstar->age);
        }

        Dprint("Det. rdot = %g (newrad=%g oldrad=%g dtm=%g)\n",
               newstar->rdot,
               newstar->radius,
               star->radius,
               model->dtm);

        /*
         * Determine stellar evolution timescales
         * and the timestep required
         */
        double dt;
        if(stardata->model.intermediate_step == FALSE)
        {
            stellar_timestep(stellar_type,
#ifdef BSE
                             newstar->age,
                             newstar->tm,
                             newstar->tn,
                             tscls,
#endif
                             &dt,
                             &model->time_remaining,
                             newstar,
                             stardata);
            Dprint("post-timestep (gave dt = %g) tremain %g\n",dt,model->time_remaining);
        }
        
        
        Dprint("set stellar timestep=%g from stellar_type=%d dt=%g time_remaining=%g MINIMUM_STELLAR_TIMESTEP %g\n",
               newstar->stellar_timestep,
               newstar->stellar_type,
               dt,
               model->time_remaining,
               MINIMUM_STELLAR_TIMESTEP);
        
        /*
         * Update stellar structure:
         * derived variables 
         */
        newstar->effective_radius =
            (newstar->roche_radius>TINY && newstar->radius>newstar->roche_radius) ? 
            Max(newstar->roche_radius,newstar->core_radius) : newstar->radius;
#ifdef XRAY_LUMINOSITY
        newstar->Xray_luminosity = Xray_luminosity(star);
#endif
#ifdef BSE
        newstar->tbgb = tscls[T_BGB];
        newstar->tms = tm;
#endif //BSE
        
#ifdef GIANT_CRICTICAL_Q_CHEN_HAN
        // save base of the giant branch luminosity and radius
        // as well as the A0 parameter
        if(Is_zero(star->A0) && (newstar->stellar_type>HERTZSPRUNG_GAP))
        {
            newstar->A0 = log10(newstar->radius);
        }
#endif// GIANT_CRICTICAL_Q_CHEN_HAN
 

        Copy_star(newstar,star);
        Dprint("SET AGE %g\n",star->age);
        Dprint("TMS set star %d = %30.22e\n",star->starnum,star->tms);
        
    } // loop over evolving stars

    /*
     * Timestep should be set for massless remnants
     * to allow logging.
     */
    Starloop(k)
    {
        SETstar(k);
        if(star->stellar_type==MASSLESS_REMNANT)
        {
            star->hybrid_HeCOWD = FALSE;
            double dt;
            stellar_timestep(star->stellar_type,
#ifdef BSE
                             star->age,
                             star->tm,
                             star->tn,
                             NULL,
#endif//BSE
                             &dt,
                             &model->time_remaining,
                             star,
                             stardata);
        }
    }
    
    if(retval == KEEP_LOOPING_RESULT)
    {
        handle_massless_remnants(stardata);
        set_kelvin_helmholtz_times(stardata);
    }
    Dprint("R=%g %g\n",stardata->star[0].radius,stardata->star[1].radius);
    Dprint("SN %d %d\n",stardata->star[0].SN_type,stardata->star[1].SN_type);

    return retval;
}


static void set_derivatives(struct star_t * const old,
                            struct star_t * const new,
                            const double dt)
{
    if(Is_not_zero(dt))
    {
        const double idt = 1.0/dt;
        new->derivative[DERIVATIVE_STELLAR_RADIUS] = 
            (new->radius - old->radius)*idt;
        new->derivative[DERIVATIVE_STELLAR_ROCHE_RADIUS] =
            (new->roche_radius - old->roche_radius)*idt;
        new->derivative[DERIVATIVE_STELLAR_CORE_MASS] = 
            (new->core_mass-old->core_mass)*idt;
        new->derivative[DERIVATIVE_STELLAR_LUMINOSITY] = 
            (new->luminosity-old->luminosity)*idt;
        new->derivative[DERIVATIVE_STELLAR_TEMPERATURE] = 
            (Teff_from_star_struct(new) - Teff_from_star_struct(old))*idt;
    }
}
