/*
 * This function initializes the binary star parameters on the first timestep
 */
#include "../binary_c.h"

void initialize_parameters(struct stardata_t * RESTRICT const stardata)
{
    struct model_t * const model = &stardata->model;
    struct common_t * const common = &stardata->common;
    struct star_t * const star1 = &stardata->star[0];
    struct star_t * const star2 = &stardata->star[1];
#ifdef PRE_MAIN_SEQUENCE
    double aperi;
#endif
    double rc = 0.0,lum = 0.0,rm = 0.0,mc = 0.0,age = 0.0;
    Star_number k = 0;
    /************************************************************/

#ifdef BINT
    BINT_init(stardata);
#endif//BINT
    
    check_nans_are_signalled();
    model->time = stardata->preferences->start_time;    
    model->dtm = model->dt = 0.0;
        
    Dprint("Initializing the parameters\n");


    Dprint("random_seed\n");

    /*
     * The random number seed can be specified as a preference, 
     * or - if the seed is not yet set - it is set by the random_seed()
     * function
     */
    if(stardata->preferences->cmd_line_random_seed!=0)
    {
        common->random_seed = stardata->preferences->cmd_line_random_seed;
        Dprint("random_seed from cmd line %ld\n",common->random_seed);
    }
    else if(common->random_seed!=0)
    {
        common->random_seed = random_seed();
        Dprint("random_seed from random_seed() %ld\n",common->random_seed);
    }

#ifdef RANDOM_SEED_LOG
    printf("random_seed=%ld\n",common->random_seed);
#ifdef FILE_LOG
    if(model->log_fp!=NULL)
    {
        fprintf(model->log_fp,"random_seed=%ld\n",common->random_seed);
    }
#endif //FILE_LOG

#ifdef RANDOM_SEED_LOG_STREAM
    if(RANDOM_SEED_LOG_STREAM!=NULL) fprintf(RANDOM_SEED_LOG_STREAM,"random_seed=%ld\n",common->random_seed);
#endif//RANDOM_SEED_LOG_STREAM

#endif//RANDOM_SEED_LOG


    
    /*
     * If the time really is zero, we require some further setup.
     */
    if(Is_zero(model->time))
    {
        evolution_time_zero_setup(stardata);
    }
    Dprint("Memcopied XZAMS ok\n");

#ifdef PRE_MAIN_SEQUENCE
    aperi = stardata->common.orbit.separation*(1.0-stardata->common.orbit.eccentricity);
#endif 


    model->sgl = FALSE; /* true for single star evolution ? */

#ifdef FORCE_SINGLE_STAR_EVOLUTION_WHEN_SECONDARY_IS_A_PLANET
    if(Less_or_equal(star2->mass,0.09))
    {
        /* Single star evoltion */
        model->sgl=TRUE;
        common->zams_eccentricity = common->eccentricity=-1.0;
        common->zams_period = common->orbit.period=0.0;
        common->separation=1E7;
        common->orbit.angular_momentum=0.0;
        common->orbit.angular_frequency=0.0;
        star2->mass=0.01;
        if(star1->omega<0.0)star1->omega=0.0;
        star2->omega=0.0;
        star2->angular_momentum=0.0;
        star2->pms_mass=0.01;
        star2->mass=0.01;
        Dprint("Star 2 is a planet: configure for single-star evolution\n");
    }
    else
#endif
    {
        common->orbit.angular_frequency = TWOPI/common->orbit.period;
        Dprint("calc Jorb from a = %g, omega = %g\n",
               common->orbit.separation,
               common->orbit.angular_frequency);
        common->orbit.angular_momentum = MASS_PRODUCT/Total_mass*sqrt(1.0-ECC_SQUARED)*Separation_squared*common->orbit.angular_frequency;
        
        if(star1->omega<0.0) star1->omega = common->orbit.angular_frequency;
        if(star2->omega<0.0) star2->omega = common->orbit.angular_frequency;
    }

    Dprint("hence Jorb = %g\n",common->orbit.angular_momentum);

    const Boolean init_sgl = model->sgl;

    /**** per star setup ****/
    Starloop(k)
    {
        SETstar(k);
        Dprint("In first k loop, k=%d (m=%g m0=%g other star m=%g m0=%g)\n",
               k,
               star->mass,
               star->phase_start_mass,
               stardata->star[Other_star(k)].mass,
               stardata->star[Other_star(k)].phase_start_mass);

        /*
         * If mass == 0 skip most of the per-star setup.
         * 
         * This happens, e.g., when inital abundances
         * are required but nothing else.
         */
        if(Is_not_zero(star->mass))
        {
            age = model->time - star->epoch;
            Dprint("Set initial age = %g from %g - epoch = %g\n",
                   age,model->time,star->epoch);
        
            Dprint("Calling stellar_timescales...");
            if(init_sgl && k==1)
            {
                star->phase_start_mass=0.0;
                star->effective_zams_mass=0.0;
                star->pms_mass=0.0;
            }
            else
            {
                star->phase_start_mass = star->mass;
                star->effective_zams_mass    = star->mass;
                star->pms_mass     = star->mass;
            }
            star->max_MS_core_mass=0.0;
            star->stellar_type_tstart=0.0;

#ifdef BSE
            double GB[GB_ARRAY_SIZE];
            double lums[LUMS_ARRAY_SIZE];
            double tscls[TSCLS_ARRAY_SIZE];
            double tm,tn;
        
            stellar_timescales(star->stellar_type,
                               star->phase_start_mass,
                               star->mass,
                               &tm,&tn,
                               tscls,
                               lums,
                               GB,
                               stardata,star);

            Dprint("Call stellar_structure : m0=%12.12e, age=%12.12e, m=%12.12e\ntm==%12.12e, tn=%12.12e\n",
                   star->phase_start_mass,
                   age,
                   star->mass,
                   tm,tn);

            stellar_structure_longcall(&star->phase_start_mass,
                                       &star->phase_start_core_mass,
                                       &age,
                                       &star->mass,
                                       &tm,
                                       &tn,
                                       tscls,
                                       lums,
                                       GB,
                                       &rm,
                                       &lum,
                                       &star->stellar_type,
                                       &mc,
                                       &star->CO_core_mass,
                                       &star->GB_core_mass,
                                       &star->max_MS_core_mass,
                                       &rc,
                                       &(star->menv),
                                       &(star->renv),
                                       &(star->k2),
                          
                                       &star->time_first_pulse,
                                       &star->num_thermal_pulses,
                                       &star->time_prev_pulse,
                                       &star->prev_tagb,
                                       &star->menv_1tp,
                                       &star->mc_1tp,
                                       &star->core_mass_no_3dup,
                                       &star->interpulse_period,
                                       &star->time_next_pulse,
                                       &star->lambda_3dup,
                                       &star->num_thermal_pulses_since_mcmin,
                                       &star->spiky_luminosity,

                                       &(star->SN_type),
                                       FALSE,
                                       star,
                                       stardata,
                                       STELLAR_STRUCTURE_CALLER_initialize_parameters
                );

            Dprint("OUT  stellar_structure : m0=%12.12e, age=%12.12e, m=%12.12e\ntm==%12.12e, tn=%12.12e\n",
                   star->phase_start_mass,
                   age,
                   star->mass,
                   tm,tn);
      
            Dprint("Setting stellar variables\n");

            star->age = age;
            Dprint("SET AGE %g\n",star->age);
            star->epoch = model->time-age;

            Dprint("Set initial epoch = %g - %g = %g\n",model->time,age,star->epoch);

            star->radius = rm;
            star->core_mass = mc;
            star->core_radius = rc;
            star->tms = tm;
            star->tbgb = tscls[T_BGB];
            star->luminosity = lum;
#else
            stellar_structure(star,
                              stardata,
                              FALSE,
                              STELLAR_STRUCTURE_CALLER_initialize_parameters);
#endif//BSE
            star->q = Q(k);
            determine_roche_lobe_radius(stardata,&stardata->common.orbit,star);
            Dprint("post-roche\n");
            star->rol0 = star->roche_radius;
        
            Dprint("setting lum to %12.12e\n",lum);

#ifdef PRE_MAIN_SEQUENCE
            /*
             * If the binary orbit is so close that the star is outside
             * its Roche lobe on the pre-main sequence, try to age the
             * star appropriately such that it fits inside its Roche lobe.
             *
             * Of course this is just a fudge...
             */

#define PMSprint if(1)printf

            Boolean pms_fit_failed = FALSE; 
	
            if(ON_MAIN_SEQUENCE(star->stellar_type) &&
               stardata->preferences->pre_main_sequence==TRUE &&
               stardata->preferences->pre_main_sequence_fit_lobes==TRUE &&
               star->radius > star->roche_radius_at_periastron)
            {
                /* age the star until it fits just inside the main sequence */


                /* find the R/RZAMS factor now */ 
                const double f = preMS_radius_factor(star->mass,0.0);

                /* hence RZAMS */
                const double RZAMS = star->radius / f;

                /*
                 * hence target factor = R/RZAMS = ROLperi/RZAMS 
                 * but this must be >1
                 */
                const double targetf = Max(1.0, 0.999 * star->roche_radius_at_periastron / RZAMS);

            
                /* 
                 * hence get the corresponding age 
                 * NB on failure will put the star on the ZAMS
                 */
                age = 1e-6 * time_from_preMS_radius_factor(stardata,
                                                           star->mass,
                                                           targetf);
                
                PMSprint("star %d : R_PMS=%g > RLperi=%g : R_ZAMS = %g : ",
                         star->starnum,star->radius,star->roche_radius_at_periastron,RZAMS);
                PMSprint("target radius factor f = %g : ",targetf);
                PMSprint("new age %g\n",age);
 
                /* if we can, calculate new radius */
                if(!Fequal(f,1.0))
                {
#ifdef BSE
                    stellar_structure_longcall(&star->phase_start_mass,
                                               &star->phase_start_core_mass,
                                               &age,
                                               &star->mass,
                                               &tm,
                                               &tn,
                                               tscls,
                                               lums,
                                               GB,
                                               &rm,
                                               &lum,
                                               &star->stellar_type,
                                               &mc,
                                               &star->CO_core_mass,
                                               &star->GB_core_mass,
                                               &star->max_MS_core_mass,
                                               &rc,
                                               &(star->menv),
                                               &(star->renv),
                                               &(star->k2),

                                               &star->time_first_pulse,
                                               &star->num_thermal_pulses,
                                               &star->time_prev_pulse,
                                               &star->prev_tagb,
                                               &star->menv_1tp,
                                               &star->mc_1tp,
                                               &star->core_mass_no_3dup,
                                               &star->interpulse_period,
                                               &star->time_next_pulse,
                                               &star->lambda_3dup,
                                               &star->num_thermal_pulses_since_mcmin,
                                               &star->spiky_luminosity,

                                               &(star->SN_type),
                                               FALSE,
                                               star,
                                               stardata,STELLAR_STRUCTURE_CALLER_initialize_parameters
                        );
            
                    star->age = age;
                    star->epoch = -age;
                    star->radius = rm;
#endif//BSE
#ifdef BINT
                    stellar_structure(star,
                                      stardata,
                                      FALSE,
                                      STELLAR_STRUCTURE_CALLER_initialize_parameters);
#endif//BINT
                    Dprint("Set epoch = - age = %g, age = %g\n",star->epoch, star->age);
                }

                PMSprint("R=%g : RLperi=%g ",
                         star->radius,
                         star->roche_radius_at_periastron);

                if(star->radius > star->roche_radius_at_periastron)
                {
                    PMSprint("FAILED");
                    pms_fit_failed = TRUE;
                }
                else
                {
                    PMSprint("SUCCESS");
                }

                PMSprint(" (sep=%g perisep=%g) age=%g age/pms_lifetime=%g (pms lifetime %g)\n",
                         stardata->common.orbit.separation,
                         stardata->common.orbit.separation*(1.0-stardata->common.orbit.eccentricity),
                         star->age,
                         star->age/(1e-6*preMS_lifetime(star->mass)),
                         1e-6*preMS_lifetime(star->mass)
                    );
            }

            if(pms_fit_failed==TRUE)
            {
                printf(">>> WARNING <<< : Pre-MS Roche lobe fitting failed, expect immediate RLOF from star %d\n",k);
            }
#endif // PRE_MAIN_SEQUENEC
           
            /*
             * If t=0 (first timestep) then set the spin,
             * otherwise we assume whatever is passed in.
             */
            if(Is_zero(model->time) && star->omega<1e-3)
            {
                calculate_rotation_variables(star,star->radius);
                if(Fequal(star->vrot0,-2.0))
                {
                    /*
                     * vrot0 = -2 : 
                     * form at the orbital velocity
                     */
                    star->omega = stardata->common.orbit.angular_frequency;
                    Dprint("from omega orb");
                }
                else
                {
                    if(Is_zero(star->vrot0))
                    {
                        /*
                         * Given zero on the command line, or nothing,
                         * use the Hurley et al 2000/2002 formula
                         * Hurley, Phd
                         * Thesis (2000), Cambridge University,  p. 63
                         */
                        if(Is_not_zero(rm))
                        {
                            star->omega = OMEGA_FROM_VKM * vrotf(star->mass) / rm;
                        }
                        else
                        {
                            star->omega = 0.0;
                        }
                        Dprint("from Hurley");
                    }
                    else if(Fequal(star->vrot0,-1.0))
                    {
                        /*
                         * vrot0 = -1 : 
                         * form at breakup velocity.
                         * NB This assumes the radius is deformed 
                         * at the equator
                         * (see calculate_rotation_variables).
                         */
                        star->omega = star->omega_crit;
                        Dprint("from omega crit");
                    } 
                    else
                    {
                        /*
                         * Use cmd-line specified vrot (km/s)
                         * and assume the star is not deformed.
                         */
                        if(Is_not_zero(rm))
                        {
                            star->omega = OMEGA_FROM_VKM * star->vrot0 / rm;
                        }
                        else
                        {
                            star->omega = 0.0;
                        }
                        Dprint("from cmd");
                    }

                    /*
                     * hence set the equatorial angular velocity
                     * NB (YEAR_LENGTH_IN_SECONDS*1e5/R_SUN ~ 45.35
                     */
                }
                calculate_rotation_variables(star,star->radius);
            }

            /* and hence set the stellar angular momentum */
            Dprint("J from omega=%g k2=%g rm=%g m=%g mc=%g rc=%g\n",
                   star->omega,
                   star->k2,
                   star->radius,
                   star->mass,
                   star->core_mass,
                   star->core_radius);

            star->angular_momentum = star->omega *
                moment_of_inertia(star,star->radius);

            /* zero derivatives */
            star->derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS]=0.0;
            star->derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN]=0.0;
            star->derivative[DERIVATIVE_STELLAR_ANGMOM]=0.0;
            star->stellar_timestep = 1e6; /* large initial timestep */
        }
    } /** end loop over stars **/
    
#if defined PRE_MAIN_SEQUENCE && defined FILE_LOG
    /*
     * Logging for the Pre-MS
     */
    if(Is_zero(model->dt) &&
       (ON_MAIN_SEQUENCE(stardata->star[0].stellar_type) ||
        ON_MAIN_SEQUENCE(stardata->star[1].stellar_type)) &&
       stardata->preferences->pre_main_sequence==TRUE)
    {
        Set_logstring(LOG_PREMAINSEQUENCE,
                      "PreMS aperi=%g t1=%g f1=%g R1=%g R1/RLperi1=%g R1/a(1-e)=%g t2=%g f2=%g R2=%g R2/RLperi2=%g R2/a(1-e)=%g R1+2/aperi=%g",
                      aperi,
                      stardata->star[0].age,
                      1e6*stardata->star[0].age/preMS_lifetime(stardata->star[0].mass),
                      stardata->star[0].radius,
                      stardata->star[0].radius/stardata->star[0].roche_radius_at_periastron,
                      stardata->star[0].radius/aperi,
                      stardata->star[1].age,
                      1e6*stardata->star[1].age/preMS_lifetime(stardata->star[1].mass),
                      stardata->star[1].radius,
                      stardata->star[1].radius/stardata->star[1].roche_radius_at_periastron,
                      stardata->star[1].radius/aperi,
                      (stardata->star[0].radius + stardata->star[1].radius)/aperi
            );
    }
#endif // PRE_MAIN_SEQUENCE && FILE_LOG


#ifdef NUCSYN
    if(stardata->preferences->initial_abunds_only==TRUE)
    {
        /* output initial abundances then exit */
        Printf("Initial abundances from mixture %u metallicity %g\n",
               stardata->preferences->initial_abundance_mix,
               stardata->common.metallicity
            );
        const static char element_strings[][4] = NUCSYN_SHORT_ELEMENT_STRINGS; 
        Isotope i;
        Ordered_isotope_loop(i)
        {
            Printf("%s%d=>%12.12e, # %u\n",
                   element_strings[stardata->store->atomic_number[i]],
                   stardata->store->nucleon_number[i],
                   star1->Xenv[i],
                   i);
        }
        Exit_or_return_void(BINARY_C_SPECIAL_EXIT,"exit init parameters: init abunds only");
    }
#endif //NUCSYN

    Dprint("Entering first timestep\n");
    
}

