#include "../binary_c.h"

/*
 * Binary_c's main loop to evolve a binary-star system in time.
 * 
 * Returns non-zero on error, otherwise 0.
 */


/* eprint is the local debugging statement */ 
#define eprint(...)                                                     \
    fprintf(stdout,"ZOOM %s dt_zoomfac %g : ",               \
            YELLOW,                                                     \
            stardata->model.dt_zoomfac);                                \
    fprintf(stdout,__VA_ARGS__);                                        \
    fflush(stdout);                                                     \
    fprintf(stdout,"%s",COLOUR_RESET);
#undef eprint
#define eprint(...) /* do nothing */

int evolve_system_binary_c(struct stardata_t * RESTRICT const stardata)
{
    eprint("EVOLVE SYSTEM stardata=%p previous_stardata=%p stardata = %p store = %p (buffer = %p, size %zu) model_time = %30.20e\n",
           stardata,
           stardata->previous_stardata,
           stardata,
           stardata->store,
           stardata->tmpstore->raw_buffer,
           stardata->tmpstore->raw_buffer_size,
           stardata->model.time
        );

    /*
     * Setup
     */
    reset_binary_c_timeout();
    Start_evolution_timer;

    /*
     * Call start_of_evolution to set things up, 
     * this is required also for restarts
     */
    start_of_evolution(stardata);
    
    eprint("Save to previous\n"); 

    /*
     * start with an evolving system, stop looping 
     * when this is FALSE. 
     * Also don't reject systems by default. 
     */
    Boolean evolving = TRUE;
    Boolean reject = FALSE;
    
    /* 
     * Set previous stardata(s)
     */
    evolution_save_to_previous(stardata);

    eprint("start evolution t=%g M1=%g M2=%g a=%g P=%g\n",
           stardata->model.time,
           stardata->star[0].mass,
           stardata->star[1].mass,
           stardata->common.orbit.separation,
           stardata->common.orbit.period
        );
    
    while(evolving == TRUE) 
    {
        stardata->model.model_number++;
        eprint("Evolution loop top : model %d\n",
               stardata->model.model_number);
               

        /*
         * Start of timestep initialization. If this returns 
         * EVOLUTION_BREAK we shouldn't perform the timestep's 
         * calculations, and we should stop evolving.
         */ 
        int evstatus = evolution_loop_top(stardata);
        
        if(evstatus == EVOLUTION_BREAK)
        {
            eprint("Break evolution\n");
            evolving = FALSE;          
        }
        else
        {
            /*
             * We can evolve for this timestep.
             *
             * Initialize things at the start of the step.
             */
            eprint("Initialize start of step\n");
            initialize_generic_system_every_timestep(stardata);
            
            /*
             * Do time evolution.
             */
            eprint("call evolution() with dtm = %g\n",stardata->model.dtm);
            evstatus = evolution(stardata,GENERIC_SYSTEM_CALL);
            eprint("evolution returned %d \"%s\"\n",
                   evstatus,
                   Evolution_loop_string(abs(evstatus)));

            /*
             * Check if we can and should reject this timestep's evolution
             * because something went wrong in evolution().
             */
            reject = check_reject_flags(stardata);
            Dprint("evolution rejected at t=%g? %s\n",
                   stardata->model.time,
                   Yesno(reject));
            
            if(reject == FALSE)
            {
                /*
                 * Evolution was successful
                 */
                eprint("evolution was successful\n");
                evolution_success(stardata);

                /*
                 * Check for Roche-lobe overflow
                 */
                eprint("test for RLOF\n");
                int rlof_state = test_for_roche_lobe_overflow(stardata);
                
                eprint("rlof_state %d %s\n",
                       rlof_state,
                       Evolution_loop_string(rlof_state));

                if(rlof_state == ROCHE_OVERFLOW ||
                   rlof_state == ROCHE_OVERFLOW_AFTER_SPIRAL_IN)
                {                    
                    /* 
                     * We are in Roche-lobe overflow
                     */
                    eprint("EVRLOF : -> RLOF (%g %g)\n",
                           stardata->star[0].radius / stardata->star[0].roche_radius,
                           stardata->star[1].radius / stardata->star[1].roche_radius
                        );

                    /*
                     * Test for a contact system. 
                     *
                     * This is one that is either truly in contact, or
                     * whose spiral-in time is shorter than the timestep.
                     */
                    if(rlof_state == ROCHE_OVERFLOW_AFTER_SPIRAL_IN ||
                       test_if_primary_still_fills_roche_lobe(stardata)==CONTACT)
                    {
                        eprint("EVRLOF -> contact\n");
                        contact_system(stardata,FALSE);
                    }
                    else
                    {
                        /*
                         * Begin RLOF
                         */
                        start_RLOF(stardata);
                    }
                }
                else
                {
                    stardata->model.in_RLOF = FALSE;
                }

                
                /*
                 * Do updates that require a forward-Euler
                 * step only.
                 */
                evolution_forward_Euler(stardata);    

              
                /*
                 * If evstatus is "STOP" we should stop evolving
                 * after we've finishsed up this timestep.
                 */
                if(abs(evstatus) == STOP)
                {
                    evolving = FALSE;
                }
            }
        }


        if(reject == TRUE)
        {
            /*
             * This step has been rejected. Call evolution_rejected
             * to do something about it, and unset the reject flag.
             */
            eprint("CALL EV REJECTED\n");
            evolution_rejected(stardata);
            eprint("POST REJECT zoom %g\n",
                   stardata->model.dt_zoomfac);
            reject = FALSE;
        }
        else
        {
            /*
             * catch events, e.g. supernovae, unstable RLOF, 
             * common-envelope evolution
             */
            if(events_pending(stardata))
            {
                eprint("\n\n\n\n>>>>>  EVENTS t=%30.20e <<<<<\n\n\n\n",
                       stardata->model.time); 
                catch_events(stardata);
            }

            /*
             * Check for evolution split before 
             * we save_to_previous. If evolution_split
             * does not return EVOLUTION_SPLIT_CONTINUE, it
             * either splits the evolution here or restores
             * the previously saved.
             */
#ifdef EVOLUTION_SPLITTING
            if(unlikely(evolution_split(stardata,&evstatus) !=
                        EVOLUTION_SPLIT_CONTINUE))
            {
                /*
                 * Keep evolving, but do nothing else: evolution_split
                 * does the administration of the splitting algorithm
                 */
                evolving = TRUE;
            }
            else
#endif // EVOLUTION_SPLITTING
            {
                /*
                 * Do logging
                 */
                Evolution_logging;

                
                /*
                 * Update timestep triggers
                 */
                timestep_increment_fixed_timesteps(stardata);

                /*
                 * Calculate the next timestep
                 */
                set_next_timestep(stardata);

                /*
                 * Save the stardata to previous_stardata, unless evolving is FALSE,
                 * in which case we want to keep the previous in place
                 * just in case there is a stardata dump to follow
                 */
                if(evolving)
                {
                    eprint("LOOP this %d prev %d\n",
                           stardata->model.model_number,
                           stardata->previous_stardata->model.model_number);
                    evolution_save_to_previous(stardata);
                }
            }
        }
    }

    /*
     * The system has finished evolving. Clean up and return.
     */
    eprint("End of evolution\n");
    end_of_evolution(stardata);
    End_evolution_timer;
    evolution_cleanup(stardata,TRUE,TRUE);
    return 0;
}
