#include "../binary_c.h"

/*
 * Evolve a detached (non-RLOF) binary star system
 */
int evolve_detached_system(struct stardata_t * RESTRICT const stardata,
                           Boolean * RESTRICT initialize)
{
    
#ifdef RLOF_ABC
    stardata->model.rlof_type=RLOF_NONE;
#endif

    /*
     * Do initialization if required
     */
    if(*initialize==TRUE)
    {
        initialize_detached_system_every_timestep(stardata);
    }
    else
    {
        *initialize = TRUE;
    }

    /* 
     * Update stellar structure, masses/angular momenta and orbit
     * for time t+dt
     */
    int status = evolution(stardata,DETACHED_SYSTEM_CALL);
    Dprint("evolution returned status %d, a=%g\n",status,stardata->common.orbit.separation);
    int stop = 0;
    
    if(status<0)
    {
        stop = -status;
        status = STOP_LOOPING_RESULT;
    }
    else
    {
        if(status != KEEP_LOOPING_RESULT)
        {
            stop = 
                status == SYSTEM_IS_BROKEN_APART ? LOOP : 
                status == STOP_LOOPING_RESULT ? STOP : 0;
        }

        /* 
         * Check for supernova
         */
        Dprint("SN? %d\n",stardata->model.supernova);
        if(stardata->model.supernova==TRUE)
        {
            stop = LOOP_NOW;
            Dprint("SN true -> stop = LOOP_NOW");
        }

        /*
         * Check for manual rejection
         */
        if(check_reject_flags(stardata) == TRUE)
        {
            stop = LOOP_REJECT;
            Dprint("Manual rejection (stop == LOOP_REJECT)\n");
        }
        
        /* 
         * Do Roche lobe and mass checks, determine mass ratios etc.
         */
        if(!stop)
        {
            /*
             * If not interpolating (to find R=RL) set the next timestep 
             */
            if(stardata->model.intpol==0) set_next_timestep(stardata);
            Dprint("Next dt %g dtm %g (intpol %d)\n",
                   stardata->model.dt,
                   stardata->model.dtm,
                   stardata->model.intpol
                );

            /* Test whether Roche lobe overflow has begun. */
            int rlof_state = test_for_roche_lobe_overflow(stardata);

            if(rlof_state == REJECT_EVOLUTION)
            {
                stop = LOOP_REJECT;
                Dprint("RLOF state stop = LOOP_REJECT\n");
                Dprint("SNe %p %p\n",
                       stardata->star[0].new_supernova,
                       stardata->star[1].new_supernova);
            }
            else if(rlof_state == ROCHE_UNSTABLE_MERGER)
            {
                /*
                 * Note: this is impossible with the new
                 * binary_c evolution algorithm, instead
                 * contact_system is called as an event.
                 */
                stardata->model.in_RLOF=FALSE;
                contact_system(stardata,FALSE);
                stop = LOOP_NOW;
            }
            else if(rlof_state == ROCHE_OVERFLOW_NONE)
            {
                /* no RLOF */
                stardata->model.in_RLOF=FALSE;

                /*
                 * Check for collision at periastron (contact system)
                 */
                if(check_for_collision_at_periastron(stardata)==TRUE) 
                {
                    stop = (contact_system(stardata,FALSE) == END_ROCHE) ? LOOP_NOW : LOOP;
                    Dprint("Contact : stop = %s\n",
                           (stop == LOOP_NOW) ? "LOOP_NOW" : "LOOP"); 
                }
                else
                {
                    /* no contact or RLOF, check loop conditions */
                    if(to_loop_or_not_to_loop(stardata)==FALSE)
                    {
                        /* failure caught: stop */
                        stop = STOP;
                        Dprint("No RLOF or contact : stop = STOP\n");
                    }
                    else
                    {
                        /* normal detached evolution continues */
                        stop = LOOP_NOW;
                        *initialize = FALSE; 
                        Dprint("No RLOF or contact : stop = LOOP_NOW\n");
                    }
                }
            }
            else if(rlof_state == ROCHE_OVERFLOW)
            {
                /* begin RLOF */
                Dprint("start RLOF");
                start_RLOF(stardata);
            }
            else if(rlof_state == STOP_LOOPING_RESULT)
            {
                /* no RLOF, but we should stop looping */
                Dprint("no RLOF, but stop = STOP");
                stardata->model.in_RLOF = FALSE;
                stop = STOP;
            }
        }
    }

    catch_events(stardata);
    
    Dprint("stop = %d = %s\n",
           stop,
           Evolution_loop_string(stop));
    
    return stop;
}

