#include "../binary_c.h"

/*
 * Subroutine to set things up at time zero, which is 
 * usually the first time a star is evolved (although, when
 * using the API, this may not be true : a star can then be 
 * restarted).
 */

void evolution_time_zero_setup(struct stardata_t * RESTRICT const stardata)
{
    Dprint("time zero setup\n");
#if defined BINT || defined NUCSYN
    Star_number k;
#endif // BINT || NUCSYN
    struct common_t * const common =  &stardata->common;
    common->init_random_seed = common->random_seed;

    Dprint("set orbital parameters\n");
    if(Is_not_zero(stardata->star[0].mass)
       &&
       Is_not_zero(stardata->star[1].mass))
    {
    
        /* 
         * if the period is zero, use the provided
         * separation, otherwise use the provided period
         */
        if(Is_zero(common->orbit.period))
        {
            common->orbit.period = calculate_orbital_period(stardata);
        }

        common->zams_period = common->orbit.period;

        /*
         * convert orbital period to years
         **/
        common->orbit.period /= YEAR_LENGTH_IN_DAYS;

        /*
         * calculate the separation 
         */
        common->orbit.separation =
            common->zams_separation = calculate_orbital_separation(stardata);
    }
    common->zams_eccentricity = common->orbit.eccentricity;

#ifdef BINT
    Starloop(k)
    {
        SETstar(k);
        /*
         * Set up parameters for the BINT interpolation
         * scheme
         */
        star->bint.XHc = 0.760 - 3.0 * stardata->common.nucsyn_metallicity;
        //0.7381; // Asplund 2009 (solar)
        star->bint.XHec = 0.240 + 2.0 * stardata->common.nucsyn_metallicity;
        //0.2485; // Asplund 2009 (solar)
    }
#endif


#ifdef NUCSYN
    Dprint("Set initial abundances (XZAMS, Xsolar)\n");
    
    /*
     * copy initial abundances into the XZAMS array
     */
    Clear_isotope_array(common->XZAMS);
    Clear_isotope_array(common->Xsolar);
       
    /* 
     * set the initial abundance mixture using the nucsyn_metallicity 
     */
    nucsyn_initial_abundances(stardata->star[0].Xenv,
                              common->nucsyn_metallicity,
                              stardata,
                              stardata->preferences->initial_abundance_mix);
      
    
    /* initial nuclear burning timesteps */
    Reaction_network i;
    for(i=0;i<NUMBER_OF_REACTION_NETWORKS;i++)
    {
        stardata->common.dtguess[i] = LONG_TIMESTEP;
    }
    
    /*
     * copy the initial abundances where they are required
     */
    Starloop(k)
    {
        SETstar(k);
        if(k!=0) Copy_abundances(stardata->star[0].Xenv,star->Xenv);
        Copy_abundances(stardata->star[0].Xenv,star->Xinit);
        Copy_abundances(stardata->star[0].Xenv,star->Xacc);
    }
    Copy_abundances(stardata->star[0].Xenv,common->XZAMS);

   
    /*
     * Pure solar, Z=0.02 abundances
     */
    nucsyn_initial_abundances(common->Xsolar,
                              0.02,
                              NULL,
                              stardata->preferences->initial_abundance_mix);

    /*
     * overwrite the initial abundance array to be 
     * the true initial abundances
     */
    Copy_abundances(stardata->star[0].Xenv,
                    stardata->preferences->the_initial_abundances);

#ifdef NANCHECKS
    nancheck("Xsolar in initialize_parameters",common->Xsolar,ISOTOPE_ARRAY_SIZE);
    nancheck("XZAMS in initialize_parameters",common->XZAMS,ISOTOPE_ARRAY_SIZE);
#endif //NANCHECKS
#endif // NUCSYN
}
