
#include "../binary_c.h"


void evolution_save_to_previous(struct stardata_t * RESTRICT const stardata)
{
    /*
     * Save current stardata into the previous stardata, and 
     * update the previous_stardatas stack if necessary
     */

    
    /*
     * If we're using a solver that has created an intermediate
     * solution, do not save.
     *
     * e.g. the RK2 solver on its first step should not be saved
     *
     * If force is TRUE then we must save, regardless of the "intermediate"
     * state. This is done e.g. on the first timestep.
     */

    Boolean save = Boolean_(stardata->model.intermediate_step == FALSE &&
                            stardata->previous_stardatas != NULL);
    
    Dprint("Save to prev? solver=%d solver_step=%d previous_stardatas=%p : %d\n",           
           stardata->preferences->solver,
           stardata->model.solver_step,
           stardata->previous_stardatas,
           save
        );
    if(save==TRUE)
    {
        /*
         * Max number of previous stardatas
         */
        const unsigned int n = Solver_n_previous_stardatas(stardata->preferences->solver);

        /*
         * We have stardata->n_previous_stardata previous stardatas
         *
         * We want to save n previous stardatas.
         */
        if(stardata->n_previous_stardatas < n)
        {
            /*
             * Allocate memory for new previous_stardata
             */
            Dprint("New previous_stardata[%u] : want %u\n",
                   stardata->n_previous_stardatas,
                   n);

            /* increase array of pointers */
            stardata->previous_stardatas = Realloc(stardata->previous_stardatas,
                                                   sizeof(struct stardata_t *)*(1+stardata->n_previous_stardatas));
            memset(stardata->previous_stardatas + (size_t)stardata->n_previous_stardatas,
                   0,
                   sizeof(struct stardata_t*) * ((size_t)n - (size_t)stardata->n_previous_stardatas));
                   
            unsigned int i;
            for(i=0;i<n;i++)
            {
                if(stardata->previous_stardatas[i] == NULL)
                {
                    stardata->previous_stardatas[i] = New_stardata;
                }
            }
            Dprint("previous_stardatas[%u] = %p\n",
                   stardata->n_previous_stardatas,
                   stardata->previous_stardatas);
            stardata->n_previous_stardatas++;
        }

        if(n == 1)
        {
#ifdef STARDATA_DIFFSTATS
            printf("Copy stardata : %5.2f %% different\n",
                   diff_struct_pc(stardata,
                                  stardata->previous_stardatas[0],
                                  sizeof(struct stardata_t)));
#endif
            
            /* copy stardata to previous_stardata with one memcpy
             * 0.22s
             */
            //diff_stardata(stardata);

            /*
            copy_stardata(
                stardata,
                stardata->previous_stardatas[0],
                COPY_STARDATA_PREVIOUS_NONE
                );
            */            

            /*
             * Use strided_memcpy to copy at most 10 chunks
             */            

            /*
             * stride/chars   runtime/s
             * 1              1.63           
             * 2              0.96
             * 4              0.67
             * 8              0.47
             * 16             0.33
             * 32             0.26
             * 64             0.24
             * 128            0.23
             * 256            0.23
             * 512            0.22
             * 1024           0.21
             * 2048           0.22
             */
            strided_memcpy(stardata->previous_stardatas[0],
                           stardata,
                           sizeof(struct stardata_t),
                           2048*sizeof(char));            
        }
        else
        {
            /*
             * We require multiple stardata copies, n steps back in time
             */
            unsigned int i;
            for(i=n-1; i>0; i--)
            {
                Dprint("Copy stardata previous[%u] (%p) to previous[%u] (%p %p)\n",
                       i-1,
                       stardata->previous_stardatas[i-1],
                       i,
                       stardata->previous_stardatas[i],
                       stardata->previous_stardatas[i]->preferences
                    );

                /* use copy_stardata to do a pure copy */
                copy_stardata(
                    stardata->previous_stardatas[i-1],
                    stardata->previous_stardatas[i],
                    COPY_STARDATA_PREVIOUS_NONE
                    );
            }
            Dprint("Copy stardata (%p) to previous[%u] (%p)\n",
                   stardata,
                   (unsigned int)0,
                   stardata->previous_stardatas[0]
                );
            copy_stardata(
                stardata,
                stardata->previous_stardatas[0],
                COPY_STARDATA_PREVIOUS_NONE
                );
        }
        stardata->previous_stardata = stardata->previous_stardatas[0];
    }


    Dprint("saved to previous at %p %p with prefs = %p\n",
           stardata->previous_stardata,
           stardata->previous_stardatas[0],
           stardata->previous_stardata->preferences);
}




