#include "../binary_c.h"
static Boolean smooth_evolution(struct stardata_t * stardata);
static Boolean MAYBE_UNUSED smooth_evolution_checker(
    struct stardata_t * stardata,
    const Boolean check_stellar_type_change,
    const Boolean check_duplicity_change,
    const Boolean check_events_pending
    );

void evolution_rejection_tests(struct stardata_t * RESTRICT const stardata)
{

    /*
     * If we canot reject, simply do nothing
     */
    const Boolean can = can_reject(stardata);
    Dprint("RLOF can reject? %d\n",can);

    if(can == TRUE)
    {
        /*
         * Here you can reject a system if it changes too much.
         *
         * What you should *not* do is compare derivatives to determine
         * whether to reject. This won't work, because the derivatives
         * are a function of the previous structure, so will not change
         * when you reject and change the timestep.
         *
         * When you reject, set stardata->model.reject to one of the REJECT_*
         * macros, and immediately return (it is not required to test further
         * even if other tests may also reject the timestep).
         */


        /*
         * Test for RLOF in this timestep, but no RLOF in the previous timestep
         */
        const unsigned int RLOF_state = test_for_roche_lobe_overflow(stardata);
        const unsigned int prev_RLOF_state = test_for_roche_lobe_overflow(stardata->previous_stardata);
        Dprint("test for RLOF rejection prev = %u, now = %u -> %s\n",
               prev_RLOF_state,
               RLOF_state,
               Evolution_loop_string(RLOF_state)
            );
        if(prev_RLOF_state == ROCHE_OVERFLOW_NONE &&
           RLOF_state != ROCHE_OVERFLOW_NONE)
        {
            Dprint("RLOF reject : went from no RLOF to RLOF\n");
            stardata->model.reject = REJECT_RLOF;
            return;
        }

        /*
         * Angular momentum change tests: don't let it
         * change by more than preferences->max_stellar_angmom_change.
         *
         * Ignore this test if max_stellar_angmom_change is zero.
         */
        if(Is_not_zero(stardata->preferences->max_stellar_angmom_change))
        {
#define Jchange(k) (fabs(1.0 - stardata->star[(k)].angular_momentum /   \
                         stardata->previous_stardata->star[(k)].angular_momentum))
        
            Dprint("check reject deltaJ %g %g (tidal factors %g %g) can reject? %s events? %s smooth? %s : %d,%d vs %d,%d : binary %d vs %d\n",
                   Jchange(0),
                   Jchange(1),
                   Is_not_zero(stardata->common.orbit.angular_frequency) ? (stardata->star[0].omega / stardata->common.orbit.angular_frequency) : 0.0,
                   Is_not_zero(stardata->common.orbit.angular_frequency) ? (stardata->star[1].omega / stardata->common.orbit.angular_frequency) : 0.0,
                   Yesno(can),
                   Yesno(events_pending(stardata)),
                   Yesno(smooth_evolution(stardata)),
                   stardata->star[0].stellar_type,
                   stardata->previous_stardata->star[0].stellar_type,
                   stardata->star[1].stellar_type,
                   stardata->previous_stardata->star[1].stellar_type,
                   _System_is_binary(stardata) ,
                   _System_is_binary(stardata->previous_stardata));

            /*
             * Do not allow stellar angular momentum to change too much
             * e.g. to avoid moving in/out of tidal lock
             */
            if(smooth_evolution(stardata))
            {
                Star_number k;
                Starloop(k)
                {
                    if(stardata->star[k].angular_momentum > MIN_ANGMOM_FOR_TIMESTEP_AND_REJECTION_TESTS &&
                       Jchange(k) > stardata->preferences->max_stellar_angmom_change)
                    {
                        stardata->star[k].reject = REJECT_ANGMOM;
                        Dprint("ZOOM REJECT star %d Yes by JCHANGE dt = %g (deltaJ/J %g : J was %g, is %g) <<<\n",
                               k,
                               stardata->model.dt,
                               Jchange(k),
                               stardata->previous_stardata->star[k].angular_momentum,
                               stardata->star[k].angular_momentum
                               
                            );
                        return;
                    }
                }
            }
        }
        
#ifdef GAIAHRD

        /*
         * Perhaps reject timestep when things change too fast to make
         * the Gaia-like HRD.
         *
         * First check we can : 
         * note we ignore stellar type 
         * changes in this check.
         */
        if(smooth_evolution_checker(stardata,FALSE,TRUE,TRUE))
        {
            /*
             * When constructing Gaia-like HRDs, do not let
             * Teff,L change too quickly
             */
            const double f = 1.99;
            const double max_dL = f * stardata->preferences->gaia_L_binwidth;
            const double max_dTeff = f * stardata->preferences->gaia_Teff_binwidth;

            /*
             * Get the next Gaia data
             */
            double L,Teff;
            Star_number primary;
            Boolean do_log,is_single;
            gaia_log(stardata,
                     &Teff,
                     &L,
                     &primary,
                     &do_log,
                     &is_single);

            /*
             * If we are going to log, check that we 
             * haven't changed too much
             */
            if(do_log == TRUE)
            {
                Boolean reject_HRD;

                /*
                 * Check L and Teff don't change too much
                 */
                if(fabs(L - stardata->common.gaia_L) > max_dL)
                {
                    /*
                      printf("L = %g vs gaia_L = %g : reject lum\n",
                           L,
                           stardata->common.gaia_L);
                    */
                    stardata->star[0].reject = REJECT_LUMINOSITY;
                    reject_HRD = TRUE;
                }
                else if(fabs(Teff - stardata->common.gaia_Teff) > max_dTeff)
                {
                    /*
                      printf("Teff = %g vs gaia_Teff = %g : reject Teff\n",
                           Teff,
                           stardata->common.gaia_Teff);
                    */
                    stardata->star[0].reject = REJECT_TEFF;
                    reject_HRD = TRUE;
                }
                else
                {
                    reject_HRD = FALSE;
                }

                /*
                 * return if rejected
                 */
                if(reject_HRD == TRUE)
                {
                    return;
                }
            }
        }
#endif//GAIAHRD
    }
}


static Boolean smooth_evolution(struct stardata_t * stardata)
{
    /*
     * Define "smooth" evolution to be when the stellar
     * types do not change, the system remains single or binary,
     * and there are no events
     */
    return Boolean_(stardata->previous_stardata != NULL &&
                    stardata->star[0].stellar_type == stardata->previous_stardata->star[0].stellar_type &&
                    stardata->star[1].stellar_type == stardata->previous_stardata->star[1].stellar_type &&
                    _System_is_binary(stardata) == _System_is_binary(stardata->previous_stardata) &&
                    events_pending(stardata) == FALSE);
}

static Boolean smooth_evolution_checker(
    struct stardata_t * stardata,
    const Boolean check_stellar_type_change,
    const Boolean check_duplicity_change,
    const Boolean check_events_pending
    )
{
    /*
     * Define "smooth" evolution to be when the stellar
     * types do not change, the system remains single or binary,
     * and there are no events
     */
    return Boolean_(
        (
            /* always check we have a previous stardata */
            stardata->previous_stardata != NULL
            )
        &&
        (
            /*
             * check for stellar type change
             */
            check_stellar_type_change == FALSE || (
                stardata->star[0].stellar_type == stardata->previous_stardata->star[0].stellar_type &&
                stardata->star[1].stellar_type == stardata->previous_stardata->star[1].stellar_type
                )
            )
        &&
        (
            /*
             * check for duplicity change
             */
            check_duplicity_change == FALSE ||
            _System_is_binary(stardata) == _System_is_binary(stardata->previous_stardata)
            )
        &&
        (
            /*
             * check for pending events
             */
            check_events_pending == FALSE ||
            events_pending(stardata) == FALSE
            )
        
        );
}
