#pragma once
#ifndef EVOLUTION_MACROS_H
#define EVOLUTION_MACROS_H

/** This file contains useful macros for the binary_c evolution loop **/


// The maximum number of goes through the evolution loop
// NB under linux the timeout (via reset_binary_c_timeout) is a better
// way to ensure the program doesn't run forever
#define MAX_EVOLUTION_NUM_LOOPS 1e8

/* See test_for_roche_lobe_overflow for the use of MAX_INTPOL */
#define MAX_INTPOL 500
#define NEARLY_MAX_INTPOL 400

#define MAX_EVOLUTION_ITERATIONS 10000

/* some more useful macros */
#define Q12 (stardata->star[0].mass/stardata->star[1].mass)
#define Q21 (stardata->star[1].mass/stardata->star[0].mass)
#define MASS_PRODUCT (stardata->star[0].mass*stardata->star[1].mass)
#define ECC_SQUARED Pow2(stardata->common.orbit.eccentricity)
#define Separation_squared Pow2(stardata->common.orbit.separation)
#define JORB_SQUARED Pow2(stardata->common.orbit.angular_momentum)

#define THERE_IS_AN_IMMEDIATE_MERGER (Is_zero(stardata->model.time) &&((Is_zero(stardata->star[0].mass)&& Is_not_zero(stardata->star[0].pms_mass))||(Is_zero(stardata->star[1].mass)&& Is_not_zero(stardata->star[1].pms_mass))))


/*
 * Return codes
 */

/* evolution loop control */
#define STATUS_OK 0
#define LOOP 1
#define LOOP_NOW 2
#define STOP 3
#define LOOP_RLOF 4
#define STOP_ROCHE 5
#define LOOP_REJECT 6

#define KEEP_LOOPING_RESULT 7
#define STOP_LOOPING_RESULT 8
#define SYSTEM_IS_BROKEN_APART 9
#define ROCHE_OVERFLOW 10
#define END_ROCHE 11
#define ROCHE_OVERFLOW_NONE KEEP_LOOPING_RESULT
#define NO_CONTACT 12
#define CONTACT 13
#define REJECT_EVOLUTION 14
#define ROCHE_UNSTABLE_MERGER 15
#define ROCHE_OVERFLOW_AFTER_SPIRAL_IN 16

#define Evolution_loop_string(S) (                                      \
        (S) == STATUS_OK ? "OK" :                                       \
        (S) == LOOP ? "Loop":                                           \
        (S) == LOOP_NOW ? "Loop now":                                   \
        (S) == STOP ? "Stop":                                           \
        (S) == LOOP_RLOF ? "Loop RLOF":                                 \
        (S) == STOP_ROCHE ? "Stop RLOF":                                \
        (S) == LOOP_REJECT ? "Reject" :                                 \
        (S) == KEEP_LOOPING_RESULT ? "Keep looping":                    \
        (S) == STOP_LOOPING_RESULT ? "Stop looping":                    \
        (S) == SYSTEM_IS_BROKEN_APART ? "System is broken apart":       \
        (S) == ROCHE_OVERFLOW ? "Start RLOF" :                   \
        (S) == END_ROCHE ? "End Roche":                                 \
        (S) == NO_CONTACT ? "No contact":                               \
        (S) == CONTACT ? "Contact" :                                    \
        (S) == REJECT_EVOLUTION ? "Reject evolution" :                  \
        (S) == ROCHE_UNSTABLE_MERGER ? "Unstable RLOF -> merger" :      \
        (S) == ROCHE_OVERFLOW_AFTER_SPIRAL_IN ? "Spiral in -> RLOF -> merger" :      \
        "Unknown"                                                       \
        )

/* evolution return codes */
#define DETACHED_SYSTEM_CALL 15
#define RLOF_SYSTEM_CALL 16
#define GENERIC_SYSTEM_CALL 17


#define Reset_num_thermal_pulses(A) if(stardata->star[(A)].num_thermal_pulses<-1.5) stardata->star[(A)].num_thermal_pulses=-1.0;


#define EVOLUTION_NO_ACTION_REQUIRED 0
#define EVOLUTION_CONTINUE 1
#define EVOLUTION_CONTINUE_RLOF 2
#define EVOLUTION_BREAK 3
#define EVOLUTION_REJECT 4
#define EVOLUTION_SHOULD_NOT_HAPPEN -666

#define Evolution_status_string(status) (                               \
    (status) == EVOLUTION_CONTINUE ? "Continue" :                       \
    (status) == EVOLUTION_REJECT ? "Reject":                            \
    (status) == EVOLUTION_CONTINUE_RLOF ? "Continue RLOF":              \
    (status) == EVOLUTION_BREAK ? "Break":                              \
    "Unknown!"                                                          \
        )

#define Evolution_logging                       \
    Dprint("calling difflog\n");                \
    evolution_difflog(stardata);                \
    Dprint("calling log every timestep\n");     \
    log_every_timestep(stardata);               

/*
 * Evolution_loop_check calls the function (with args) Function, 
 * and depending on its return value, either issues
 * a continue, break or does nothing.
 *
 * NB evolution_split must be called before logging functions
 * e.g. evolution_difflog and log_every_timestep.
 *
 * NB (Function_call) should be called ONCE to return an
 *    integer, set in evstatus.
 *    Do NOT use (Function_call) in printf or Dprint statements!
 */
#define Evolution_loop_check(Function_call)                             \
    {                                                                   \
        RLOFing_system = FALSE;                                         \
        if(0)printf("\n\nEVOLUTION LOOP ***************\n");            \
                                                                        \
        /* call function : this calls evolution() */                    \
        const int evstatus = (Function_call);                           \
                                                                        \
        /* check status */                                              \
        int status = loop_check(stardata,evstatus);                     \
                                                                        \
        Dprint("RLOF loop check gave status %d = %s\n",                 \
               status,                                                  \
               Evolution_loop_string(evstatus)                          \
            );                                                          \
        if(status!=EVOLUTION_REJECT)                                    \
        {                                                               \
            Dprint("Not rejected : calling nucsyn\n");                  \
            evolution_nucsyn(stardata);                                 \
            Dprint("Calling na nchecks\n");                             \
            evolution_nanchecks(stardata);                              \
            Dprint("Check for split\n");                                \
            if(evolution_split(stardata,&status))                       \
            {                                                           \
                status = EVOLUTION_REJECT;                              \
            }                                                           \
            else                                                        \
            {                                                           \
                Dprint("Logging\n");                                    \
                Evolution_logging;                                      \
                timestep_increment_fixed_timesteps(stardata);           \
                Dprint("Save to previous\n");                           \
                evolution_save_to_previous(stardata);                   \
            }                                                           \
        }                                                               \
                                                                        \
        Dprint("continue or break : status = %d = %s\n",status,         \
               Evolution_status_string(status));                        \
        if(status == EVOLUTION_CONTINUE)                                \
        {                                                               \
            continue;                                                   \
        }                                                               \
        else if(status == EVOLUTION_REJECT)                             \
        {                                                               \
            continue;                                                   \
        }                                                               \
        else if(status == EVOLUTION_CONTINUE_RLOF)                      \
        {                                                               \
            RLOFing_system = TRUE;                                      \
            continue;                                                   \
        }                                                               \
        else if(status == EVOLUTION_BREAK)                              \
        {                                                               \
            evolving = FALSE;                                           \
            break;                                                      \
        }                                                               \
    }


#if (defined(TIMED) && defined(HAVE_FTIME))
#define Start_evolution_timer                   \
    struct timeb start_time,end_time;;          \
    ftime(&start_time);
#else
#define Start_evolution_timer
#endif

#if (defined TIMED && HAVE_FTIME)
#define End_evolution_timer                                             \
    ftime(&end_time);                                                   \
    printf("TIME took %lf seconds\n",                                   \
           ((double)end_time.time+(double)end_time.millitm/1000)-       \
           ((double)start_time.time+(double)start_time.millitm/1000));
#else
#define End_evolution_timer
#endif


#define EVOLUTION_SPLIT_CONTINUE 0
#define EVOLUTION_SPLIT_RESTART 2
#define EVOLUTION_SPLIT_NEW_SPLIT 1


#endif /* EVOLUTION_MACROS_H */
