#include "../binary_c.h"

#ifdef BINT
#ifdef __TODO
static double common_envelope_lambda(struct stardata_t * stardata,
                                     struct star_t * star);
static double orbital_energy(struct star_t * donor,
                             struct star_t * accretor,
                             struct orbit_t * orbit);
static star_t * stripped_core(struct stardata_t * stardata,
                              struct star_t * star);
#endif//__TODO

int common_envelope_evolution_BINT (
    struct star_t * donor_star,
    struct star_t * accretor_star,
    struct stardata_t *stardata
    )
{
#ifdef __TODO

    /* 
     * Common envelope evolution : BINT algorithm
     *
     * The "donor" star is the one that initiates the Roche-lobe
     * overflow that leads to the formation of a common envelope.
     * 
     * The other star, which is normally a dwarf, is the accretor.
     */
    Exit_binary_c(BINARY_C_ALGORITHM_OUT_OF_RANGE,
                  "TODO:  implement common envelope evolution in BINT\n");

    /*
     * Back up stars passed in for later logging
     */
    struct star_t donor_backup = New_star_from(donor_star);
    struct star_t accretor_backup = New_star_from(accretor_star);

    /*
     * Save orbit passed in, we work on orbit
     */
    struct orbit_t * orbit = Malloc_from(stardata->common.orbit,
                                         struct orbit_t);
    
    if(stardata->preferences->comenv_prescription == COMENV_BSE ||
       stardata->preferences->comenv_prescription == COMENV_NANDEZ2016)
    {
        /*
         * Alpha-lambda prescriptions
         */
        double Eorb_i = orbital_energy(donor_star,
                                       accretor_star,
                                       stardata->orbit);
        double Eorb_f;
        
        if(stardata->preferences->comenv_prescription == COMENV_BSE)
        {
            /*
             * Hurley et al. 2002 prescription
             *
             * This is a classical alpha-lambda based on the 
             * giant donor's binding energy
             */

            /*
             * Calculate binding energy of the giant donor 
             */
            double Ebind_donor = GIANT_LIKE_STAR(donor_star->stellar_type) ?
                (donor_star->mass * (donor_star->mass - donor_star->core_mass)/
                 (donor_star->radius * common_envelope_lambda(stardata,donor_star,accretor))) : 0.0;

            /*
             * Calculate the binding energy of the accretor, if it's a giant.
             */
            double Ebind_accretor = GIANT_LIKE_STAR(accretor_star->stellar_type) ?
                (accretor_star->mass * (accretor_star->mass - accretor_star->core_mass)/
                 (accretor_star->radius * common_envelope_lambda(stardata,accretor,donor_star))) : 0.0;

            /*
             * Total binding energy of the envelope
             */
            double Ebind_i = Ebind_accretor + Ebind_donor;

            /*
             * Energy donated to the envelope
             */
            deltaE = Ebind_i / stardata->preferences->alpha_ce;
            
        }
        else if(stardata->preferences->comenv_prescription == COMENV_NANDEZ2016)
        {
            /*
             * Nandez et al. 2016 prescription
             *
             * This has the total alpha*lambda calculated according to their
             * prescription.
             */
            double delta_E =
                donor_star->mass * (donor_star->mass - donor_star->core_mass) /
                (donor_star->radius * calc_alpha_lambda_Nandez2016(donor_star->mass,
                                                                   accretor_star->core_mass,
                                                                   donor_star->core_mass));
            if(GIANT_LIKE_STAR(accretor))
            {
                Ebind_i +=
                    accretor_star->mass * (accretor_star->mass - accretor_star->core_mass) /
                    (accretor_star->radius * calc_alpha_lambda_Nandez2016(accretor_star->mass,
                                                                          donor_star->core_mass,
                                                                          accretor_star->core_mass));
            }

            deltaE = Ebind_i / alpha_lambda;
        }
        else
        {
            Exit_binary_c(BINARY_C_ALGORITHM_OUT_OF_RANGE,
                          "Unknown alpha-lambda comenv_prescription %d\n",
                          stardata->preferences->comenv_prescription);
        }

        /*
         * Inject energy from the orbital decay into the final orbit
         * to see if it is bound.
         */
        Eorb_f = EorbI + deltaE;

        /*
         * HPhd : Eq. 3.139
         */
        orbit->separation = -2.0 * Eorb_f / (donor_star->core_mass * Effective_core_mass(accretor)); 
    }
    else if(stardata->preferences->comenv_prescription==COMENV_NELEMANS_TOUT)
    {
        /*
         * Nelemans and Tout prescription
         */
        orbit->separation = nelemans_separation(stardata->common.orbit.separation,
                                                donor_star->mass,
                                                accretor_star->mass,
                                                Effective_core_mass(donor_star),
                                                Effective_core_mass(accretor),
                                                stardata->preferences->nelemans_gamma,
                                                stardata);
    }
    else
    {
        Exit_binary_c(BINARY_C_ALGORITHM_OUT_OF_RANGE,
                      "Unknown comenv_prescription %d\n",
                      stardata->preferences->comenv_prescription);
    }


    Boolean merge_stars = FALSE;
    
    /*
     * Construct the stripped cores of the stars
     */
    struct star_t * donor_core = stripped_core(donor_star);
    struct star_t * accretor_core = stripped_core(accretor);
    
    if(Less_or_equal(orbit->separation,0.0))
    {
        /*
         * separation is zero or negative, implies a merged system
         */
        merge_stars = TRUE;
    }
    else
    {
        /*
         * Separation > 0:
         * Construct a binary system made up of the stripped core
         * of the donor giant and the (possibly stripped) companion 
         */
    
    
        /*
         * We now know the separation, update Roche lobe radii
         * and check for Roche lobe overflow of the cores
         */
        determine_roche_lobe_radius(stardata,orbit,donor_core);
        determine_roche_lobe_radius(stardata,orbit,accretor_core);
        merge_stars = Boolean_(donor_core->radius > donor_core->roche_radius ||
                               accretor_core->radius > accretor_core->roche_radius);

        if(merge_stars == FALSE)
        {
            /*
             * Perhaps just fill the Roche lobes of the remnants? 
             */
            
        }
    }

    if(merge_stars == TRUE)
    {
        /*
         * Merge the stars.
         */
        merge_stars(orbit,donor_star,accretor);

        /*
         * Remove energy deltaE which came from the orbit.
         */
        
    }
    else
    {
        /*
         * Cores do not merge: 
         * Make a binary system which contains the stripped cores
         * at the appropriate separation.
         */
        Copy_star(donor_core,donor_star);
        Copy_star(accretor_core,accretor);
        memcpy(&stardata->common.orbit,orbit,sizeof(struct orbit_t));
    }

    

    
    Safe_free(donor_core);
    Safe_free(accretor_core);



#if defined NUCSYN && defined NUCSYN_STELLAR_POPULATIONS_ENSEMBLE

    /*
     * If there is a merger, make the comenv type negative
     */
    if(stardata->model.coalesce == TRUE)
    {
        stardata->model.comenv_type =
            donor_backup->stellar_type == HG ? NUCSYN_ENSEMBLE_COMENV_HG_MERGER :
            donor_backup->stellar_type == GIANT_BRANCH ? NUCSYN_ENSEMBLE_COMENV_GB_MERGER :
            donor_backup->stellar_type == CHeB ? NUCSYN_ENSEMBLE_COMENV_CHeB_MERGER :
            donor_backup->stellar_type == EAGB ? NUCSYN_ENSEMBLE_COMENV_EAGB_MERGER :
            donor_backup->stellar_type == TPAGB ? NUCSYN_ENSEMBLE_COMENV_TPAGB_MERGER :
            donor_backup->stellar_type == HeHG ? NUCSYN_ENSEMBLE_COMENV_HeHG_MERGER :
            donor_backup->stellar_type == HeGB ? NUCSYN_ENSEMBLE_COMENV_HeGB_MERGER :
            0;
        
        stardata->model.comenv_type *= -1;
    }
    else
    {
        stardata->model.comenv_type =
            donor_backup->stellar_type == HG ? NUCSYN_ENSEMBLE_COMENV_HG_DETACHED :
            donor_backup->stellar_type == GIANT_BRANCH ? NUCSYN_ENSEMBLE_COMENV_GB_DETACHED :
            donor_backup->stellar_type == CHeB ? NUCSYN_ENSEMBLE_COMENV_CHeB_DETACHED :
            donor_backup->stellar_type == EAGB ? NUCSYN_ENSEMBLE_COMENV_EAGB_DETACHED :
            donor_backup->stellar_type == TPAGB ? NUCSYN_ENSEMBLE_COMENV_TPAGB_DETACHED :
            donor_backup->stellar_type == HeHG ? NUCSYN_ENSEMBLE_COMENV_HeHG_DETACHED :
            donor_backup->stellar_type == HeGB ? NUCSYN_ENSEMBLE_COMENV_HeGB_DETACHED :
            0;
    }

    /* 0 is a failure : shouldn't ever happen */
    if(stardata->model.comenv_type==0)
    {
        Exit_binary_c(BINARY_C_WRONG_ARGUMENT,"comenv_type set to 0, i.e. unknown, from donor_backup->stellar_type=%d\n",
                      donor_backup->stellar_type);
    }
#endif // NUCSYN

#endif // __TODO    
    
    return 0;
}

#ifdef __TODO

static double common_envelope_lambda(struct stardata_t * stardata,
                                     struct star_t * star,
                                     struct star_t * companion)
{
    return common_envelope_lambda(star->phase_start_mass,
                                  star->core_mass,
                                  star->luminosity,
                                  star->radius,
                                  star->rzams,
                                  star->menv / (star->mass - star->core_mass),
                                  star->stellar-type,
#ifdef NUCSYN
                                  star->Xenv,
#endif
                                  stardata);
}

static double orbital_energy(struct star_t * donor,
                             struct star_t * accretor,
                             struct orbit_t * orbit)
{
    return
        Is_zero(orbit->separation) ? 
        LARGE_ENERGY :
        (-donor->core_mass *
         (
             GIANT_LIKE_STAR(accretor_star->stellar_type) ?
             accretor_star->core_mass :
             accretor_star->mass
             )
         /(2.0 * orbit->separation * (1.0 - Pow2(orbit->eccentricity)))
            );
}

static star_t * stripped_core(struct stardata_t * stardata,
                              struct star_t * star)
{
    /*
     * Convert the star into its stripped core.
     *
     * Return a pointer to the core structure. 
     */
    struct star_t * core = New_star_from(star);
    core->mass = core->core_mass;
    stellar_structure(core,
                      stardata,
                      FALSE,
                      STELLAR_STRUCTURE_CALLER_common_envelope_evolution);
    return core;
}
        
#endif//__TODO
#endif//BINT

