#include "../binary_c.h"

double Pure_function wind_djorb_dt(const struct stardata_t * RESTRICT const stardata,
                                   double orbital_angular_frequency)
{
    /*
     * Rate of change of orbital angular momentum because of 
     * wind mass transfer.
     *
     * Returns the correct sign, i.e. negative for angular momentum loss.
     */

    double djorb_dt = 0.0;
    double mtot = stardata->star[0].mass + stardata->star[1].mass;
    
    double mdot_system = 
        /* mass loss : derivatives are negative, so this term is positive */
        -(stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS]+
          stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS])
        /* mass gain: derivatives are positive, so this term is negative */
        -(stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN]+
          stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN]);
    
    double Jorb = Separation_squared*orbital_angular_frequency*
        Sqrt_eccfac*stardata->star[0].mass*stardata->star[1].mass/mtot;

    if(stardata->preferences->wind_angular_momentum_loss == WIND_ANGMOM_LOSS_BSE)
    {
        /* original Hurley/Tout (BSE) formalism */
        djorb_dt =  stardata->preferences->wind_djorb_fac*
            (
                (
                    -stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS]+
                    stardata->star[0].q*stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN]
                    )
                *Mass_squared(1)
                +
                (
                    -stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS]+
                    stardata->star[1].q*stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN]
                    )
                *Mass_squared(0)
                )
            * (Jorb/(mtot*stardata->star[0].mass*stardata->star[1].mass));
        /*
          printf("TOUT %g %g %g %g %g %g %g %g %g\n",
          stardata->model.time,
          djorb_dt,
          stardata->star[0].mass,
          stardata->star[1].mass,
          stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS],
          stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN],
          stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS],
               stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN],
               Jorb
            );
        */
    }
    else if(stardata->preferences->wind_angular_momentum_loss == WIND_ANGMOM_LOSS_SPHERICALLY_SYMMETRIC)
    {
        /*  
         * CAB + SdM 2010
         *   Assume that this takes away the specific AM of the mass losing
         *   star.  
         *
         *   In case you want to improve this: This assumption may be alright
         *   for fast winds. For slow winds the AM loss is possibly
         *   considerably more if the wind can still interact with orbit. It
         *   may also feed some AM back into the orbit of the second
         *   star. see e.g Hurley et al thesis and Bonacic et al 2008 for an
         *   improved description.
         *
         * RGI:
         * The derivatives are positive. 
         * Why is the first term the wind loss - accretion *on the other star*?
         * Makes no sense to me!
         */
        double dm_windloss_0 = -stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS];
        double dm_windloss_1 = -stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS];
        double dm_windaccr_0 = stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN];
        double dm_windaccr_1 = stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN];
        
        double dm_lost_from_star_0 =  dm_windloss_0 - dm_windaccr_1;
        double dm_lost_from_star_1 =  dm_windloss_1 - dm_windaccr_0;


        djorb_dt = Jorb * 1.0/(mtot * stardata->star[0].mass * stardata->star[1].mass) 
            * ( dm_lost_from_star_0 * Mass_squared(1) + dm_lost_from_star_1 * Mass_squared(0));
    }
    else if(stardata->preferences->wind_angular_momentum_loss == WIND_ANGMOM_LOSS_LW ||
            stardata->preferences->wind_angular_momentum_loss == WIND_ANGMOM_LOSS_LW_HYBRID)
    {
        if(stardata->preferences->wind_angular_momentum_loss == WIND_ANGMOM_LOSS_LW)
        {
            /* 
             * dJorb = lw * dM * Jorb/(M1+M2)
             */
            djorb_dt = stardata->preferences->lw * mdot_system * Jorb / mtot;
        }
        else if(stardata->preferences->wind_angular_momentum_loss == WIND_ANGMOM_LOSS_LW_HYBRID)
        {
            // assume faster wind is donor (not always true!)
            double vw1 = Max(stardata->star[0].vwind,stardata->star[1].vwind)*1e-5;

            // <vorb>
            double vorb = 1e-5*sqrt ( GRAVITATIONAL_CONSTANT * 
                                      (stardata->star[0].mass+stardata->star[1].mass)*M_SUN/
                                      (stardata->common.orbit.separation*R_SUN));

            if(vw1 > vorb)
            {
                /* 
                 * fast wind : hence dJorb = lw * dM * Jorb/(M1+M2)
                 * is slow, assume conservative mass transfer (djorb_dt=0)
                 */
                djorb_dt = stardata->preferences->lw*mdot_system*Jorb/mtot;
            }
        }
    }
    return(-djorb_dt);
}
