#include "../binary_c.h"

/*
 * Update the mass/angular momenta of stars, also evolves circumbinary discs.
 *
 * NB if RLOF is unstable, its outcome is dealt with elsewhere
 */

#define OUTVAR stardata->star[1].phase_start_mass

int update_masses_angular_momenta_and_structure(struct stardata_t * RESTRICT const stardata,
                                                const Boolean RLOF)
{
    Dprint("Main loop var=%g: set kw1,kw2 dtm=%g\n",
           OUTVAR,stardata->model.dtm);
    Nancheck(stardata->common.orbit.angular_momentum);
    
    stardata->model.dt = 1e6 * stardata->model.dtm;
    
    /*
     * Can only do things if the timestep 
     * is non-zero. Often the stellar evolution
     * algorithm will set the timestep to zero
     * to update the structures without changing the 
     * masses.
     */
    if(Is_not_zero(stardata->model.dtm))
    {
        /*
         * calculate wind loss/gain rates and 
         * associated stellar/orbital derivatives
         */
        Dprint("Main loop: wind loss var=%g dtm=%g\n",OUTVAR,stardata->model.dtm);
        calc_wind_loss_and_gain_rates(stardata,RLOF);
        Dprint("Main loop: initial_mass_loss var=%g dtm=%g\n",OUTVAR,stardata->model.dtm);

        /*
         * Calculate RLOF rates: the stability of 
         * RLOF is determined prior to being here.
         */
        if(RLOF == TRUE)
        {
            Dprint("In RLOF : calculate derivatives");
            struct RLOF_orbit_t * RLOF_orbit = Malloc(sizeof(struct RLOF_orbit_t));
            RLOF_init_dM_orbit(stardata,RLOF_orbit);
            RLOF_stars;
            const double mass_transfer_rate = RLOF_mass_transfer_rate(donor->radius,
                                                                      donor->roche_radius,
                                                                      donor->mass,
                                                                      stardata,
                                                                      donor);
            
            Dprint("update masses etc. RLOF : rate = %g\n",mass_transfer_rate);
            RLOF_mass_transferred_in_one_orbit(stardata,
                                               mass_transfer_rate,
                                               RLOF_orbit);
            RLOF_orbit->dM_RLOF_transfer = -RLOF_orbit->dM_RLOF_lose;
            RLOF_orbit->dM_RLOF_accrete = RLOF_orbit->dM_RLOF_transfer; 
            RLOF_dM_orbit_to_time_derivatives(stardata,RLOF_orbit);

            if(0)printf("RLOF in one orbit lose=%g transfer=%g accrete=%g (wind =%g, %g; M1=%g M2=%g sep=%g)\n",
                        RLOF_orbit->dM_RLOF_lose,
                        RLOF_orbit->dM_RLOF_transfer,
                        RLOF_orbit->dM_RLOF_accrete,
                        RLOF_orbit->dM_other[0],
                        RLOF_orbit->dM_other[1],
                        donor->mass,
                        accretor->mass,
                        stardata->common.orbit.separation
                );
            Safe_free(RLOF_orbit);
            Dprint("post-RLOF\n");
        }
    
        Dprint("Main loop: update_mass_spin_epoch var=%g dtm=%g\n",OUTVAR,stardata->model.dtm);

        /* limit accretion rates (also novae and updates stellar types) */
        limit_accretion_rates(stardata);
        Dprint("post-limit\n");
        
        /* Save wind accretion luminosities */
        set_wind_accretion_luminosities(stardata);
        Dprint("post-wind accretion luminosities\n"); 
        
        /* Calculate angular momentum and eccentricity derivatives */ 
        angular_momentum_and_eccentricity_derivatives(stardata,RLOF);
        Dprint("post angmom and ecc derivs\n");
    }
    else if(RLOF == TRUE && Is_zero(stardata->model.dtm))
    {
        Dprint("RLOF and dtm!=0\n");
        /*
         * If in RLOF, but with a timestep of zero (e.g. post-CE),
         * we still require the next timestep to be non-zero otherwise
         * the evolution algorithm will lock up.
         */
        stardata->model.dtm = RLOF_MINIMUM_TIMESTEP;
    }

#ifdef DISCS
    /*
     * Evolve discs
     */
    evolve_discs(stardata);
#endif
    
    /* apply stellar derivatives (which may be zero) */
    apply_stellar_mass_and_angular_momentum_derivatives(stardata,RLOF);
    Dprint("post apply stellar\n");
    
    /* apply other derivatives (which also may be zero) */
    apply_other_stellar_derivatives(stardata,RLOF);
        
    /* apply orbital derivatives */
    Dprint("Main loop: update and check ecc var=%g, dtm=%g\n",OUTVAR,stardata->model.dtm);
    Nancheck(stardata->common.orbit.angular_momentum);
    const Boolean retval =
        apply_orbital_angular_momentum_and_eccentricity_derivatives(stardata,
                                                                    RLOF);

    Dprint("post-apply orb and ecc : J=%g NaN? %d\n",
           stardata->common.orbit.angular_momentum,
           isnan(stardata->common.orbit.angular_momentum));
    Nancheck(stardata->common.orbit.angular_momentum);
    
    return retval;
}
