#include "../binary_c.h"
#include "tides.h"


/*
 * This function calculates the derivatives of 
 * stellar and orbital angular momentum, as well 
 * as orbital eccentricity, because of tidal interactions
 */

void tides(struct stardata_t * const stardata,
           const Boolean RLOF_boolean,
           const Star_number k,
           const double radius,
           struct star_t * const star,
           struct star_t * const companion)
{
    double f2,f3,f4,f5,raa2,raa6,tc,f,tcqr,rg2,ttid;
    double cq     = companion->q;
    double ecc2   = ECC_SQUARED;
    double omecc2 = 1.0 - ecc2; // 1-e^2
    double sqome2 = sqrt(omecc2); // sqrt(1-e^2)
    double sqome3 = Pow3(sqome2); // (1-e^2)^(3/2)
    double sq6    = Pow3(omecc2); // (1-e^2)^3
    double isqome2_13 = 1.0/(sq6*sq6*sqome2);
    double starI  = moment_of_inertia(star,radius);
    const int ndonor MAYBE_UNUSED = stardata->model.ndonor;
    
    /*
     * Calculate tidal effects : circularization, orbital changes and spin up.
     */
    Dprint("Calculate circularization/shrinkage/spinup (tides) %d\n",DO_TIDES);

#ifdef IMPROVED_EULER_TIDES
    Boolean use_improved_Euler =
        Boolean_(stardata->previous_stardata!=NULL &&
                 _System_is_single(stardata) == _System_is_single(stardata->previous_stardata));
#endif
    
    if(DO_TIDES)
    {
        raa2 = Pow2(radius/stardata->common.orbit.separation);
        raa6 = Pow3(raa2);

        /*
         * Hut's polynomials.
         */
        if(Is_really_zero(ecc2))
        {
            f5 = f4 = f3 = f2 = 1.0;
        }
        else
        {
            f5 = 1.0+ecc2*(3.0+ecc2*0.3750);
            f4 = 1.0+ecc2*(1.50+ecc2*0.1250);
            f3 = 1.0+ecc2*(3.750+ecc2*(1.8750+ecc2*7.8125e-02));
            f2 = 1.0+ecc2*(7.50+ecc2*(5.6250+ecc2*0.31250));
        }
        /*
         * f1 is not required
         * 
         * f1 = 1.0+ecc2*(15.50+ecc2*(31.8750+ecc2*(11.56250+ecc2*0.3906250)));
         */

        Dprint("Hut polynomials: f2=%g f3=%g f4=%g f5=%g\n",f2,f3,f4,f5);

	/*
	printf("Tides : radiative? %d : convective? %d (pms? %d age=%g pmst=%g CONVECTIVE_PMS=%d RADIATIVE_MAIN_SEQUENCE=%d)\n",
	       USE_RADIATIVE_DAMPING,
	       !USE_RADIATIVE_DAMPING && USE_CONVECTIVE_DAMPING,
	       stardata->preferences->pre_main_sequence,
	       star->age*1e6,
	       preMS_lifetime(star->mass),
	       CONVECTIVE_PMS,
	       RADIATIVE_MAIN_SEQUENCE
	    );
	*/
	
        if(USE_RADIATIVE_DAMPING)
        {
            /*
             * Radiative damping (Zahn, 1977, A&A, 57, 383 and 1975, A&A, 41, 329).
             */
            f = 1.9782e+04*radius*sqrt(star->mass/Pow5(stardata->common.orbit.separation))*
                E2(stardata,star)*pow(1.0+cq,5.0/6.0);

            tcqr = f*cq*raa6;
            Dprint("tcqr = %g * %g * %g = %g\n",
                   f,cq,raa6,tcqr);
            rg2 = star->k2;
            Dprint("Zahn/Izzard radiative damping tcqr=%g rg2=%g\n",tcqr,rg2);
        }
        else if(USE_CONVECTIVE_DAMPING)
        {
            /*
             * Convective damping (Hut, 1981, A&A, 99, 126).
             */

            if(RLOF_boolean==TRUE) star->renv = Max(TINY,Min(star->renv,radius-star->core_radius));

            tc = MR23YR*pow((star->menv*star->renv*(radius-0.50*star->renv)/
                             (3.0*star->luminosity)),1.0/3.0);

            ttid = TWOPI/(1.0e-10 + fabs(stardata->common.orbit.angular_frequency - star->omega));
            
            /*
             * This is correct! It is wrong in Jarrod's BSE code.
             * DO NOT CHANGE IT (see MNRAS 329 897 eq 32)
             */

            double im=1.0/star->mass;
            double itc=1.0/tc;

            f = Pow2(ttid*itc*0.5);
            f = Min(1.0,f);
            tcqr = 2.0*f*cq*raa6*star->menv*im*itc*(1.0/21.0);
            rg2 = star->k2*Max(1e-30,Envelope_mass(k))*im;
            Dprint("Convective (Hut) damping tcqr=%g cq=%g rg2=%g raa6=%g k2=%g menv=%g Envelope_mass=%g im=%g itc=%g\n",
                   tcqr,cq,raa6,rg2,star->k2,star->menv,Envelope_mass(k),im,itc);
        }
        else
        {
            /*
             * Degenerate damping (Campbell, 1984, MNRAS, 207, 433)
             */
            f = 7.33e-09*pow(star->luminosity/star->mass,5.0/7.0);
            tcqr = f*Pow2(raa2*cq)/(1.0+cq);
            rg2 = K3;
            Dprint("Degenerate (Campbell) damping tcqr=%g rg2=%g\n",tcqr,rg2);
        }

        /*
         * Modify tidal timescale by strength factor
         */
        tcqr *= stardata->preferences->tidal_strength_factor;
          
        Dprint("Modulate tcqr by TIDAL_STRENGTH_FACTOR=%g -> tcqr=%g\n",
               stardata->preferences->tidal_strength_factor,tcqr);

        /*
         * Circularization.
         */
        star->derivative[DERIVATIVE_STELLAR_ECCENTRICITY] =
            -27.0*tcqr*(1.0+cq)*raa2*(stardata->common.orbit.eccentricity*isqome2_13)*
            (f3 - 0.61111111111*sqome3*f4*star->omega/
             stardata->common.orbit.angular_frequency) ;

        stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_TIDES] += star->derivative[DERIVATIVE_STELLAR_ECCENTRICITY];

        Dprint("TIDES %d at t=%g de/dt=%g from tcqr=%g cq=%g raa2=%g e=%g isqome2_13=%g f3=%g f4=%g omega*=%g omegaO=%g\n",
               star->starnum,
               stardata->model.time,
               stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_TIDES],
               tcqr,cq,raa2,
               stardata->common.orbit.eccentricity,
               isqome2_13,f3,f4,star->omega,stardata->common.orbit.angular_frequency
            );
        

        /* 
         * Save timescale 
         * (this will be calculated twice (for each star), and only the second
         *  time is it correct, i.e. for both stars)
         */
        stardata->common.orbit.tcirc = 

            /* no change -> long timescale : Note if e=0 then de/dt = 0 also */
            Is_zero(stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_TIDES]) ? LONG_TIDAL_TIME :

            /* estimate from timescale = e / de/dt */
            (Is_not_zero(stardata->common.orbit.eccentricity) ? 
             fabs(stardata->common.orbit.eccentricity/stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_TIDES]) : 0.0);
        
        /*
         * Spin up of star.
         */
        const double sqome3f5 = sqome3*f5;
        const double c = 3.0*cq*tcqr/(rg2*Pow6(omecc2))*sqome3f5;
        Dprint("c= %g * %g / (%g * %g) * %g = %g\n",
               3.0*cq,
               tcqr,
               rg2,
               Pow6(omecc2),
               sqome3f5,
               c);
        const double x = star->omega;
        const double y = f2*stardata->common.orbit.angular_frequency/sqome3f5;

//#undef EXPONENTIAL_TIDES
#ifdef EXPONENTIAL_TIDES
        /*
         * Solve the differential equation analytically, and use the derivative that
         * would have were it linear.
         *
         * This is similar to the method of Guderley and Hsu 1972
         * https://www.ams.org/journals/mcom/1972-26-117/S0025-5718-1972-0298952-7/
         * for solving stiff terms analytically
         */
        star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] =
            (y - x)/stardata->model.dt  * (1.0 - exp(- c * stardata->model.dt));
        
#else
        /*
         * Or assume a linear solution, as in BSE (tends to be unstable when tides are strong).
         */
        star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] =
            c * (y - x);
#endif//EXPONENTIAL_TIDES        

        Dprint("Spin up domega/dt = %g from c=%g cq=%g tcqr=%g rg2=%g omecc2=%g f2=%g sqome3f5=%g omega_orb=%g omega_star=%g\n",
               star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES],
               c,
               cq,
               tcqr,
               rg2,
               omecc2,
               f2,
               sqome3f5,
               stardata->common.orbit.angular_frequency,
               star->omega
            );
        Nancheck(star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES]);
        

        /* 
         * Calculate the equilibrium spin at which no angular momentum 
         * can be transferred. The stellar angular velocity derivative
         * cannot exceed the amount which would bring us to this equilibrium
         * value, because when omega = omega_eq we have d(omega)/dt = 0.
         */
        if(stardata->model.dt>TINY)
        {
            const double eqspin = stardata->common.orbit.angular_frequency*f2/sqome3f5;
            Dprint("eqspin = %g, omega = %g : domega/dt was %g\n",
                   eqspin,
                   star->omega,
                   star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES]);
            if(Is_not_zero(eqspin))
            {
                double delta_omega_dt = (eqspin - star->omega)/stardata->model.dt;
                double domega = stardata->model.dt * star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES];
                double new_omega = star->omega + domega;
                if(star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] > 0.0 &&
                   new_omega < eqspin)
                {
                    /*
                     * We're spinning up (domega/dt>0) and currently slower than eqspin
                     * (omega < eqspin) : domega/dt must be < delta_omega_dt
                     */
                    star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] = 
                        Min(star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES],
                            delta_omega_dt);
                }
                else if(star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] < 0.0 &&
                        new_omega > eqspin)
                {
                    /*
                     * We're spinning down (domega/dt<0) and currently faster than eqspin
                     * (omega > eqspin) : domega/dt must be > delta_omega_dt
                     */
                    star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] = 
                        Max(star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES],
                            delta_omega_dt);
                }
            }
            Dprint("eqspin = %g, domega/dt is now %g\n",
                   eqspin,
                   star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES]);

            /*
             * Do not allow omega to change sign because of tides : 
             * tides can never do this.
             */
            const double omega_estimate = star->omega + 
                star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] * stardata->model.dt;
            if(0 && star->omega * omega_estimate < 0.0)
            {
                /*
                 * Instead, assume omega = 0 on the next step
                 */
                Dprint("Truncate : omega = %g, domega/dt was %g (would make omega = %g) ",
                       star->omega,
                       star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES],
                       omega_estimate); 
                star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] = 
                    - star->omega / stardata->model.dt;
                Dprint("is now %g (c = %g, x = %g, y = %g, omega_orb = %g, linear rate %g)\n",
                       star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES],
                       c,
                       x,
                       y,
                       stardata->common.orbit.angular_frequency,
                       c*(y-x)
                    );
            }
        }

        /*
         * Calculate angular momentum changes: what is taken from the star is
         * put into the orbit, and vice versa
         */
        double dj_dt = starI * star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES];

        star->derivative[DERIVATIVE_STELLAR_ANGMOM_TIDES] = +dj_dt;
        stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_TIDES] -= dj_dt;
        double tspin =  fabs(star->omega/star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES]);
        
        Dprint("Jdot%d Calc djt from starI=%g domega/dt=%g (omega now %g, estimate next omega %g): dj/dt=%g (J now %g, estimate next J %g) : tcirc = %g, tspin = %g\n",
               star->starnum,
               starI,
               star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES],
               star->omega,
               star->omega + star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] * stardata->model.dt,
               dj_dt,
               star->angular_momentum,
               star->angular_momentum + dj_dt * stardata->model.dt,
               stardata->common.orbit.tcirc,
               tspin
            );
        
        star->tide_spin_lock =
            Boolean_(tspin < stardata->preferences->minimum_timestep);

        if(0 && star->tide_spin_lock)
        {
            /*
             * Lock tides
             */
            star->derivative[DERIVATIVE_STELLAR_ANGMOM_TIDES] = 0.0;
            star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] = 0.0;
        }
    }
}
