#include "../binary_c.h"

/*
 * Function to check if the star has overflowed its Roche lobe, and
 * take appropriate action to interpolate back to the time where it
 * did.
 *
 * returns:
 * ROCHE_OVERFLOW_NONE  : there is no RLOF
 * ROCHE_OVERFLOW : indicates that we are just starting RLOF
 * ROCHE_OVERFLOW_AFTER_SPIRAL_IN : indicates that RLOF will start in the next timestep after some spiral in
 */

unsigned int test_for_roche_lobe_overflow(struct stardata_t * RESTRICT const stardata)
{
    unsigned int retval;

    if(!System_is_binary)
    {
        /*
         * If we're a single-star system, we cannot have RLOF
         */
        retval = ROCHE_OVERFLOW_NONE;
    }
    else
    {
        /*
         * Set model->ndonor    to the donor, 
         *     model->naccretor to accretor
         */
        set_ndonor_and_naccretor(stardata);
        RLOF_stars;

        if(stardata->preferences->RLOF_interpolation_method == 
           RLOF_INTERPOLATION_BINARY_C)
        {
            /*
             * Check for one star overflowing
             */
            Dprint("intermediate? %s, in_RLOF = %s, overflowing? %s [R0/RL0 = %g, R1/RL1 = %g] spiral time %g vs mindt %g\n",
                   Yesno(stardata->model.intermediate_step),
                   Yesno(stardata->model.in_RLOF),
                   Yesno(RLOF_overflowing(stardata,RLOF_ENTRY_THRESHOLD)),
                   stardata->star[0].radius / stardata->star[0].roche_radius,
                   stardata->star[1].radius / stardata->star[1].roche_radius,
                   spiral_in_time(stardata),
                   stardata->preferences->minimum_timestep);

            Boolean spiralin = FALSE;
            if(RLOF_overflowing(stardata,1.0))
            {
                /* 
                 * We are overflowing a Roche lobe
                 */
                retval = ROCHE_OVERFLOW;
            }
            else if(spiral_in_time(stardata) < stardata->preferences->minimum_timestep)
            {
                /*
                 * special case : compact object RLOF, which isn't time
                 * resolved, should occur in this timestep. Catch it 
                 * now before contact happens.
                 *
                 * The donor should be "less evolved" than the accretor,
                 * if not swap the stars.
                 */
                if(COMPACT_OBJECT(donor->stellar_type)&&
                   COMPACT_OBJECT(accretor->stellar_type)&&
                   donor->stellar_type > accretor->stellar_type)
                {
                    Swap_stars(donor,accretor);
                }

                retval = ROCHE_OVERFLOW_AFTER_SPIRAL_IN;
                spiralin = TRUE;
                Dprint("RLOF spiral in time %g : min dt %g\n",
                       spiral_in_time(stardata),
                       stardata->preferences->minimum_timestep);
            }
            else
            {
                /* no RLOF */
                retval = ROCHE_OVERFLOW_NONE;
            }
            Dprint("RLOF check t = %20.12e J = %20.12e, a %20.12e : 0-> %d M=%g Mc=%g R=%g RL=%g : 1-> %d M=%g Mc=%g R=%g RL=%g : q %g %g : rlf %g %g -> %s %s\n",
                   stardata->model.time,
                   stardata->common.orbit.angular_momentum,
                   stardata->common.orbit.separation,
                   stardata->star[0].stellar_type,
                   stardata->star[0].mass,
                   stardata->star[0].core_mass,
                   stardata->star[0].radius,
                   stardata->star[0].roche_radius,
                   stardata->star[1].stellar_type,
                   stardata->star[1].mass,
                   stardata->star[1].core_mass,
                   stardata->star[1].radius,
                   stardata->star[1].roche_radius,
                   (stardata->star[0].q),
                   (stardata->star[1].q),
                   rl(stardata->star[0].q),
                   rl(stardata->star[1].q),
                   retval == REJECT_EVOLUTION ? "reject" :
                   retval == ROCHE_OVERFLOW ? "RLOF" :
                   retval == ROCHE_OVERFLOW_NONE ? "no RLOF" :
                   "unknown",
                   spiralin == TRUE ? "spiralin" : ""
                );
        }
        else
        {
            /*
             * Use the BSE method to interpolate R to RL
             */
            retval = interpolate_R_to_RL(stardata);
        }
    }

    return retval;
}
