#include "../binary_c.h"

/*
 * Function to detect stellar spin beyond the breakup velocity
 *
 * The in_RLOF boolean activates slightly different code in the case
 * we are in RLOF (e.g. the accretion prevention code)
 *
 * Note: 
 * You can CHANGE derivative[ORBIT_ANGMOM]
 * and 
 * You can SET the stellar excretion disc derivative
 * but 
 * do NOT set or change the STELLAR_ANGMOM derivative
 *
 * Returns TRUE if overspin occurs, FALSE otherwise. 
 */
Boolean prevent_overspin(struct star_t * const star,
                         struct stardata_t * const stardata,
                         const Boolean in_RLOF)
{
#ifdef NANCHECKS
    if(isnan(star->angular_momentum) || isnan(star->omega))
    {
        Backtrace;
        Exit_binary_c(BINARY_C_EXIT_NAN,
                      "NAN at entrance to prevent_overspin : star %d : Jspin=%g omega=%g \n",
                      star->angular_momentum,
                      star->omega
            ); 
    }
#endif
    Boolean overspin;
    
    /* timestep */
    double dt = in_RLOF==TRUE ? (stardata->model.dtm*1e6) : stardata->model.dt;
    
    Dprint("star %d, in RLOF? %d (R=%g, RL=%g) -> dt = %g; init J = %g\n",
           star->starnum,
           in_RLOF,
           star->radius,
           star->roche_radius,
           dt,
           star->angular_momentum);


    if(
        /* we can do nothing if the timestep is zero */
        Less_or_equal(dt,0.0) ||
        stardata->model.intpol>0 ||
        /* we can do nothing if we have no spin */
        Is_zero(star->omega)
        ) return FALSE; 

    double jcrit = breakup_angular_momentum(star);
    
    /* 
     * Calculate the angular momentum the star would 
     * like to have if all the previously calculated angular
     * momentum derivatives were applied.
     */
    double jstar = Max(MINIMUM_STELLAR_ANGMOM,
                       star->angular_momentum + Jdot_net(star)*dt);
    
    /*
     * Detect overspin
     */
    const double j_excess = jstar - jcrit;
    overspin = jstar > jcrit ? TRUE : FALSE;

    Dprint("Star %d Jcrit = %g, Jstar = %g + %g * %g = %g : overspin? %s\n",
           star->starnum,
           jcrit,
           star->angular_momentum,
           Jdot_net(star),
           dt,
           jstar,
           Yesno(overspin));
    
    if(overspin == TRUE)
    {
        /*
         * Star is spinning too fast : determine what we should do
         */
        if(stardata->preferences->overspin_algorithm == OVERSPIN_BSE)
        {
            /*
             * BSE algorithm returns excess angular momentum to the orbit
             *
             * This is equivalent to saying the material forms a disc
             * around the accreting star, and this disc is tidally coupled 
             * to the stars, hence can "give back" its angular momentum.
             *
             * It also assumes no mass is lost from the disc.
             */
            stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM] += j_excess/dt;
            star->derivative[DERIVATIVE_STELLAR_ANGMOM_TIDES] -= j_excess/dt;
        }
        else if(stardata->preferences->overspin_algorithm == OVERSPIN_MASSLOSS)
        {
            /*
             * Remove excess angular momentum by mass loss,
             * e.g. a decretion disc
             */
    
            if(stardata->preferences->rotationally_enhanced_mass_loss==ROTATIONALLY_ENHNACED_MASSLOSS_ANGMOM ||
               stardata->preferences->rotationally_enhanced_mass_loss==ROTATIONALLY_ENHNACED_MASSLOSS_LANGER_FORMULA_AND_ANGMOM)
            {
                
                /* 
                 * Calculate specific angular momentum of accreted material:
                 * assume keplerian disk 
                 **** Update me : use accreted specific angular momentum!
                 */
                double r = star->radius > star->roche_radius ? star->effective_radius : star->radius;
                double l_kep = Pow2(r) * star->omega_crit;

                /* calculate mass to be ejected */
                double m_eject_required = j_excess / l_kep ;
                double m_eject = m_eject_required;

                Dprint("Star %d : J excess = %g : r = %g, l_kep = %g,  must eject %g Msun\n",
                       star->starnum,j_excess,r,l_kep,m_eject_required);

                /*
                 * Limit m_eject to the 2% of the envelope mass 
                 * in any one timestep. In an ideal time-evolution 
                 * scheme, this would not be required. In theory we
                 * could reject the stellar evolution step and redo
                 * with a shorter timestep, but is this really necessary?
                 */
                m_eject = 
                    Min(m_eject,
                        0.02*
                        (star->mass - star->core_mass));
        
                /*
                 * Eject material in a decretion disc.
                 *
                 * Reduce the star's angular momentum by dj_eject. 
                 * This is only < dj if the 2% envelope limit is reached.
                 */
                double dj_eject = j_excess * m_eject / m_eject_required;

                Dprint("Actually ejecting %g Msun, with dJ = %g\n",
                       m_eject,dj_eject);

                /* rates */
                double mdot = -  m_eject / dt;
                double jdot = - dj_eject / dt;

                /* remove mass and angular momentum from the star */
                star->derivative[DERIVATIVE_STELLAR_MASS_DECRETION_DISC]   += mdot;
                star->derivative[DERIVATIVE_STELLAR_ANGMOM_DECRETION_DISC] += jdot;

                /*
                 * Use gamma passed in, just as we would with any other
                 * non-conservative mass loss 
                 */
                double gamma = non_conservative_gamma(stardata,
                                                      Other_star_struct(star),
                                                      star);

                double Lorb = Angular_momentum_from_stardata / Total_mass;
                double jdot1 = mdot * Lorb * gamma;
        
                Dprint("Remove Jorb at rate = %g\n",jdot1);

                /*
                 * This is "the same" based on Carlo's wind/angmom mass loss
                 * for only one star. Gives similar, but not identical, results.
                 * I do not (yet) know why.
                 */
                /*
                  double mtot = stardata->star[0].mass + stardata->star[1].mass;
                  double Jorb = Pow2(stardata->common.orbit.separation)*
                  stardata->common.orbit.angular_frequency*
                  Sqrt_eccfac*stardata->star[0].mass*stardata->star[1].mass/mtot;
                  double jdot2 =
                  Jorb * 1.0/(mtot * stardata->star[0].mass * stardata->star[1].mass) 
                  * ( mdot * Mass_squared(1));

                  printf("jdot 1 = %g, 2 = %g\n",jdot1,jdot2);
                */

                stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_NONCONSERVATIVE_LOSS] +=
                    jdot1;
                stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM] += jdot1;
        
                Dprint("Star %d rotates > breakup : j_accrete=%g j_excess=%g l_kep=%g m_eject=%g new J/Jcrit = %g : mdot noncon %g\n",
                       star->starnum,
                       jdot,
                       j_excess,
                       l_kep,
                       m_eject,
                       (star->angular_momentum - dj_eject)/jcrit,
                       stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_NONCONSERVATIVE_LOSS]
                    );        
            }
        }
    }
    return overspin;
}

