
#include "../binary_c.h"

#ifdef BSE

//#undef DEBUG
//#define DEBUG 1
//#define MERGED_STELLAR_TYPE_WARNING

#ifdef NUCSYN

/*
 * the source of the envelope, and hence surface abundances, depends
 * on the stellar types (k1,k2) of the stars.
 */
#define ENVELOPE_SOURCE_STAR1 1
#define ENVELOPE_SOURCE_STAR2 2
#define ENVELOPE_SOURCE_BOTH 3
#define ENVELOPE_SOURCE_TZ 4

#endif //NUCSYN


void mix_stars_BSE(struct star_t * const stars, 
                   struct stardata_t * const stardata)
{
    /**************************************
     *
     *     Author : J. R. Hurley
     *     Date :   7th July 1998
     *
     *     Evolution parameters for mixed star.
     *     ------------------------------------
     *
     * C version, 27/01/2001, Rob Izzard
     *
     * Nucleosynthesis added, 2002/3 Rob Izzard
     */
    int merged_stellar_type;
    double m01,m02,m03,m1,m2,m3,age1,age2,age3;
    double MAYBE_UNUSED mc1,mc2,mc3;
    double tms1,tms2,tms3,tn,mcconv1,mcconv2,mcconv3,mcCO1,mcCO2,mcCO3;
#ifdef NUCSYN
    Abundance *X1,*X2;
    double menv1,menv2;


    /*
     * env_src tells us from where any ejected material comes
     *
     * 1 : star 1
     * 2 : star 2
     * 3 : both stars
     * 4 : Thorne-Zytkow object (treated differently) 
     */
    int env_src;
#endif
    double tscls[TSCLS_ARRAY_SIZE],lums[LUMS_ARRAY_SIZE],GB[GB_ARRAY_SIZE];

    Dprint("in mix k1=%d k2=%d\n",stars[0].stellar_type,stars[1].stellar_type);


    /*
     * Choose star numbers such that 
     * k1 = stellar_type(i1) >= k2 = stellar_type(i2)
     */
    const Star_number i1 = stars[0].stellar_type>=stars[1].stellar_type ? 0 : 1;
    const Star_number i2 = Other_star(i1);

    /*
     * Specify case index for collision treatment, with k1>k2.
     */  
    const Stellar_type k1 = stars[i1].stellar_type;
    const Stellar_type k2 = stars[i2].stellar_type;

    /* 
     * Stellar type of the merger star
     */
    merged_stellar_type = stardata->store->collision_matrix[k1][k2];
    Dprint("collision matrix[%d][%d] gave %d\n",
           k1,k2,
           merged_stellar_type);


#ifdef COWD_QUIET_MERGER
    if(k1==COWD && k2==COWD)
    {
        /* 
         * assume COWD-COWD merger ignites carbon 
         * and ejects the least massive WD
         */
        merged_stellar_type = ONeWD;        
    }
#endif  
            
#ifdef NS_BH_AIC_LOG
    /*
     * Do we count NSNS mergers in a BH as a NSNS event?
     * Say YES perhaps we should...
     */
    if((stars[i1].stellar_type==NEUTRON_STAR)&&
       (stars[i2].stellar_type==NEUTRON_STAR))
    {
        printf("NSNS p=%g t=%g in mix\n",stardata->model.probability,
               stardata->model.time);
        char c[STRING_LENGTH];
        sprintf(c,"NSNS at time t=%g\n",stardata->model.time);
        output_string_to_log(stardata,c);
    }
#endif

    // nucleosynthesis depends on k1 and k2
#ifdef NUCSYN
    Dprint("Star %d : m=%g, mc=%g, kw=%d\n",1,stars[0].mass,stars[0].core_mass,stars[0].stellar_type); 
    Dprint("Star %d : m=%g, mc=%g, kw=%d\n",2,stars[1].mass,stars[1].core_mass,stars[1].stellar_type); 

#ifdef NUCSYN_STRIP_AND_MIX
    mix_down_to_core(&(stardata->star[0]));
    mix_down_to_core(&(stardata->star[1]));
#endif

    /* 
     * Decide where the new envelope comes from
     */
    if((k1<=TPAGB && k2<=TPAGB)
        ||
        (NAKED_HELIUM_STAR(k1) && NAKED_HELIUM_STAR(k2))
        ||
        (WHITE_DWARF(k1) && WHITE_DWARF(k2))
        )
    {
        env_src = ENVELOPE_SOURCE_BOTH;
    }
    else if(k1>=NEUTRON_STAR && k2<NEUTRON_STAR) 
    {
        /*
         * Note: this assumes the Eddington limit applies
         * ... if it doesn't, you may get some other kind of object
         */
        env_src = ENVELOPE_SOURCE_TZ;
    }
    else if(k1>k2)
    {
        env_src = ENVELOPE_SOURCE_STAR2;
    }
    else
    {
        env_src = ENVELOPE_SOURCE_STAR1;
    }

    Dprint("env_src=%d : %s\n",
           env_src,
           (env_src == ENVELOPE_SOURCE_BOTH ? "both stars" :
           env_src == ENVELOPE_SOURCE_STAR1 ? "star 1" :
           env_src == ENVELOPE_SOURCE_STAR2 ? "star 2" :
            "Thorne-Zytkow object"));
           
#endif//NUCSYN

#ifdef MERGED_STELLAR_TYPE_WARNING
    if(merged_stellar_type>100)
    {
        fprintf(stderr,
                "warning : mix merged_stellar_type>100 (merged_stellar_type=%d, k1=%d, k2=%d)\n",
                merged_stellar_type,
                k1,
                k2);
    }
#endif

    /*
     * Determine evolution time scales for first star.
     */
    Dprint("Determine evolution time scales for first star.");
    m01 = stars[i1].mass;
    m1 = stars[i1].mass;
    mc1 = stars[i1].core_mass;
    mcconv1 = stars[i1].max_MS_core_mass;
    mcCO1 = stars[i1].CO_core_mass;
    age1 = stars[i1].age;
    stellar_timescales(k1,
                       m01,
                       m1,
                       &tms1,
                       &tn,
                       tscls,
                       lums,
                       GB,
                       stardata,
                       stars);

    /*
     * Obtain time scales for second star.
     */
    Dprint("Determine evolution time scales for second star.");
    m02 = stars[i2].mass;
    m2 = stars[i2].mass;
    mc2 = stars[i2].core_mass;
    mcconv2 = stars[i2].max_MS_core_mass;
    mcCO2 = stars[i2].CO_core_mass;
    age2 = stars[i2].age;
    stellar_timescales(k2,
                       m02,
                       m2,
                       &tms2,
                       &tn,
                       tscls,
                       lums,
                       GB,
                       stardata,
                       stars+1);                         
    Dprint("MIX IN m1=%g m2=%g (m1+m2=%g)\n",m1,m2,m1+m2);


#ifdef NUCSYN
    /*
     * Assume the merging event is enough to mix the envelope and 
     * accretion layer of each star (if an accretion layer exists)
     */
    nucsyn_mix_accretion_layer_and_envelope(stardata,
                                            &stars[i1],
                                            m1 - mc1);
    nucsyn_mix_accretion_layer_and_envelope(stardata,
                                            &stars[i2],
                                            m2 - mc2);

    /*
     * new envelope abundances
     */
    X1 = stars[i1].Xenv;
    X2 = stars[i2].Xenv;
    
    /*
     * Mix the two stars' envelopes according to env_src
     * Envelope masses are menv1 and menv2
     */
    if(env_src==ENVELOPE_SOURCE_STAR1)
    {
        /*
         * All the envelope comes from star i1, star i2 sinks
         * without mixing into the envelope
         * the envelope on i1 is already on i1 so we don't have to do
         * anything
         */
    }
    else if(env_src==ENVELOPE_SOURCE_STAR2)
    {
        /*
         * All the envelope comes from star i2
         */
        Copy_abundances(X2,stars[i1].Xenv);
    }
    else if(env_src==ENVELOPE_SOURCE_BOTH)
    {
        /*
         * All the envelope comes from either
         * 1) the completely mixed envelopes or
         * 2) in the case where a star is a WD, the mixture of
         * the WD and the envelope of the other star or
         * 3) complete mixing in the case where neither star
         * has a core (in this case the envelope is 
         */
        if(WHITE_DWARF(k2))
        {
            /*
             * for WD mixing into the envelope treat all WD as envelope
             */
            menv2=m2;
        }
        else
        {
            /*
             * otherwise the envelope mass is the difference between
             * the total mass and the core mass. In the case of MS stars
             * this is the total mass because the core mass is zero.
             */
            menv2=m2-stars[i2].core_mass;
        }
        
        if(WHITE_DWARF(k1))
        {
            menv1=m1;
        }
        else
        {
            menv1=m1-stars[i1].core_mass;
        }

        nucsyn_dilute_shell(menv1,
                            X1,
                            menv2,
                            X2);

    }
    else if(env_src==ENVELOPE_SOURCE_TZ)
    {
        /* Thorne-Zytkow object */
        if(k2 < NEUTRON_STAR)
        {
            /* 
             * Star i2 is totally lost to the ISM 
             */
            Dprint("calc yields T-Z object mix i2=%d\n",i2);
            struct star_t * tzo = &stardata->star[i2];
            nucsyn_TZ_surface_abundances(tzo);
            nucsyn_calc_yields(stardata,
                               tzo,
                               m2,
                               tzo->Xenv,
                               0.0,
                               NULL,
                               i2,
                               YIELD_NOT_FINAL,
                               NUCSYN_SOURCE_THORNE_ZYTKOW);
        }
    }
    else
    {
        /*
         * all other cases, completely mix the stars
         */
        nucsyn_dilute_shell(m1,
                            X1,
                            m2,
                            X2);
    }
#endif // NUCSYN

    /* 
     * Check for planetary systems - defined as HeWDs and low-mass WDs!
     */
    Dprint("Planet check: merged_stellar_type=%d\n",merged_stellar_type);
    if(k1==HeWD && m1<0.05)
    {
        merged_stellar_type = k2;
        if(k2<=MAIN_SEQUENCE)
        {
            merged_stellar_type = MAIN_SEQUENCE;
            age1 = 0.0;
        }
    }
    else if(k1>=COWD && merged_stellar_type==TPAGB && m1<0.5)
    {
        merged_stellar_type = HeGB;
    }
    if(k2==HeWD && m2<0.05)
    {
        merged_stellar_type = k1;
        if(k1 <= MAIN_SEQUENCE)
        {
            merged_stellar_type = MAIN_SEQUENCE;
            age2 = 0.0;
        }
    }

    if(merged_stellar_type >= 100)
    {
        Dprint("Forced merger instead of comenv (ok if envelope mass is small)\n");
        merged_stellar_type-=100;
    }

    /*
     * Specify total mass of the merged object.
     */
    if(POST_SN_OBJECT(merged_stellar_type) &&
       !POST_SN_OBJECT(k2))
    {
        /*
         * NS/BH merging with a less compact object.
         *
         * Assume the least compact object is ejected,
         * usually through a Thorne-Zytkow phase.
         */
        m3 = m1;
    }
    else
    {
        /*
         * Assume the two stars just merge.
         */
        m3 = m1 + m2;
    }

    m03 = m01 + m02;
    age3 = 0.0;

    Dprint("merged_stellar_type = %d with total mass %g\n",
           merged_stellar_type,
           m3);
    
    /*
     * Evaluate apparent age and other parameters
     * depending on stellar type.
     * Also log and set up supernovae.
     */    
    if(merged_stellar_type <= MAIN_SEQUENCE)
    {
        /*  
         * Specify new age based on complete mixing.
         */
        if(k1==HeMS)
        {
            merged_stellar_type = HeMS;
        }
        mc3 = 0.0;
        mcCO3 = 0.0;
        mcconv3 = 0.0; //todo
        Dprint("Merged MS star timescales");
        stellar_timescales(merged_stellar_type,
                           m03,
                           m3,
                           &tms3,
                           &tn,
                           tscls,
                           lums,
                           GB,
                           stardata,
                           stars+0);
        
#ifdef SELMA_BETTER_TREATMENT_OF_MS_MERGERS
        //double age3_settling_of_cores, age3_completemixing, age3_oldimplementation;
        double fracMSage_moreEvolvedStar, fracMSage_lessEvolvedStar;    
        double mass_moreEvolvedStar, mass_lessEvolvedStar;
        double mcore_moreEvolvedStar, mcore_lessEvolvedStar, mcore_merger;

        //Assume that a fraction f_massloss_during_merger       

        m03 *= (1.0-stardata->preferences->f_massloss_during_MS_mergers);
        m3 *= (1.0-stardata->preferences->f_massloss_during_MS_mergers);

        /*
         * Check which star is the most evolved star (can be the
         * secondary, for example systems that experienced slow case A
         * mass transfer )
         */
        if (age1/tms1 > age2/tms2)
        {         
            fracMSage_moreEvolvedStar = age1/tms1;
            fracMSage_lessEvolvedStar = age2/tms2;
            mass_moreEvolvedStar = m1;
            mass_lessEvolvedStar = m2;
        }
        else
        {
            fracMSage_moreEvolvedStar = age2/tms2;
            fracMSage_lessEvolvedStar = age1/tms1;
            mass_moreEvolvedStar = m2;
            mass_lessEvolvedStar = m1;
        }

        /*
         * Get the size of the convective core at this moment. 
         *
         * As a proxy I currently use the size that the core will have
         * at the end of the mains sequence / beginning of the
         * Hertzsprung gap.  This ignores the effect that the
         * convective core is typically larger at zero age and slowly
         * retreats.
         */
        mcore_moreEvolvedStar =  
            effective_core_mass_fraction_of_MS_stars(mass_moreEvolvedStar, 
                                                      stardata->common.metallicity);
        
        mcore_lessEvolvedStar = 
            effective_core_mass_fraction_of_MS_stars(mass_lessEvolvedStar, 
                                                      stardata->common.metallicity);
  
        mcore_merger = effective_core_mass_fraction_of_MS_stars(m3, 
                                                                 stardata->common.metallicity);
        

        /*
         * Assume that the core of the most evolved star settles in
         * the center of the merger and that the other core will form
         * the layers on top of that. 
         *
         * Then assume that the core of the merger produces will be
         * convective and mixes. Also assume that the relative age is
         * an indicator of the helium abundance in the core,
         * i.e. age/tms ~ Xhe_center and Xhe_center = helium present
         * in layer of the original star that will become part of the
         * new core / mcore_3
         *
         * Caveats: (I) in reality there may be more helium present in
         * the progenitor stars because we underestimate the current
         * core size, and their may be helium present in layers above
         * the core (II) This may in particular the case for the
         * primary star in slow case A mergers. 
         * 
         */
                  
        /*
         * This parameter regulates an option for extra mixing of
         * the merger product. The new (convective) core is always
         * assumed to be mixed. If [mixing_in_MS_mergers > 0] an
         * extra fraction of the H-rich of the envelope of the
         * merger is mixed with the core. [mixing_in_MS_mergers = 1]
         * implies that the entire envelope (after mass loss during
         * the merger itself is mixed).
         */
        double mass_mixed_region = mcore_merger;
        if (stardata->preferences->mixingpar_MS_mergers > 0.0) 
        {
            mass_mixed_region += (1.0-mcore_merger)*stardata->preferences->mixingpar_MS_mergers;
        }
                
        if (mass_mixed_region > mcore_moreEvolvedStar + mcore_lessEvolvedStar) 
        {  
            /* new core contains both progenitor cores and some fresh material */
            age3 = tms3 * (  mcore_moreEvolvedStar * fracMSage_moreEvolvedStar +  
                             mcore_lessEvolvedStar * fracMSage_lessEvolvedStar ) / mass_mixed_region ;
        }
        /*
         * If a high fraction of mass is lost during the merger, the
         * new mixed region might be smaller than the original
         * cores. Assume that the core of the most evolved star sinks
         * to the center. (This might not be true for very extreme
         * mass ratios, where the entropy of the core of the companion
         * is lower than the evolved primary.) Evert Glebbeek find
         * that this may happen for mass ratioas < 0.1 (in models
         * without over shooting). I find that the mass ratios should
         * be more extreme, based on models with overshooting.  In
         * other words, these are binaries that I do not even consider
         * in the population synthesis.
         */
        else if (mass_mixed_region > mcore_moreEvolvedStar)
        {  
            /*
             * new core contains core of more evolved star 
             * plus part of the core of the less evolved star
             */
            age3 = tms3 * ( mcore_moreEvolvedStar * fracMSage_moreEvolvedStar +  
                            (mass_mixed_region-mcore_moreEvolvedStar)*fracMSage_lessEvolvedStar) 
                /mass_mixed_region;
        }
        else
        {  
            /*
             * new core only contains material from the
             * core of the more evolved star (i.e. have the same relative age)
             */ 
            age3 = tms3 * fracMSage_moreEvolvedStar;
        }
        
#else
        /*
         * Old implementation (not recommended).  This assumes that 
         * - core helium fraction changes linearly with time
         * - Both stars have core mass fractions of 0.1 
         *   (severely underestimating it for high mass stars)
         * - The helium present in the cores of the original stars 
         *   is mixed throughout the merger
         *
         * This assuption leads to highly  rejuvenated stars, in most cases setting the 
         * relative age of the merger to just a few percent. 
         * SdM
         */
        age3 = 0.10*tms3*(age1*m1/tms1 + age2*m2/tms2)/m3;
#endif //SELMA_BETTER_TREATMENT_OF_MS_MERGERS
    }
    else if(GIANT_LIKE_STAR(merged_stellar_type)) 
    {
        //mc3 = mc1 + mc2;
        mc3 = m1;
        mcCO3 = mcCO1 + mcCO2;
        mcconv3 = mcconv1 + mcconv2;
        
        if(merged_stellar_type == CHeB)
        {
            age3 = age1/tms1;
        }
        giant_age(&mc3,
                  m3,
                  &merged_stellar_type,
                  &m03,
                  &age3,
                  stardata,
                  &(stardata->star[i1]));
        Dprint("post-gntage : %d\n",merged_stellar_type);
    }
    else if(merged_stellar_type==HeMS)
    {
        mc3 = 0.0;
        mcCO3 = 0.0;
        mcconv3 = 0.0; // todo
        Dprint("Merged HeMS star timescales");
        stellar_timescales(merged_stellar_type,
                           m03,
                           m3,
                           &tms3,
                           &tn,
                           tscls,
                           lums,
                           GB,
                           stardata,
                           stars+0);
        age3 = age2 * tms3/tms2 * m2/m3;
    }
    else if(WHITE_DWARF(merged_stellar_type))
    {
        /*
         * Ensure that a new WD has the initial mass set correctly.
         */

        if(stardata->preferences->WDWD_merger_algorithm == WDWD_MERGER_ALGORITHM_BSE)
        {
            /*
             * BSE algorithm of Hurley et al. (2002)
             */
            m03 = m3;
            mc3 = m3;
            mcCO3 = mcCO1 + mcCO2;
            mcconv3 = 0.0;

#ifdef COWD_QUIET_MERGER
            if(k1==COWD && k2==COWD)
            {
                /*
                 * Assume that the accreting COWD ignites carbon
                 * and is blown off in the 'explosion'
                 */
                Boolean kmax = m1>m2 ? FALSE : TRUE;
                m3 = kmax==FALSE ? m1 : m2;
                m03 = kmax==FALSE ? m01 : m02;
            }
#endif
            if(merged_stellar_type<ONeWD &&
               More_or_equal(m3,stardata->preferences->chandrasekhar_mass))
            {
                m3 = 0.0;
                /* 
                 * M > MCh SNIa
                 */
                Dprint("Yield SNIa m=%g\n",m1+m2);
                stars[0].SN_type = SN_IA_CHAND_Coal;
                struct star_t * news = 
                    new_supernova(stardata,&stars[0],&stars[1],NULL);
                if(news)
                {
                    stellar_structure_make_massless_remnant(news);
                }
            }
        
        }
        else if(stardata->preferences->WDWD_merger_algorithm == WDWD_MERGER_ALGORITHM_PERETS2019)
        {
            /*
             * Perets' algorithm (2019)
             */
            m03 = m3;
            mc3 = m3;
            mcCO3 = mcCO1 + mcCO2;
            mcconv3 = 0.0;

            if(
                /* both must be COWDs */
                stars[0].stellar_type == COWD &&
                stars[1].stellar_type == COWD &&
                /* one must be a hybrid */
               (stars[1].hybrid_HeCOWD == TRUE ||
                stars[1].hybrid_HeCOWD == TRUE) &&
                /* and the merged mass must be 0.9<M<1.9 */
                m3 > 0.9 &&
                m3 < 1.9
                // total M > 1.2 -> normal Ia
                // < 1.2 subluminous Ia
                // need the other star to exceed the hybrid mass to make a disc
                )
            {
                Dprint("Yield SNIa m=%g\n",m1+m2);

                /*
                 * Hagai says: 
                 * If M>1.2Msun we get a Ia.
                 * Around 1.1-1.2 it's in between.  
                 * If M<1.1Msun we get a subluminous Ia.
                 *
                 * We may need the mass of the non-hybrid star to exceed that
                 * of the hybrid, but for now ignore this condition.
                 */
                if(m3 < 1.2)
                {
                    stars[0].SN_type = SN_IA_Hybrid_HeCOWD_subluminous;
                }
                else
                {
                    stars[0].SN_type = SN_IA_Hybrid_HeCOWD;
                }
                struct star_t * news = 
                    new_supernova(stardata,&stars[0],&stars[1],NULL);
                if(news)
                {
                    /* explosion is complete, leave nothing behind */
                    stellar_structure_make_massless_remnant(news);
                }
            }
               

        }
        else if(stardata->preferences->WDWD_merger_algorithm == WDWD_MERGER_ALGORITHM_CHEN2016)
        {
            /*
             * Chen's algorithm (2016)
             */
            /* TODO */
            mcCO3 = mcCO1 + mcCO2; // prevent compiler warning
            mcconv3 = 0.0; // prevent compiler warning
            m03 = m3; // prevent compiler warning
            mc3 = m3; // prevent compiler warning

            Exit_binary_c(BINARY_C_ALGORITHM_OUT_OF_RANGE,
                          "WDWD merger algorithm %d of Chen 2016 not yet implemented\n",
                          stardata->preferences->WDWD_merger_algorithm);
      
        }
        else
        {
            mcCO3 = mcCO1 + mcCO2; // prevent compiler warning
            mcconv3 = 0.0; // prevent compiler warning
            m03 = m3; // prevent compiler warning
            mc3 = m3; // prevent compiler warning
            Exit_binary_c(BINARY_C_ALGORITHM_OUT_OF_RANGE,
                          "Unknown WDWD merger algorithm %d\n",
                          stardata->preferences->WDWD_merger_algorithm);
        }
    }
    else if(POST_SN_OBJECT(merged_stellar_type))
    {
        /* 
         * Merger of two stars to give a NS or BH.
         *
         * We log NSNS/NSBH/BHBH as "supernovae" just in case
         * you ever want to kick them or model their "light" curves.
         * 
         * We also trap Thorne-Zytkow object formation.
         */

        /*
         * First, if the new neutron star exceeds the 
         * maximum mass limit, convert to a black hole.
         */
        if(merged_stellar_type==NEUTRON_STAR &&
           m3 > stardata->preferences->max_neutron_star_mass)
        { 
            merged_stellar_type = BLACK_HOLE;
        }

        Dprint("Final NS/BH object of stellar type %d merged with k2=%d",
               merged_stellar_type,k2);

        if(k2 <= ONeWD)
        {
            /*
             * Thorne-Zytkow object when a NS or BH 
             * mixes with a less dense star.
             * 
             * Assume fast mass loss of the "envelope" hence
             * we are left with the NS or BH afterwards,
             * even in the case of a WD merging with a NS/BH.
             */
#ifdef ANTI_TZ
            /* flag a TZ object and save its mass and its NS mass */
            stars->TZ_object=TRUE;
            stars->TZ_mass = m1+m2;
            stars->TZ_NS_mass = m1;
            stars->TZ_channel=TZ_CHANNEL_MIX;
#endif
            m03 = m1;
            m3 = m1;

            stars[0].SN_type = SN_TZ;
            struct star_t * news = 
                new_supernova(stardata,&stars[0],&stars[1],NULL);
            Dprint("new stellar structure at %p\n",news);
            if(news)
            {
                news->stellar_type = merged_stellar_type;
            }
        }
        else
        {
            /*
             * Compact object - NS/BH merger
             */
            if(k1==NEUTRON_STAR && k2==NEUTRON_STAR)
            {
                stars[0].SN_type = SN_NS_NS;
            }
            else if(k2==NEUTRON_STAR && k1==BLACK_HOLE)
            {
                stars[0].SN_type = SN_BH_NS;
            }
            else if(k1==BLACK_HOLE && k2==BLACK_HOLE)
            {
                stars[0].SN_type = SN_BH_BH;
            }
            else
            {
                Exit_binary_c(BINARY_C_ALGORITHM_BRANCH_FAILURE,
                              "Mix stars : NS/BH merger unknown (stellar types in %d %d)",
                              k1,k2);
            }

            Dprint("Made new SN with NS/BH remnant (%d)",
                   merged_stellar_type);

            struct star_t * news = 
                new_supernova(stardata,&stars[0],&stars[1],NULL);

            Dprint("new stellar structure at %p\n",news);
            if(news)
            {
                news->stellar_type = merged_stellar_type;
            }
            /* 
             * Set post-SN mass
             */
            news->mass = m3;
            news->core_mass = m3;
            news->phase_start_mass = m3;

#ifdef NUCSYN
            nucsyn_set_remnant_abunds(&(stars[0].Xenv[0]));
#endif
        }
        mc3 = m3;
        mcCO3 = m3;
        mcconv3 = 0.0;
    }
    else if(merged_stellar_type == MASSLESS_REMNANT)
    {
        m3 = 0.0;
        mc3 = 0.0;
        mcCO3 = 0.0;
        mcconv3 = 0.0;

        Dprint("Merged and exploded to a massless remnant");

        stars[0].SN_type = SN_IA_He;
        struct star_t * news = 
            new_supernova(stardata,&stars[0],&stars[1],NULL);
        if(news)
        {
            stellar_structure_make_massless_remnant(news);
        }
#ifdef NUCSYN
        /*
         * This can, according to p103 of Jarrod's thesis,
         * only be a HeWD-HeWD merger. Do we really believe
         * these explode?
         */
        nucsyn_set_remnant_abunds(&(stars[0].Xenv[0]));
#endif //NUCSYN
    }
    else if(merged_stellar_type>100)
    {
        /* 
         * Common envelope case which should only be used after 
         * common_envelope_evolution
         */
        merged_stellar_type = k1;
        age3 = age1;
        m3 = m1;
        mc3 = m1;
        m03 = m01;
        mcCO3 = mcCO1 + mcCO2;
        mcconv3 = mcconv1 + mcconv2;
    }
    else
    {
        /*       
         * This should not be reached.
         */  
        mc3 = 0.0;
        mcCO3 = 0.0;
        mcconv3 = 0.0;
        Exit_binary_c(BINARY_C_MIX_MERGED_STELLAR_TYPE_NOT_CAUGHT,
                      "Error in mix.c: \"merged_stellar_type not caught\"! k1=%d, k2=%d -> k3=%d\n",
                      k1,k2,merged_stellar_type);
    }

    /*
     * Put the result in star 1. Make star 2 a massless remnant.
     */

#if defined NUCSYN && defined NUCSYN_STRIP_AND_MIX
    double mass_lost = m1 + m2 - m3;
    nucsyn_mix_stars(stardata,mass_lost,m3);
#endif // NUCSYN && NUCSYN_STRIP_AND_MIX

    /*
     * If there has been a SN, let the event 
     * handler take care of updating the star.
     * Its mass and core mass are set in news above.
     */
    stars[0].mass = m3;
    stars[0].core_mass = mc3;
    stars[0].CO_core_mass = mcCO3;
    stars[0].max_MS_core_mass = mcconv3;
    
    if(stars[0].SN_type != SN_NONE)
    {
        stars[0].mass = m3;
        stars[0].phase_start_mass = m3;
    }
    else
    {
        stars[0].stellar_type = merged_stellar_type;
        stars[0].phase_start_mass = m03;
        stars[0].age = age3;
    }

    /* 
     * Companion becomes massless and hence the 
     * system becomes single.
     */
    stars[1].mass = 0.0;
    stars[1].core_mass = 0.0;
    stars[1].CO_core_mass = 0.0;
    stars[1].GB_core_mass = 0.0;
    stars[1].max_MS_core_mass = 0.0;
    stars[1].stellar_type = MASSLESS_REMNANT;
    stardata->model.sgl = TRUE;

    /*
     * Preserve the compact core type
     */
    if(stars[0].stellar_type < Max(stars[0].compact_core_type,
                                   stars[1].compact_core_type))
    {
        if(WHITE_DWARF(stars[0].stellar_type))
        {
            stars[0].stellar_type = Max(stars[0].compact_core_type,
                                        stars[1].compact_core_type);
        }
    } 

    /*
     * Turn off mass transfer now the stars have merged
     */
    Zero_stellar_derivatives(&stardata->star[0]);
    Zero_stellar_derivatives(&stardata->star[1]);

#ifdef NUCSYN
    stardata->star[0].dmacc = 0.0;
    stardata->star[1].dmacc = 0.0;

    /*
     * if the star is a (hydrogen) MS star then reset its zams mass
     * and treat the star as an evolved ms star
     */
    if(merged_stellar_type <= MAIN_SEQUENCE)
    {
        stars->effective_zams_mass = m3;
        Copy_abundances(stars[i1].Xenv,stars[i1].Xinit);
    }
    Dprint("MIXOUT abunds star 1 H1=%g He4=%g\n",
            stars[0].Xenv[XH1],stars[1].Xenv[XHe4]);
    Dprint("Star %d : m=%g, mc=%g, kw=%d\n",
           1,stars[0].mass,stars[0].core_mass,stars[0].stellar_type); 
    Dprint("Star %d : m=%g, mc=%g, kw=%d\n",
           2,stars[1].mass,stars[1].core_mass,stars[1].stellar_type); 
      
#endif //NUCSYN

    /*
     * Reset AGB parameters after a merger
     */
    stardata->star[0].num_thermal_pulses=-1;
    stardata->star[1].num_thermal_pulses=-1;
    
}
#endif//BSE
