#include "../binary_c.h"
/*
 * Update masses, angular momenta and other stellar 
 * structure variables for time t+dt.
 *
 * NB This may change the stellar types as well, e.g.
 *    when hydrogen is accreted on top of a helium star or 
 *    white dwarf.
 *
 * system_type can be DETACHED_SYSTEM_CALL, RLOF_SYSTEM_CALL or
 * GENERIC_SYSTEM_CALL
 */

int mass_angmom_and_evolutionary_changes(struct stardata_t * RESTRICT const stardata,
                                         const Evolution_system_type system_type)
{
    int stop = 0;

    /*
     * Set mass and angular momentum derivatives to zero
     * before recalculating them
     */
    zero_stellar_mass_and_angmom_derivatives(stardata);

    if(system_type == GENERIC_SYSTEM_CALL)
    {
        /*
         * Generic system
         */
        Dprint("generic system");

        /*
         * test for RLOF, act depending on its stability
         */
        const Boolean RLOF = test_for_roche_lobe_overflow(stardata) == ROCHE_OVERFLOW ? TRUE : FALSE;

        Dprint("RLOF at t=%g? %s\n",
               stardata->model.time,
               Yesno(RLOF));
        
        int stability = (RLOF == FALSE ?
                         RLOF_IMPOSSIBLE :
                         RLOF_stability_tests(stardata));

        Dprint("STABLE at t=%g? RLOF %d, stability %d \"%s\"\n",
               stardata->model.time,
               RLOF,
               stability,
               RLOF_stability_string(stability)
            );
        
        if(stability != RLOF_IMPOSSIBLE &&
           stability != RLOF_STABLE &&
           stability != RLOF_INSTABILITY_BLOCKED)
        {
            /*
             * Unstable mass transfer : 
             *
             * the function RLOF_unstable_mass_transfer will take 
             * care of changes to the stellar masses and updates
             * to structure by queueing an event which is then 
             * handled at the end of the timestep.
             *
             * Note that this trigger is blocked on intermediate
             * steps.
             */
            Dprint("update masses UNSTABLE RLOF\n");
            RLOF_unstable_mass_transfer(stardata,stability,FALSE);
        }
        else
        {
            /* 
             * Update the system with stable or no mass transfer
             */
            Dprint("update masses %s\n",RLOF==TRUE?"RLOF":(RLOF==FALSE?"detached":"unknown"));
            update_masses_angular_momenta_and_structure(stardata,
                                                        RLOF);
        }

        /*
         * Test for non-RLOFing collision
         */
        if(RLOF==FALSE &&
           check_for_collision_at_periastron(stardata) == TRUE)
        {
            Dprint("CONTACT RLOF=%d %g %g \n",
                   RLOF,
                   stardata->star[0].radius/stardata->star[0].roche_radius,
                   stardata->star[1].radius/stardata->star[1].roche_radius
                );
            contact_system(stardata,FALSE);
        }
    }
    else if(system_type == RLOF_SYSTEM_CALL)
    {
        Dprint("RLOFing system mass changes");
        /*
         * RLOFing system mass changes
         */
        int stability = RLOF_stability_tests(stardata);
        
        if(stability == RLOF_IMPOSSIBLE ||
           stability == RLOF_STABLE)
        {
            /* 
             * stable mass transfer, or no RLOF at all
             */
            if(update_masses_angular_momenta_and_structure(stardata,TRUE)==END_RLOF)
            {
                Dprint("Caught END_RLOF\n");
                stop = LOOP;
            }
        }
        else
        {
            Dprint("Caught RLOF -> unstable mass transfer\n");
            /* 
             * unstable mass transfer : we still must call the update function
             * so the derivatives are set to something *but* because the mass
             * transfer will be caught as an event we call with FALSE to 
             * prevent the timestep being rejected for some other reason
             */
            update_masses_angular_momenta_and_structure(stardata,FALSE);
            stop = RLOF_unstable_mass_transfer(stardata,stability,FALSE) == 0
                ? LOOP : STATUS_OK;
            Dprint("unstable at %g : stop = %d %s : pending? %d, stability %d\n",
                   stardata->model.time,
                   stop,
                   Evolution_loop_string(stop),
                   events_pending(stardata),
                   stability
                );
        }
    }
    else 
    {
        Dprint("Detached system mass changes");
        /*
         * Detached system mass changes
         */  
        if(update_masses_angular_momenta_and_structure(stardata,FALSE)==FALSE)
        {
            Dprint("Called main_loop - was FALSE\n");
            stop = LOOP;
        }
        else
        {
            Dprint("main_loop returned TRUE\n");
        }
    }
    
    return stop;
}
