#include "../binary_c.h"
#include "donor_types.h"

/*
  static Pure_function double MM93_k(const double m1,
  const double m2,
  const double xL);
*/

void limit_accretion_rates(struct stardata_t * RESTRICT const stardata)
{

    /* 
     * Limit accretion rates by e.g.
     * Super-Eddington mass ejection, novae, 
     * this kind of thing.
     */
    Star_number k;
    Starloop(k)
    {
        struct star_t * accretor = &stardata->star[k];
        struct star_t * donor = &stardata->star[Other_star(k)];
        
        Dprint("limit acc rate star %d : is currently %g\n",
               k,
               Mdot_net(accretor));
        
        /*
         * Osaki 1985 based on Meyer & Meyer-Hofmeister 1983
         */

        /*
          double k = MM93_k(accretor->mass,
          donor->mass,
          donor->roche_radius/stardata->common.orbit.separation);
          double kappa0 = 0.1;
          double T = Teff_from_star_struct(donor);
          double rho0 = GRAVITATIONAL_CONSTANT*donor->mass*M_SUN/
          (kappa0 * Pow2(donor->radius*R_SUN) * GAS_CONSTANT * T);
          double Q = 2.0 * PI * GAS_CONSTANT * T * 
          Pow3(stardata->common.orbit.separation*R_SUN)/
          (GRAVITATIONAL_CONSTANT * M_SUN * (accretor->mass + donor->mass) * k); 
          double cs = sqrt(GAS_CONSTANT * T);
          double H = Pow2(donor->radius * R_SUN) * GAS_CONSTANT * T/
          (GRAVITATIONAL_CONSTANT * M_SUN * donor->mass);
                        
          double dr = R_SUN*(donor->roche_radius - donor->radius);
          double mdot = 1.0/exp(1.0) * 
          Q * cs * rho0 * exp(-dr/H);
          mdot *= YEAR_LENGTH_IN_SECONDS / M_SUN;
          accretor->derivative[DERIVATIVE_STELLAR_MASS_RLOF_GAIN] += mdot;
        */
        
        /*
         * Limit mass accretion rate.
         * Only applies if the accretion rate > 0
         */
        if(Positive_nonzero(Mdot_net(accretor)))
        {
            Dprint("accretion rate is > 0\n"); 
            /* the actual limits */
            double accretion_limit[NUMBER_OF_ACCRETION_LIMITS];

            /* booleans to determine whether to apply these limits */
            Boolean apply_accretion_limit[NUMBER_OF_ACCRETION_LIMITS];

            /*
             * Accretor limits
             */
        
            /* Hachisu limit for accretion */
            if(
                stardata->preferences->hachisu_disk_wind == TRUE &&
                WHITE_DWARF(accretor->stellar_type) &&
                Mdot_net(accretor) > Hachisu_max_rate(donor,accretor)
                )
            {
                apply_accretion_limit[ACCRETION_LIMIT_HACHISU] = TRUE;
                accretion_limit[ACCRETION_LIMIT_HACHISU] =
                    Hachisu_max_rate(donor,accretor);
            }
            else
            {
                accretion_limit[ACCRETION_LIMIT_HACHISU] = 0.0;
                apply_accretion_limit[ACCRETION_LIMIT_HACHISU] = FALSE;
            }

            /* Eddington limit for accretion */
            accretion_limit[ACCRETION_LIMIT_EDDINGTON] =
                eddington_limit_for_accretion(stardata,
                                              accretor->radius,
                                              Hydrogen_mass_fraction);
            apply_accretion_limit[ACCRETION_LIMIT_EDDINGTON] = TRUE;
            
            /* Accretor thermal limit */
            accretion_limit[ACCRETION_LIMIT_THERMAL] =
                thermal_limit_for_accretion(accretor,stardata);
            
            /* only for MS, HG or CHeB stars (as in BSE) */
            apply_accretion_limit[ACCRETION_LIMIT_THERMAL] =
                Boolean_(ON_MAIN_SEQUENCE(accretor->stellar_type)  ||
                         accretor->stellar_type == HERTZSPRUNG_GAP ||
                         accretor->stellar_type == CHeB);

            /* Dynamical accretor limit */
            accretion_limit[ACCRETION_LIMIT_DYNAMICAL] = 
                dynamical_limit_for_accretion(accretor,stardata);
            apply_accretion_limit[ACCRETION_LIMIT_DYNAMICAL] = TRUE;
            
            
            /* log limit hitting */
            Boolean hit_accretion_limit[NUMBER_OF_ACCRETION_LIMITS];
            int i;
            for(i=0;i<NUMBER_OF_ACCRETION_LIMITS;i++)
            {
                hit_accretion_limit[i] = Boolean_(Mdot_net(accretor) > accretion_limit[i]);
            }
        
            /* impose accretion limit */
            for(i=0;i<NUMBER_OF_ACCRETION_LIMITS;i++)
            {
                double excess_Mdot = Mdot_net(accretor) - accretion_limit[i];
                Dprint("impose limit %d %s: Mdot_net = %g, excess_Mdot = %g vs limit %g : apply? %s : over limit? %s\n",
                       i,
                       Accretion_limit_string(i),
                       Mdot_net(accretor),
                       excess_Mdot,
                       accretion_limit[i],
                       Yesno(apply_accretion_limit[i]),
                       Yesno(excess_Mdot > 0.0)
                    );

                /*
                 * Mass loss if a limit is hit 
                 */
                if(apply_accretion_limit[i] == TRUE && excess_Mdot > 0.0)
                {
                    hit_accretion_limit[i] = TRUE;

                    Dprint("Star %d exceeded accretion limit %d (%s) : mdot acc %g, limit %g\n",
                           accretor->starnum,
                           i,
                           Accretion_limit_string(i),
                           Mdot_net(accretor),
                           accretion_limit[i]
                        );
                    
                    /*
                     * Convert excess_Mdot mass to non-conservative
                     * loss.
                     */
                    accretor->derivative[DERIVATIVE_STELLAR_MASS_NONCONSERVATIVE_LOSS] -=
                        excess_Mdot;

                    excess_Mdot = 0.0;
                }
            }
            
            /* legacy logging */
            if(hit_accretion_limit[ACCRETION_LIMIT_EDDINGTON] == TRUE)
            {
                stardata->model.supedd = TRUE;
            }

            /*
             * Accretion onto a compact object may lead to mass loss 
             * (e.g. novae)
             */
            double accretion_rate = Mdot_net(accretor);
                        
            if(accretion_rate > 0.0 &&
               COMPACT_OBJECT(accretor->stellar_type) &&
               !COMPACT_OBJECT(donor->stellar_type))
            {
                if(WHITE_DWARF(accretor->stellar_type))
                {
                    /*
                     * Accretion onto a white dwarf
                     *
                     * Two rates are of interest: 
                     *
                     * steady_burn_rate : below this, burning is steady
                     * 
                     * new_envelope_rate : above this, a new stellar envelope forms
                     *
                     * These rates depend on the composition of the donated material.
                     * At present we distinguish hydrogen-rich from helium-rich.
                     * 
                     * See Claeys et al. (2014) for details.
                     * He-rich: see Nomoto (1982).
                     *
                     * Note that, at present, we do not distinguish 
                     * between different accretor types.
                     */
                    double steady_burn_rate;
                    double new_envelope_rate;
                    compact_object_accretion_limits(stardata,
                                                    accretor,
                                                    donor,
                                                    &steady_burn_rate,
                                                    &new_envelope_rate);

                    Nova_type nova_type MAYBE_UNUSED = NOVA_TYPE_HYDROGEN;
                    if(HYDROGEN_DONOR)
                    {
                        /*
                         * Check helium burning limits:
                         * it's possible to accrete above the steady_burn_limit
                         * for hydrogen, but below the steady_burn_limit for helium,
                         * in which case we should have helium novae.
                         *
                         * In this case we should enforce novae, just tag them as helium
                         * novae rather than hydrogen.
                         */
                        const Stellar_type st_was = donor->stellar_type;
                        double steady_burn_rate_He;
                        double new_envelope_rate_He;

                        /*
                         * Find the accretion limits for helium
                         */
                        donor->stellar_type = HeMS;
                        compact_object_accretion_limits(stardata,
                                                        accretor,
                                                        donor,
                                                        &steady_burn_rate_He,
                                                        &new_envelope_rate_He);

                        /*
                         * Check that we're in the steady H-burning regime,
                         * but should have helium novae, this implies that
                         * steady_burn_rate_He > steady_burn_rate_H.
                         *
                         * Adjust the steady_burn_rate to force novae,
                         * but tag them as NOVA_TYPE_HELIUM.
                         */
                        if(accretion_rate > steady_burn_rate &&
                           accretion_rate < steady_burn_rate_He)
                        {
                            steady_burn_rate = steady_burn_rate_He;
                            nova_type = NOVA_TYPE_HELIUM;
                        }
                        
                        /* restore stellar type */
                        donor->stellar_type = st_was;
                    }

                    Dprint("MDOT = %g vs limits steady = %g new env = %g timestep %g\n",
                           accretion_rate,
                           steady_burn_rate,
                           new_envelope_rate,
                           stardata->model.dt);

                    
                    Dprint("At t=%g : novae? %s (acc %g, burn %g, newenv %g)\n",
                            stardata->model.time,
                            Yesno(accretion_rate < steady_burn_rate),
                            accretion_rate,
                            steady_burn_rate,
                            new_envelope_rate
                        );

                    /*
                     * Check for novae
                     */
                    if(accretion_rate < steady_burn_rate)
                    {
                        /*
                         * Unsteady burning which may lead to novae, hence
                         * mass and angular momentum loss.
                         *
                         * When a nova occurs, the white dwarf retains a fraction
                         * f of the surface hydrogen layer.
                         *
                         * The rest, (1-f) * dM, is lost.
                         *
                         * f = 0 means the white dwarf does not change mass.
                         * f = 1 means the white dwarf increases mass at the maximum rate.
                         *
                         * Angular momentum is dealth with in 
                         * angular_momentum_and_eccentricity_derivatives()
                         */
                        
                        const double f = nova_retention_fraction(stardata,
                                                                 accretor,
                                                                 donor,
                                                                 accretion_rate,
                                                                 steady_burn_rate);
                        
                        Dprint("nova_retention_factor f = %g\n",f);

                        
                        /*
                         * Accretion-induced irradiation 
                         * TODO 
                         */
                        //double Lac = GRAVITATIONAL_CONSTANT * accretion_rate * accretor->mass * Pow2(M_SUN)/
                        //(2.0 * accretor->radius * R_SUN);
                        //double Lirr = Pow2(donor->radius/stardata->common.orbit.separation)*0.25 * Lac;
                        //double xi = Lirr / (L_SUN * donor->luminosity * 0.5);

                        /*
                         * Fraction of a spherical shell that is
                         * intercepted by the donor star
                         */
                        const double geometric_accretion_fraction = Pow2(donor->radius)/
                            (4.0*Pow2(stardata->common.orbit.separation));

                        /*
                         * Reverse accretion
                         * M1 = WD, M2 = donor, q=M2/M1
                         * Shara+ 1986 Eq.11
                         * see also Martin, Livio and Schaefer (2011)
                         */
                        const double q = donor->mass/accretor->mass;

                        accretor->nova_beta =
                            Fequal(stardata->preferences->beta_reverse_nova,
                                   BETA_REVERSE_NOVAE_GEOMETRY) ?
                            // set from geometric argument
                            geometric_accretion_fraction :
                            // or manually
                            stardata->preferences->beta_reverse_nova;

                        /*
                         * Frictional angular momentum loss (FAML)
                         * Shara+ 1986 Eq.20 but with beta replaced by the true 
                         * geometric factor. Should this be the same beta?
                         * This is not clear from the paper! 
                         */
                        const double v_rd = orbital_velocity(stardata); // orbital velocity (km/s)
                        const double v_w = 1000.0; // nova velocity (km/s)
                        const double vrat = v_rd/v_w;
                        accretor->nova_faml = stardata->preferences->nova_faml_multiplier *
                            Pythag2(vrat,1.0) * geometric_accretion_fraction *
                            1.0 / (q*(1.0 + q)) / accretor->mass;
                        

                        /*
                         * Calculate the recurrence time, i.e. the time
                         * between successive nova explosions
                         */
                        const double recurrence_time =
                            nova_recurrence_time(accretion_rate,
                                                 accretor->mass);
                            
                        /*
                         * Now we have calculated the properties of the
                         * novae, decide whether we should model them 
                         * individually (with short timesteps), or in an
                         * average way (with long timesteps). 
                         */
                        if(stardata->preferences->individual_novae == TRUE)
                        {
                            /*
                             * Novae are modelled individually
                             *
                             * First calculate the critical mass for ignition.
                             */
                            double dMcrit = nova_explosion_layer_mass(stardata,accretor);

                            Dprint("DM %g : mdotcritdwarf %g\n",
                                   dMcrit,dwarf_nova_critical_mass_transfer_rate(stardata,donor,accretor));
                            
                            Dprint("t=%g Nova layer dm_novaH was %g, now %g (+= (accretion_rate = %g * dt = %g) = %g) of M = %g : dMcrit = %g\n",
                                        stardata->model.time,
                                        accretor->dm_novaH,
                                        accretor->dm_novaH + accretion_rate * stardata->model.dt,

                                        accretion_rate,
                                        stardata->model.dt,
                                        accretion_rate*stardata->model.dt,
                                    
                                        accretor->mass,
                                        dMcrit
                                );

                            /*
                             * Increase the surface hydrogen layer mass.
                             */
                            accretor->derivative[DERIVATIVE_STELLAR_H_LAYER_MASS] = accretion_rate;
                            
                            /*
                             * Check if we have sufficient mass for explosion
                             */
                            if(accretor->dm_novaH > dMcrit)
                            {
                                /*
                                 * Trigger a nova event
                                 */
                                struct binary_c_nova_event_t * new_nova =
                                    Malloc(sizeof(struct binary_c_nova_event_t));
                                new_nova->accretor = accretor;
                                new_nova->donor = donor;
                                new_nova->accretion_rate = accretion_rate;
                                new_nova->steady_burn_rate = steady_burn_rate;
                                new_nova->f = f;
                                accretor->nova = NOVA_STATE_EXPLODING;
                                if(Add_new_event(stardata,
                                                 BINARY_C_EVENT_NOVA,
                                                 &nova_event_handler,
                                                 &nova_erase_event_handler,
                                                 new_nova,
                                                 NORMAL_EVENT)
                                   != BINARY_C_EVENT_DENIED
                                    )
                                {
                                    /*
                                     * Nova event added
                                     */
                                }
                                else
                                {
                                    /*
                                     * Failed to allocate new event data
                                     */
                                    Safe_free(new_nova);
                                }

                                /*
                                 * If we're exploding now, set the next timeout
                                 */
                                stardata->model.nova_timeout = stardata->model.time +
                                    NOVA_TIMEOUT_FACTOR * // usually 2
                                    1e-6 * // convert yr to Myr
                                    recurrence_time;
                            }
                            else
                            {
                                /* 
                                 * Phase between novae.
                                 *
                                 * If there has been a previous nova explosion, i.e.
                                 * accretor->nova is not NOVA_STATE_NONE, 
                                 * then we set accretor->nova to NOVA_STATE_QUIESCENT.
                                 *
                                 * If there has not yet been a nova, but we expect one
                                 * soon, set to NOVA_STATE_PRE_EXPLOSION.
                                 */
                                accretor->derivative[DERIVATIVE_STELLAR_MASS_NOVA] = 0.0;
                                accretor->nova =
                                    (accretor->nova == NOVA_STATE_QUIESCENT ||
                                     accretor->nova == NOVA_STATE_EXPLODING) ?
                                    NOVA_STATE_QUIESCENT :
                                    NOVA_STATE_PRE_EXPLOSION;
                            }
                        }
                        else
                        {
                            
                            /*
                             * Novae are modelled as average events, which are put in 
                             * the normal mass loss.
                             *
                             * We set nova to be NOVA_STATE_EXPLODING on the assumption
                             * that the timestep is long so there are going to be some
                             * explosions, but it's not really relevant.
                             */
                            accretor->derivative[DERIVATIVE_STELLAR_MASS_NOVA] += 
                                - (1.0-f) * accretion_rate;
                            accretor->nova = NOVA_STATE_EXPLODING;
                            Dprint("accretion rate %g -> dM/dt nova = %g\n",
                                   accretion_rate,
                                    accretor->derivative[DERIVATIVE_STELLAR_MASS_NOVA]);

                        }


                        /*
                         * Count novae
                         */
                        accretor->derivative[DERIVATIVE_STELLAR_NUM_NOVAE] =
                            1.0 / recurrence_time; 

                        
                        /*
                          donor->derivative[DERIVATIVE_STELLAR_MASS_IRRADIATIVE_LOSS] =
                          - 1.1e18 * (xi/4.0) * pow(1.0 + donor->mass,2.0/3.0) * pow(donor->mass,2.833) 
                          * YEAR_LENGTH_IN_SECONDS / M_SUN *
                          stardata->preferences->nova_irradiation_multiplier;
                        */

                        /*
                          Dprint("IRRAD Lac=%g, Lirr=%g, Ldonor = %g, %g vs Mdot %g\n",
                          Lac/L_SUN,
                          Lirr/L_SUN,
                          donor->luminosity,
                          donor->derivative[DERIVATIVE_STELLAR_MASS_IRRADIATIVE_LOSS],
                          accretion_rate);
                        */
                        /*
                         * Nova-induced mass transfer back onto the companion
                         */
                        donor->derivative[DERIVATIVE_STELLAR_MASS_NOVA] += 
                            accretor->nova_beta * -accretor->derivative[DERIVATIVE_STELLAR_MASS_NOVA];
                    }
                    else
                    {
                        /*
                         * We have steady burning.
                         * 
                         * Accretion rate exceeds the steady burning rate
                         * either burning is steady, or the star forms
                         * a new envelope.
                         *
                         * Note: we do not treat the case where
                         * there is steady accretion of hydrogen, but then
                         * helium flashes underneath which would leave to 
                         * helium novae. These should be caught here (todo!).
                         */
                        accretor->nova = NOVA_STATE_NONE;
                        
                        if(accretion_rate < new_envelope_rate)
                        {
                            /*
                             * Possibly a supersoft X-ray source : allow steady burning, do nothing here
                             */
                        }
                        else
                        {
                            /*
                             * Make a new giant envelope :
                             * do not alter the mass accretion rate.
                             * This is handled in apply_stellar_mass_and_angular_momentum.c
                             */
                        }
                    }
                }
                else
                {
                    /*
                     * Accretion onto a neutron star/black hole, what to do?
                     */
                    accretor->nova = NOVA_STATE_NONE;
                }
            }
            else
            {
                /*
                 * No longer a compact object : cannot have novae
                 */
                accretor->nova = NOVA_STATE_NONE;
            }
        } // accretion rate > 0 check
        else
        {
            /*
             * Accretion rate is zero or negative, i.e. mass
             * is constant or being lost.
             */


            /*
             * If we were previously having novae, keep the state
             * at NOVA_STATE_QUIESCENT until nova_timeout has been reached
             */
            accretor->nova =
                (
                    accretor->nova == NOVA_STATE_QUIESCENT ||
                    accretor->nova == NOVA_STATE_EXPLODING ||
                    stardata->model.time < stardata->model.nova_timeout 
                    ) 
                ? NOVA_STATE_QUIESCENT : NOVA_STATE_NONE;
        }
    }    
}

