#include "../binary_c.h"

void calc_wind_loss_and_gain_rates(struct stardata_t * RESTRICT const stardata,
                                   const Boolean RLOF_bool)
{

    /* 
     * Set the stellar wind mass loss and gain rates.
     *
     * If RLOF_bool is TRUE then you're in RLOF, which uses a 
     * different radius (the Roche radius, rather than the stellar radius,
     * stored in star->effective_radius) rather than the usual star->radius.
     */
    if(stardata->preferences->wind_mass_loss != FALSE)
    {
        double radius,rlperi,vwind2,ivsqm,omv2;
        Star_number k;

        Starloop(k)
        {
            SETstar(k);
            if(star->stellar_type!=MASSLESS_REMNANT)
            {
                /*
                 * Calculate wind mass loss from the previous timestep.
                 */      
                rlperi = star->roche_radius*
                    (1.0-stardata->common.orbit.eccentricity);

                Dprint("rlperi = %12.12g from roche_radius=%12.12g, ecc=%12.12g\n",
                       rlperi,
                       star->roche_radius,
                       stardata->common.orbit.eccentricity);


                radius = RLOF_bool==FALSE ? star->radius : star->effective_radius;

                Dprint("star %d: kw=%d, lum=%12.12g, radius=%12.12g, mass=%12.12g, mc=%12.12g, rlperi=%12.12g, z=%12.12g\n",k,
                       star->stellar_type,
                       star->luminosity,
                       radius,
                       star->mass,
                       star->core_mass,
                       rlperi,
                       stardata->common.metallicity);

                star->derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS] =
                    - calc_stellar_wind_mass_loss(star->stellar_type,
                                                  star->luminosity,
                                                  radius,
                                                  star->mass,
                                                  star->core_mass,
                                                  rlperi,
                                                  stardata->common.metallicity,
                                                  star->omega,
                                                  stardata,k);
                
                Dprint("from calc_stellar_wind_loss dmr(%d)=%12.12g\n",
                       k,star->derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS]);


                /*
                 * Binary systems have wind accretion
                 */
                if(System_is_binary)
                {
                    /*
                     * Calculate how much of wind mass loss from companion will be
                     * accreted (Boffin & Jorissen, A&A 1988, 205, 155).
                     */

                    /* V1.10 different version of the above */
                    vwind2 = 2.0*BETA*BONDI_HOYLE_CONVERSION_FACTOR*star->mass/radius;

#ifdef USE_WIND_VELOCITY_OF_VW93_ON_TPAGB

                    /*
                     * CAB:
                     *      Use the wind velocity calculated with fitting formulae (e.g.: VW93-K02 on the TPAGB)
                     *      at the beginning of the TPAGB vwind might be 0, which can give problems in M_acc;
                     *      If vwind=0, then put vwind = 5km/s, apprximately the minimum observed 
                     *      (see e.g. Vassiliadis&Wood, 1993)
                     *      If you don't define this function, when you calculate the mass accretion rate
                     *      you will use for vwind2 a fraction of escape velocity
                     */
                    if(star->stellar_type==TPAGB)
                    {
                        /* vwind in stardata->star is in cm/s (5 km/s = 5e5 cm/s) */
                        star->vwind=Max(5e5,star->vwind);
        
                        /*
                         * ... locally in (km/s)^2, because later in the
                         * formula for derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN] these units are required  
                         */
                        vwind2 = Pow2(star->vwind*1e-5);
                    }
/***********************************************************************************************/

#endif /* USE_WIND_VELOCITY_OF_VW93_ON_TPAGB */

                    /* NB Jarrod also uses a FIXED wind velocity BETA */
                    //vwind2=Pow2(BETA);
 
                    /* or this form (in fortran!)
                       c               vwind2 = (beta*(1.0d0
                       c     &              -exp(-((sep-rad(k))/2*rad(k))**2)))**2
                    */
                    double vorb2 = orbital_velocity_squared(stardata);
                    ivsqm = 1.0/sqrt(1.0-ECC_SQUARED);

                    omv2 = pow((1.0 + vorb2/vwind2),1.5);

                    stardata->star[Other_star(k)].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN] =
                        ivsqm*stardata->preferences->Bondi_Hoyle_accretion_factor*-stardata->star[k].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS]*
                        Pow2((BONDI_HOYLE_CONVERSION_FACTOR*stardata->star[Other_star(k)].mass/vwind2))/
                        (2.0*Separation_squared*omv2);
                    
#ifdef WRLOF_MASS_TRANSFER
                    stardata->star[Other_star(k)].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN] =
                        Max(stardata->star[Other_star(k)].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN],
                            WRLOF_rate(stardata,k));
#endif // WRLOF_MASS_TRANSFER
                        
                    /* perhaps limit accretion rate */
                    stardata->star[Other_star(k)].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN] =
                        Min(stardata->star[Other_star(k)].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN],
                            (RLOF_bool==TRUE ? 1.0 : 0.8) * -star->derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS]);


                    Dprint("wind accretion: acc2=%g dmt(%d)=%g [%3.2f pc of companion wind loss] (from ivsqm=%g dmr=%12.12g ac1=%12.12g mass=%g vwind2=%g sep=%g omv2=%g\n",
                           stardata->preferences->Bondi_Hoyle_accretion_factor,
                           Other_star(k),
                           stardata->star[Other_star(k)].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN],
                           100.0*stardata->star[Other_star(k)].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN]/
                           Max(1e-20,-fabs(stardata->star[k].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS])),
                           ivsqm,
                           -star->derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS],
                           BONDI_HOYLE_CONVERSION_FACTOR,
                           stardata->star[Other_star(k)].mass,
                           vwind2,stardata->common.orbit.separation,omv2);
                }
                else
                {
                    /* detached system : no mass gain */
                    stardata->star[Other_star(k)].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN] = 0.0;
                }
            }
            else
            {
                // massless : no wind and no accretion on the other star
                star->derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS]=0.0;
                stardata->star[Other_star(k)].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN]=0.0;
            }
        }
    }
    Dprint("Mass loss/gain rates : 0 %g %g : 1 %g %g\n", 
           stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS],
           stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN],
           stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS],
           stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN]);
}
