#include "../binary_c.h"


void apply_stellar_mass_and_angular_momentum_derivatives(struct stardata_t * RESTRICT const stardata,
                                                         const Boolean RLOF)
{
    /*
     * Apply stellar mass and angular momentum derivatives
     *
     * Changes the masses and angular momenta of the stars,
     * and also (perhaps) the stellar type, epoch etc. 
     * appropriate for the new mass and accreted material.
     */
    Star_number k;

    /*
     * Timestep
     */
    double dt = RLOF==TRUE ? (stardata->model.dtm*1e6) : stardata->model.dt;
    double dtm = stardata->model.dtm;

#ifdef REVERSE_TIME
    if(stardata->preferences->reverse_time == TRUE)
    {
        dt *= -1;
        dtm *= -1;
    }
#endif//REVERSE_TIME
    
#if DEBUG==1
    if(0&&Debug_expression)
        show_derivatives(stardata);
#endif
    Evolving_STARLOOP(k)
    {
        SETstar(k);

        /*
         * Change in angular momentum
         */
        star->derivative[DERIVATIVE_STELLAR_ANGMOM] =
            Jdot_net(star);

        apply_derivative(stardata,
                         &star->angular_momentum,
                         star->derivative,
                         dt,
                         DERIVATIVE_STELLAR_ANGMOM);

        /*
         * Preserve sanity in case something goes wrong.
         * Note that this breaks angular momentum conservation.
         */
        Clamp(star->angular_momentum,
              MINIMUM_STELLAR_ANGMOM,
              MAXIMUM_STELLAR_ANGMOM);

        /*
         * Change in mass
         */
        star->derivative[DERIVATIVE_STELLAR_MASS] = Mdot_net(star);

        apply_derivative(stardata,
                         &star->mass,
                         star->derivative,
                         dt,
                         DERIVATIVE_STELLAR_MASS);
                
        if(star->mass < 0.0)
        {
            show_derivatives(stardata);
            Exit_binary_c(2,
                          "M<0 with dt=%g dM/dt=%g\n",
                          dt,
                          star->derivative[DERIVATIVE_STELLAR_MASS]
                );
        }
#ifdef MINIMUM_STELLAR_MASS
        star->mass = Max(MINIMUM_STELLAR_MASS,
                         star->mass);
#endif
        
#ifdef TRIM_SUPERMASSIVE_STARS
        if(star->mass > MAXIMUM_STELLAR_MASS)
        {
            strip_supermassive_star(stardata,star,star->mass);
            star->mass = MAXIMUM_STELLAR_MASS;
        }
#endif


        /* core growth(s) */
        {
            apply_derivative(stardata,
                             &star->core_mass,
                             star->derivative,
                             dt,
                             DERIVATIVE_STELLAR_HE_CORE_MASS);
            apply_derivative(stardata,
                             &star->core_mass_no_3dup,
                             star->derivative,
                             dt,
                             DERIVATIVE_STELLAR_HE_CORE_MASS_NO_TDUP);
            apply_derivative(stardata,
                             &star->CO_core_mass,
                             star->derivative,
                             dt,
                             DERIVATIVE_STELLAR_CO_CORE_MASS);
        }

        /*
         * Thermal pulses
         */
        {
            apply_derivative(stardata,
                             &star->num_thermal_pulses,
                             star->derivative,
                             dt,
                             DERIVATIVE_STELLAR_NUM_THERMAL_PULSES);
            apply_derivative(stardata,
                             &star->num_thermal_pulses_since_mcmin,
                             star->derivative,
                             dt,
                             DERIVATIVE_STELLAR_NUM_THERMAL_PULSES_SINCE_MCMIN);
        }
        
        
        /*
         * Core mass cannot exceed the total mass 
         */
        star->core_mass = Min(star->mass,
                              star->core_mass);
        /*
         * CO core mass cannot exceed the total core mass
         */
        star->CO_core_mass = Min(star->core_mass,
                                 star->CO_core_mass);
        
        /*
         * Reset phase start mass on accretion
         */
        if(ON_EITHER_MAIN_SEQUENCE(star->stellar_type) ||
           (star->stellar_type==HERTZSPRUNG_GAP && 
            RLOF==FALSE &&
#ifdef REJUVENATE_HERTZSPRUNG_GAP_STARS
            1
#else
            0
#endif
               ))
        {
            star->phase_start_mass = star->mass;

#ifdef MINIMUM_STELLAR_MASS
            star->phase_start_mass = Max(MINIMUM_STELLAR_MASS,
                                         star->phase_start_mass);
#endif // MINIMUM_STELLAR_MASS
            
            if(
                (RLOF==FALSE)
                ||
                (RLOF==TRUE &&
                 star->stellar_type==HERTZSPRUNG_GAP &&
                 stellar_structure_giant_branch_helium_ignition(stardata,star)
                    )
                )
            {
#ifdef BSE
                /*
                 * Update epoch
                 */
                double m0 = star->phase_start_mass;
            Dprint("Set initial mass to %g (was %g)\n",star->phase_start_mass,m0);

                double GB[GB_ARRAY_SIZE];
                double lums[LUMS_ARRAY_SIZE];
                double tscls[TSCLS_ARRAY_SIZE];
                double tm,tn;

                stellar_timescales(star->stellar_type,
                                   star->phase_start_mass,
                                   star->mass,
                                   &tm,
                                   &tn,
                                   tscls,
                                   lums,
                                   GB,
                                   stardata,
                                   star);
                Dprint("stellar_timescales gave tm=%30.22e tn=%30.22e\n",tm,tn);

                if(RLOF==TRUE)
                {
                    if(GB[GB_MC_BGB]<star->core_mass)
                    {
                        star->phase_start_mass = m0;
                    }
                }
                else
                {
                    if(star->stellar_type==HERTZSPRUNG_GAP)
                    {
                        if((GB[GB_MC_BGB]<star->core_mass)||
                           (m0 > stardata->common.metallicity_parameters[ZPAR_MASS_FGB]))
                        {
                            star->phase_start_mass = m0;
                            Dprint("Overridden to m0=%g for HG star\n",m0);
                        }
                        else
                        {
                            double agex =
                                tm + (tscls[T_BGB] - tm)*
                                (star->age - star->tms)/
                                (star->tbgb - star->tms);
                            star->epoch = stardata->model.time - agex;
                        } 
                        Dprint("set epoch (apply1) = %g (tm=%g tbgb=%g age=%g tms=%g)\n",
                               star->epoch,
                               tm,
                               tscls[T_BGB],
                               star->age,
                               star->tms
                            );
                    }
                    else
                    {
                        star->epoch = stardata->model.time - 
                            star->age * tm / star->tms;
                        Dprint("set epoch (apply2) star %d type %d = %30.22e - %30.22e * %30.22e/%30.22e = %30.22e\n",
                               star->starnum,
                               star->stellar_type,
                               stardata->model.time,
                               star->age,
                               tm,
                               star->tms,
                               star->epoch);
                    } 
                }
#endif//BSE
            }
        }

        if(stardata->preferences->individual_novae == TRUE)
        {
            /*
             * Grow the surface hydrogen layer mass when we have 
             * individual novae
             */
            apply_derivative(stardata,
                             &star->dm_novaH,
                             star->derivative,
                             dt,
                             DERIVATIVE_STELLAR_H_LAYER_MASS);

        }

        /*
         * Count novae
         */
        apply_derivative(stardata,
                         &star->num_novae,
                         star->derivative,
                         dt,
                         DERIVATIVE_STELLAR_NUM_NOVAE);
        
        /*
         * Update the stellar type as a result of accretion: 
         * 
         * This only happens if the accretion rate exceeds
         * the new envelope rate, or if hydrogen accretes onto
         * a naked helium star.
         * 
         * Novae are dealt with elsewhere.
         */
        struct star_t * accretor = star;
        struct star_t * donor = Other_star_struct(star);
        double accretion_rate = Mdot_net(accretor);
        
        if(accretion_rate > 0.0)
        {
            /*
             * Hydrogen accretion onto a helium star
             * 
             * Replaced RLOF_naked_helium_star_secondary
             */
            if(NAKED_HELIUM_STAR(accretor->stellar_type) &&
               donor->stellar_type <= TPAGB)
            {
                accretor->stellar_type =
                    // HeMS has a helium core -> CHeB
                    accretor->stellar_type == HeMS ? CHeB :
                    // HeHG has a CO core with helium shell -> EAGB
                    accretor->stellar_type == HeHG ? EAGB :
                    //HeGB has CO core with thin helium shell -> TPAGB
                    TPAGB; 
                                
                double mcx;
                if(accretor->stellar_type==CHeB)
                {
                    accretor->age /= accretor->tms;
                    mcx = accretor->mass;
                }
                else
                {
                    mcx = accretor->core_mass;
                    
                    /*
                     * Reset number of thermal pulses
                     */
                    if(accretor->stellar_type==TPAGB)
                    {
                        accretor->num_thermal_pulses = -1;
                    }
                }
                
                /*
                 * Core mass is the whole of the helium star.
                 * The envelope is the accreted hydrogen.
                 */
                accretor->phase_start_mass = Max(accretor->mass,
                                                 accretor->phase_start_mass);

#ifdef BSE
                giant_age(&mcx,
                          accretor->mass,
                          & accretor->stellar_type,
                          & accretor->phase_start_mass,
                          & accretor->age,
                          stardata,
                          accretor);
#endif//BSE
                accretor->core_mass = Min(accretor->mass,mcx);
                accretor->CO_core_mass = Min(accretor->core_mass,
                                             accretor->CO_core_mass);
                accretor->epoch = stardata->model.time + 
                    dtm - accretor->age;
            }
#ifdef DISCS_ACCRETION_ONTO_RGB_FUDGE_BROKEN
            else if(accretor->stellar_type == GIANT_BRANCH &&
                    accretor->core_mass < 0.46)
            {
                /* this is likely wrong! */
#ifdef BSE
                giant_age(&accretor->core_mass,
                          accretor->mass,
                          & accretor->stellar_type,
                          & accretor->phase_start_mass,
                          & accretor->age,
                          stardata,
                          accretor);
#endif//BSE
                accretor->core_mass = Min(accretor->mass,mcx);
                accretor->CO_core_mass = Min(accretor->core_mass,
                                             accretor->CO_core_mass);
                accretor->epoch = stardata->model.time - accretor->age;
            }
#endif//DISCS_ACCRETION_ONTO_RGB_FUDGE_BROKEN
            else if(COMPACT_OBJECT(accretor->stellar_type))
            {
                /*
                 * Accretion onto a compact object, 
                 * i.e. white dwarf, neutron star or black hole
                 */
                double steady_burn_rate;
                double new_envelope_rate;
                compact_object_accretion_limits(stardata,
                                                accretor,
                                                donor,
                                                &steady_burn_rate,
                                                &new_envelope_rate);

                if(Mdot_net(accretor) > new_envelope_rate)
                {
                    /*
                     * WD accretion leads to a new envelope
                     */
                    if(WHITE_DWARF(accretor->stellar_type))
                    {
                        Stellar_type new_stellar_type =
                            donor->stellar_type <= TPAGB ?
                            // hydrogen accretion, new type depends on core type (He or HeCO hybrid or CO/ONe) 
                            (accretor->stellar_type == HeWD ? GIANT_BRANCH :
                             (accretor->hybrid_HeCOWD == TRUE ? EAGB : TPAGB)) :
                            // helium accretion
                            HeGB; // helium donor -> always a helium giant (core is CO)
                        
                        /*
                         * Reset number of thermal pulses
                         */
                        if(accretor->stellar_type == TPAGB)
                        {
                            accretor->num_thermal_pulses = -1;
                        }
#ifdef BSE
                        giant_age(&accretor->core_mass,
                                  accretor->mass,
                                  &new_stellar_type,
                                  &accretor->phase_start_mass,
                                  &accretor->age,
                                  stardata,
                                  accretor);
#endif // BSE
                        accretor->core_mass = Min(accretor->mass,accretor->mass);
                        accretor->CO_core_mass = Min(accretor->core_mass,
                                                     accretor->CO_core_mass);

                        accretor->epoch = stardata->model.time + 
                            dtm - accretor->age;
                        Dprint("Set accretor (star %d) epoch = %g\n",
                               accretor->starnum,
                               accretor->epoch);

                        /*
                         * Set the new stellar type
                         */
#ifdef NUCSYN
#ifdef NUCSYN_STRIP_AND_MIX
                        accretor->strip_and_mix_disabled=TRUE;
#endif
                        if(new_stellar_type!=accretor->stellar_type &&
                           new_stellar_type < HeWD)
                        {
                            /* 
                             * stellar type has been changed in GNTAGE, 
                             * perhaps to a He star
                             *
                             * The accreted layer is now the new surface.
                             */
                            Copy_abundances(accretor->Xacc,
                                            accretor->Xenv);
                        }
#endif //NUCSYN
                        star->stellar_type = new_stellar_type;
                    }
                    else
                    {
                        /*
                         * A NS or BH can form a red giant, i.e. a Thorne-Zytkow
                         * object. This is immediately destroyed in the current
                         * code, but could be given a stellar type of its own
                         * if required at a later date (if so, giant_age needs
                         * an update).
                         */
                    } 
                }
            }
        }
    }

    

    calculate_spins(stardata);


}
