#include "../binary_c.h"

void angular_momentum_and_eccentricity_derivatives(
    struct stardata_t * RESTRICT const stardata,
    const Boolean RLOF_boolean)
{
    Star_number k; 
    /*
     * Calculate angular momentum and eccentricity derivatives
     * arising from the stars and the orbit.
     *
     * 1) Orbital : Wind loss
     * 2) Stellar : a) Wind mass loss and gain
     *    Stellar : b) Magnetic braking
     * 3) Both    : RLOF mass transfer
     *              novae
     * 4) Both    : Tidal interactions
     * 5) Orbital : non-conservative RLOF
     * 6) Orbital : Gravitational radiation
     * 7) Both    : wind-fed circumbinary discs
     * 8) Stars   : Artificial 
     * 9) Both    : Prevent breakup (overspin) of the stars
     *     
     * The RLOF-specific parts are activated if RLOF_boolean==TRUE
     */

    /*
     * First, zero the angular momentum and eccentricity derivatives, 
     * and assume a long timescale for tides
     */

    Zero_orbit_and_system_derivatives(stardata);
    Starloop(k)
    { 
        SETstar(k);
        Zero_stellar_angmom_derivatives(star);
    }
    stardata->common.orbit.tcirc = LONG_TIDAL_TIME;
    
    if(System_is_binary)
    {
        /*
         * Eccentricity changes because of wind loss/gain:
         * not active in RLOF, because then we assume e=0 
         */
        if(RLOF_boolean == FALSE)
        {
            stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_WINDS] =
                -wind_decc_dt(stardata);
        }

        /* 
         * Orbital angular momentum changes because of wind loss
         */
        stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_WIND_LOSS] += 
            wind_djorb_dt(stardata,stardata->common.orbit.angular_frequency);

#ifdef DISCS
        /*
         * Circumbinary disc(s)
         */
        if(stardata->common.ndiscs)
        {
            int i;
            Discdebug(2,
                      "derivs for %d discs (M=%g)\n",
                      stardata->common.ndiscs,
                      stardata->common.ndiscs?stardata->common.discs[0].M:0.0 
                      
                );
            for(i=0;i<stardata->common.ndiscs;i++)
            {
                /*
                 * Tides remove angular momentum from the binary
                 * and put it in the disc
                 *
                 * Factor of YEAR_LENGTH_IN_SECONDS / ANGULAR_MOMENTUM_CGS converts
                 * to code units.
                 *
                 * Note: what's stored are the total changes in the time
                 *       the disc has existed during binary_c's timestep,
                 *       which is disc->dT. Thus we must scale for the 
                 *       complete binary_c timestep, which is where the 
                 *       factor ftdisc comes in. If the disc exists for the 
                 *       whole binary_c timestep, ftdisc == 1.
                 *
                 */

                struct disc_t * disc = & stardata->common.discs[i];
                double dt = stardata->model.dtm * 1e6 * YEAR_LENGTH_IN_SECONDS;
                double ftdisc = disc->dT / dt;
                if(disc->dT > TINY && dt > TINY)
                {
                    double jdot = disc->Jdot_binary * ftdisc *  
                        YEAR_LENGTH_IN_SECONDS / ANGULAR_MOMENTUM_CGS;
                    stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_CBDISC] += jdot;

                    /*
                     * Decide how to split the mass 
                     * (and angmom?) between the two stars
                     *
                     * Note that f is the fraction of the accretion rate from
                     * the disc that falls onto star 0 (the "primary") given
                     * q = M2/M1.
                     *
                     * If f < -0.5, set the accretion rate to zero.
                     */
                    double f = disc_inner_edge_accretion_f(stardata);

                    if(f>-0.5)
                    {
                        double mdot = disc->Mdot_binary * ftdisc *  
                            YEAR_LENGTH_IN_SECONDS / M_SUN;
#ifdef ___MOREDEBUGGING
                        if(0)printf("DMBIN deriv mdot %g Msun/y (ftdisc = %g/%g = %g, disc->M=%g) : DM=%g : f = %g : 1-f = %g\n",
                                    mdot,
                                    disc->dT/YEAR_LENGTH_IN_SECONDS,
                                    dt,
                                    ftdisc,
                                    disc->M/M_SUN,
                                    mdot * stardata->model.dtm * 1e6,
                                    f,
                                    1.0-f
                            );
#endif// ___MOREDEBUGGING
                        stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_CBDISC_GAIN] =
                            f * mdot;
                        stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_CBDISC_GAIN] =
                            (1.0-f) * mdot;
                    }
                    else
                    {
                        stardata->star[0].derivative[DERIVATIVE_STELLAR_MASS_CBDISC_GAIN] = 0.0;
                        stardata->star[1].derivative[DERIVATIVE_STELLAR_MASS_CBDISC_GAIN] = 0.0;
                    }
                    
                    // TODO : what about the stellar spin up?

                    /*
                     * Resonant interaction with the circumbinary
                     * disc drives binary eccentricity
                     */
                    double edot = disc->edot_binary * ftdisc * YEAR_LENGTH_IN_SECONDS;
                    
                    stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_CBDISC] = edot;

                    /* systematic loss from the disc */
                    stardata->star[0].derivative[DERIVATIVE_SYSTEM_CBDISC_MASS_LOSS] =
                        disc->Mdot_ejected
                        * YEAR_LENGTH_IN_SECONDS / M_SUN;
                }
            } // loop over discs
        }
#endif // DISCS
    } // system_is_binary

    /*
     * Stellar angular momentum loss from winds, including
     * magnetic braking, and tides.
     */
    Evolving_STARLOOP(k)
    {
        SETstars(k);

        /* stellar spin changes from wind loss/gain */
        stellar_wind_angmom(stardata,RLOF_boolean,k);

        /* spin down by magnetic braking */
        magnetic_braking(stardata,star);

#ifdef OMEGA_CORE
        /* couple core and envelope */
        core_envelope_coupling(stardata,star);
#endif
        
        Dprint("ANGMOM%d %d jdot=%30.22e jspin=%30.22e dj=%30.22e dt=%30.22e\n",
               star->starnum,
               star->stellar_type,
               star->derivative[DERIVATIVE_STELLAR_ANGMOM],
               star->angular_momentum,
               star->derivative[DERIVATIVE_STELLAR_ANGMOM] * stardata->model.dt,
               stardata->model.dt
            );
    }

    
    if(System_is_binary)
    {
        /*
         * Orbital angular momentum loss because of 
         * nova explosions. Only activate if nova mass 
         * derivative is < 0.0.
         *
         * Algorithm is in Shara+ 1986 Eq.12
         *
         * The nova_beta and nova_faml parameters are set
         * in limit_accretion_rates.c. 
         */
        Starloop(k)
        {
            SETstars(k);
            if(Is_not_zero(star->derivative[DERIVATIVE_STELLAR_MASS_NOVA]))
            {
                nova_angular_momentum_changes(stardata,
                                              star,
                                              star->derivative[DERIVATIVE_STELLAR_MASS_NOVA],
                                              &star->derivative[DERIVATIVE_STELLAR_ANGMOM_NOVA],
                                              &stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_NOVA]);
            }
        }

        
        /* 
         * Calculate stellar spin up/down because of accretion
         * and/or loss of mass during RLOF 
         */
        if(RLOF_boolean) RLOF_stellar_angmom_derivative(stardata);
     
        Evolving_STARLOOP(k)
        {
            SETstars(k);

            /* tidal spin up/down and circularization */
            tides(stardata,
                  RLOF_boolean,
                  k,
                  RLOF_boolean == TRUE ? star->effective_radius : star->radius,
                  star,companion);
        }

        /* orbital angular momentum losses because of RLOF transfer */
        non_conservative_angular_momentum_loss(stardata,RLOF_boolean);

        /*
         * For very close systems include orbital angular 
         * momentum loss mechanisms e.g. gravitational radiation.
         */
        angular_momentum_loss_mechanisms_for_close_systems(stardata);

        /*
         * Circumbinary discs from winds
         */
#ifdef DISCS_CIRCUMBINARY_FROM_WIND
        disc_stellar_wind_to_cbdisc(stardata);
#endif // DISCS_CIRCUMBINARY_FROM_WIND

    }

    /*
     * Arificial derivatives
     */
    Starloop(k)
    {
        if(
            /* start time must be > 0 to be valid */
            (stardata->preferences->accretion_start_time > 0.0 &&
           More_or_equal(stardata->model.time,
                         stardata->preferences->accretion_start_time))
           &&
           /* if end time < 0 we ignore it and accrete forever */
           (stardata->preferences->accretion_end_time < 0.0 ||
            stardata->model.time < stardata->preferences->accretion_end_time))
        {
            stardata->star[k].derivative[DERIVATIVE_STELLAR_MASS_ARTIFICIAL] =
                k==0 ?
                stardata->preferences->mass_accretion_rate1 :
                stardata->preferences->mass_accretion_rate2;
            stardata->star[k].derivative[DERIVATIVE_STELLAR_ANGMOM_ARTIFICIAL] =
                k==0 ?
                stardata->preferences->angular_momentum_accretion_rate1 :
                stardata->preferences->angular_momentum_accretion_rate2;
        }
        else
        {
            stardata->star[k].derivative[DERIVATIVE_STELLAR_MASS_ARTIFICIAL] = 0.0;
            stardata->star[k].derivative[DERIVATIVE_STELLAR_ANGMOM_ARTIFICIAL] = 0.0;
        }
        
    }
    stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_ARTIFICIAL] =
        stardata->preferences->angular_momentum_accretion_rate_orbit;

    /*
     * Summed derivatives
     */
    stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM] =
        Jdot_orbit(stardata)
        +
        stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_ARTIFICIAL];
    
    stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY] =
        edot_orbit(stardata);

    if(0)
        printf("EDOT is grav=%g + tides=%g + winds=%g + cbdisc=%g = %g\n",
               stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_GRAVITATIONAL_RADIATION],
               stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_TIDES],
               stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_WINDS],
               stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_CBDISC],
               stardata->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY]
            );
    Dprint("ANGMOM deriv %g\n", stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM]);
    
    /* 
     * Calculate rotation variables and ensure that the stars
     * do not spin up beyond break-up, and
     * transfer the excess angular momentum back to the orbit.
     */
    calculate_spins(stardata);

    Dprint("ANGMOM deriv %g\n", stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM]);
    
    Evolving_STARLOOP(k)
    {
        SETstar(k);

        double effective_radius = star->radius > star->roche_radius ? star->effective_radius : star->radius;
        calculate_rotation_variables(star,
                                     effective_radius);
        prevent_overspin(star,stardata,RLOF_boolean);
    }
    Dprint("ANGMOM deriv %g\n", stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM]);
        
}
