#include "../binary_c.h"
#include "donor_types.h"

/*
 * Minimum stable accretion rate on a white dwarf
 * 
 * Returns the rate in Msun/year

 *
 * Note: these are rates limited by nuclear burning only.
 * Other limits (e.g. thermal, Eddington, etc.) are imposed elsehwere.
 */

double WD_min_accretion_rate(struct stardata_t * RESTRICT const stardata,
                             struct star_t * const donor,
                             struct star_t * const accretor)
{
    /*
     * Choose rate based on donor stellar type
     */
    double rate =
        HYDROGEN_DONOR ? stardata->preferences->WD_accretion_rate_novae_upper_limit_hydrogen_donor :
        HELIUM_DONOR ? stardata->preferences->WD_accretion_rate_novae_upper_limit_helium_donor :
        stardata->preferences->WD_accretion_rate_novae_upper_limit_other_donor;

    if(rate < -TINY)
    {
        /* 
         * Use rate algorithm
         */
        const int algorithm = Map_float_algorithm(rate);
        Dprint("mapped rate %g to algorithm %d \"%s\"\n",
               rate,
               algorithm,
               Donor_rate_algorithm_string(algorithm));
        
        switch(algorithm)
        {
            
        case DONOR_RATE_ALGORITHM_BSE:
            /*
             * BSE (Hurley et al. 2002) algorithm
             */
            if(HYDROGEN_DONOR)
            {
                rate = 1.03e-7;
            }
            else
            {
                /* 
                 * Rate may be limited elsewhere
                 */
                rate = VERY_SMALL_MASS_TRANSFER_RATE;
            }
            break;
            
        case DONOR_RATE_ALGORITHM_CLAEYS2014:
            /*
             * Claeys et al. (2014) algorithm
             */
            if(HYDROGEN_DONOR)
            {
                Abundance X;
#ifdef NUCSYN
                X = donor->Xenv[XH1];
#else
                X = 0.7;
#endif
                rate =
                    (1.0/8.0) *
                    5.3e-7 *
                    (1.7-X)/Max(1e-10,X) * 
                    Max(0.0, accretor->mass - 0.4);
            }
            else if(HELIUM_DONOR)
            {
                /*
                 * 3.98107e-8, compiler will pre-compute this
                 */
                rate = pow(10.0,-7.4); 
            }
            else
            {
                /* what to do for other donors? */
                rate = 0.0;
            }
            break;
                
        default:
            rate = 0.0; // avoid compiler warnings
            Exit_binary_c(BINARY_C_SETUP_UNKNOWN_ARGUMENT,
                          "Unknown accretion rate novae upper limit method %g (mapped to algorithm %d) \n",
                          stardata->preferences->WD_accretion_rate_novae_upper_limit_hydrogen_donor,
                          algorithm);
            
            
        }
    }
    
    rate = Max(VERY_SMALL_MASS_TRANSFER_RATE,rate);
    return rate;
}

