#include "../binary_c.h"
#include "donor_types.h"


/*
 * Maximum stable accretion rate on a white dwarf
 * 
 * Returns the rate in Msun/year
 *
 * Note: these are rates limited by nuclear burning only.
 * Other limits (e.g. thermal, Eddington, etc.) are imposed elsehwere.
 */
double WD_max_accretion_rate(struct stardata_t * RESTRICT const stardata,
                             struct star_t * const donor,
                             struct star_t * const accretor)
{
    /*
     * Choose rate based on donor stellar type
     */
    double rate = 
        HYDROGEN_DONOR ? stardata->preferences->WD_accretion_rate_new_giant_envelope_lower_limit_hydrogen_donor : 
        HELIUM_DONOR ? stardata->preferences->WD_accretion_rate_new_giant_envelope_lower_limit_helium_donor :
        stardata->preferences->WD_accretion_rate_new_giant_envelope_lower_limit_other_donor;
        
    if(rate < -TINY)
    {
        /* 
         * Use rate algorithm
         */
        const int algorithm = Map_float_algorithm(rate);
        Dprint("mapped rate %g to algorithm %d \"%s\"\n",
               rate,
               algorithm,
               Donor_rate_algorithm_string(algorithm));
     
        switch(algorithm)
        {
            
        case DONOR_RATE_ALGORITHM_BSE:
            /*
             * BSE (Hurley et al. 2002) algorithm
             */
            if(HYDROGEN_DONOR)
            {
                rate = 2.71e-7;
            }
            else
            {
                /*
                 * Rate may be limited elsewhere
                 */
                rate = VERY_LARGE_MASS_TRANSFER_RATE;
            }
            break;
            
        case DONOR_RATE_ALGORITHM_CLAEYS2014:
            /*
             * Claeys et al. (2014) algorithm
             */
            rate = Hachisu_max_rate(donor,accretor);
            break;
                
        default:
            rate = 0.0; // avoid compiler warnings
            Exit_binary_c(BINARY_C_SETUP_UNKNOWN_ARGUMENT,
                          "Unknown accretion rate new giant envelope limit method %g\n",
                          stardata->preferences->WD_accretion_rate_new_giant_envelope_lower_limit_hydrogen_donor);
            
            
        }
    }
    
    return Max(VERY_SMALL_MASS_TRANSFER_RATE,rate);
}
