#pragma once
#ifndef STELLAR_TYPES_H
#define STELLAR_TYPES_H
/*
 * The binary_c stellar population nucleosynthesis framework.
 *
 * Contact: r.izzard@surrey.ac.uk or rob.izzard@gmail.com
 *
 * http://personal.ph.surrey.ac.uk/~ri0005/binary_c.html
 * https://gitlab.eps.surrey.ac.uk/ri0005/binary_c
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-announce
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-devel
 * https://twitter.com/binary_c_code
 * https://www.facebook.com/groups/149489915089142/
 *
 * Please see the files README, LICENCE and CHANGES,
 * and the doc/ directory for documentation.
 *
 * Stellar types for binary_c, based on the definitions of
 * Hurley et al. "Evolution of Binary Stars and the
 * Effect of Tides on Binary Populations", MNRAS 2002 and
 * references therein.
 *
 * Also included are function macros for indentifying sets
 * of stellar types.
 */
#include "binary_c_parameters.h" 
#include "binary_c_macros.h"


#define LOW_MASS_MS 0
#define LOW_MASS_MAIN_SEQUENCE 0
#define MS 1
#define MAIN_SEQUENCE 1

#define HG 2
#define HERTZSPRUNG_GAP HG

#define GIANT_BRANCH 3
#define FIRST_GIANT_BRANCH GIANT_BRANCH
#define CHeB 4
#define CORE_HELIUM_BURNING CHeB
#define EAGB 5
#define EARLY_ASYMPTOTIC_GIANT_BRANCH EAGB
#define TPAGB 6
#define THERMALLY_PULSING_ASYMPTOTIC_GIANT_BRANCH TPAGB

#define HeMS 7
#define NAKED_MAIN_SEQUENCE_HELIUM_STAR HeMS
#define HeHG 8
#define NAKED_HELIUM_STAR_HERTZSPRUNG_GAP HeHG
#define HeGB 9
#define NAKED_HELIUM_STAR_GIANT_BRANCH HeGB

#define HeWD 10
#define HELIUM_WHITE_DWARF HeWD
#define COWD 11
#define CARBON_OXYGEN_WHITE_DWARF COWD
#define ONeWD 12
#define OXYGEN_NEON_WHITE_DWARF ONeWD

#define NS 13
#define NEUTRON_STAR NS
#define BH 14
#define BLACK_HOLE BH
#define MASSLESS_REMNANT 15

#define NUMBER_OF_STELLAR_TYPES 16

/*
 * Strings for logging
 */
#define STELLAR_TYPES                           \
    "Deeply or fully convective low mass MS",   \
        "Main Sequence",                        \
        "Hertzsprung Gap",                      \
        "First Giant Branch",                   \
        "Core Helium Burning",                  \
        "First Asymptotic Giant Branch",        \
        "Second Asymptotic Giant Branch",       \
        "Main Sequence Naked Helium",           \
        "Hertzsprung Gap Naked Helium",         \
        "Giant Branch Naked Helium",            \
        "Helium White Dwarf",                   \
        "Carbon/Oxygen White Dwarf",            \
        "Oxygen/Neon White Dwarf",              \
        "Neutron",                              \
        "Black Hole",                           \
        "Massless"

/* 
 * Stellar type string from stellar type S
 */
#define Stellar_type_string(S)                                          \
    (                                                                   \
        (S)==LOW_MASS_MAIN_SEQUENCE ? "Convective low mass MS" :        \
        (S)==MAIN_SEQUENCE ? "Main Sequence" :                          \
        (S)==HERTZSPRUNG_GAP ? "Hertzsprung Gap" :                      \
        (S)==GIANT_BRANCH ? "First Giant Branch" :                      \
        (S)==CORE_HELIUM_BURNING ? "Core Helium Burning" :              \
        (S)==EAGB ? "Early Asymptotic Giant Branch" :                   \
        (S)==TPAGB ? "Thermally-Pulsing Asymptotic Giant Branch" :      \
        (S)==HeMS ? "Main Sequence Naked Helium" :                      \
        (S)==HeHG ? "Hertzsprung Gap Naked Helium" :                    \
        (S)==HeGB ? "Giant Branch Naked Helium" :                       \
        (S)==HeWD ? "Helium White Dwarf" :                              \
        (S)==COWD ? "Carbon/Oxygen White Dwarf" :                       \
        (S)==ONeWD ? "Oxygen/Neon White Dwarf" :                        \
        (S)==NEUTRON_STAR ? "Neutron star":                             \
        (S)==BLACK_HOLE ? "Black Hole" :                                \
        (S)==MASSLESS_REMNANT ? "Massless Remnant" :                    \
        "Unknown"                                                       \
        )

/*
 * Stellar type string from star struct.
 *
 * Using the star struct means we can include
 * hybrid types.
 */
#define Stellar_type_string_from_star(STAR)                             \
    (                                                                   \
        (STAR)->stellar_type==LOW_MASS_MAIN_SEQUENCE ? "Convective low mass MS" : \
        (STAR)->stellar_type==MAIN_SEQUENCE ? "Main Sequence" :         \
        (STAR)->stellar_type==HERTZSPRUNG_GAP ? "Hertzsprung Gap" :     \
        (STAR)->stellar_type==GIANT_BRANCH ? "First Giant Branch" :     \
        (STAR)->stellar_type==CORE_HELIUM_BURNING ? "Core Helium Burning" : \
        (STAR)->stellar_type==EAGB ? "Early Asymptotic Giant Branch" :  \
        (STAR)->stellar_type==TPAGB ? "Thermally-Pulsing Asymptotic Giant Branch" : \
        (STAR)->stellar_type==HeMS ? "Main Sequence Naked Helium" :     \
        (STAR)->stellar_type==HeHG ? "Hertzsprung Gap Naked Helium" :   \
        (STAR)->stellar_type==HeGB ? "Giant Branch Naked Helium" :      \
        (STAR)->stellar_type==HeWD ? "Helium White Dwarf" :             \
        ((STAR)->stellar_type==COWD && (STAR)->hybrid_HeCOWD==TRUE) ? "Carbon/Oxygen/Helium Hybrid White Dwarf" : \
        (STAR)->stellar_type==COWD ? "Carbon/Oxygen White Dwarf" :      \
        (STAR)->stellar_type==ONeWD ? "Oxygen/Neon White Dwarf" :       \
        (STAR)->stellar_type==NEUTRON_STAR ? "Neutron star":            \
        (STAR)->stellar_type==BLACK_HOLE ? "Black Hole" :               \
        (STAR)->stellar_type==MASSLESS_REMNANT ? "Massless Remnant" :   \
        "Unknown"                                                       \
        )

/*
 * Short stellar type string from the stellar type.
 *
 * Note that this means hybrid types are not labelled as 
 * such.
 */
#define Short_stellar_type_string(S)                    \
    (                                                   \
        (S) == LOW_MASS_MAIN_SEQUENCE ? "LMMS" :        \
        (S) == MAIN_SEQUENCE ? "MS" :                   \
        (S) == HERTZSPRUNG_GAP ? "HG" :                 \
        (S) == GIANT_BRANCH ? "GB" :                    \
        (S) == CORE_HELIUM_BURNING ? "CHeB" :           \
        (S) == EAGB ? "EAGB" :                          \
        (S) == TPAGB ? "TPAGB" :                        \
        (S) == HeMS ? "HeMS" :                          \
        (S) == HeHG ? "HeHG" :                          \
        (S) == HeGB ? "HeGB" :                          \
        (S) == HeWD ? "HeWD" :                          \
        (S) == COWD ? "COWD" :                          \
        (S) == ONeWD ? "ONeWD" :                        \
        (S) == NEUTRON_STAR ? "NS":                     \
        (S) == BLACK_HOLE ? "BH" :                      \
        (S) == MASSLESS_REMNANT ? "REM" :               \
        "Unknown"                                       \
        )

/*
 * Short stellar type string from star struct.
 *
 * Use of the star struct means we can include hybrid 
 * types.
 */
#define Short_stellar_type_string_from_star(STAR)                       \
    (                                                                   \
        (STAR)->stellar_type == LOW_MASS_MAIN_SEQUENCE ? "LMMS" :       \
        (STAR)->stellar_type == MAIN_SEQUENCE ? "MS" :                  \
        (STAR)->stellar_type == HERTZSPRUNG_GAP ? "HG" :                \
        (STAR)->stellar_type == GIANT_BRANCH ? "GB" :                   \
        (STAR)->stellar_type == CORE_HELIUM_BURNING ? "CHeB" :          \
        (STAR)->stellar_type == EAGB ? "EAGB" :                         \
        (STAR)->stellar_type == TPAGB ? "TPAGB" :                       \
        (STAR)->stellar_type == HeMS ? "HeMS" :                         \
        (STAR)->stellar_type == HeHG ? "HeHG" :                         \
        (STAR)->stellar_type == HeGB ? "HeGB" :                         \
        (STAR)->stellar_type == HeWD ? "HeWD" :                         \
        ((STAR)->stellar_type == COWD && (STAR)->hybrid_HeCOWD == TRUE) ? "COWDhybrid" : \
        (STAR)->stellar_type == COWD ? "COWD" :                         \
        (STAR)->stellar_type == ONeWD ? "ONeWD" :                       \
        (STAR)->stellar_type == NEUTRON_STAR ? "NS":                    \
        (STAR)->stellar_type == BLACK_HOLE ? "BH" :                     \
        (STAR)->stellar_type == MASSLESS_REMNANT ? "REM" :              \
        "Unknown"                                                       \
        )


/* White dwarf 10,11,12 */
#define WHITE_DWARF(A) (((A)>HeGB)&&((A)<NEUTRON_STAR))

/* White dwarf CO or ONe */
#define CONeWD(A)                               \
    ((A) == ONeWD ||                            \
     (A) == COWD)

/* Later than a white dwarf */
#define LATER_THAN_WHITE_DWARF(A) ((A)>HeGB)

/* stars with "cores" according to the BSE definition */ 
#define HAS_BSE_CORE(A)                         \
    ((A) >= HG &&                               \
     (A) <  NEUTRON_STAR &&                     \
     (A) != HeMS)

/* Naked He Star 7,8,9 */
#define NAKED_HELIUM_STAR(A)                    \
    ((A)>TPAGB &&                               \
     (A)<HeWD)

/* carbon-core stars */
#define CARBON_CORE_STAR(A)                     \
    ((A)>=CHeB &&                               \
     (A)<HeWD &&                                \
     (A)!=HeMS) 

/* dammit! I wish we could call two things "MAIN_SEQENCE" but we can't! */
/* NB Helium main sequence is separate from the hydrogen main sequence */
/* Main sequence 0,1 */
#define ON_MAIN_SEQUENCE(A)                     \
    ((A) == LOW_MASS_MS ||                      \
     (A) == MAIN_SEQUENCE)

/* Either MS 0,1,7 */
#define ON_EITHER_MAIN_SEQUENCE(A)              \
    ((A) == LOW_MASS_MS ||                      \
     (A) == MAIN_SEQUENCE ||                    \
     (A) == HeMS)

/* Post MS He star 8,9 */
#define POST_MS_NAKED_He_STAR(A)                \
    ((A) == HeHG ||                             \
     (A) == HeGB)

/* Giant 2,3,4,5,6,8,9 (cf. Hurley+2002 Eq.61) */
#define GIANT_LIKE_STAR(A)                      \
    ((A) > MAIN_SEQUENCE &&                     \
     (A) < HeWD &&                              \
     (A) != HeMS)

/* Asymptotic Giant Branch 5,6 */
#define AGB(A)                                  \
    ((A) == EAGB ||                             \
     (A) == TPAGB)

/* Giant branch 3,5,6 */
#define ON_GIANT_BRANCH(A)                      \
    ((A) == FIRST_GIANT_BRANCH ||               \
     AGB(A))

/* either giant branch 3,5,6,9 */
#define ON_EITHER_GIANT_BRANCH(A)               \
    (ON_GIANT_BRANCH(A) ||                      \
     (A) == HeGB)

#define HELIUM_CORED(A)                         \
    (A == CHeB ||                               \
     NAKED_HELIUM_STAR(A) ||                    \
     A == HeWD)

#define COMENV_OFFSET (+100)

#define COMPACT_OBJECT(A)                       \
    ((A)>HeGB &&                                \
     (A)<MASSLESS_REMNANT)

#define NUCLEAR_BURNING(A) ((A)<HeWD)

#define POST_NUCLEAR_BURNING(A) ((A)>HeGB)

#define MASSIVE_REMNANT(A)                      \
    ((A)>HeGB &&                                \
     (A)<MASSLESS_REMNANT)

#define GRB_PROGENITOR(A)                       \
    ((A) == NS ||                               \
     (A) == BH)

#define POST_SN_OBJECT(A)                       \
    ((A) == NS ||                               \
     (A) == BH)

#define IS_A_PLANET(A)                                  \
    (stardata->star[(A)].stellar_type == LMMS &&        \
     stardata->star[(A)].mass<MAXIMUM_PLANET_MASS))

#define WAS_A_STAR(A)                                           \
    (stardata->star[(A)].pms_mass > MAXIMUM_PLANET_MASS)

#define MASSLESS(A)                                             \
    (stardata->star[(A)].stellar_type  ==  MASSLESS_REMNANT)

#define EVOLVING(A) (!(MASSLESS(A)))

#define IS_A_STAR(A) (EVOLVING(A) &&                                    \
                      stardata->star[(A)].mass>MAXIMUM_PLANET_MASS)

#define NUCLEAR_BURNING_BINARY                          \
    (NUCLEAR_BURNING(stardata->star[0].stellar_type) || \
     NUCLEAR_BURNING(stardata->star[1].stellar_type))


/* Filter for convective envelopes */
#define CONVECTIVE_ENVELOPE(A)                  \
    (ON_EITHER_GIANT_BRANCH(A) ||               \
     (A) == HG ||                               \
     (A) == HeHG)

/**************************************************/

/* Here are the system types */

/* You can define these any way you like! */

/* k is the star number - assumes you have access to stardata */

/* These are the ways to define the systems */

/* Extended stellar types string array */
#define EXT_STELLAR_TYPES                       \
    STELLAR_TYPES,"Blue Straggler"

/************************************************************/
/* These are the types for the individual stars */
#define SINGLE_UNDEF 666
#define CHECK_BLUE_STRAGGLER(k)                                 \
    (stardata->common.tm<stardata->model.time                   \
     &&                                                         \
     ON_MAIN_SEQUENCE(stardata->star[(k)].stellar_type))
#define BLUE_STRAGGLER 16

/************************************************************/
/* These are the binary system types */
#define SYSTEM_UNDEF 0
#define BINARY_MS_MS 1
#define BINARY_MS_HG 2
#define BINARY_MS_GB 3
#define BINARY_MS_EAGB 4
#define BINARY_MS_TPAGB 5
#define BINARY_STAR_SYSTEM -2
#define SINGLE_STAR_SYSTEM -1
/************************************************************/

#define EITHER_STAR_EVOLVING                    \
    (EVOLVING(0) ||                             \
     EVOLVING(1))

#define EITHER_STAR_MASSLESS                    \
    (MASSLESS(0) ||                             \
     MASSLESS(1))

#define NEITHER_STAR_MASSLESS                   \
    ( (!MASSLESS(0)) &&                         \
      (!MASSLESS(1)) )

#define NOTHING_LEFT (MASSLESS(0) && MASSLESS(1))

#define OSTAR(A)                                                \
    ((spectral_type(&(stardata->star[(A)])) == SPECTRAL_TYPE_O) \
     &&                                                         \
     (stardata->star[(A)].stellar_type<TPAGB))

#endif /* STELLAR_TYPES_H */
