#pragma once
#ifndef BINARY_MATHS_H
#define BINARY_MATHS_H

/*
 * The binary_c stellar population nucleosynthesis framework.
 *
 * Contact: r.izzard@surrey.ac.uk or rob.izzard@gmail.com
 *
 * http://personal.ph.surrey.ac.uk/~ri0005/binary_c.html
 * https://gitlab.eps.surrey.ac.uk/ri0005/binary_c
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-announce
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-devel
 * https://twitter.com/binary_c_code
 * https://www.facebook.com/groups/149489915089142/
 *
 * Please see the files README, LICENCE and CHANGES,
 * and the doc/ directory for documentation.
 *
 *
 * This file contains macros used in binary_c that are
 * mathematical in nature.
 */

#include <math.h>
#include <float.h>
#include <stdarg.h>

/*
 * If defined, allow monochecks in generic_bisect.
 */
#define BISECT_DO_MONOCHECKS

/* 
 * define USE_FABS to force the use of fabs instead of 
 * comparisons to +TINY and -TINY. This is usually slow.
 */
//#define USE_FABS

/* 
 * defined ZERO_CHECKS_ARE_FUNCTIONS to make Is_zero etc. functions,
 * otherwise they are macros 
 *
 * NB This is ignored if USE_FABS is defined
 */
//#define ZERO_CHECKS_ARE_FUNCTIONS

/*
 * Use inline functions for simple pow(x,y) operations
 */
//#define POWER_OPERATIONS_ARE_FUNCTIONS

/* 
 * Timings for 1000 sytems
# 123.17,124.58,125.47 standard run : no fabs, all others functions
# 117.94,118.92,120.57 no fabs, all macros
# 119.64,120.25,120.73 macros with fabs
# 120.80,123.88,124.80 all macros, except min/max are functions[C
# 117.80,119.01,120.97 all macros, except pow calls arefunctions
*/

/************************************************************/

/* small and large numbers */
#define ZERO (0.0)
#define TINY (1e-14)
#define VERY_TINY (1e-20)
#define REALLY_TINY (1E-200) 
#define LARGE_FLOAT (1E200)

/* geometric constants */
#define PI (3.141592653589793238462643383279)
#define TWOPI (2.0*PI)

/* swap macro : A and B must be of the same type */
#define Swap(A,B) {const Autotype(A) temp=(A);(A)=(B);(B)=temp;}

/* macros to define less and more than operations */
#define Less_than(A,B) ((A)<(B))
#define More_than(A,B) ((A)>(B))

/* macros to define less and more than taking TINY into account */
#define Really_less_than(A,B) (Less_than((A),(B))&&!Fequal((A),(B)))
#define Really_more_than(A,B) (More_than((A),(B))&&!Fequal((A),(B)))

/*
 * Min and Max of two numbers of generic type with only
 * one evalutation of each, taken from 
 * https://gcc.gnu.org/onlinedocs/gcc/Typeof.html
 */

#define Max(A,B) ({                             \
            const Autotype(A) _a = (A);         \
            const Autotype(B) _b = (B);         \
            More_than(_a,_b) ? _a : _b;         \
        })
#define Min(A,B) ({                             \
            const Autotype(A) _a = (A);         \
            const Autotype(B) _b = (B);         \
            Less_than(_a,_b) ? _a : _b;         \
        })
    
/* max, min of several floating point numbers */
#define Max3(A,B,C) Max((A),Max((B),(C)))
#define Max4(A,B,C,D) Max((A),Max3((B),(C),(D)))
#define Max5(A,B,C,D,E) Max((A),Max4((B),(C),(D),(E)))
#define Min3(A,B,C) Min((A),Min((B),(C)))
#define Min4(A,B,C,D) Min((A),Min3((B),(C),(D)))
#define Min5(A,B,C,D,E) Min((A),Min4((B),(C),(D),(E)))

/* NB minx should not be NULL */
#define MinX(ax,ay,bx,by,_minx) ({                              \
            const Autotype(ax) _ax = (ax);                      \
            const Autotype(bx) _bx = (bx);                      \
            const Autotype(ay) _ay = (ay);                      \
            const Autotype(by) _by = (by);                      \
            *_minx = Less_than((_ay),(_by)) ? _ax : _bx;        \
            Min(_ay,_by);                                       \
        })

/* NB maxx should not be NULL */
#define MaxX(ax,ay,bx,by,_minx) ({                              \
            const Autotype(ax) _ax = (ax);                      \
            const Autotype(bx) _bx = (bx);                      \
            const Autotype(ay) _ay = (ay);                      \
            const Autotype(by) _by = (by);                      \
            *_maxx = More_than((_ay),(_by)) ? _ax : _bx;        \
            Max(_ay,_by);                                       \
        })

/*
 * Which_MIN and Which_MAX return the minimum and maximum (respectively)
 * of the list of doubles passed in and sets *which to the index
 * (0-based) of the minimum in the list.
 */
static inline double _which_Min(unsigned int * which,
                                unsigned int n,
                                ...);
static inline double _which_Max(unsigned int * which,
                                unsigned int n,
                                ...);
static inline double _which_Min(unsigned int * which,
                                unsigned int n,
                                ...)
{
    va_list args;
    va_start(args,n);
    
    double min = LARGE_FLOAT;
    unsigned int i;
    for(i=0;i<n;i++)
    {
        double x = va_arg(args,double);
        if(Less_than(x, min))
        {
            *which = i;
            min = x;
        }
    }
    va_end(args);
    return min;
}
static inline double _which_Max(unsigned int * which,
                                unsigned int n,
                                ...)
{
    va_list args;
    va_start(args,n);
    
    double max = -LARGE_FLOAT;
    unsigned int i;
    for(i=0;i<n;i++)
    {
        double x = va_arg(args,double);
        if(More_than(x, max))
        {
            *which = i;
            max = x;
        }
    }
    va_end(args);
    return max;
}
     
/* convenience macros for accessing _which_Min and _which_Max */
#define Which_MIN2(I,A,B) _which_Min(&(I),2,(A),(B))
#define Which_Min3(I,A,B,C) _which_Min(&(I),3,(A),(B),(C))
#define Which_Min4(I,A,B,C,D) _which_Min(&(I),4,(A),(B),(C),(D))
#define Which_Min5(I,A,B,C,D,E) _which_Min(&(I),5,(A),(B),(C),(D),(E))
#define Which_MAX2(I,A,B) _which_Max(&(I),2,(A),(B))
#define Which_Max3(I,A,B,C) _which_Max(&(I),3,(A),(B),(C))
#define Which_Max4(I,A,B,C,D) _which_Max(&(I),4,(A),(B),(C),(D))
#define Which_Max5(I,A,B,C,D,E) _which_Max(&(I),5,(A),(B),(C),(D),(E))


/*
 * Macro to calculate the fractional difference between two numbers
 * with a check to make sure we avoid 1/0 and calculate the maximum
 * difference (assumes floats/doubles)
 */
#define Signed_diff(A,B) (((A)-(B))/Max(Min(fabs(A),fabs(B)),(TINY)))

/*
 * Macro to calculate the absolute fractional difference between two numbers
 * with a check to make sure we avoid 1/0 and calculate the maximum
 * difference (assumes floats/doubles)
 */
#define Abs_diff(A,B) (fabs((A)-(B))/Max(Min(fabs(A),fabs(B)),(TINY)))

/*
 * Use Abs_diff to see if A and B are the same within
 * some threshold EPS 
 */
#define Float_same_within_eps(A,B,EPS) (Abs_diff((A),(B))<=(EPS))

/*
 * Apply Float_same_within_eps with the double-precision 
 * floating point epsilon. NB we assume that we never use
 * single-precision in binary_c
 */
#define Float_same_within_dbl_epsilon(A,B) Float_same_within_eps((A),(B),(DBL_EPSILON))
#define Float_same_within_ten_dbl_epsilon(A,B) Float_same_within_eps((A),(B),(10.0*(DBL_EPSILON)))

/*
 * Macro to calculate the absolute % difference between two numbers
 * with a check to make sure we avoid 1/0 and calculate the maximum
 * % difference (assumes floats/doubles)
 */
#define Percent_diff(A,B) (100.0*Signed_diff((A),(B)))
#define Abs_percent_diff(A,B) (100.0*Abs_diff((A),(B)))

// fast bit-shift equivalent to pow(2,A) but for integers
#define Integer_power_of_two(A) (1<<(A))

/* integer power macro (uses inline function) */
#define Power_of_integer(A,B) (powab(A,B))
static inline int powab(const int a,const int b) Pure_function;
static inline int powab(const int a,const int b){int r=1;unsigned int i;for(i=0;i<b;i++){r*=a;}return r;}


/*
 * Safe(r) pow(x,y) to avoid over/underflows.
 * Of course, this is a VERY crude fix. You're
 * better off fixing the underlying problem.
 */
#define Saferpow(X,Y) (pow(X,Max(-100.0,Min(100.0,Y))))

/*
 * Useful macros for floating point comparison, on the 
 * assumption that numbers are ~ 1.0.
 *
 * Less_or_equal is supposed to be <= for floats, More_or_equal is >=, Fequal
 * is ==. The idea is that if the two numbers are similar in magnitude, A*TINY
 * will always provide an accurate test of equality. Of course, you have to
 * choose TINY appropriately - with (say) 30 bits of mantissa, that's 9E-10 so
 * 1E-20 is plenty small enough! To then get a relatively small number,
 * multiply TINY by either A or B (since they are similar!).
 * 
 * The Fequal works in the same way, with a > or < check as well.
 *
 * There are two sets of macros: with and without fabs. 
 *
 * The fabs function call is potentially slow, and not really necessary.
 * You might want to disable it (see options at the top of this file).
 *
 * If your numbers are likely to not be ~1.0, you should use 
 * Float_same_within_dbl_epsilon and Float_same_within_ten_dbl_epsilon above.
 */

#ifdef USE_FABS

/* Macros for comparisons using fabs */
#define Is_zero(A) (fabs((A))<TINY)
#define Is_really_zero(A) (fabs((A))<REALLY_TINY)
#define Is_not_zero(A) (fabs((A))>TINY)
#define Is_really_not_zero(A) (fabs((A))>TINY)
#define Fequal(A,B) (Is_zero((A)-(B)))
#define More_or_equal(A,B) (((A)>=(B))||(Fequal((A),(B))))
#define Less_or_equal(A,B) More_or_equal((B),(A))
#define Abs_more_than(A,B) (fabs((A))>(B))
#define Abs_more_than_or_equal(A,B) (Abs_more_than((A),(B))||Fequal((A),(B)))
#define Abs_less_than(A,B) (fabs((A))<(B))
#define Abs_less_than_or_equal(A,B) (Abs_less_than((A),(B))||Fequal((A),(B)))


#else//USE_FABS

/* Macros for comparisons without using fabs */
#define Is_zero_macro(A) (((A)<TINY)&&(-(A)<TINY))
#define Is_really_zero_macro(A) (((A)<REALLY_TINY)&&(-(A)<REALLY_TINY))

#ifdef ZERO_CHECKS_ARE_FUNCTIONS
#define Is_zero(A) (func_is_zero(A))
#define Is_really_zero(A) (func_is_really_zero(A))
#else
#define Is_zero(A) (Is_zero_macro(A))
#define Is_really_zero(A) (Is_really_zero_macro(A))
#endif

static inline int func_is_zero(const double a) Pure_function;
static inline int func_is_really_zero(const double a) Pure_function;
static inline int func_is_zero(const double a){return(Is_zero_macro(a));}
static inline int func_is_really_zero(const double a){return(Is_really_zero_macro(a));}

/* derived macros */
#define Is_not_zero(A) (!(Is_zero((A))))
#define Is_really_not_zero(A) (!(Is_really_zero((A))))
#define Fequal(A,B) (Is_zero((A)-(B)))
#define More_or_equal(A,B) (((A)>=(B))||(Fequal((A),(B))))
#define Less_or_equal(A,B) More_or_equal((B),(A))
#define Neither_is_zero(A,B) (Is_not_zero(A) &&    \
                           Is_not_zero(B))
#define Neither_is_really_zero(A,B) (Is_really_not_zero(A) &&      \
                                  Is_really_not_zero(B))

// alternative > and < for fabs(x)<y
#define Abs_more_than(A,B) (((A)>(B))||(-(A)>(B)))
#define Abs_more_than_or_equal(A,B) (Abs_more_than((A),(B))||Fequal((A),(B)))
#define Abs_less_than(A,B) ((-(B)<(A))&&((A)<(B)))
#define Abs_less_than_or_equal(A,B) (Abs_less_than((A),(B))||Fequal((A),(B)))
#endif//USE_FABS

/* find the sign of a number */
#define Sign(X) (Is_zero(X) ? +1.0 : (X)/fabs(X))
#define Sign_is_really(X) (Is_really_zero(X) ? +1.0 : (X)/fabs(X))


/*
 * Sign macros, but give 0 when X is zero
 */
#define Sign_int(X) ((int)(Is_zero(X) ? 0 : ((X)/fabs(X) > 0.0 ? +1 : -1)))
#define Sign_is_really_int(X) ((int)(Is_really_zero(X) ? 0 : ((X)/fabs(X) > 0.0 ? +1 : -1)))


/* range macros */
#define In_range(A,B,C) ((More_or_equal((A),(B)))&&(Less_or_equal((A),(C))))
#define Limit_range(A,B,C) (Max((B),Min((C),(A))))
#define Fix_range(A,B,C) ((A)=(Limit_range((A),(B),(C))))
#define Clamp(A,B,C) ((A)=(Limit_range((A),(B),(C))))

/* Pythagoras in 2D or 3D */
#define Pythag2_squared(X,Y) (((X)*(X) + (Y)*(Y)))
#define Pythag2(X,Y) (sqrt(Pythag2_squared((X),(Y))))
#define Pythag3_squared(X,Y,Z) (((X)*(X) + (Y)*(Y) + (Z)*(Z)))
#define Pythag3(X,Y,Z) (sqrt(Pythag3_squared((X),(Y),(Z))))
/* ditto with coordinate structs */
#define Pythag2_coord(C) (sqrt(Pow2((C).x) + Pow2((C).y)))
#define Pythag3_coord(C) (sqrt(Pow2((C).x) + Pow2((C).y) + Pow2((C).z)))

/*
 * power macros
 *
 * p means point, e.g. 1p5 = 1.5 = 3/2
 * d means divide, e.g. 1d4 = 1/4 = 0.25
 */
#define _Pow2_macro(X) ((X)*(X))
#define _Pow3_macro(X) ((X)*(X)*(X))
#define _Pow4_macro(X) ((X)*(X)*(X)*(X))
#define _Pow5_macro(X) ((X)*(X)*(X)*(X)*(X))
#define _Pow6_macro(X) ((X)*(X)*(X)*(X)*(X)*(X))
#define _Pow7_macro(X) ((X)*(X)*(X)*(X)*(X)*(X)*(X))
#define _Pow8_macro(X) ((X)*(X)*(X)*(X)*(X)*(X)*(X)*(X))
#define _Pow9_macro(X) ((X)*(X)*(X)*(X)*(X)*(X)*(X)*(X)*(X))
#define _Pow1p5_macro(X) ((X)*sqrt(X))
#define _Pow2p5_macro(X) ((X)*(X)*sqrt(X))
#define _Pow3p5_macro(X) ((X)*(X)*(X)*sqrt(X))
#define _Pow1d4_macro(X) (sqrt(sqrt(X)))
#define _Pow5d4_macro(X) ((X)*sqrt(sqrt(X)))
#define _Pow5p5_macro(X) (_Pow5_macro(X)*sqrt(X))
#define _Pow3d4_macro(X) ((X)/(_Pow1d4_macro(X)))
#define _Pow2d3_macro(X) (_Pow2_macro(cbrt(X)))
#define _Pow4d3_macro(X) (_Pow4_macro(cbrt(X)))

#ifdef POWER_OPERATIONS_ARE_FUNCTIONS
/* use functions for simple power operations */
#define Pow2(A) func_pow2(A)
#define Pow3(A) func_pow3(A)
#define Pow4(A) func_pow4(A)
#define Pow5(A) func_pow5(A)
#define Pow6(A) func_pow6(A)
#define Pow7(A) func_pow7(A)
#define Pow8(A) func_pow8(A)
#define Pow9(A) func_pow9(A)
#define Pow2p5(A) func_pow2p5(A)
#define Pow3p5(A) func_pow3p5(A)
#define Pow1p5(A) func_pow1p5(A)
#define Pow1d4(A) func_pow1d4(A)
#define Pow5d4(A) func_pow5d4(A)
#define Pow1p5(A) func_pow1p5(A)
#define Pow5p5(A) func_pow5p5(A)
#define Pow3d4(A) func_pow3d4(A)
#define Pow2d3(A) func_pow2d3(A)
#define Pow4d3(A) func_pow4d3(A)

#define Power_function static inline double Constant_function
    
Power_function func_pow2(const double x);
Power_function func_pow3(const double x);
Power_function func_pow4(const double x);
Power_function func_pow5(const double x);
Power_function func_pow6(const double x);
Power_function func_pow7(const double x);
Power_function func_pow8(const double x);
Power_function func_pow9(const double x);
Power_function func_pow2p5(const double x);
Power_function func_pow3p5(const double x);
Power_function func_pow1d4(const double x);
Power_function func_pow5d4(const double x);
Power_function func_pow5p5(const double x);
Power_function func_pow3d4(const double x);
Power_function func_pow2d3(const double x);
Power_function func_pow4d3(const double x);
Power_function func_pow2(const double x) {return _Pow2_macro(x);}
Power_function func_pow3(const double x) {return _Pow3_macro(x);}
Power_function func_pow4(const double x) {return _Pow4_macro(x);}
Power_function func_pow5(const double x) {return _Pow5_macro(x);}
Power_function func_pow6(const double x) {return _Pow6_macro(x);}
Power_function func_pow7(const double x) {return _Pow7_macro(x);}
Power_function func_pow8(const double x) {return _Pow8_macro(x);}
Power_function func_pow9(const double x) {return _Pow8_macro(x);}
Power_function func_pow1p5(const double x) {return _Pow1p5_macro(x);}
Power_function func_pow2p5(const double x) {return _Pow2p5_macro(x);}
Power_function func_pow3p5(const double x) {return _Pow3p5_macro(x);}
Power_function func_pow1d4(const double x) {return _Pow1d4_macro(x);}
Power_function func_pow5d4(const double x) {return _Pow5d4_macro(x);}
Power_function func_pow5p5(const double x) {return _Pow5p5_macro(x);}
Power_function func_pow3d4(const double x) {return _Pow3d4_macro(x);}
Power_function func_pow2d3(const double x) {return _Pow2d3_macro(x);}
Power_function func_pow4d3(const double x) {return _Pow4d3_macro(x);}
#else
/* use macros for simple power operations */
#define Pow2(A) _Pow2_macro(A)
#define Pow3(A) _Pow3_macro(A)
#define Pow4(A) _Pow4_macro(A)
#define Pow5(A) _Pow5_macro(A)
#define Pow6(A) _Pow6_macro(A)
#define Pow7(A) _Pow7_macro(A)
#define Pow8(A) _Pow8_macro(A)
#define Pow9(A) _Pow9_macro(A)
#define Pow2p5(A) _Pow2p5_macro(A)
#define Pow3p5(A) _Pow3p5_macro(A)
#define Pow1p5(A) _Pow1p5_macro(A)
#define Pow1d4(A) _Pow1d4_macro(A)
#define Pow5d4(A) _Pow5d4_macro(A)
#define Pow1p5(A) _Pow1p5_macro(A)
#define Pow5p5(A) _Pow5p5_macro(A)
#define Pow3d4(A) _Pow3d4_macro(A)
#define Pow2d3(A) _Pow2d3_macro(A)
#define Pow4d3(A) _Pow4d3_macro(A)

#endif //POWER_OPERATIONS_ARE_FUNCTIONS

/*
 * Compare with DBL_MIN to make sure
 * numbers are non-zero but either positive
 * or negative
 */
#define SMALLEST_POSITIVE_NUMBER (DBL_MIN)
#define Positive_nonzero(A) ((A)>(SMALLEST_POSITIVE_NUMBER))
#define Negative_nonzero(A) ((A)<(-SMALLEST_POSITIVE_NUMBER))
#define Nonzero(A) (Positive_nonzero(A)||Negative_nonzero(A))


/*
 * Polynomial forms
 */

#define Quadratic(X,A,B,C) (                    \
        (A)+                                    \
        (B)*(X)+                                \
        (C)*Pow2(X)                             \
        )

#define Cubic(X,A,B,C,D) (                      \
        Quadratic((X),(A),(B),(C)) +            \
        (D)*Pow3(X)                             \
        )
                        
#define Quartic(X,A,B,C,D,E) (                  \
        Cubic((X),(A),(B),(C),(D)) +            \
        (E)*Pow4(X)                             \
        )
             
#define Quintic(X,A,B,C,D,E,F) (                \
        Quartic((X),(A),(B),(C),(D),(E)) +      \
        (F)*Pow5(X)                             \
        )


#define Quadratic_array(X,A) (                  \
        ((A)[0])+                               \
        ((A)[1])*(X)+                           \
        ((A)[2])*Pow2(X)                        \
        )

#define Cubic_array(X,A) (                      \
        Quadratic_array((X),(A)) +              \
        ((A)[3])*Pow3(X)                        \
        )
                        
#define Quartic_array(X,A) (                    \
        Cubic_array((X),(A)) +                  \
        ((A)[4])*Pow4(X)                        \
        )
             
#define Quintic_array(X,A) (                    \
        Quartic_array((X),(A)) +                \
        ((A)[5])*Pow5(X)                        \
        )

/*
 * Macro to evalutate Y if X is nonzero
 */
#define Eval_if_nonzero(X,Y) (Is_not_zero(X) ? (Y) : 0.0)

/*
 * 10^X - the opposite of log10
 */
#define Pow10(X) (pow(10.0,(X)))

/*
 * "Safe" pow(10,x) which first checks if x is NaN, if it
 * is, do nothing
 */
#define Safepow10(X) (isnan(X) ? (X) : Pow10(X)))

/*
 * Macro to check if X is a non-zero number
 */
#define Is_nonzero_number(X)                    \
    ((Is_really_not_zero(X) &&                  \
      !isnan(X) && !isinf(X)))


#endif // BINARY_MATHS_H
