#pragma once
#ifndef BINARY_MACROS_H
#define BINARY_MACROS_H

/*
 * The binary_c stellar population nucleosynthesis framework.
 *
 * Contact: r.izzard@surrey.ac.uk or rob.izzard@gmail.com
 *
 * http://personal.ph.surrey.ac.uk/~ri0005/binary_c.html
 * https://gitlab.eps.surrey.ac.uk/ri0005/binary_c
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-announce
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-devel
 * https://twitter.com/binary_c_code
 * https://www.facebook.com/groups/149489915089142/
 *
 * Please see the files README, LICENCE and CHANGES,
 * and the doc/ directory for documentation.
 *
 * This file contains most of the macros used in binary_c, 
 * and loads in macros from many other header files. You
 * should not include this file directly, include binary_c.h
 * instead.
 */

#include "binary_c_code_options.h"
#include "binary_c_debug.h"
#include "binary_c_function_map.h"
#include "binary_c_types.h"
#include "binary_c_maths.h"
#include "binary_c_string_macros.h"
#include "binary_c_parameters.h"
#include "binary_c_function_macros.h"
#include "timestep/timestep_macros.h"
#include "setup/metallicity_parameters.h"
#include "binary_c_error_codes.h"
#include "binary_c_version.h"
#include "wind/wind_macros.h"
#include "binary_c_derivatives.h"
#include "stellar_colours/stellar_colour_macros.h"
#include "events/events_macros.h"
#include "supernovae/sn.h"
#include "file/file_macros.h"
#include "disc/disc_macros.h"
#include "disc/disc_function_macros.h"
#include "nucsyn/nucsyn_macros.h"
#include "evolution/rejection_macros.h"

/* buffering states */
#define INTERNAL_BUFFERING_OFF 0
#define INTERNAL_BUFFERING_PRINT 1
#define INTERNAL_BUFFERING_STORE 2

/* batchmode controls */
#define BATCHMODE_ERROR -1
#define BATCHMODE_OFF 0 
#define BATCHMODE_ON 1
#define BATCHMODE_ON_AND_ECHO 2
#define BATCHMODE_LIBRARY 3
#define BATCH_SUBMODE_NORMAL 0
#define BATCH_SUBMODE_FIN 1

/* detailed_logging : probably not used much (may be removed!) */
#define DETAILED_OUTPUT_FILENAME "/tmp/c_binary_details.dat"

/* number of bytes in a MByte, GByte, etc. */
#define __KILOBYTE_MULTIPLIER 1000
#define KILOBYTE (__KILOBYTE_MULTIPLIER)
#define MEGABYTE (KILOBYTE*__KILOBYTE_MULTIPLIER)
#define GIGABYTE (MEGABYTE*__KILOBYTE_MULTIPLIER)
#define TERABYTE (GIGABYTE*__KILOBYTE_MULTIPLIER)
#define PETABYTE (TERABYTE*__KILOBYTE_MULTIPLIER)
#define EXABYTE (PETABYTE*__KILOBYTE_MULTIPLIER)

/* for pedants */
#define __KIBIBYTE_MULTIPLIER 1024
#define KIBIBYTE (__KIBIBYTE_MULTIPLIER)
#define MEBIBYTE (KIBIBYTE*__KIBIBYTE_MULTIPLIER)
#define GIBIBYTE (MEBIBYTE*__KIBIBYTE_MULTIPLIER)
#define TEBIBYTE (GIBIBYTE*__KIBIBYTE_MULTIPLIER)
#define PEBIBYTE (TEBIBYTE*__KIBIBYTE_MULTIPLIER)
#define EXIBYTE (PEBIBYTE*__KIBIBYTE_MULTIPLIER)

/*
 * Default log filename : can be overridden with command
 * line option --log_filename. 
 */
#ifdef CGI_EXTENSIONS
#define DEFAULT_LOG_OUTPUT_FILENAME "/tmp/c_log-www.dat"
#else//CGI_EXTENSIONS
#define DEFAULT_LOG_OUTPUT_FILENAME "/tmp/c_log.dat"
#endif //CGI_EXTENSIONS

#ifdef MEMMAP
/* Use mmap() for the log rather than a file */
#define BIGLOG_FILENAME "/tmp/c_biglog.dat"
/* Local biglog size is in local_binary_macros.h */
#define BIGLOG_SIZE (LOCAL_BIGLOG_SIZE*MEGABYTE)
#define BIGLOG_LINES (BIGLOG_SIZE/MEMMAP_STRING_LENGTH)

/* Note that MEMMAP_STRING_LENGTH must be <= STRING_LENGTH */
#define MEMMAP_STRING_LENGTH ((size_t)82)
#endif /* MEMMAP */

#define HRDIAG_FILENAME "/tmp/c_binary_hrdiag.dat"
#include "stellar_structure/stellar_structure_macros.h"

#ifdef NUCSYN
/* nucleosynthesis yields also need supernovae/sn.h */
#include "supernovae/sn.h"
#endif //NUCSYN

#include "binary_c_physical_constants.h"

#define MINIMUM_SEPARATION_TO_BE_CALLED_BINARY (100.0/R_SUN)
#define MAXIMUM_SEPARATION_TO_BE_CALLED_BINARY (1e15*R_SUN)


#define MIN_OBSERVATIONAL_STELLAR_MASS (0.01)
#define MAX_OBSERVATIONAL_BINARY_ORBITAL_PERIOD (1e7*YEAR_LENGTH_IN_DAYS)

#define OBSERVABLE_RADIAL_VELOCITY_MIN_DEFAULT 1.0

#define TINY_ECCENTRICITY 1e-10

/* 
 * Conversion factor for angular momentum : code units -> cgs 
 * J (code) * ANGULAR_MOMENTUM_CGS = J (cgs)
 */
#define ANGULAR_MOMENTUM_CGS (M_SUN*R_SUN*R_SUN/YEAR_LENGTH_IN_SECONDS)

#define NGTV -1.0
#define NGTV2 -2.0

#define Total_mass ((double)(stardata->star[0].mass+stardata->star[1].mass))
#define K2 0.1
#define K3 0.21
#define MR23YR 0.4311
#define MINIMUM_STELLAR_ANGMOM 1e-50
#define MAXIMUM_STELLAR_ANGMOM 1e100
#define MINIMUM_ORBITAL_ANGMOM 0.0
#define MAXIMUM_ORBITAL_ANGMOM 1e100
#define UNDEFINED_ORBITAL_ANGULAR_FREQUENCY 1e50

/* minimum metallicity used when log10(Z) is required */
#define MINIMUM_METALLICITY 1e-20

/* Handy for debugging */
#define UNUSED_FLOAT -666.6
#define NO_RESULT_YET_FLOAT -666.6
#define UNUSED_INT -666

/** labels used for logging **/

#define LOG_LABELS {                            \
        /* 0 */"OFF_MS",                        \
        /* 1 */"INITIAL ",                      \
        /* 2 */"TYPE_CHNGE",                    \
        /* 3 */"BEG_RCHE 1>2",                  \
        /* 4 */"COMENV  ",                      \
        /* 5 */"CONTACT ",                      \
        /* 6 */"COALESCE",                      \
        /* 7 */"END_RCHE 1!>2",                 \
        /* 8 */"GNTAGE  ",                      \
        /* 9 */"NO_REMNT",                      \
        /* 10 */"MAX_TIME",                     \
        /* 11 */"DISRUPT ",                     \
        /* 12 */"BEG_SYMB",                     \
        /* 13 */"END_SYMB",                     \
        /* 14 */"BEG_BSS ",                     \
        /* 15 */"END_BSS",                      \
        /* 16 */"q-inv",                        \
        /* 17 */"shrinkAGB",                    \
        /* 18 */"SN" ,                          \
        /* 19 */"Split",                        \
        /* 20 */"Start Barium Giant",           \
        /* 21 */"End Barium Giant",             \
        /* 22 */"Start novae",                  \
        /* 23 */"End novae",                    \
        /* 24 */"Start rapid rotator 1",        \
        /* 25 */"End rapid rotator 1",          \
        /* 26 */"Start rapid rotator 2",        \
        /* 27 */"End rapid rotator 2",          \
        /* 28 */"BEG_RCHE 2>1",                 \
        /* 29 */"END_RCHE 2!>1",                \
        /* 30 */"Start artificial accretion",   \
        /* 31 */"End artificial accretion",     \
        /* 32 */"Start medium rotator 1",       \
        /* 33 */"End medium rotator 1",         \
        /* 34 */"Start medium rotator 2",       \
        /* 35 */"End medium rotator 2",         \
        /* 36 */"Start tidal lock 1",           \
        /* 37 */"End tidal lock 1",             \
        /* 38 */"Start tidal lock 2",           \
        /* 39 */"End tidal lock 2",             \
        /* 40 */"Circularized",                 \
        /* 41 */"Start Carbon Star 1",          \
        /* 42 */"Start Carbon Star 2",          \
        /* 43 */"End Carbon Star 1",            \
        /* 44 */"End Carbon Star 2",            \
        /* 45 */"Start CEMP Star 1",          \
        /* 46 */"Start CEMP Star 2",          \
        /* 47 */"End CEMP Star 1",            \
        /* 48 */"End CEMP Star 2",            \
    }

#define OFF_MS_LABEL 0
#define INITIAL_LABEL 1
#define TYPE_CHNGE_LABEL 2
#define BEG_RCHE12_LABEL 3
#define COMENV_LABEL 4
#define CONTACT_LABEL 5
#define COALESCE_LABEL 6
#define END_RCHE12_LABEL 7
#define GNTAGE_LABEL 8
#define NO_REMNT_LABEL 9
#define MAX_TIME_LABEL 10
#define DISRUPT_LABEL 11
#define BEG_SYMB_LABEL 12
#define END_SYMB_LABEL 13
#define BEG_BSS_LABEL 14
#define END_BSS_LABEL 15
#define Q_INV_LABEL 16
#define SHRINK_AGB_LABEL 17
#define SN_LABEL 18
#define SPLIT_LABEL 19
#define BEG_BARIUM_LABEL 20
#define END_BARIUM_LABEL 21
#define BEG_NOVAE_LABEL 22
#define END_NOVAE_LABEL 23
#define BEG_RAPID_ROTATOR_LABEL1 24
#define END_RAPID_ROTATOR_LABEL1 25
#define BEG_RAPID_ROTATOR_LABEL2 26
#define END_RAPID_ROTATOR_LABEL2 27
#define BEG_RCHE_21_LABEL 28
#define END_RCHE_21_LABEL 29
#define START_ARTIFICIAL_ACCRETION_LABEL 30
#define END_ARTIFICIAL_ACCRETION_LABEL 31
#define BEG_MEDIUM_ROTATOR_LABEL1 32
#define END_MEDIUM_ROTATOR_LABEL1 33
#define BEG_MEDIUM_ROTATOR_LABEL2 34
#define END_MEDIUM_ROTATOR_LABEL2 35
#define BEG_TIDAL_LOCK_LABEL1 36
#define END_TIDAL_LOCK_LABEL1 37
#define BEG_TIDAL_LOCK_LABEL2 38
#define END_TIDAL_LOCK_LABEL2 39
#define CIRCULARIZED_LABEL 40
#define BEG_CARBON_LABEL1 41
#define BEG_CARBON_LABEL2 42
#define END_CARBON_LABEL1 43
#define END_CARBON_LABEL2 44
#define BEG_CEMP_LABEL1 45
#define BEG_CEMP_LABEL2 46
#define END_CEMP_LABEL1 47
#define END_CEMP_LABEL2 48
#define NLOG_LABELS 49

#define LOG_NONE 0
#define LOG_COMENV_PRE 1
#define LOG_PREMAINSEQUENCE 2
#define LOG_PROBABILITY 3
#define LOG_PERIASTRON_COLLISION 4
#define LOG_SN_AND_KICK 5
#define LOG_OFF_MS 6
#define LOG_STELLAR_TYPE_CHANGE 7
#define LOG_COMENV_POST 8
#define LOG_SPLIT_EVOLUTION 9
#define LOG_CIRCUMBINARY_DISC 10
#define LOG_ROCHE 11
#define LOG_DISC 12
#define LOG_CONTACT 13
#define LOG_RANDOM 14
#define LOG_REJECT 15
#define LOG_TERMINATE 16
#define NLOGFLAGS 17

#define LOCK_shrinkAGB 1
#define LOCK_BSS 2
#define LOCK_NO_REMNANT 3
#define LOCK_RLOF 4
#define LOCK_SYMBIOTIC 5
#define LOCK_BOTH_POST_MS 6
#define LOCK_BARIUM_STAR 7
#define LOCK_NOVAE 8
#define LOCK_RAPID_ROTATOR1 9
#define LOCK_RAPID_ROTATOR2 10
#define LOCK_ARTIFICIAL_ACCRETION 11
#define LOCK_MEDIUM_ROTATOR1 12
#define LOCK_MEDIUM_ROTATOR2 13
#define LOCK_TIDAL_LOCK1 14
#define LOCK_TIDAL_LOCK2 15
#define LOCK_CARBON_STAR1 16
#define LOCK_CARBON_STAR2 17
#define LOCK_CEMP_STAR1 18
#define LOCK_CEMP_STAR2 19
#define NLOCKFLAGS 20

/* The sizes of arrays used in the code */
/* NB add one to fortran array size so we can use 1 based indices (sigh) */
#define GIANT_BRANCH_PARAMETERS_SIZE 116
#define MAIN_SEQUENCE_PARAMETERS_SIZE 144
#define NUMBER_OF_METALLICITY_PARAMETERS 15

#define GB_ARRAY_SIZE 12
#define LUMS_ARRAY_SIZE 9
#define TSCLS_ARRAY_SIZE 15
#define COLLISION_MATRIX_SIZE 15

/* constants used in set_metallicity_parameters and/or zfuncs */
#define ZPAR_M2 (2.0)
#define ZPAR_M2_3 (8.0)
#define ZPAR_M2_3_5 (11.313708499) 
#define ZPAR_M3 (16.0)
#define LALPH_MCUT (2.0)
#define RGAMM_M1 (1.0)


#ifdef DEBUG
/** indentation global variable, makes debugging information **/
/** easier to see **/
#define INDENTATION_CHARACTER "->"
#endif /* DEBUG */

/** stellar types contains the star types and handy macros for them **/
#include "binary_c_stellar_types.h"

/* Some wolf-rayet stuff */
#define WR_START 0
#define WR_END 1
#define POST_WR_PRE_SUPERNOVA 2
#define POST_WR_POST_SUPERNOVA 3
#define WR_MEMLOG_LINES 4
#define WR_LOG_FILENAME "/tmp/wr_start_stop_log.dat"

/* Black Hole mass prescritions */
#define BH_HURLEY2002 0
#define BH_BELCZYNSKI 1
#define BH_SPERA2015 2
#define BH_FRYER12_DELAYED 3
#define BH_FRYER12_RAPID 4

/*
 * Max split depth is the maximum depth of splitting that can occur in
 * distributed kicking : see binary_star_functions/distributed_kick.c and
 * adaptive_grid/get_signal_from_filelog.c
 */
#define MAX_SPLIT_DEPTH 10

// different wolf rayet type winds
#define WR_WIND_BSE 0 
#define WR_WIND_MAEDER_MEYNET 1
#define WR_WIND_NUGIS_LAMERS 2
#define WR_WIND_ELDRIDGE 3

/* different GB winds */
#define GB_WIND_REIMERS 0
#define GB_WIND_SCHROEDER_CUNTZ_2005 1

/* different AGB winds */
#define AGB_WIND_VW93_KARAKAS 0
#define AGB_WIND_VW93_ORIG 1
#define AGB_WIND_REIMERS 2
#define AGB_WIND_BLOECKER 3
#define AGB_WIND_VAN_LOON 4
#define AGB_WIND_ROB_CWIND 5
#define AGB_WIND_VW93_KARAKAS_CARBON_STARS 6
#define AGB_WIND_VW93_ORIG_CARBON_STARS 7
#define AGB_WIND_MATTSSON 8

/*
 * TMP_LOCATION is the location of your /tmp direction - it gets prepended so
 * no trailing slash please - a value of "" leaves /tmp as your temporary
 * directory (the usual case)
 */
#define TMP_LOCATION ""

/* timescale macros - corresponding to those in calc_lum_and_evol_time */
#define T_MS 0
#define T_BGB 1
#define T_HE_IGNITION 2
#define T_HE_BURNING 3
#define T_GIANT_TINF_1 4
#define T_GIANT_TINF_2 5
#define T_GIANT_TX 6
#define T_EAGB_TINF_1 7
#define T_EAGB_TINF_2 8
#define T_EAGB_T 9
#define T_TPAGB_TINF_1 10
#define T_TPAGB_TINF_2 11
#define T_TPAGB_TX 12
#define T_TPAGB_FIRST_PULSE 13
#define T_TMCMAX 14

#define TIMESCALES_STRINGS {                            \
                            "MS",                       \
                            "BGB",                      \
                            "HE_IGNITION",              \
                            "HE_BURNING",               \
                            "GIANT_TINF1",              \
                            "GIANT_TINF2",              \
                            "GIANT_T",                  \
                            "EAGB_TINF_1",              \
                            "EAGB_TINF_2",              \
                            "EAGB_T",                   \
                            "TPAGB_TINF_1",             \
                            "TPAGB_TINF_2",             \
                            "TPAGB_T",                  \
                            "TPAGB_FIRST_PULSE",        \
                            "T_TMCMAX"                  \
    }

/* ditto for luminosity macros */
#define L_ZAMS 1
#define L_END_MS 2
#define L_BGB 3
#define L_HE_IGNITION 4
#define L_HE_BURNING 5
#define L_LMX 6
#define L_BAGB 7
#define L_TPAGB_FIRST_PULSE 8

/* and GB */
#define GB_EFFECTIVE_AH 1
#define GB_A_H_HE 2
#define GB_B 3
#define GB_D 4
#define GB_p 5
#define GB_q 6
#define GB_Mx 7
#define GB_A_HE 8
#define GB_MC_BGB 9
#define GB_AXEL_1 10
#define GB_AXEL_2 11
#define GB_SMOOTH_1 10
#define GB_SMOOTH_2 11

/* commonly used metallicity parameters */
#define ZPAR_METALLICITY 0
#define ZPAR_MASS_MS_HOOK 1 
#define ZPAR_MASS_HE_FLASH 2
#define ZPAR_MASS_FGB 3

#define Sqrt_eccfac sqrt(Max(1.0-stardata->common.orbit.eccentricity*stardata->common.orbit.eccentricity,0.0))

/* spectral types */
#define Spectral_Type int
#define Long_Spectral_Type double
#define SPECTRAL_TYPE_UNDEFINED -1
#define SPECTRAL_TYPE_O 0
#define SPECTRAL_TYPE_B 1
#define SPECTRAL_TYPE_A 2
#define SPECTRAL_TYPE_F 3
#define SPECTRAL_TYPE_G 4
#define SPECTRAL_TYPE_K 5
#define SPECTRAL_TYPE_M 6
#define SPECTRAL_TYPE_STRINGS {"O","B","A","F","G","K","M"}

#define INTERPOLATE_DO_NOT_USE_CACHE 0
#define INTERPOLATE_USE_CACHE 1

/* Which function to check? */
#ifndef SYSCHECK
#define SYSCHECK /* */
#endif //!SYSCHECK

/* R star merger types */
#define RSTAR_HE_GB 1
#define RSTAR_CHeB_GB 2
#define RSTAR_HeWD_GB 3
#define RSTAR_CH 4
#define RSTAR_HeWD_HG 5
#define RSTAR_GB_HG 6
#define RSTAR_HG_HG 7
#define RSTAR_AGB 8
#define RSTAR_He 9
#define RSTAR_CONeWD 10
#define RSTAR_GB_GB 11
#define RSTAR_OTHER 12


/* common envelope prescriptions */
#define COMENV_BSE 0 
#define COMENV_NELEMANS_TOUT 1
#define COMENV_NANDEZ2016 2

/* common envelope spin algorithms */
#define COMENV_EJECTION_SPIN_METHOD_DO_NOTHING 0
#define COMENV_EJECTION_SPIN_METHOD_SYNCHRONIZE 1 

#define COMENV_MERGER_SPIN_METHOD_SPECIFIC 0
#define COMENV_MERGER_SPIN_METHOD_CONSERVE_ANGMOM 1
#define COMENV_MERGER_SPIN_METHOD_CONSERVE_OMEGA 2
#define COMENV_MERGER_SPIN_METHOD_BREAKUP 3

#define REMNANT_LUMINOSITY 1e-10
#define REMNANT_RADIUS 1e-10

/* rotationally enhanced mass-loss prescriptions */
#define ROTATIONALLY_ENHNACED_MASSLOSS_NONE 0
#define ROTATIONALLY_ENHNACED_MASSLOSS_LANGER_FORMULA 1
#define ROTATIONALLY_ENHNACED_MASSLOSS_ANGMOM 2
#define ROTATIONALLY_ENHNACED_MASSLOSS_LANGER_FORMULA_AND_ANGMOM 3

/* a very large dM/dt (Msun/y) */
#define VERY_LARGE_MASS_TRANSFER_RATE (1e50)

/* a very small dM/dt (Msun/y) */
#define VERY_SMALL_MASS_TRANSFER_RATE (1e-50)

/*
 * minimum envelope mass for RLOF  
 * (only when using BSE style RLOF interpolation : binary_c
 *  defaults to a more stable algorithm)
 */
#define MINIMUM_ENVELOPE_MASS_FOR_RLOF 0.05

/*
 * If the separation drops below this during BSE-style interpolation,
 * simply allow the system to merge.
 */
#define MINIMUM_RLOF_INTERPOLATION_SEPARATION 1e-6

/* RLOF methods */
#define RLOF_METHOD_BSE 0
#define RLOF_METHOD_ADAPTIVE 1
#define RLOF_METHOD_RITTER 2
#define RLOF_METHOD_CLAEYS 3

#define RLOF_RATE_UNDEF 0
#define RLOF_RATE_BSE 1
#define RLOF_RATE_ADAPTIVE 2

/* RLOF interpolation methods */
#define RLOF_INTERPOLATION_BINARY_C 0
#define RLOF_INTERPOLATION_BSE 1

/* critical mass transfer rate algorithms */
#define DONOR_RATE_ALGORITHM_CLAEYS2014 -1
#define DONOR_RATE_ALGORITHM_BSE -2

/* Qcrit on the giant branch algorithms */
#define QCRIT_GB_BSE -1
#define QCRIT_GB_HJELLMING_WEBBINK -2
#define QCRIT_GB_Q_NO_COMENV -3
#define QCRIT_GB_CHEN_HAN_TABLE -4
#define QCRIT_GB_CHEN_HAN_FORMULA -5
#define QCRIT_GB_DEFAULT QCRIT_GB_BSE
#define QCRIT_GB_GE2015 -6
#define QCRIT_GB_VOS2018 -7

/*
 * critical q by stellar type : 
 * negative means use an appropriate algorithm 
 */
#define QCRIT_DEFAULTS 0,0,4.0,-1,

#define RLOF_NEITHER_STAR -1

/* RLOF cases and stability */
#define RLOF_NONE 0
#define RLOF_A 1
#define RLOF_B 2
#define RLOF_C 3
#define RLOF_OTHER 4

#define RLOF_INSTABILITY_BLOCKED -2
#define RLOF_IMPOSSIBLE -1
#define RLOF_STABLE 0
#define RLOF_UNSTABLE 1
#define RLOF_UNSTABLE_LOW_MASS_MS_COMENV 2
#define RLOF_UNSTABLE_GIANT_COMENV 3
#define RLOF_UNSTABLE_WD_COMENV 4
#define RLOF_UNSTABLE_NS 5
#define RLOF_UNSTABLE_BH 6
#define RLOF_UNSTABLE_MS_MERGER 7
#define RLOF_UNSTABLE_VERY_LARGE_DONOR 8

#define RLOF_STRINGS ".","A","B","C","x"


/* how do we lose angular momentum? */
#define WIND_ANGMOM_LOSS_BSE 0
#define WIND_ANGMOM_LOSS_LW 1 
#define WIND_ANGMOM_LOSS_LW_HYBRID 2 
#define WIND_ANGMOM_LOSS_SPHERICALLY_SYMMETRIC 3

#define KICK_RANDOM 0 
#define KICK_STRAIGHT_UP 1
#define KICK_FORWARD 2
#define KICK_BACKWARD 3
#define KICK_INWARD 4
#define KICK_OUTWARD 5

/* E2 prescriptions */
#define E2_HURLEY_2002 0
#define E2_IZZARD 1

/* Wind-RLOF methods */
#define WRLOF_NONE 0
#define WRLOF_Q_DEPENDENT 1
#define WRLOF_QUADRATIC 2

#ifdef USE_ANSI_COLOURS
/* ANSI Colours */
#define BLACK "\e[0;30m"

#define RED "\e[0;31m"
#define GREEN "\e[0;32m"
#define YELLOW "\e[0;33m"
#define ORANGE "\e[0;33m"
#define BLUE "\e[0;34m"
#define MAGENTA "\e[0;35m"
#define CYAN "\e[0;36m"
#define WHITE "\e[0;37m"
#define BOLD "\e[0;1m"

#define BRIGHT_RED "\e[1;31m"
#define BRIGHT_GREEN "\e[1;32m"
#define BRIGHT_YELLOW "\e[1;33m"
#define BRIGHT_BLUE "\e[1;34m"
#define BRIGHT_MAGENTA "\e[1;35m"
#define BRIGHT_CYAN "\e[1;36m"
#define BRIGHT_WHITE "\e[1;37m"
#define BRIGHT_BOLD "\e[1;1m"

#define COLOUR_RESET "\e[0;0m"
#define DEBUG_COLOURS
#define FILE_LOG_COLOURS
#else//USE_ANSI_COLOURS
#define BLACK ""
#define RED ""
#define BRIGHT_RED ""
#define GREEN ""
#define BRIGHT_GREEN ""
#define YELLOW ""
#define BRIGHT_YELLOW ""
#define BLUE ""
#define BRIGHT_BLUE ""
#define MAGENTA ""
#define BRIGHT_MAGENTA ""
#define CYAN ""
#define BRIGHT_CYAN ""
#define WHITE ""
#define BRIGHT_WHITE ""
#define BOLD ""
#define BRIGHT_BOLD ""
#define COLOUR_RESET ""
#endif//USE_ANSI_COLOURS

/* Thorne-Zytkow channels */
#define TZ_CHANNEL_COMENV 1
#define TZ_CHANNEL_MIX 2

/* RLOF control flags */
#define RLOF_NONE 0 
#define RLOF_NORMAL 1
#define RLOF_COMENV_MERGER 2
#define RLOF_COMENV_DISRUPT 3
#define RLOF_DYNAMICAL_WD 4
#define RLOF_SHORT_GRB 5
#define RLOF_BLACK_HOLE_MERGER 6
#define RLOF_MERGER 7
#define RLOF_DYNAMICAL_MS 8

#define END_RLOF 0
#define CONTINUE_RLOF 1

#define GALACTIC_MODEL_NELEMANS2004 0

/*
 * AGB core calculation algorithms
 */
#define AGB_CORE_ALGORITHM_DEFAULT 0
#define AGB_CORE_ALGORITHM_HURLEY 1
#define AGB_CORE_ALGORITHM_KARAKAS 2


/*
 * AGB radius calculation algorithms
 */
#define AGB_RADIUS_ALGORITHM_DEFAULT 0
#define AGB_RADIUS_ALGORITHM_HURLEY 1
#define AGB_RADIUS_ALGORITHM_KARAKAS 2

/*
 * AGB luminosity calculation algorithms
 */
#define AGB_LUMINOSITY_ALGORITHM_DEFAULT 0
#define AGB_LUMINOSITY_ALGORITHM_HURLEY 1
#define AGB_LUMINOSITY_ALGORITHM_KARAKAS 2

/*
 * AGB third dredge up algorithms
 */
#define AGB_THIRD_DREDGE_UP_ALGORITHM_DEFAULT 0
#define AGB_THIRD_DREDGE_UP_ALGORITHM_HURLEY 1
#define AGB_THIRD_DREDGE_UP_ALGORITHM_KARAKAS 2
#define AGB_THIRD_DREDGE_UP_ALGORITHM_STANCLIFFE 3


/*
 * Angular momentum in RLOF transfer model
 */
#define RLOF_ANGULAR_MOMENTUM_TRANSFER_MODEL_BSE 0
#define RLOF_ANGULAR_MOMENTUM_TRANSFER_MODEL_CONSERVATIVE 1

/*
 * RLOF gamma when non-conservative
 */
#define RLOF_NONCONSERVATIVE_GAMMA_DONOR -1
#define RLOF_NONCONSERVATIVE_GAMMA_ISOTROPIC -2

/*
 * Graviational radiation models
 */
#define GRAVITATIONAL_RADIATION_BSE 0
#define GRAVITATIONAL_RADIATION_BSE_WHEN_NO_RLOF 1
#define GRAVITATIONAL_RADIATION_NONE 2
#define GRAVITATIONAL_RADIATION_LANDAU_LIFSHITZ 3
#define GRAVITATIONAL_RADIATION_LANDAU_LIFSHITZ_WHEN_NO_RLOF 4


/*
 * Stack options
 */
#define STACK_WARNING 0
#define STACK_SET 1
#define STACK_DO_NOTHING 2

/* 
 * fprintf file format filters
 */
#define FILE_FILTER_NOTHING 0
#define FILE_FILTER_STRIP_ARGUMENTS 1

/*
 * Common envelope prescriptions 
 * (other than the default). Comment out
 * to not build the code, or define as a number
 * which can be passed in as stardata->preferences->lambda_ce
 */
#define LAMBDA_CE_DEWI_TAURIS -1.0
#define LAMBDA_CE_WANG_2016 -2.0
#define LAMBDA_CE_POLYTROPE -3.0

/*
 * Number of Lagrange points
 */
#define NUMBER_OF_LAGRANGE_POINTS 5

/*
 * Debugging numbers for generic_bisect.
 */
#define BISECTOR_DISC_TVISC 0
#define BISECTOR_DISC_J 1
#define BISECTOR_DISC_F 2
#define BISECTOR_DISC_RHALFJ 3
#define BISECTOR_DISC_PRESSURE_RADIUS 4
#define BISECTOR_COMENV_DM 5
#define BISECTOR_DISC_OWEN_RADIUS 6
#define BISECTOR_DISC_VISCOUS 7
#define BISECTOR_DISC_MASS_RADIUS 8
#define BISECTOR_DISC_RIN_MIN 9
#define BISECTOR_DISC_BISECTION_ROOTER 10

// to be removed:
#define BISECTOR_DISC_PHALFJ 11

/*
 * Turn monochecks on or off.
 */
#define BISECT_NO_MONOCHECKS 0
#define BISECT_USE_MONOCHECKS 1
#define BISECT_USE_MONOCHECKS_FATAL 2


#define BISECTOR_STRINGS                        \
    "Tvisc0",                                   \
        "J",                                    \
        "F",                                    \
        "RhalfJ",                               \
        "PfromR",                               \
        "ComenvDM",                             \
        "OwenRadius",                           \
        "Viscous",                              \
        "RhalfM",                               \
        "RinMin",                               \
        "Disc_bi_rooter"

#define BISECT_RETURN_FAIL -1.0



#define NONCONSERVATIVE_TYPE_NONE 0
#define NONCONSERVATIVE_TYPE_SUPER_EDDINGTON 1
#define NONCONSERVATIVE_DISC_WIND 2

/*
 * Nova mechanisms
 */
#define NOVA_RETENTION_ALGORITHM_CONSTANT 0
#define NOVA_RETENTION_ALGORITHM_CLAEYS2014 1
#define NOVA_RETENTION_ALGORITHM_HILLMAN2015 2

/* muiltiplier to make the Roche radius very large */
#define HUGE_ROCHE_MULTIPLIER (1e10)

/* a large separation to mimic single stars */
#define VERY_LARGE_SEPARATION 1e50

/* an impossibly large stellar mass */
#define IMPOSSIBLY_LARGE_MASS 1e50

/*
 * Accretion limitors
 */
#define ACCRETION_LIMIT_EDDINGTON 0
#define ACCRETION_LIMIT_THERMAL 1
#define ACCRETION_LIMIT_DYNAMICAL 2
#define ACCRETION_LIMIT_HACHISU 3
#define NUMBER_OF_ACCRETION_LIMITS 4


#define HACHISU_IGNORE_QCRIT -1.0

/*
 * Donor limitors
 */
#define DONOR_LIMIT_THERMAL 0
#define DONOR_LIMIT_DYNAMICAL 1
#define DONOR_LIMIT_ENVELOPE 2
#define DONOR_LIMIT_HG_FRACTIONAL_ENVELOPE 3
#define NUMBER_OF_DONOR_LIMITS 4
#define DONOR_LIMIT_STRINGS {                   \
        "Thermal",                              \
        "Dynamical",                            \
        "Envelope",                             \
        "HG fractional envelope",               \
    }


/*
 * Nova states
 */
#define NOVA_STATE_NONE 0
#define NOVA_STATE_PRE_EXPLOSION 1
#define NOVA_STATE_EXPLODING 2
#define NOVA_STATE_QUIESCENT 3
#define NOVA_STATE_STRINGS {                    \
        "None",                                 \
        "Pre-explosion",                        \
        "Exploding",                            \
        "Quiescent"                             \
    }

/*
 * TPAGB luminosity algorithm
 */
#define TPAGB_LUMINOSITY_AVERAGE 0
#define TPAGB_LUMINOSITY_DIPS 1

/* Opacity algorithms */
#define OPACITY_ALGORITHM_PACZYNSKI 0
#define OPACITY_ALGORITHM_FERGUSON_OPAL 1
#define OPACITY_ALGORITHM_STARS 2

/* Equation of state algorithms */
#define EQUATION_OF_STATE_PACZYNSKI 0

/*
 * Stellar structure algorithms
 */ 
#define STELLAR_STRUCTURE_ALGORITHM_MODIFIED_BSE 0
#define STELLAR_STRUCTURE_ALGORITHM_NONE 1
#define STELLAR_STRUCTURE_ALGORITHM_EXTERNAL_FUNCTION 2
#define STELLAR_STRUCTURE_ALGORITHM_BINT 3

/*
 * Integer to describe the stardata dump file format
 */
#define STARDATA_DUMP_FORMAT 1


/*
 * A real millisecond
 */
#ifdef CPUFREQ
#define millisecond 1e3*((float)CPUFREQ)
#endif

/*
 * An SVN version number to signal that SVN is undefined
 */
#define SVN_REVISION_UNDEFINED -1.0

/*
 * Solar symbol
 *
 * Use either:
 * U+2609 ☉ 
 * or 
 * U+2299 ⊙
 */
#define SOLAR_SYMBOL "☉"

/*
 * Convenience macros
 */
#define L_SOLAR "L☉"
#define M_SOLAR "M☉"
#define R_SOLAR "R☉"
#define J_CGS "g cm^2 s^-1"

/*
 * Magnetic braking algorithms
 */
#define MAGNETIC_BRAKING_ALGORITHM_HURLEY_2002 0
#define MAGNETIC_BRAKING_ALGORITHM_ANDRONOV_2003 1
#define MAGNETIC_BRAKING_ALGORITHM_BARNES_2010 2
#define MAGNETIC_BRAKING_ALGORITHM_RAPPAPORT_1983 3

/*
 * Default gamma for Rappaport type systemsx
 */
#define MAGNETIC_BRAKING_GAMMA_DEFAULT 3.0

/*
 * Small envelope algorithms
 */
#define SMALL_ENVELOPE_METHOD_BSE 0
#define SMALL_ENVELOPE_METHOD_MILLER_BERTOLAMI 1

/*
 * For stellar_angular_momentum.c
 */
#define ANGULAR_MOMENTUM_STAR 0
#define ANGULAR_MOMENTUM_CORE 1
#define ANGULAR_MOMENTUM_ENVELOPE 2

/*
 * WD WD merger algorithms
 */

#define WDWD_MERGER_ALGORITHM_BSE 0
#define WDWD_MERGER_ALGORITHM_PERETS2019 1
#define WDWD_MERGER_ALGORITHM_CHEN2016 2

/*
 * Solvers
 */
#define SOLVER_FORWARD_EULER 0
#define SOLVER_RK2 1
#define SOLVER_RK4 2
#define SOLVER_PREDICTOR_CORRECTOR 3

#ifdef CODESTATS
#define CODESTAT_MEMCPY 0
#define CODESTAT_MALLOC 1
#define CODESTAT_CALLOC 2
#define CODESTAT_REALLOC 3
#define CODESTAT_MEMSET 4
#define NUMBER_OF_CODESTATS 5
#endif // CODESTATS


#define UPDATE_TIME TRUE
#define DO_NOT_UPDATE_TIME FALSE

/*
 * Actions for copy_stardata()
 */
#define COPY_STARDATA_PREVIOUS_NONE 0
#define COPY_STARDATA_PREVIOUS_COPY 1
#define COPY_STARDATA_MAINTAIN_FROM_STACK_POINTERS 2
#define COPY_STARDATA_MAINTAIN_TO_STACK_POINTERS 3

/*
 * Rotational velocity algorithms
 */
#define VROT_NON_ROTATING 1e-10
#define VROT_BSE 0
#define VROT_BREAKUP -1
#define VROT_SYNC -2

/*
 * Nova settings
 */
#define BETA_REVERSE_NOVAE_GEOMETRY -1.0

/*
 * Macro to ignore artificial accretion
 */
#define ARTIFICIAL_ACCRETION_IGNORE -1.0

/*
 * Maximum stellar angular momentum change per timestep
 */
#define MAX_STELLAR_ANGMOM_CHANGE_DEFAULT 0.05

/*
 * Minimum angular momentum for timestep/rejection tests
 */
#define MIN_ANGMOM_FOR_TIMESTEP_AND_REJECTION_TESTS 1e-2

/*
 * Nova types
 */
#define NOVA_TYPE_HYDROGEN 0
#define NOVA_TYPE_HELIUM 1

/*
 * Number of letters in the (English) alphabet
 */
#define NUMBER_OF_LETTERS_IN_THE_ALPHABET 26

/*
 * Number of alphabetical cases
 */
#define NUMBER_OF_ALPHABETICAL_CASES 2

/*
 * Alpha-args parameters list size (see parse_arguments.c)
 */
#define ALPHA_ARGS_SIZE \
    (NUMBER_OF_LETTERS_IN_THE_ALPHABET * NUMBER_OF_ALPHABETICAL_CASES)

/*
 * Default value for effective_metallicity and nucsyn_metallicity
 */
#define DEFAULT_TO_METALLICITY (-1.0)


/*
 * Overspin algorithms
 */
#define OVERSPIN_BSE 0
#define OVERSPIN_MASSLOSS 1

#endif /* BINARY_MACROS_H */
