#pragma once
#ifndef BINARY_C_FUNCTION_MACROS_H
#define BINARY_C_FUNCTION_MACROS_H

/*
 * The binary_c stellar population nucleosynthesis framework.
 *
 * Contact: r.izzard@surrey.ac.uk or rob.izzard@gmail.com
 *
 * http://personal.ph.surrey.ac.uk/~ri0005/binary_c.html
 * https://gitlab.eps.surrey.ac.uk/ri0005/binary_c
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-announce
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-devel
 * https://twitter.com/binary_c_code
 * https://www.facebook.com/groups/149489915089142/
 *
 * Please see the files README, LICENCE and CHANGES,
 * and the doc/ directory for documentation.
 *
 *
 * This file contains the main group of function macros
 * defined for binary_c.
 */

#include "./events/events_function_macros.h"

/*
 * Logical not
 */
#define Not(A) ((A)==TRUE ? FALSE : TRUE)

/*
 * Floating point precision : 
 * http://christian-seiler.de/projekte/fpmath/
 */
#if defined FPU_PRECISION && !defined NO_FPU_CONTROL
#define Set_FPU_precision fpu_control_t fpu_oldcw,fpu_cw;       \
    _FPU_GETCW(fpu_oldcw);                                      \
    fpu_cw = (fpu_oldcw &                                       \
              ~ _FPU_EXTENDED &                                 \
              ~ _FPU_DOUBLE &                                   \
              ~ _FPU_SINGLE) |                                  \
        _FPU_DOUBLE;                                            \
    _FPU_SETCW(fpu_cw);

#else
#define Set_FPU_precision /* */
#endif // FPU_PRECISION

/*
 * Floating point policy : this is determined 
 * by the FPU_CAPTURE_* macros, defined in binary_c_code_options.h
 */

#ifdef FPU_CAPTURE_INVALID
#define FPU_POLICY_INVALID | FE_INVALID
#else
#define FPU_POLICY_INVALID
#endif
#ifdef FPU_CAPTURE_INEXACT
#define FPU_POLICY_INEXACT | FE_INEXACT
#else
#define FPU_POLICY_INEXACT
#endif

#ifdef FPU_CAPTURE_OVERFLOWS
#define FPU_POLICY_OVERFLOWS | FE_OVERFLOW
#else
#define FPU_POLICY_OVERFLOWS
#endif

#ifdef FPU_CAPTURE_UNDERFLOWS
#define FPU_POLICY_UNDERFLOWS | FE_UNDERFLOW
#else
#define FPU_POLICY_UNDERFLOWS
#endif
#ifdef FPU_CAPTURE_DIVIDE_BY_ZERO
#define FPU_POLICY_DIVIDE_BY_ZERO | FE_DIVBYZERO
#else
#define FPU_POLICY_DIVIDE_BY_ZERO
#endif

#define FPU_POLICY                              \
    FPU_POLICY_INVALID                          \
    FPU_POLICY_OVERFLOWS                        \
    FPU_POLICY_UNDERFLOWS                       \
    FPU_POLICY_DIVIDE_BY_ZERO                   \
    FPU_POLICY_INEXACT

#define Set_FPU_policy /**/
//feenableexcept(0 FPU_POLICY)

#define Fpstring(X)                                     \
    (                                                   \
        fpclassify(X)==FP_NAN ?       "NaN" :           \
        fpclassify(X)==FP_INFINITE ?  "Inf" :           \
        fpclassify(X)==FP_ZERO ?      "Zero" :          \
        fpclassify(X)==FP_SUBNORMAL ? "Subnormal" :     \
        fpclassify(X)==FP_NORMAL ?    "Normal" :        \
        "Unknown"                                       \
        )

#ifdef STACK_CHECKS
#define Set_stack_size                                                  \
    if(STACK_ACTION != STACK_DO_NOTHING)                                \
    {                                                                   \
        struct rlimit rl;                                               \
        if(likely(getrlimit(RLIMIT_STACK,&rl)==0))                      \
        {                                                               \
            unsigned long int stacksize = STACK_SIZE*1000L*1000L;       \
            if(rl.rlim_cur!=RLIM_INFINITY && rl.rlim_cur < stacksize)   \
            {                                                           \
                printf("Stack is too small: currently %ld (max %ld)\n", \
                       (long int) rl.rlim_cur,                          \
                       (long int)rl.rlim_max);                          \
                if(STACK_ACTION == STACK_SET)                           \
                {                                                       \
                    rl.rlim_cur = stacksize;                            \
                    if(setrlimit(RLIMIT_STACK,&rl)!=0)                  \
                    {                                                   \
                        fprintf(stderr,                                 \
                                "Unable to set stack to %ld MB (%ld bytes)\n", \
                                (long int)STACK_SIZE,                   \
                                (long int)STACK_SIZE*1024L*1024L);      \
                        Exit_binary_c_no_stardata(                      \
                            BINARY_C_STACK_FAIL,                        \
                            "Unable to set stack to %ld MB (%ld bytes)\n", \
                            (long int)STACK_SIZE,                       \
                            (long int)STACK_SIZE*1024L*1024L);          \
                    }                                                   \
                    else                                                \
                    {                                                   \
                        getrlimit(RLIMIT_STACK,&rl);                    \
                        printf("Stack set to %ld\n",                    \
                               (long int) rl.rlim_cur);                 \
                    }                                                   \
                }                                                       \
            }                                                           \
        }                                                               \
        else                                                            \
        {                                                               \
            fprintf(stderr,"Failed to determine current stack limit\n"); \
        }                                                               \
    }
#else
#define Set_stack_size /* */
#endif


/* Convert a C (integer) expression to Boolean type */
#define Boolean_(EXPR) ((EXPR) ? (TRUE) : (FALSE))

/*
 * Set, unset and append log strings
 * N = log number
 */
#define Set_logstring(N,...)                    \
    {                                           \
        set_logstring(N,                        \
                      stardata,                 \
                      __VA_ARGS__);             \
    }
#define Append_logstring(N,...)                 \
    {                                           \
        append_logstring(N,                     \
                         stardata,              \
                         __VA_ARGS__);          \
    }
#define Unset_logstring(N,...)                  \
    {                                           \
        set_logstring(N,                        \
                      stardata,                 \
                      NULL);                    \
    }

#define Append_logstring_debug1 if(0)printf("Append logstring: len of %p (%s) is %d, add Max %d chars of \"%s\" to %p which is \"%s\"\n", \
                                            stardata->model.logflagstrings[(N)], \
                                            stardata->model.logflagstrings[(N)], \
                                            istring,                    \
                                            (STRING_LENGTH-2-istring),  \
                                            cstring,                    \
                                            stardata->model.logflagstrings[(N)], \
                                            stardata->model.logflagstrings[(N)]);

#define Append_logstring_debug2 if(0)printf("post-Append strlen %d\n",  \
                                            strlen(stardata->model.logflagstrings[(N)]));        




    
/*
 * Stellar structure calculations
 */


/*
 * macro definition of the index of the other star
 */
#define Other_star(N) ((N)==0 ? 1 : 0)
#define Other_star_struct(S) (&stardata->star[Other_star((S)->starnum)])

/*
 * Effective core mass
 */
#define Effective_core_mass(S) (                        \
        (ON_EITHER_MAIN_SEQUENCE((S)->stellar_type)) ?  \
        ((S)->mass) :                                   \
        ((S)->core_mass)                                \
        )

/* T_eff - the effective temperature - A is the star number */
#define Teff_from_luminosity_and_radius(L,R) (1000.0*sqrt(sqrt((1130.0*(L)/(Pow2(R))))))


/* T_eff - the effective temperature of star N */
#define Teff(N) (Teff_from_luminosity_and_radius(               \
                     stardata->star[(N)].luminosity,            \
                     stardata->star[(N)].radius))

/* same for a star struct (STAR) */
#define Teff_from_star_struct(STAR)                     \
    (Teff_from_luminosity_and_radius(                   \
        ((STAR)->luminosity),                           \
        ((STAR)->radius)))

#define HRdiagram(N)                            \
    log10(Teff(N)),                             \
        (log10(stardata->star[(N)].luminosity))

#define Envelope_mass(N) (stardata->star[(N)].mass-stardata->star[(N)].core_mass)

#define Q(N) (Is_zero(stardata->star[Other_star(N)].mass) ? 1e50 :      \
              stardata->star[(N)].mass/stardata->star[Other_star(N)].mass)

#define Mass_squared(B) (stardata->star[(B)].mass*stardata->star[(B)].mass)

#define Convective_star(N) (stardata->star[(N)].mass<MAX_CONVECTIVE_MASS && \
                            stardata->star[(N)].mass>MIN_CONVECTIVE_MASS)

#ifdef RLOF_RADIATION_CORRECTION
#define Roche_radius(Q,N) ((Q)<TINY ? 1e100 : ((N)*rl((Q),stardata->preferences->RLOF_f)))
#else
#define Roche_radius(Q,N) ((Q)<TINY ? 1e100 : ((N)*rl(Q)))
#endif
#define Roche_radius3(M1,M2,N) ((M2)<TINY ? 1e100 : Roche_radius(((M1)/(M2)),(N)))

/* surface hydrogen */
#ifdef NUCSYN
#define Hydrogen_mass_fraction (donor->Xenv[XH1])
#else
#define Hydrogen_mass_fraction (0.760 - 3.0 * stardata->common.metallicity)
#endif


/*
 * Decimal spectral type converter
 */
#define Float_spectral_type_to_integer(N)               \
    ((int)                                              \
     ((N)+0.05))
#define Float_spectral_type_to_subclass(N)                              \
    ((int)                                                              \
     ((int)(10.0 *((N)-Float_spectral_type_to_integer(N))+0.5)))

/*
 * Macros for making and copying stardata_t structs
 */
#define New_stardata (Malloc(sizeof(struct stardata_t)))
#define New_clear_stardata (Calloc(sizeof(struct stardata_t),1))

#ifdef _DEP
#define Copy_stardata(FROM,TO)                                          \
    printf("CS : %p to %p\n",(FROM),(TO));                              \
    fflush(NULL);                                                       \
    ((struct stardata_t*)memcpy((TO),(FROM),sizeof(struct stardata_t)))
#endif //_DEP

/* move stardata */
#define Move_stardata(FROM,TO)                                          \
    ((struct stardata_t*)memmove((TO),(FROM),sizeof(struct stardata_t)))

/* shallow stardata copy */
#define New_stardata_from(STARDATA)                                     \
    copy_stardata((STARDATA),(New_stardata),COPY_STARDATA_PREVIOUS_NONE)

/* clear stardata */
#define Clear_stardata(STARDATA)                        \
    memset((STARDATA),0,sizeof(struct stardata_t))

#define Copy_previous_stardatas(FROM,TO)                                \
    {                                                                   \
                                                                        \
        /* previous_stardatas is a new copy */                          \
        printf("malloc previous_stardatas %d\n",                        \
               (FROM)->n_previous_stardatas);                           \
        fflush(NULL);                                                   \
        (TO)->previous_stardatas =                                      \
            Malloc(sizeof(struct stardata_t *) *                        \
                   (FROM)->n_previous_stardatas);                       \
                                                                        \
        /* copy previous stardatas from FROM to TO */                   \
        int __i;                                                        \
        for(__i=0;__i<(FROM)->n_previous_stardatas;__i++)               \
        {                                                               \
            printf("Copy previous stardata %d = %p\n",                  \
                   __i,                                                 \
                   (FROM)->previous_stardatas[__i]);                    \
            fflush(NULL);                                               \
            (TO)->previous_stardatas[__i] =                             \
                (FROM)->previous_stardatas[__i] == NULL ?               \
                NULL :                                                  \
                New_stardata_from((FROM)->previous_stardatas[__i]);     \
        }                                                               \
                                                                        \
        /* administration */                                            \
        (TO)->n_previous_stardatas = (FROM)->n_previous_stardatas;      \
        (TO)->previous_stardata = (TO)->previous_stardatas[0];          \
                                                                        \
    }

/*
 * NB memcpy is faster than memmove
 */
#define Swap_stardatas(STARDATA1,STARDATA2)     \
    {                                           \
        struct stardata_t * _s = New_stardata;  \
        Copy_stardata((STARDATA2),_s);          \
        Copy_stardata((STARDATA1),(STARDATA2)); \
        Copy_stardata(_s,(STARDATA1));          \
        Safe_free(_s);                          \
    }


/*
 * Macros as above, but including the option to copy
 * the previous_stardatas stack.
 */
#define New_deep_stardata_from(STARDATA)                \
    ({                                                  \
        struct stardata_t * s = New_clear_stardata;     \
        copy_stardata((STARDATA),                       \
                      s,                                \
                      COPY_STARDATA_PREVIOUS_COPY);     \
        s;                                              \
    })

#define Clear_deep_stardata(STARDATA)                   \
    {                                                   \
        free_previous_stardatas(STARDATA);              \
        erase_events(STARDATA);                         \
        (STARDATA)->n_previous_stardatas = 0 ;          \
        (STARDATA)->previous_stardata = NULL;           \
        memset((STARDATA),0,sizeof(struct stardata_t)); \
    }

#define Swap_deep_stardatas(STARDATA1,STARDATA2)        \
    {                                                   \
        struct stardata_t * _s = New_stardata;          \
        Copy_deep_stardata((STARDATA2),_s);             \
        Copy_deep_stardata((STARDATA1),(STARDATA2));    \
        Copy_deep_stardata(_s,(STARDATA1));             \
        Safe_free(_s);                                  \
    }




/*
 * Macros for making and copying star_t structs.
 * NB in the Swap_stars use memcpy because it's 
 *    faster than memmove. 
 */
#define New_star (Malloc(sizeof(struct star_t)))
#define New_clear_star (Calloc(sizeof(struct star_t),1))
#define Copy_star(FROM,TO) memcpy((TO),(FROM),sizeof(struct star_t))
#define Move_star(FROM,TO) memmove((TO),(FROM),sizeof(struct star_t))
#define New_star_from(STAR) Copy_star((STAR),(New_star))
#define Clear_star(STAR) memset((STAR),0,sizeof(struct star_t))
#define Swap_stars(STAR1,STAR2)                 \
    {                                           \
        struct star_t * _s = New_star;          \
        Copy_star((STAR2),_s);                  \
        Copy_star((STAR1),(STAR2));             \
        Copy_star(_s,(STAR1));                  \
        Safe_free(_s);                          \
    }



/*
 * loop over the stars using variable A (which
 * you have to declare as an integer)
 *
 * Starloop loops over both stars
 * 
 * Evolving_STARLOOP loops but only on stars that are evolving
 * (stellar type < MASSLESS_REMNANT)
 *
 * SETstar sets a struct called 'star' to be star A
 * SETstars does the same, and calls the other star 'companion'
 */
#define Starloop(N) for((N)=0;(N)<NUMBER_OF_STARS;(N)++)

#define Evolving_STARLOOP(N)                            \
    (N)=-1;                                             \
    while(                                              \
        ( (N) = next_evolving_star( (N),stardata) )!=-1 \
        )

#define Nuclear_burning_STARLOOP(N)                             \
    (N)=-1;                                                     \
    while(                                                      \
        ( (N) = next_nuclear_burning_star( (N),stardata) )!=-1  \
        )


#define SETstar(N)                                      \
    struct star_t * const star MAYBE_UNUSED =           \
        (struct star_t * const) &(stardata->star[(N)]);

#define SETstarp(N)                                             \
    struct star_t * const star MAYBE_UNUSED =                   \
        (struct star_t * const)&((*stardata)->star[(N)]);

#define SETstars(N)                                                     \
    struct star_t * const star MAYBE_UNUSED =                           \
        (struct star_t * const)&(stardata->star[(N)]);                  \
    struct star_t * const companion MAYBE_UNUSED =                      \
        (struct star_t * const)&(stardata->star[Other_star(N)]);


/*
 * structs and integers often used in RLOF routines
 * (with suppression of unused variable warnings)
 */ 
#define RLOF_stars_structs                      \
    struct star_t * accretor MAYBE_UNUSED =     \
        &(stardata->star[naccretor]);           \
    struct star_t * donor MAYBE_UNUSED =        \
        &(stardata->star[ndonor]);

#define RLOF_stars                              \
    const int ndonor MAYBE_UNUSED =             \
        stardata->model.ndonor ;                \
    const int naccretor MAYBE_UNUSED =          \
        stardata->model.naccretor;              \
    RLOF_stars_structs;

#define Timer_seconds(N) (((float)(N))/((1e6*(float)CPUFREQ)))
#define Timer_microseconds(N) (((float)(N))/(((float)CPUFREQ)))

/************************************************************/
/* Data tables */

/* verbosity flag */
#define DTVB 0

/*
 * Functions to count the number of items in, and
 * define a new, data table.
 * TARGET is an (empty) pointer which is Malloced to the 
 * appropriate size.
 */ 
#define DataTableN(T) (sizeof(T)/sizeof(double))
#define DataTableNLines(T,P,D) (sizeof(T)/(sizeof(double)*((P)+(D))))


/*
 * Make a new data table at TARGET with original data at either
 * 
 * POINTER (for NewDataTable_from_Pointer which sets from a pre-allocated memory location) or
 * pr
 * ARRAY (for NewDataTable_from_Array, which sets from a static array)
 * new table is an empty struct data_table_t pointer at TARGET,
 * with NDATA data items and NPARAM parameters on each 
 * of NLINES lines of table.
 *
 * Usually the TARGET is a NULL pointer, but could (in principle)
 * already exist, in which case the Malloc is skipped.
 */

#define NewDataTable_from_Pointer(POINTER,TARGET,NPARAM,NDATA,NLINES)   \
    {                                                                   \
        (TARGET) = (struct data_table_t *)                              \
            Malloc(sizeof(struct data_table_t));                        \
        (TARGET)->data = (POINTER);                                     \
        (TARGET)->nlines = (int) (NLINES);                              \
        (TARGET)->ndata = (int) (NDATA);                                \
        (TARGET)->nparam = (int) (NPARAM);                              \
        if(DTVB)                                                        \
            fprintf(stderr,                                             \
                    "Alloc table at %p size %ld nparam %d ndata %d nlines %d\n", \
                    TARGET,                                             \
                    (long int)sizeof((TARGET)->data),                   \
                    (int)(TARGET)->nparam,                              \
                    (int)(TARGET)->ndata,                               \
                    (int)(TARGET)->nlines                               \
                );                                                      \
        if((NDATA)<=0 || (NPARAM)<=0 || (NLINES)<=0)                    \
        {                                                               \
            Exit_binary_c_no_stardata(                                  \
                BINARY_C_OUT_OF_RANGE,                                  \
                "Data table create attempt failed : one of NDATA=%d NPARAM=%d NLINES=%d is <= 0", \
                (NDATA),                                                \
                (NPARAM),                                               \
                (NLINES));                                              \
        }                                                               \
    }


/*
 * This version requires _C99.
 * Do not try to shrink this to a one-line version that calls the pointer
 * macro. This is technically possible, but the ARRAY_DATA is expanded in the process
 * which is not what you want.
 *
 * Note that this version breaks valgrind, which seems confused by it,
 * even though it's no problem for gcc. Weird!
 */
#define NewDataTable_from_Array_C99(ARRAY_DATA,TARGET,NPARAM,NDATA,NLINES) \
    {                                                                   \
        double * __p = (double[]){ ARRAY_DATA };                        \
        if(DTVB)fprintf(stderr,"__p = %p allocd %ld\n",                 \
                        __p,                                            \
                        (long int)Malloc_usable_size(__p));             \
        NewDataTable_from_Pointer(__p,TARGET,NPARAM,NDATA,NLINES);      \
    }


/*
 * This version does not require _C99, but does require an
 * extra memcpy and hence probably more RAM
 */
#define NewDataTable_from_Array(ARRAY_DATA,TARGET,NPARAM,NDATA,NLINES)  \
    {                                                                   \
        if(DTVB)fprintf(stderr,"cast [%d]\n",(NPARAM+NDATA)*NLINES);    \
        const double __newdata[(NPARAM+NDATA)*NLINES] = { ARRAY_DATA }; \
        if(DTVB)fprintf(stderr,"alloc size %ld\n",                      \
                        (long int)sizeof(__newdata));                   \
        double * __p = (double*) Malloc(sizeof(__newdata));             \
        if(DTVB)fprintf(stderr,"__p = %p allocd %ld\n",                 \
                        __p,                                            \
                        (long int)Malloc_usable_size(__p));             \
        if(DTVB)fprintf(stderr,"copy %p to %p, size %ld\n",             \
                        __newdata,                                      \
                        __p,                                            \
                        (long int)sizeof(__newdata));                   \
        memcpy(__p,__newdata,sizeof(__newdata));                        \
        if(DTVB)fprintf(stderr,"Allocate from pointer %d %d %d\n",      \
                        NPARAM,NDATA,NLINES);                           \
        NewDataTable_from_Pointer(__p,TARGET,NPARAM,NDATA,NLINES);      \
    }


#define ShowDataTable(TABLE)                                            \
    {                                                                   \
        int _i;                                                         \
        for(_i=0;_i<(TABLE)->nlines;_i++)                               \
        {                                                               \
            int _j;                                                     \
            for(_j=0;_j<(TABLE)->nparam;_j++)                           \
            {                                                           \
                _printf("p(%d)=%g ",                                    \
                        _j,                                             \
                        *((TABLE)->data+_i*                             \
                          ((TABLE)->nparam+(TABLE)->ndata)+_j));        \
            }                                                           \
            for(_j=(TABLE)->nparam;                                     \
                _j<((TABLE)->nparam+(TABLE)->ndata);_j++)               \
            {                                                           \
                _printf("d(%d)=%g ",                                    \
                        _j-(TABLE)->nparam,                             \
                        *((TABLE)->data+_i*                             \
                          ((TABLE)->nparam+(TABLE)->ndata)+_j));        \
            }                                                           \
            _printf("\n");                                              \
        }                                                               \
    }                                           


/*
 * Wrapper to rinterpolate() to use the data_table_t structs.
 */

#define Interpolate(DATA_TABLE,X,R,CACHE_N)                             \
    if(DTVB)fprintf(stderr,                                             \
                    "Interpolate %p data=%p nparam=%d ndata=%d nlines=%d ncache==%d\n", \
                    (DATA_TABLE),                                       \
                    (DATA_TABLE)->data,                                 \
                    (DATA_TABLE)->nparam,                               \
                    (DATA_TABLE)->ndata,                                \
                    (DATA_TABLE)->nlines,                               \
                    (CACHE_N));                                         \
    if(stardata->tmpstore->rinterpolate_data==NULL)                     \
    {                                                                   \
        rinterpolate_alloc_dataspace(&stardata->tmpstore->              \
                                     rinterpolate_data);                \
    }                                                                   \
    rinterpolate((DATA_TABLE)->data,                                    \
                 stardata->tmpstore->rinterpolate_data,                 \
                 (DATA_TABLE)->nparam,                                  \
                 (DATA_TABLE)->ndata,                                   \
                 (DATA_TABLE)->nlines,                                  \
                 (X),                                                   \
                 (R),                                                   \
                 (CACHE_N)                                              \
        )

/*
 * Size of the data footprint in a data table
 */
#define Data_table_data_size(S) (sizeof(double)*(S)->nlines*((S)->ndata+(S)->nparam))

/*
 * Copy data table from source S to destination D 
 */
#define Copy_data_table(S,D)                                            \
    {                                                                   \
        if(DTVB)fprintf(stderr,"copy table %p to %p, dest data = %p ; source data=%p nparam=%d ndata=%d nlines=%d\n", \
                        (S),                                            \
                        (D),                                            \
                        ((D)==NULL?NULL:(D)->data),                     \
                        (S)->data,                                      \
                        (S)->nparam,                                    \
                        (S)->ndata,                                     \
                        (S)->nlines);                                   \
        if((D)==NULL)                                                   \
        {                                                               \
            if(DTVB)fprintf(stderr,"calloc new table\n");               \
            (D)=Calloc(1,sizeof(struct data_table_t));                  \
        }                                                               \
        if((D)->data==NULL)                                             \
        {                                                               \
            if(DTVB)fprintf(stderr,"realloc %ld bytes\n",               \
                            (long int)Data_table_data_size(S));         \
            (D)->data=Realloc((D)->data,Data_table_data_size(S));       \
        }                                                               \
        (D)->nlines = (S)->nlines;                                      \
        (D)->ndata = (S)->ndata;                                        \
        (D)->nparam = (S)->nparam;                                      \
        memcpy((D)->data,                                               \
               (S)->data,                                               \
               Data_table_data_size(S));                                \
        if(DTVB)fprintf(stderr,                                         \
                        "Copied data table %p to %p data=%p nparam=%d ndata=%d nlines=%d\n", \
                        (S),                                            \
                        (D),                                            \
                        (D)->data,                                      \
                        (D)->nparam,                                    \
                        (D)->ndata,                                     \
                        (D)->nlines);                                   \
    }

/*
 * Malloc a new data table in empty pointer P
 */
#define New_data_table(P)                       \
    (P) = Malloc(sizeof(struct data_table_t));


#define Delete_data_table(P)                    \
    if(P)                                       \
    {                                           \
        Safe_free((P)->data);                   \
        Safe_free(P);                           \
    }

/*
 * Push Printf through the binary_c buffer.
 *
 * Works for up to 100 arguments.
 *
 * Does nothing (no-op) if no arguments are given.
 */
#define Get_printf_macro(                                       \
    _0_,                                                        \
    _1_,  _2_,  _3_,  _4_,  _5_,  _6_,  _7_,  _8_, _9_, _10_,   \
    _11_, _12_, _13_, _14_, _15_, _16_, _17_, _18_, _19_, _20_, \
    _21_, _22_, _23_, _24_, _25_, _26_, _27_, _28_, _29_, _30_, \
    _31_, _32_, _33_, _34_, _35_, _36_, _37_, _38_, _39_, _40_, \
    _41_, _42_, _43_, _44_, _45_, _46_, _47_, _48_, _49_, _50_, \
    _51_, _52_, _53_, _54_, _55_, _56_, _57_, _58_, _59_, _60_, \
    _61_, _62_, _63_, _64_, _65_, _66_, _67_, _68_, _69_, _70_, \
    _71_, _72_, _73_, _74_, _75_, _76_, _77_, _78_, _79_, _80_, \
    _81_, _82_, _83_, _84_, _85_, _86_, _87_, _88_, _89_, _90_, \
    _91_, _92_, _93_, _94_, _95_, _96_, _97_, _98_, _99_, _100_, \
    NAME,...) NAME

#define Printf_no_op(...) /* no-op */
#define Printf_n(...) binary_c_buffered_printf(stardata,FALSE,__VA_ARGS__);
#define Printf_n_deslash(...) binary_c_buffered_printf(stardata,TRUE,__VA_ARGS__);
#define Printf(...)                                             \
    Get_printf_macro(                                           \
    _0, ##__VA_ARGS__,                                          \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_n,Printf_n,Printf_n,Printf_n,Printf_n,               \
    Printf_no_op)(__VA_ARGS__)

/* PRINTF = Printf for backwards compatibility */
#define PRINTF(...) Printf(__VA_ARGS__)
#define Printf_deslash(...)                                             \
    Get_printf_macro(                                                   \
        _0, ##__VA_ARGS__,                                              \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash,Printf_n_deslash, \
        PRINTFN_DESLASHOOP)(__VA_ARGS__)


#define Clear_error_buffer                                      \
    if(stardata!=NULL &&                                        \
       stardata->tmpstore != NULL &&                            \
       stardata->tmpstore->error_buffer != NULL)                  \
    {                                                           \
        clear_error_buffer(stardata);                           \
    }


#define Clear_printf_buffer                                     \
    if(likely(stardata!=NULL &&                                 \
              stardata->preferences!=NULL &&                    \
              stardata->preferences->internal_buffering))       \
    {                                                           \
        clear_printf_buffer(stardata);                          \
    }

    
/*
 * Wrapper for nancheck
 */
#ifdef NANCHECKS
#define nancheck(...) _nancheck(stardata,__VA_ARGS__)
#else
#define nancheck(...) /* NANCHECKS not defined: nancheck is disabled */
#endif

/*
 * The fraction of blackbody flux between energies E0 and E1
 * given a blackbody at temperature T
 */

#define Blackbody_flux_fraction(T,E0,E1)                \
    (blackbody_fraction((E0)/PLANCK_CONSTANT,(T)) -     \
     blackbody_fraction((E1)/PLANCK_CONSTANT,(T)))

/*
 * Reduce a 'T' or 'F' string depending on 
 * whether X is TRUE or FALSE
 */
#define True_or_false(X) ((X)==TRUE ? "T" : "F")
/*
 * Ditto for yes or no
 */
#define Yes_or_no(X) ((X)==TRUE ? "Y" : "N")


/*
 * Check for an eccentricity that is valid
 */
#define Valid_eccentricity(E) Boolean_(In_range((E),0.0,1.0))


#ifdef USE_GSL

/*
 * GSL convenience macros.
 *
 * NB vectors are assumed to be 3D.
 */
#define Set_GSL_vector(V,X,Y,Z)                 \
    {                                           \
        gsl_vector_set((V),0,(X));              \
        gsl_vector_set((V),1,(Y));              \
        gsl_vector_set((V),2,(Z));              \
    }
#define Set_GSL_4vector(V,W,X,Y,Z)              \
    {                                           \
        gsl_vector_set((V),0,(W));              \
        gsl_vector_set((V),1,(X));              \
        gsl_vector_set((V),2,(Y));              \
        gsl_vector_set((V),3,(Z));              \
    }
#define Set_GSL_vector_array(V,X)               \
    {                                           \
        gsl_vector_set((V),0,((X)[0]));         \
        gsl_vector_set((V),1,((X)[1]));         \
        gsl_vector_set((V),2,((X)[2]));         \
    }
#define Set_GSL_4vector_array(V,X)              \
    {                                           \
        gsl_vector_set((V),0,((X)[0]));         \
        gsl_vector_set((V),1,((X)[1]));         \
        gsl_vector_set((V),2,((X)[2]));         \
        gsl_vector_set((V),3,((X)[3]));         \
    }
#define Get_GSL_vector(V,X,Y,Z)                 \
    {                                           \
        (X)=gsl_vector_get((V),0);              \
        (Y)=gsl_vector_get((V),1);              \
        (Z)=gsl_vector_get((V),2);              \
    }
#define Get_GSL_4vector(V,W,X,Y,Z)              \
    {                                           \
        (W)=gsl_vector_get((V),0);              \
        (X)=gsl_vector_get((V),1);              \
        (Y)=gsl_vector_get((V),2);              \
        (Z)=gsl_vector_get((V),3);              \
    }
#define Get_GSL_vector_array(V,X)               \
    {                                           \
        (X)[0] = gsl_vector_get((V),0);         \
        (X)[1] = gsl_vector_get((V),1);         \
        (X)[2] = gsl_vector_get((V),2);         \
    }
#define Get_GSL_4vector_array(V,X)              \
    {                                           \
        (X)[0] = gsl_vector_get((V),0);         \
        (X)[1] = gsl_vector_get((V),1);         \
        (X)[2] = gsl_vector_get((V),2);         \
        (X)[3] = gsl_vector_get((V),3);         \
    }

/*
 * Here, ARGS are a va_list, but PARAMS are
 * a (void * p) which is a pointer to a GSL_args struct.
 * 
 * This in turn contains the pointer to the args list.
 *
 * You can access the error value through the integer pointer 
 * int * GSL_error : the VA_ARGS should be used to set this 
 * in a variable which is returned.
 */
#define Map_GSL_params(PARAMS,VA_ARGS)          \
    va_list (VA_ARGS);                          \
    va_copy(                                    \
        (VA_ARGS),                              \
        ((struct GSL_args_t *)(PARAMS))->args   \
        );

/*
 * Copy a set of GSL_args while also copying the va_list
 * properly with va_copy.
 * 
 * DEST and SRC are cast to a GSL_args_t * (so can be void *).
 */
#define Copy_GSL_args(SRC,DEST)                         \
                                                        \
    memcpy(((struct GSL_args_t *)(DEST)),               \
           ((struct GSL_args_t *)(SRC)),                \
           sizeof(struct GSL_args_t));                  \
                                                        \
    va_copy(((struct GSL_args_t *)(DEST))->args,        \
            ((struct GSL_args_t *)(SRC))->args));       \


#endif//USE_GSL

/*
 * Macro and version checks
 */

/*
 * Macro to convert a macro to a string
 */ 
#define Stringify(item) "" #item

/*
 * Macro to expand a macro and then convert to a string
 */
#define Stringify_macro(item) (Stringify(item))

/*
 * Macros to count and add numbers of arguments in 
 * variadic lists.
 *
 * Number_of_arguments works but gives warnings on strings.
 *
 * Instead, use Get_argument_count but this only works up 
 * to N=100 - this should be sufficient.
 */
#define Number_of_arguments(...)                        \
    (sizeof((int[]){0,##__VA_ARGS__})/sizeof(int)-1)

#define Sum_of_arguments(...) \
    sum(Number_of_arguments(__VA_ARGS__), ##__VA_ARGS__)

#define Get_argument_count(...)                                         \
    __Internal_get_argument_count_private(                              \
        0, ## __VA_ARGS__,                                              \
        100, 99, 98, 97, 96, 95, 94, 93, 92, 91,                         \
        90, 89, 88, 87, 86, 85, 84, 83, 82, 81,                         \
        80, 79, 78, 77, 76, 75, 74, 73, 72, 71,                         \
        70, 69, 68, 67, 66, 65, 64, 63, 62, 61,                         \
        60, 59, 58, 57, 56, 55, 54, 53, 52, 51,                         \
        50, 49, 48, 47, 46, 45, 44, 43, 42, 41,                         \
        40, 39, 38, 37, 36, 35, 34, 33, 32, 31,                         \
        30, 29, 28, 27, 26, 25, 24, 23, 22, 21,                         \
        20, 19, 18, 17, 16, 15, 14, 13, 12, 11,                         \
        10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0)

#define __Internal_get_argument_count_private(                  \
    _0_, _1_, _2_, _3_, _4_, _5_, _6_, _7_, _8_, _9_, _10_,     \
    _11_, _12_, _13_, _14_, _15_, _16_, _17_, _18_, _19_, _20_, \
    _21_, _22_, _23_, _24_, _25_, _26_, _27_, _28_, _29_, _30_, \
    _31_, _32_, _33_, _34_, _35_, _36_, _37_, _38_, _39_, _40_, \
    _41_, _42_, _43_, _44_, _45_, _46_, _47_, _48_, _49_, _50_, \
    _51_, _52_, _53_, _54_, _55_, _56_, _57_, _58_, _59_, _60_, \
    _61_, _62_, _63_, _64_, _65_, _66_, _67_, _68_, _69_, _70_, \
    _71_, _72_, _73_, _74_, _75_, _76_, _77_, _78_, _79_, _80_, \
    _81_, _82_, _83_, _84_, _85_, _86_, _87_, _88_, _89_, _90_, \
    _91_, _92_, _93_, _94_, _95_, _96_, _97_, _98_, _99_, _100_, \
    count, ...) count

/*
 * Test macro to see if it defined, and print 
 * an option description message (whether is it defined or not)
 */
#define Macrotest(macro,...)                            \
    Printf("%s is %s ",                                 \
           "" #macro,                                   \
           (strcmp("" #macro,                           \
                   Stringify(macro)) ? "on" : "off" )); \
    Printf(__VA_ARGS__);                                \
    Printf("\n");


/*
 * As above, but only print the message string if the macro is defined
 * and true
 */
#define Macrotestif(macro,...)                          \
    Printf("%s is %s ",                                 \
           "" #macro,                                   \
           (strcmp("" #macro,                           \
                   Stringify(macro)) ? "on" : "off" )); \
    if(strcmp("" #macro, Stringify(macro)))             \
    {                                                   \
        Printf(__VA_ARGS__);                            \
    }                                                   \
    Printf("\n");
/* end */

/*
 * ... and if defined and false.
 */
#define Macrotestifnot(macro,...)                       \
    Printf("%s is %s ",                                 \
           "" #macro,                                   \
           (strcmp("" #macro,                           \
                   Stringify(macro)) ? "on" : "off" )); \
    if(!strcmp("" #macro, Stringify(macro)))            \
    {                                                   \
        Printf(__VA_ARGS__);                            \
    }                                                   \
    Printf("\n");
/* end */

/*
 * Macros to show other macros
 */
#define Show_float_macro(macro,...)             \
    Printf_deslash("%s is %50.30e ",            \
                   "" #macro,                   \
                   (double)(macro));            \
    Printf(__VA_ARGS__);                        \
    Printf("\n");
#define Show_string_macro(macro,...)            \
    Printf_deslash("%s is %s ",                 \
                   "" #macro,                   \
                   Stringify_macro(macro));     \
    Printf(__VA_ARGS__);                        \
    Printf("\n");
#define Show_int_macro(macro,...)               \
    Printf_deslash("%s is %d ",                 \
                   "" #macro,                   \
                   (int)(macro));               \
    Printf(__VA_ARGS__);                        \
    Printf("\n");
#define Show_long_int_macro(macro,...)          \
    Printf_deslash("%s is %ld ",                \
                   "" #macro,                   \
                   (long int)(macro));          \
    Printf(__VA_ARGS__);                        \
    Printf("\n");


/*
 * Macro to determine the AGB core algorithm 
 */
#ifdef NUCSYN
#define AGB_Core_Algorithm                                              \
    ((stardata->preferences->AGB_core_algorithm==AGB_CORE_ALGORITHM_DEFAULT || \
      stardata->preferences->AGB_core_algorithm==AGB_CORE_ALGORITHM_KARAKAS) \
     ? AGB_CORE_ALGORITHM_KARAKAS : AGB_CORE_ALGORITHM_HURLEY)
#else//NUCSYN
#define AGB_Core_Algorithm                                              \
    ((stardata->preferences->AGB_core_algorithm==AGB_CORE_ALGORITHM_DEFAULT) \
     ? AGB_CORE_ALGORITHM_HURLEY : AGB_CORE_ALGORITHM_KARAKAS)
#endif//NUCSYN

/*
 * Macro to determine the AGB radius algorithm
 */
#ifdef NUCSYN
#define AGB_Radius_Algorithm                                            \
    ((stardata->preferences->AGB_radius_algorithm==AGB_RADIUS_ALGORITHM_DEFAULT || \
      stardata->preferences->AGB_radius_algorithm==AGB_RADIUS_ALGORITHM_KARAKAS) \
     ? AGB_RADIUS_ALGORITHM_KARAKAS : AGB_RADIUS_ALGORITHM_HURLEY)
#else//NUCSYN
#define AGB_Radius_Algorithm                                            \
    ((stardata->preferences->AGB_radius_algorithm==AGB_RADIUS_ALGORITHM_DEFAULT) \
     ? AGB_RADIUS_ALGORITHM_HURLEY : AGB_RADIUS_ALGORITHM_KARAKAS)
#endif//NUCSYN

/*
 * Macro to determine the AGB luminosity algorithm
 */
#ifdef NUCSYN
#define AGB_Luminosity_Algorithm                                        \
    ((stardata->preferences->AGB_luminosity_algorithm==AGB_LUMINOSITY_ALGORITHM_DEFAULT || \
      stardata->preferences->AGB_luminosity_algorithm==AGB_LUMINOSITY_ALGORITHM_KARAKAS) \
     ? AGB_LUMINOSITY_ALGORITHM_KARAKAS : AGB_LUMINOSITY_ALGORITHM_HURLEY)
#else//NUCSYN
#define AGB_Luminosity_Algorithm                                        \
    ((stardata->preferences->AGB_luminosity_algorithm==AGB_LUMINOSITY_ALGORITHM_DEFAULT) \
     ? AGB_LUMINOSITY_ALGORITHM_HURLEY : AGB_LUMINOSITY_ALGORITHM_KARAKAS)
#endif//NUCSYN

/*
 * Macro to determine the AGB third dredge up algorithm
 */
#ifdef NUCSYN
#define AGB_Third_Dredge_Up_Algorithm (                                 \
        stardata->preferences->AGB_3dup_algorithm==AGB_THIRD_DREDGE_UP_ALGORITHM_STANCLIFFE ? \
        AGB_THIRD_DREDGE_UP_ALGORITHM_STANCLIFFE :                      \
        ((stardata->preferences->AGB_3dup_algorithm==AGB_THIRD_DREDGE_UP_ALGORITHM_DEFAULT || \
          stardata->preferences->AGB_3dup_algorithm==AGB_THIRD_DREDGE_UP_ALGORITHM_KARAKAS) \
         ? AGB_THIRD_DREDGE_UP_ALGORITHM_KARAKAS : AGB_THIRD_DREDGE_UP_ALGORITHM_HURLEY) \
        )
#else//NUCSYN
#define AGB_Third_Dredge_Up_Algorithm (                                 \
        stardata->preferences->AGB_3dup_algorithm==AGB_THIRD_DREDGE_UP_ALGORITHM_STANCLIFFE ? \
        AGB_THIRD_DREDGE_UP_ALGORITHM_STANCLIFFE :                      \
        ((stardata->preferences->AGB_3dup_algorithm==AGB_THIRD_DREDGE_UP_ALGORITHM_DEFAULT) \
         ? AGB_THIRD_DREDGE_UP_ALGORITHM_HURLEY : AGB_THIRD_DREDGE_UP_ALGORITHM_KARAKAS))
#endif//NUCSYN


/*
 * Macros to change the state of buffering
 */
#define Save_buffering_state                                            \
    int __buf = stardata->preferences->internal_buffering;              \
    int __batchmode = stardata->preferences->batchmode;

#define Restore_buffering_state                                         \
    stardata->preferences->internal_buffering = __buf;                  \
    stardata->preferences->batchmode = __batchmode;


/*
 * Map a va_arg without having to specify its TYPE twice
 */
#define Map_varg(TYPE,NAME,ARGLIST)             \
    TYPE NAME = va_arg(ARGLIST,TYPE);

/*
 * Bisector function result
 *
 * Given a CONSTRAINT, we want VARIABLE to equal it,
 * at which point the bisection function should give zero.
 * 
 * The function must be monotonically increasing with
 * VARIABLE, and preferably not be infinite.
 * 
 * Thus, if CONSTRAINT is zero we just
 * compare VARIABLE to CONSTRAINT directly.
 *
 * If the function is monotonically decreasing, 
 * use Bisection_result_inverse, in which we switch
 * CONSTRAINT and VARIABLE, while this time checking
 * that VARIABLE is non-zero.
 */
#define Bisection_result(CONSTRAINT,            \
                         VARIABLE)              \
    (                                           \
        Is_really_not_zero(CONSTRAINT) ?        \
        ((VARIABLE) / (CONSTRAINT) - 1.0) :     \
        ((VARIABLE) - (CONSTRAINT))             \
        )

#define Bisection_result_inverse(CONSTRAINT,    \
                                 VARIABLE)      \
    (Bisection_result(                          \
        (VARIABLE),                             \
        (CONSTRAINT)                            \
        ))

/*
 * perhaps use custom isnan function?
 */
#ifdef USE_NATIVE_ISNAN
#define _Nanchecker(X) isnan(X)
#else
/*
 * remove isnan which is sometimes defined as 
 * a macro, and call my_isnan instead
 */
#undef isnan
#define isnan(X) my_isnan(X)
#define _Nanchecker(X) my_isnan(X)
#endif // USE_NATIVE_ISNAN



#define Angular_momentum_cgs_to_code(J) ((J) / ANGULAR_MOMENTUM_CGS)
#define Angular_momentum_code_to_cgs(J) ((J) * ANGULAR_MOMENTUM_CGS)

#define Angular_momentum_derivative_cgs_to_code(Jdot)                   \
    (Angular_momentum_cgs_to_code(Jdot) * YEAR_LENGTH_IN_SECONDS)
#define Angular_momentum_derivative_code_to_cgs(Jdot)                   \
    (Angular_momentum_code_to_cgs(Jdot) / YEAR_LENGTH_IN_SECONDS)


#ifdef TRIPLE

/*
 * Macros to give outer/inner label depending
 * on the component integer, X
 */
#define Component_string(X) (                           \
        (X) == TRIPLE_COMPONENT_OUTER ? "Outer" :       \
        (X) == TRIPLE_COMPONENT_INNER ? "Inner" :       \
        "Unknown"                                       \
        )
#define component_string(X) (                           \
        (X) == TRIPLE_COMPONENT_OUTER ? "outer" :       \
        (X) == TRIPLE_COMPONENT_INNER ? "inner" :       \
        "unknown"                                       \
        )

#endif//TRIPLE


/*
 * How many previous stardatas do we need?
 */
#define Solver_n_previous_stardatas(S)          \
    ((S)==SOLVER_FORWARD_EULER ? 1 :            \
     (S)==SOLVER_RK2 ? 1 :                      \
     1)

#ifdef CODESTATS
#define Increment_codestat(N)                   \
    increment_codestat((const int)(N),          \
                       (const char*)__FILE__,   \
                       (const int)__LINE__);
#else
#define Increment_codestat(N) /* do nothing */
#endif


#define RLOF_stability_string(X)                                        \
    (                                                                   \
        (X)==RLOF_IMPOSSIBLE ? "Impossible" :                           \
        (X)==RLOF_INSTABILITY_BLOCKED ? "Instability blocked" :         \
        (X)==RLOF_STABLE ? "Stable" :                                   \
        (X)==RLOF_UNSTABLE ? "Unstable" :                               \
        (X)==RLOF_UNSTABLE_LOW_MASS_MS_COMENV ? "Unstable (low mass MS comenv)" : \
        (X)==RLOF_UNSTABLE_GIANT_COMENV ? "Unstable (giant comenv)" :   \
        (X)==RLOF_UNSTABLE_WD_COMENV ? "Unstable (WD comenv)" :         \
        (X)==RLOF_UNSTABLE_NS ? "Unstable (Neutron star)" :             \
        (X)==RLOF_UNSTABLE_BH ? "Unstable (Black hole)" :               \
        (X)==RLOF_UNSTABLE_MS_MERGER ? "Unstable (MS merger)" :         \
        (X)==RLOF_UNSTABLE_VERY_LARGE_DONOR ? "Unstable (Very large donor)" : \
        "RLOF stability unknown!"                                       \
        )

#define Donor_rate_algorithm_string(X)                          \
    (                                                           \
        (X) == DONOR_RATE_ALGORITHM_CLAEYS2014 ? "Claeys2014" : \
        (X) == DONOR_RATE_ALGORITHM_BSE ? "BSE" :               \
        "Donor rate algorithm unknown"                          \
        )

#define Qcrit_GB_algorithm_string(X)                                    \
    (                                                                   \
        (X) == QCRIT_GB_BSE ? "BSE" :                                   \
        (X) == QCRIT_GB_HJELLMING_WEBBINK ? "Hjellming and Webbink 1987" : \
        (X) == QCRIT_GB_Q_NO_COMENV ? "No comenv" :                     \
        (X) == QCRIT_GB_CHEN_HAN_TABLE ? "Chen and Han (table) 2008":   \
        (X) == QCRIT_GB_CHEN_HAN_FORMULA ? "Chen and Han (formula) 2008": \
        (X) == QCRIT_GB_GE2015 ? "Ge et al. 2015" :                     \
        (X) == QCRIT_GB_VOS2018 ? "Vos et al 2018" :                    \
        "Unknown Qcrit GB algorithm"                                    \
        )


#define Map_float_algorithm(X)                  \
    ((int)((X)-0.4))

#define Accretion_limit_string(N)                               \
    (                                                           \
        (N) == ACCRETION_LIMIT_EDDINGTON ? "Eddington" :        \
        (N) == ACCRETION_LIMIT_THERMAL ? "Thermal" :            \
        (N) == ACCRETION_LIMIT_DYNAMICAL ? "Dynamical" :        \
        (N) == ACCRETION_LIMIT_HACHISU ? "Hachisu" :            \
        "Unknown"                                               \
        )

/*
 * enable this to log fflush calls
 *
#undef fflush
#define fflush(X) printf("Flush %s\n",X==stdout ? "stdout" : X==NULL ? "NULL" : X==stderr ? "stderr" : "?");Backtrace;fflush(stdout);fflush(X)
*/
/*
 * If we have GNU C extensions, we can use the native Elvis 
 * operator. 
 *
 * Otherwise, use the usual ternary with a temporary variable.
 *
 * Note: we can use typeof here because the calls to memoize_search_result
 *       and memoize_store_results are cast to RESULT_TYPE* pointers in 
 *       the Memoize macros.
 */
#ifdef __GNUC__
#define Elvis(A,B) ((A) ?: (B))
#else
#define Elvis(A,B) ({                           \
            typeof(A) * __x = (A);              \
            __x ? __x : (B);                    \
        })
#endif//__GNUC__

#define Elvis3(A,B,C) (                         \
        Elvis((A),                              \
              Elvis((B),(C)))                   \
        )


/*
 * Given a number of bytes, N, return a float in bytes, Kbytes, 
 * Mbytes or Gbytes, along with a unit string, suitable for 
 * a format string like "%6.2f %s".
 *
 * Two versions are provided: one for decimal (power ten, kilobytes)
 * counting and one for binary (power two) counting (e.g. kibibytes)
 */
#define Mem_size_string_decimal(N)                              \
    (                                                           \
        (float)                                                 \
        (                                                       \
            (N) < KILOBYTE ? (N) :                              \
            (N) < MEGABYTE ? ((float)(N)/(float)KILOBYTE) :     \
            (N) < GIGABYTE ? ((float)(N)/(float)MEGABYTE) :     \
            (N) < TERABYTE ? ((float)(N)/(float)GIGABYTE) :     \
            (N) < PETABYTE ? ((float)(N)/(float)TERABYTE) :     \
            ((float)(N) / (float)PETABYTE)                      \
            )                                                   \
        )                                                       \
    ,                                                           \
        (                                                       \
            (N) < KILOBYTE ? ((N) == 1 ? "Byte" : "Bytes") :    \
            (N) < MEGABYTE ? "KBytes" :                         \
            (N) < GIGABYTE ? "MBytes" :                         \
            (N) < TERABYTE ? "GBytes" :                         \
            (N) < PETABYTE ? "TBytes" :                         \
            "PBytes"                                            \
            )

#define Mem_size_string_binary(N)                               \
    (                                                           \
        (float)                                                 \
        (                                                       \
            (N) < KIBIBYTE ? (N) :                              \
            (N) < MEBIBYTE ? ((float)(N)/(float)KIBIBYTE) :     \
            (N) < GIBIBYTE ? ((float)(N)/(float)MEBIBYTE) :     \
            (N) < TEBIBYTE ? ((float)(N)/(float)GIBIBYTE) :     \
            (N) < PEBIBYTE ? ((float)(N)/(float)TEBIBYTE) :     \
            ((float)(N) / (float)PEBIBYTE)                      \
            )                                                   \
        )                                                       \
    ,                                                           \
        (                                                       \
            (N) < KIBIBYTE ? ((N) == 1 ? "Byte" : "Bytes") :    \
            (N) < MEBIBYTE ? "KiBytes" :                        \
            (N) < GIBIBYTE ? "MiBytes" :                        \
            (N) < TEBIBYTE ? "GiBytes" :                        \
            (N) < PEBIBYTE ? "TiBytes" :                        \
            "PiBytes"                                           \
            )

/*
 * Macro to define whether a system is single (or not)
 */
/*
#define _System_is_single(S) \
    (Boolean_(                                                          \
        (                                                               \
            (S)->model.sgl == TRUE ||                                   \
            (S)->star[1].stellar_type == MASSLESS_REMNANT ||            \
            (S)->star[0].stellar_type == MASSLESS_REMNANT ||            \
            (S)->common.orbit.eccentricity < -TINY ||                   \
            (S)->common.orbit.separation <                              \
            MINIMUM_SEPARATION_TO_BE_CALLED_BINARY ||                   \
            (S)->common.orbit.separation >                              \
            MAXIMUM_SEPARATION_TO_BE_CALLED_BINARY ||                   \
            Less_or_equal((S)->common.orbit.angular_momentum,           \
                          MINIMUM_ORBITAL_ANGMOM) ||                    \
            More_or_equal((S)->common.orbit.eccentricity,1.0))))
#define _System_is_binary(S) (((_System_is_single(S))==TRUE) ? FALSE : TRUE)
*/


/*
 * The same logic, but using A OR B = NOT (A AND B)
 * which is usually faster. We define the binary first, then
 * single is just Not(single)
 */
#define _System_is_binary(S)                                    \
    (Boolean_(                                                  \
    (                                                           \
        (S)->model.sgl == FALSE &&                              \
        (S)->star[1].stellar_type != MASSLESS_REMNANT &&        \
        (S)->star[0].stellar_type != MASSLESS_REMNANT &&        \
        (S)->common.orbit.eccentricity > -TINY &&               \
        (S)->common.orbit.separation >                          \
        MINIMUM_SEPARATION_TO_BE_CALLED_BINARY &&               \
        (S)->common.orbit.separation <                          \
        MAXIMUM_SEPARATION_TO_BE_CALLED_BINARY &&               \
        ((S)->common.orbit.angular_momentum >                   \
         MINIMUM_ORBITAL_ANGMOM) &&                             \
        (S)->common.orbit.eccentricity < 1.0                    \
        )))
#define _System_is_single(S) (Not(_System_is_binary(S)))

#define System_is_single _System_is_single(stardata)
#define System_is_binary _System_is_binary(stardata)

/* 
 * Define an observable binary as one in which 
 * either star has a radial velocity above 
 * the threshold, and that System_is_binary is TRUE.
 */
#define Observable_binary                                               \
    (                                                                   \
    System_is_binary &&                                                 \
    (                                                                   \
        (                                                               \
            radial_velocity_K(stardata,90,1)>                           \
            stardata->preferences->observable_radial_velocity_minimum   \
            )                                                           \
        ||                                                              \
        (                                                               \
            radial_velocity_K(stardata,90,2)>                           \
            stardata->preferences->observable_radial_velocity_minimum   \
            )                                                           \
        )                                                               \
        )


#define Angular_momentum_from_stardata \
    (Pow2(stardata->common.orbit.separation) *                          \
     sqrt(Max(1.0-Pow2(stardata->common.orbit.eccentricity),0.0)) *     \
     stardata->common.orbit.angular_frequency *                         \
     stardata->star[0].mass * stardata->star[1].mass /                  \
     (stardata->star[0].mass + stardata->star[1].mass))

#define Bisector_string(N) (                                            \
        (abs(N)) == BISECTOR_DISC_TVISC ? "Tvisc0" :                    \
        (abs(N)) == BISECTOR_DISC_J ? "J" :                             \
        (abs(N)) == BISECTOR_DISC_F ? "F" :                             \
        (abs(N)) == BISECTOR_DISC_RHALFJ ? "RhalfJ" :                   \
        (abs(N)) == BISECTOR_DISC_PRESSURE_RADIUS ? "RfromP" :          \
        (abs(N)) == BISECTOR_COMENV_DM ? "ComenvDM" :                   \
        (abs(N)) == BISECTOR_DISC_OWEN_RADIUS ? "OwenRadius" :          \
        (abs(N)) == BISECTOR_DISC_VISCOUS ? "Viscous" :                 \
        (abs(N)) == BISECTOR_DISC_MASS_RADIUS ? "RfromM" :              \
        (abs(N)) == BISECTOR_DISC_RIN_MIN ? "RinMin" :                  \
        (abs(N)) == BISECTOR_DISC_BISECTION_ROOTER ? "Disc_bi_rooter" : \
        "Unknown")


#define Bisector_error_string(N) (                                      \
        (N) == BINARY_C_BISECT_ERROR_NONE ? "None" :                    \
        (N) == BINARY_C_BISECT_ERROR_MAXED_OUT ? "Maxed out" :          \
        (N) == BINARY_C_BISECT_ERROR_BRACKET_FAILED ? "Bracket failed" : \
        "Unknown")


/* Time of tpagb start */
#define TPAGB_start_time(star) ((star)->time_first_pulse)

/* Time of the next pulse on the TPAGB */
#define Next_pulse(star) (More_or_equal(tagb, (star)->time_next_pulse))

/* this evaluates to TRUE if we're on the first pulse */
#define First_pulse(star) ((star)->num_thermal_pulses<-0.5)


/*
 * Convert X to the appropriate solar unit,
 * where TYPE is L, M or R
 */
#define Solarunit(X,TYPE) ((X) / TYPE##_SUN)

/*
 * Convert X to the appropriate solar unit,
 * and return a pair of variables: 
 *
 * 1) the variable X divided by the appropriate solar constant
 * 2) the solar constant string
 *
 * Type is L, M or R
 *
 * You can thus do something like:
 *
 * r = 123.45 * R_SUN;
 * printf("Radius is %g %s",Solar(r,R));
 *
 * to output something like 
 *
 * "Radius is 123.45 R☉"
 */
#define Solar(X,TYPE) Solarunit((X),TYPE) , TYPE##_SOLAR

#ifdef CODESTATS
#define Codestat_string(N)                      \
    (                                           \
        (N)==CODESTAT_MEMCPY ? "memcpy" :       \
        (N)==CODESTAT_MALLOC ? "malloc" :       \
        (N)==CODESTAT_CALLOC ? "calloc" :       \
        (N)==CODESTAT_REALLOC ? "realloc" :     \
        (N)==CODESTAT_MEMSET ? "memset" :       \
        "unknown")
#endif


#define AGB_Luminosity_Algorithm_String(N) (                    \
        (N) == AGB_LUMINOSITY_ALGORITHM_DEFAULT ? "Default" :   \
        (N) == AGB_LUMINOSITY_ALGORITHM_HURLEY ? "Hurley" :     \
        (N) == AGB_LUMINOSITY_ALGORITHM_KARAKAS ? "Karakas" :   \
        "Unknown" )

#define AGB_Third_Dredge_Up_Algorithm_String(N) (                       \
        (N) == AGB_THIRD_DREDGE_UP_ALGORITHM_DEFAULT ? "Default" :      \
        (N) == AGB_THIRD_DREDGE_UP_ALGORITHM_HURLEY ? "Hurley" :        \
        (N) == AGB_THIRD_DREDGE_UP_ALGORITHM_KARAKAS ? "Karakas" :      \
        (N) == AGB_THIRD_DREDGE_UP_ALGORITHM_STANCLIFFE ? "Stancliffe" : \
        "Unknown" )

#define AGB_Core_Algorithm_String(N) (                  \
        (N) == AGB_CORE_ALGORITHM_DEFAULT ? "Default" : \
        (N) == AGB_CORE_ALGORITHM_HURLEY ? "Hurley" :   \
        (N) == AGB_CORE_ALGORITHM_KARAKAS ? "Karakas" : \
        "Unknown" )

#define AGB_Radius_Algorithm_String(N) (                        \
        (N) == AGB_RADIUS_ALGORITHM_DEFAULT ? "Default" :       \
        (N) == AGB_RADIUS_ALGORITHM_HURLEY ? "Hurley" :         \
        (N) == AGB_RADIUS_ALGORITHM_KARAKAS ? "Karakas" :       \
        "Unknown" )

#define Batchmode_is_on(N) ((N)>(BATCHMODE_OFF))
#define Batchmode_is_off(N) ((N)<=(BATCHMODE_OFF))

/*
 * Macro to prevent us using memcmp and associatedly forgetting
 * the ==0
 */
#define Memory_equal(A,B,C) (memcmp((A),(B),(C))==0)


/*
 * ASCII convenience macros
 */
#define ASCII_upper_case(N) Boolean_((N)>64 && (N)<91)
#define ASCII_lower_case(N) Boolean_((N)>96 && (N)<123)
#define ASCII_letter(N) (ASCII_lower_case(N) || ASCII_upper_case(N))

/*
 * ASCII letters to integer index mapping, where a-z are 0-25, and 
 * A-Z are 26-51.
 */
#define ASCII_letter_to_index(A)                                        \
    ({                                                                  \
        const unsigned int u =                                          \
            ASCII_lower_case(arg[0])==TRUE ? 0 : 1;                     \
        const unsigned int offset =                                     \
            (const unsigned int) u == 0 ? 'a' : 'A';                    \
        (A) - offset + NUMBER_OF_LETTERS_IN_THE_ALPHABET * u;           \
    })


/*
 * fflush everything and exit
 */
#define Flexit                                  \
    {                                           \
        fflush(NULL);                           \
        _exit(0);                               \
    }

/*
 * long int converters
 */
#define Long_int_to_int(N)                      \
    ((int)(                                     \
        (N) > INT_MAX ? INT_MAX :               \
        (N) < INT_MIN ? INT_MIN :               \
        (N)                                     \
        ))
#define Long_int_to_short_int(N)                \
    ((short int)(                               \
        (N) > SHRT_MAX ? SHRT_MAX :             \
        (N) < SHRT_MIN ? SHRT_MIN :             \
        (N)                                     \
        ))
#define Long_int_to_unsigned_int(N)             \
    ((unsigned int)(                            \
        (N) > UINT_MAX ? UINT_MAX :             \
        (N) < 0 ? 0 :                           \
        (N)                                     \
        ))
#define Long_int_to_unsigned_short_int(N)       \
    ((unsigned short int)(                      \
        (N) > USHRT_MAX ? USHRT_MAX :           \
        (N) < 0 ? 0 :                           \
        (N)                                     \
        ))
#define Long_int_to_boolean(N)                  \
    ((Boolean)(                                 \
        (N) == 0 ? FALSE : TRUE                 \
        ))


#endif // BINARY_C_FUNCTION_MACROS_H

