#pragma once
#ifndef BINARY_C_DERIVATIVES_H
#define BINARY_C_DERIVATIVES_H
/*
 * The binary_c stellar population nucleosynthesis framework.
 *
 * Contact: r.izzard@surrey.ac.uk or rob.izzard@gmail.com
 *
 * http://personal.ph.surrey.ac.uk/~ri0005/binary_c.html
 * https://gitlab.eps.surrey.ac.uk/ri0005/binary_c
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-announce
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-devel
 * https://twitter.com/binary_c_code
 * https://www.facebook.com/groups/149489915089142/
 *
 * Please see the files README, LICENCE and CHANGES,
 * and the doc/ directory for documentation.
 *
 * This file coantains macros labelling derivatives in binary_c,
 * and function macros to calculate combinations of derivatives.
 */




/* 
 * Stellar mass loss and gain by winds:
 *
 * The derivatives are of the "correct sign 
 * such that M(t+dt) = M(t) + dM/dt * dt
 * i.e.
 * ..._WIND_LOSS is negative
 * ..._WIND_GAIN is positive
 *
 * ..._MASS is the total change.
 */

#define DERIVATIVE_STELLAR_MASS 0
#define DERIVATIVE_STELLAR_MASS_WIND_LOSS 1
#define DERIVATIVE_STELLAR_MASS_WIND_GAIN 2

/*
 * Radius, luminosity and core mass derivatives.
 */
#define DERIVATIVE_STELLAR_RADIUS 3
#define DERIVATIVE_STELLAR_LUMINOSITY 4
#define DERIVATIVE_STELLAR_CORE_MASS 5
#define DERIVATIVE_STELLAR_TEMPERATURE 6

/*
 * Mass loss and gain
 */
#define DERIVATIVE_STELLAR_MASS_RLOF_LOSS 7
#define DERIVATIVE_STELLAR_MASS_RLOF_GAIN 8
#define DERIVATIVE_STELLAR_MASS_NOVA 9
#define DERIVATIVE_STELLAR_MASS_RLOF_TRANSFER 10
#define DERIVATIVE_STELLAR_MASS_DISC_GAIN 11
#define DERIVATIVE_STELLAR_MASS_DISC_LOSS 12
#define DERIVATIVE_STELLAR_MASS_DECRETION_DISC 13
#define DERIVATIVE_STELLAR_MASS_CBDISC_GAIN 14         
#define DERIVATIVE_STELLAR_MASS_NONCONSERVATIVE_LOSS 15
#define DERIVATIVE_STELLAR_MASS_IRRADIATIVE_LOSS 16
#define DERIVATIVE_STELLAR_MASS_ARTIFICIAL 17

/*
 * Angular momentum loss and gain
 */
#define DERIVATIVE_STELLAR_ANGMOM_RLOF_LOSS 18
#define DERIVATIVE_STELLAR_ANGMOM_RLOF_GAIN 19
#define DERIVATIVE_STELLAR_ANGMOM_WIND_LOSS 20
#define DERIVATIVE_STELLAR_ANGMOM_WIND_GAIN 21
#define DERIVATIVE_STELLAR_ANGMOM_TIDES 22
#define DERIVATIVE_STELLAR_ANGMOM_MAGNETIC_BRAKING 23
#define DERIVATIVE_STELLAR_ANGMOM_DECRETION_DISC 24
#define DERIVATIVE_STELLAR_ANGMOM_CBDISC_GAIN 25
#define DERIVATIVE_STELLAR_ANGMOM_NOVA 26
#define DERIVATIVE_STELLAR_ANGMOM_NONCONSERVATIVE_LOSS 27
#define DERIVATIVE_STELLAR_ANGMOM_ARTIFICIAL 28
#define DERIVATIVE_STELLAR_ANGMOM_CORE_ENVELOPE_COUPLING 29
#define DERIVATIVE_STELLAR_ANGMOM 30

#define DERIVATIVE_STELLAR_ECCENTRICITY 31
#define DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES 32

#define DERIVATIVE_STELLAR_HE_CORE_MASS 33
#define DERIVATIVE_STELLAR_GB_CORE_MASS 34
#define DERIVATIVE_STELLAR_CO_CORE_MASS 35

/*
 * Derivatives for the BINT algorithm
 */
#define DERIVATIVE_STELLAR_CENTRAL_HYDROGEN 36
#define DERIVATIVE_STELLAR_CENTRAL_HELIUM 37
#define DERIVATIVE_STELLAR_CENTRAL_CARBON 38


/*
 * Other derivatives
 */
#define DERIVATIVE_STELLAR_HE_CORE_MASS_NO_TDUP 39
#define DERIVATIVE_STELLAR_NUM_THERMAL_PULSES 40
#define DERIVATIVE_STELLAR_NUM_THERMAL_PULSES_SINCE_MCMIN 41
#define DERIVATIVE_STELLAR_H_LAYER_MASS 42
#define DERIVATIVE_STELLAR_ROCHE_RADIUS 43
#define DERIVATIVE_STELLAR_NUM_NOVAE 44

#define NUMBER_OF_STELLAR_DERIVATIVES 45

#define STELLAR_DERIVATIVE_STRINGS              \
    {                                           \
        "M (Total)",                            \
        "M Wind Loss",                          \
        "M Wind Gain",                          \
        "Radius",                               \
        "Luminosity",                           \
        "Core Mass",                            \
        "Stellar temperature",                  \
        "M RLOF Loss",                          \
        "M RLOF Gain",                          \
        "M Nova",                               \
        "M RLOF Transfer",                      \
        "M Disk Gain",                          \
        "M Disk Loss",                          \
        "M Decretion Disc Loss",                \
        "M CBdisc Gain",                        \
        "M Non-conservative Loss",              \
        "M Irradiative Loss",                   \
        "M artificial",                         \
        "J RLOF Loss",                          \
        "J RLOF Gain",                          \
        "J Wind Loss",                          \
        "J Wind Gain",                          \
        "J Tides",                              \
        "J Magnetic Braking",                   \
        "J Decretion Disk",                     \
        "J CBdisc gain",                        \
        "J Nova",                               \
        "J non-conservative",                   \
        "J artificial",                         \
        "J core-envelope coupling",             \
        "J (Total)",                            \
        "eccentricity",                         \
        "angular velocity (tides)",             \
        "He core mass",                         \
        "GB core mass",                         \
        "CO core mass",                         \
        "Central hydrogen",                     \
        "Central helium",                       \
        "Central carbon",                       \
        "He core mass (no 3DUP)",               \
        "Number of thermal pulses",             \
        "Number of thermal pulses since mcmin", \
        "nova H layer mass",                    \
        "Roche radius",                         \
        "Number of novae"                       \
    }

#define Show_stellar_derivatives(S,SEPARATOR)                           \
    {                                                                   \
        int iii;                                                        \
        static const char * c[100] = STELLAR_DERIVATIVE_STRINGS;        \
        printf("star %d type %d : ",                                    \
               (S)->starnum,                                            \
               (S)->stellar_type);                                      \
        for(iii=0;iii<NUMBER_OF_STELLAR_DERIVATIVES;iii++)              \
        {                                                               \
            printf("%d %s = %g %s",                                     \
                   iii,                                                 \
                   c[iii],                                              \
                   (S)->derivative[iii],                                \
                   iii==NUMBER_OF_STELLAR_DERIVATIVES-1?"":(SEPARATOR)); \
        }                                                               \
        printf("\n");                                                   \
    }

#define Zero_stellar_derivatives(S)                             \
    {                                                           \
        int iii;                                                \
        for(iii=0;iii<NUMBER_OF_STELLAR_DERIVATIVES;iii++)      \
        {                                                       \
            (S)->derivative[iii]=0.0;                           \
        }                                                       \
    }



/* system */



#define DERIVATIVE_ORBIT_ANGMOM 0
#define DERIVATIVE_ORBIT_SEMI_MAJOR_AXIS 1
#define DERIVATIVE_ORBIT_ECCENTRICITY 2

#define DERIVATIVE_ORBIT_ANGMOM_GRAVITATIONAL_RADIATION 3
#define DERIVATIVE_ORBIT_ANGMOM_WIND_LOSS 4
#define DERIVATIVE_ORBIT_ANGMOM_WIND_GAIN 5
#define DERIVATIVE_ORBIT_ANGMOM_RLOF_LOSS 6
#define DERIVATIVE_ORBIT_ANGMOM_RLOF_GAIN 7
#define DERIVATIVE_ORBIT_ANGMOM_CBDISC 8
#define DERIVATIVE_ORBIT_ANGMOM_TIDES 9
#define DERIVATIVE_ORBIT_ANGMOM_NONCONSERVATIVE_LOSS 10
#define DERIVATIVE_ORBIT_ANGMOM_NOVA 11
#define DERIVATIVE_ORBIT_ANGMOM_ARTIFICIAL 12

#define DERIVATIVE_ORBIT_ECCENTRICITY_GRAVITATIONAL_RADIATION 13
#define DERIVATIVE_ORBIT_ECCENTRICITY_TIDES 14
#define DERIVATIVE_ORBIT_ECCENTRICITY_WINDS 15
#define DERIVATIVE_ORBIT_ECCENTRICITY_CBDISC 16

#define DERIVATIVE_SYSTEM_CBDISC_MASS_LOSS 17

#define DERIVATIVE_SYSTEM_TEST 18

#define NUMBER_OF_SYSTEM_DERIVATIVES 19

/* 
 * macro that is TRUE for a derivative that can be solved
 * exponentially, assuming dx/dt = x/tau where tau = constant.
 *
 * Good examples are tidal spin up/down and angular momentum 
 * loss from the system (the latter because as J->0 the stars
 * merge, at which point J=0 and everything stops).
 *
 * Such a solver still needs short timesteps (so we can say
 * tau ~ constant) but should not suffer from the problem where
 * the x overshoots and becomes negative.
 *
 * S = stardata
 * P = pointer to table of derivatives 
 * N = derivative number
 */

#define Exponential_derivative(S,P,N)                   \
    (                                                   \
    (                                                   \
    (P)==(S)->model.derivative &&                       \
        (                                               \
            (N) == DERIVATIVE_ORBIT_ECCENTRICITY ||     \
            (N) == DERIVATIVE_ORBIT_ANGMOM              \
            )) ||                                       \
    (                                                   \
        (N) == DERIVATIVE_STELLAR_ANGMOM ||             \
        (N) == DERIVATIVE_STELLAR_MASS                  \
        )                                               \
        )
#undef Exponential_derivative
#define Exponential_derivative(S,P,N)  (FALSE)

//#undef Exponential_derivative
//#define Exponential_derivative(S,P,N) (FALSE)

#define Jdot_orbit(S) (                                                 \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_GRAVITATIONAL_RADIATION] + \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_WIND_LOSS] +      \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_WIND_GAIN] +      \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_RLOF_LOSS] +      \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_RLOF_GAIN] +      \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_CBDISC] +         \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_TIDES] +          \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_NOVA] +           \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_NONCONSERVATIVE_LOSS] \
        )

#define edot_orbit(S) (                                                 \
        (S)->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_GRAVITATIONAL_RADIATION] + \
        (S)->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_TIDES] +    \
        (S)->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_WINDS] +    \
        (S)->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_CBDISC]     \
        )

#define Zero_stellar_mass_derivatives(STAR)                             \
    {                                                                   \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS] =         \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN] =     \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_RLOF_LOSS] =     \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_RLOF_GAIN] =     \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_NOVA] =          \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_RLOF_TRANSFER] = \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_DISC_GAIN] =     \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_DISC_LOSS] =     \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_DECRETION_DISC] = \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_CBDISC_GAIN] =   \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_NONCONSERVATIVE_LOSS] = \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_ARTIFICIAL] =    \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS_IRRADIATIVE_LOSS] = \
            (STAR)->derivative[DERIVATIVE_STELLAR_H_LAYER_MASS] =       \
            (STAR)->derivative[DERIVATIVE_STELLAR_MASS] =               \
            0.0;                                                        \
    } 

#define Zero_stellar_angmom_derivatives(STAR)                           \
    {                                                                   \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_WIND_LOSS] =   \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_WIND_GAIN] =   \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_LOSS] =   \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_GAIN] =   \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_TIDES] =        \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_MAGNETIC_BRAKING] = \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_DECRETION_DISC] = \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_CBDISC_GAIN] = \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_NOVA] =        \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_NONCONSERVATIVE_LOSS] = \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_ARTIFICIAL] =  \
            (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM] =             \
            0.0;                                                        \
    } 

#define Zero_orbit_and_system_derivatives(S) {                          \
        (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_GRAVITATIONAL_RADIATION] = \
            (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_WIND_LOSS] =  \
            (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_WIND_GAIN] =  \
            (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_RLOF_LOSS] =  \
            (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_RLOF_GAIN] =  \
            (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_CBDISC] =     \
            (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_TIDES] =      \
            (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_NOVA] =       \
            (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_ARTIFICIAL] = \
            (S)->model.derivative[DERIVATIVE_ORBIT_ANGMOM_NONCONSERVATIVE_LOSS] = \
            (S)->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_GRAVITATIONAL_RADIATION] = \
            (S)->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_TIDES] = \
            (S)->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_WINDS] = \
            (S)->model.derivative[DERIVATIVE_ORBIT_ECCENTRICITY_CBDISC] = \
            (S)->model.derivative[DERIVATIVE_SYSTEM_CBDISC_MASS_LOSS] = \
            0.0; }

#define ORBIT_AND_SYSTEM_DERIVATIVE_STRINGS     \
    {                                           \
        "J (total)",                            \
        "e (total)",                            \
        "a",                                    \
        "J Gravitational Radiation",            \
        "J Wind Loss",                          \
        "J Wind Gain",                          \
        "J RLOF Loss",                          \
        "J RLOF Gain",                          \
        "J Circumbinary Disc",                  \
        "J tides",                              \
        "J nonconservative",                    \
        "J nova",                               \
        "J artificial",                         \
        "e Gravitational Radiation",            \
        "e Tides",                              \
        "e Winds",                              \
        "e Circumbinary Disc",                  \
        "Mdot Circumbinary Disc",               \
        "Test"                                  \
    }

/*
 * Stellar accretion rates
 */
#define NEGLIGIBLE_ACCRETION_RATE 1e-40

/* mass gain only */
#define Mdot_gain(STAR)                                                 \
    (                                                                   \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_WIND_GAIN] +         \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_RLOF_GAIN] +         \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_DISC_GAIN] +         \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_CBDISC_GAIN]  +      \
        Max(0.0,(STAR)->derivative[DERIVATIVE_STELLAR_MASS_NOVA]) +     \
        Max(0.0,(STAR)->derivative[DERIVATIVE_STELLAR_MASS_ARTIFICIAL]) \
        )

/* mass loss only */
#define Mdot_loss(STAR)                                                 \
    (                                                                   \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS] +         \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_RLOF_LOSS] +         \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_DISC_LOSS] +         \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_DECRETION_DISC] +    \
        (STAR)->derivative[DERIVATIVE_STELLAR_MASS_NONCONSERVATIVE_LOSS]+ \
        Min(0.0,(STAR)->derivative[DERIVATIVE_STELLAR_MASS_NOVA]) +     \
        Min(0.0,(STAR)->derivative[DERIVATIVE_STELLAR_MASS_ARTIFICIAL]) \
        )

/* net gain of mass */
#define Mdot_net(STAR) ( Mdot_gain(STAR) + Mdot_loss(STAR) )

/* angular momentum gain only : these terms should be positive */
#define Jdot_gain(STAR)                                                 \
    (                                                                   \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_WIND_GAIN] +       \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_GAIN] +       \
        Max(0.0,(STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_TIDES]) +  \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_DECRETION_DISC] +  \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_CBDISC_GAIN] +     \
        Max(0.0,(STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_CORE_ENVELOPE_COUPLING]) \
        )

/* angular momentum loss only : these terms should be negative */
#define Jdot_loss(STAR)                                                 \
    (                                                                   \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_LOSS] +       \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_WIND_LOSS] +       \
        Min(0.0,(STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_TIDES]) +  \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_MAGNETIC_BRAKING] + \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_DECRETION_DISC] +  \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_NOVA] +            \
        (STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_NONCONSERVATIVE_LOSS] + \
        Min(0.0,(STAR)->derivative[DERIVATIVE_STELLAR_ANGMOM_CORE_ENVELOPE_COUPLING]) \
        )

/* net angular momentum */
#define Jdot_net(STAR) ( Jdot_gain(STAR) + Jdot_loss(STAR) )

/*
 * Stellar loss rates
 */

#define generic_derivative(STAR,N) ((STAR)->derivative[N])
#define mass_derivative(STAR,N) (generic_derivative((STAR),N))
#define angular_momentum_derivative(STAR,N) (generic_derivative((STAR),N))
#define specific_angular_momentum_derivative(STAR,N) (generic_derivative((STAR),N))


#define DERIVATIVE_MODEL 0
#define DERIVATIVE_STAR0 1
#define DERIVATIVE_STAR1 2


#endif // BINARY_C_DERIVATIVES_H
