#pragma once
/*
 * The binary_c stellar population nucleosynthesis framework.
 *
 * Contact: r.izzard@surrey.ac.uk or rob.izzard@gmail.com
 *
 * http://personal.ph.surrey.ac.uk/~ri0005/binary_c.html
 * https://gitlab.eps.surrey.ac.uk/ri0005/binary_c
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-announce
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-devel
 * https://twitter.com/binary_c_code
 * https://www.facebook.com/groups/149489915089142/
 *
 * Please see the files README, LICENCE and CHANGES,
 * and the doc/ directory for documentation.
 *
 *
 * This file contains debugging information and macros.
 */

#ifndef BINARY_C_DEBUG_H
#define BINARY_C_DEBUG_H

#include "binary_c_string_macros.h"

/*
 * Do you want debugging output?
 *
 * DEBUG = 1 : debugging output
 * DEBUG = 0 : debug functions are 'built' but not called
 * DISABLE_DPRINT defined : debug functions are disabled
 *
 * NB If DEBUG is zero then usually the functions are not built
 * because the compiler is clever enough to recognise that if(0)
 * is always false. Useful!
 *
 * See below for further options.
 */
#define DEBUG 0

/*
 * To remove debugging possibility from the code, globally, hence
 * speed it up (and reduce the executable size) enable this. Note, this
 * will be undefined if DEBUG is set above, but if DEBUG is set on a per-file
 * basis, you have to enable DISABLE_DPRINT here.
 *
 * Normally, you want to enable DISABLE_DPRINT (and disable DEBUG) for 
 * production code. For ANY testing, do not DISABLE_DPRINT (although DEBUG
 * is optional).
 *
 * NOTE: with gcc (4.1.2 and later have been tested) because the 
 * compiler is clever, and the debug statements are prepended with
 * "if(DEBUG)", the print statements are not built into the code. 
 * (if(DEBUG) is if(0) which is clearly never true) For this reason,
 * you should not have to define DISABLE_DPRINT unless you have an 
 * ancient compiler ...
 * (Intel's compiler is similarly clever although its executables
 *  seem to be larger, and faster)
 */
//#define DISABLE_DPRINT


/* an expression to dictate when we use debugging : 
 * usually (1) but you can use an expression which
 * must be TRUE for Dprint to do anything.
 * 
 * e.g. the following:12.3482943844 
 * #define Debug_expression (stardata->model.time>=259.0)
 * will only output when the time is >= 259 Myr.
 * 
 * Use stardata instead of stardata: usually these are 
 * the same, but stardata is defined everywhere as a global
 * variable.
 */
#define Debug_expression (1)

/*
 * If you define Debug_stop_expression, and it is at any time TRUE,
 * then binary_c will stop. 
 *
 * This expression is checked every time Dprint is used (i.e. every
 * call to debug_fprintf).
 *
 * Note that Debug_expression is not required to be TRUE. 
 * If you require Debug_expression to be TRUE, then you can 
 * set this as part of the Debug_stop_expression.
 */


//#define Debug_stop_expression ( stardata->common.nucsyn_metallicity > 0 )
#undef Debug_stop_expression 

/*
 * Debug_show_expression, if defined, is shown at the beginning
 * of every Dprint. It should take the form of a format statement
 * followed by variables (or expressions) just as you would 
 * pass as arguments to printf.
 *
 * If Debug_expression is 0, or stardata is NULL in debug_fprintf(),
 * this is not shown.
 */

#define Debug_show_expression " #=%ld Jorb=%g a=%g P=%g omegaorb=%g ",  \
        (long int)stardata->model.model_number,                         \
        stardata->common.orbit.angular_momentum,                        \
        stardata->common.orbit.separation,                              \
        stardata->common.orbit.period,                                  \
        stardata->common.orbit.angular_frequency

//#undef Debug_show_expression

/* 
 * DEBUG_STREAM is either a stream (stdout, stderr).
 *
 * If you want Dprint to output to Printf and hence be captured
 * in the output buffer, define DEBUG_STREAM_BUFFER (and DEBUG_STREAM
 * is ignored by Dprint, although possibly is used elsewhere).
 */
#ifndef DEBUG_STREAM
#define DEBUG_STREAM stdout
#define DEBUG_STREAM_BUFFER
#define MAX_DEBUG_PRINT_SIZE 2048
#endif // !DEBUG_STREAM


/* enable line number output from Dprint */
#define DEBUG_LINENUMBERS

/* enable filenames */
#define DEBUG_SHOW_FILENAMES 


/*
 * If enabled, DEBUG_FAIL_ON_NAN bails out binary_c when a 
 * nan is printed. Note that this can be overridden by setting
 * preferences->allow_debug_nan to TRUE (usually temporarily). 
 */
#define DEBUG_FAIL_ON_NAN

/*
 * If enabled, DEBUG_FAIL_ON_INF bails out binary_c when an 
 * inf is printed. Note that this can be overridden by setting
 * preferences->allow_debug_inf to TRUE (usually temporarily). 
 */
//#define DEBUG_FAIL_ON_INF

/*
 * Sometimes you'll be searching for a nan (not a number) in the debug
 * output. Enable DEBUG_REMOVE_NAN_FROM_FILENAMES to remove the nan from
 * filenames (e.g. remnant -> remn_nt). This adds a little overhead, but
 * who cares? You're in DEBUG mode anyway, you can't expect it to be fast.
 */
#define DEBUG_REMOVE_NAN_FROM_FILENAMES

/*
 * Enable caller information, if this is possible (requires BACKTRACE
 * and building with -rdynamic, -g etc. (use ./configure debug)
 */
//#define DEBUG_CALLER
#define DEBUG_REMOVE_NAN_FROM_CALLER
#define DEBUG_CALLER_LINE

#ifdef DEBUG_CALLER
/*
 * DEBUG_CALLER_DEPTH is the depth in the backtrace that one must 
 * go 
 */
#define DEBUG_CALLER_DEPTH 3
#endif // DEBUG_CALLER

#if (defined DEBUG_CALLER) && (defined DEBUG_CALLER_LINE)
#define ADDR2LINE
#endif // DEBUG_CALLER && DEBUG_CALLER_LINE

/*
 * Action for the Debug_stop_expression
 */
#ifdef Debug_stop_expression
#define _Debug_stop                                                     \
    if(stardata != NULL &&                                              \
       Debug_stop_expression &&                                         \
       stardata->store->debug_stopping == 0)                            \
    {                                                                   \
        fflush(NULL);                                                   \
        stardata->store->debug_stopping = 1;                            \
        Exit_binary_c(BINARY_C_DEBUG_STOP,                              \
                      "Stopping binary_c because Debug_stop_expression (%s) is TRUE.\n", \
                      Stringify_macro(Debug_stop_expression));          \
    }
#else
#define _Debug_stop /* */
#endif // Debug_stop_expression


/*
 * The all-important Dprint macro
 */
#ifndef DISABLE_DPRINT
#define Dprint_to_pointer(P,NEWLINE,...)                        \
    if( ( DEBUG ) && ( stardata!=NULL && Debug_expression ) )   \
    {                                                           \
        debug_fprintf((P),                                      \
                      __FILE__,                                 \
                      __LINE__,                                 \
                      (NEWLINE),                                \
                      __VA_ARGS__);                             \
    }

#define Dprint(...)                                     \
    {                                                   \
        Dprint_to_pointer(stardata,TRUE,__VA_ARGS__);   \
        _Debug_stop;                                    \
    }

#define Dprint_no_newline(...)                          \
    {                                                   \
        Dprint_to_pointer(stardata,FALSE,__VA_ARGS__);  \
        _Debug_stop;                                    \
    }

#else
#define Dprint(...) _Debug_stop;
#endif // !DISABLE_PRINT

#ifdef NANCHECKS
/* nan check */
#define Nancheck(A) {                                                   \
    if(isnan(A) || isinf(A))                                            \
    {                                                                   \
        fflush(stdout);                                                 \
        fflush(stderr);                                                 \
        printf("NaN detected in %s (file %s, line %d)\n",               \
               Stringof(A),__FILE__,__LINE__);                          \
        fflush(stdout);                                                 \
        fprintf(stderr,"NaN detected in %s (file %s, line %d)\n",       \
                Stringof(A),__FILE__,__LINE__);                         \
        fflush(stderr);                                                 \
        kill(0,SIGSEGV);                                                \
        Exit_binary_c(BINARY_C_EXIT_NAN,                                         \
                      "NaN detected in %s (file %s, line %d)\n",        \
                      Stringof(A),__FILE__,__LINE__);                   \
    }                                                                   \
    }
#else
/* do nothing if no Nancheck is defined, but also do not fail to compile */
#define Nancheck(A) /* do nothing */
#endif


#define FLUSH_LOG

/*
 * Define SHOW_STARDATA to use the contents of stardata.h to show
 * the contents of the stardata struct. 
 */
//#define SHOW_STARDATA

#endif // BINARY_C_DEBUG_H
