#include "../binary_c.h"
#include "RLOF.h"

void RLOF_stellar_angmom_derivative(struct stardata_t * RESTRICT const stardata)
{
    RLOF_stars;

    /*
     * Calculate stellar angular momentum derivatives because of
     * RLOF mass transfer.
     */

    /*
     * Remove angular momentum from the donor and add it
     * to the orbit.
     */
    double dj_dt_from_donor = 
        donor->derivative[DERIVATIVE_STELLAR_MASS_RLOF_LOSS] * 
        Pow2(donor->effective_radius)*donor->omega;
    
    donor->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_LOSS] += dj_dt_from_donor;
    stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_RLOF_LOSS] -= dj_dt_from_donor;
    Dprint("jdot (RLOF stream) from donor -= %30.20e (RLOF)\n",dj_dt_from_donor);
    
    /*
     * Determine maximum accreted angular momentum depending on whether
     * there is a disk or not and whether there are novae or not. 
     * This quantity is positive.
     */
    double dj_dt_onto_accretor;

    if(stardata->preferences->RLOF_angular_momentum_transfer_model==
       RLOF_ANGULAR_MOMENTUM_TRANSFER_MODEL_BSE)
    {
        double r_acc;
        
        if(RLOF_test_for_accretion_disc(stardata) == TRUE)
        {
            /* 
             * Accretion is through a circumstellar disc.
             *
             * Alter spin of the secondary by assuming that material falls onto
             * the star from the inner edge of a Keplerian accretion disk and that the
             * system is in a steady state. Adjust the primary spin accordingly so that
             * tidal interaction between primary and orbit will conserve total angular
             * momentum.
             */
            r_acc = accretor->effective_radius; 
            Dprint("disc accretion : accretor radius %g, effective radius %g\n",
                   accretor->radius,
                   accretor->effective_radius);
        }
        else
        {
            /*
             * No accretion disk : direct impact accretion.
             *
             * Calculate the angular momentum of the transferred material by 
             * using the radius of the disk (see Ulrich & Burger) that would 
             * have formed if allowed. 
             */
            r_acc = 1.7*accretor->rmin;
            Dprint("no accretion disk : accretor's rmin=%g vs radius=%g\n",
                   accretor->rmin,
                   accretor->radius);
        }
        
        dj_dt_onto_accretor = 
            donor->derivative[DERIVATIVE_STELLAR_MASS_RLOF_TRANSFER]
            * TWOPI * AU_IN_SOLAR_RADII * sqrt(AU_IN_SOLAR_RADII * r_acc * accretor->mass);
    }
    else if(stardata->preferences->RLOF_angular_momentum_transfer_model==
            RLOF_ANGULAR_MOMENTUM_TRANSFER_MODEL_CONSERVATIVE)
    {
        /*
         * Simple conservative model of angular momentum transfer
         */
        dj_dt_onto_accretor = dj_dt_from_donor;
    }
    else
    {
        /* fail */
        Exit_binary_c(BINARY_C_WRONG_ARGUMENT,"Unknown RLOF angular momentum transfer model\n");
    }

    Dprint("RLOF ang mom ratio : donated/accreted = %g/%g = %g\n",
           dj_dt_from_donor,
           dj_dt_onto_accretor,
           Is_zero(dj_dt_onto_accretor) ? 0.0 :
           (dj_dt_from_donor/dj_dt_onto_accretor));

    /*
     * Save the specific angular momentum in case we later
     * wish to remove mass
     */
    accretor->l_acc = 
        Is_zero(donor->derivative[DERIVATIVE_STELLAR_MASS_RLOF_TRANSFER]) ? 0.0 :
        dj_dt_onto_accretor/ donor->derivative[DERIVATIVE_STELLAR_MASS_RLOF_TRANSFER];
    
    Dprint("DJT Accrete : (accretor=%d) djt2=%30.20e (donor->derivative[DERIVATIVE_STELLAR_MASS_RLOF_TRANSFER]=%30.20e) l_acc=%30.20e\n",
           accretor->starnum,
           dj_dt_onto_accretor,
           donor->derivative[DERIVATIVE_STELLAR_MASS_RLOF_TRANSFER],
           accretor->l_acc
        );
 
    /*
     * Take angular momentum from the orbit and accrete it
     * onto the accretor
     */
    stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM_RLOF_GAIN] -= dj_dt_onto_accretor;
    accretor->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_GAIN] += dj_dt_onto_accretor;
    
    Dprint("jdot (RLOF onto accretor) += %30.20e (RLOF)\n",dj_dt_onto_accretor);

}
