#include "../binary_c.h"
#include "RLOF.h"


void RLOF_dynamical_transfer_from_WD_to_remnant(struct stardata_t * RESTRICT const stardata,
                                                struct RLOF_orbit_t * const RLOF_orbit)
{
    /*
     * Dynamic transfer from a white dwarf.  
     *
     * Secondary has stellar type *WD, NS or BH.
     *
     * The result is a merged object or a destroyed 
     * object, depending on the composition (hence stellar
     * type).
     */
    RLOF_stars;
    set_kelvin_helmholtz_times(stardata);

    /* Eddington limit for accretion on to the secondary in one orbit. */
    double Eddington_rate =  eddington_limit_for_accretion(stardata,
                                                           accretor->radius,
                                                           Hydrogen_mass_fraction);
    /* Dynamical timescale for the primary.*/
    double tdyn = dynamical_timescale(donor);

    double taum = sqrt(donor->tkh*tdyn);

    /*
     * Dynamical transfer: all the mass
     */
    RLOF_orbit->dM_RLOF_lose = -donor->mass;

    RLOF_orbit->dM_RLOF_transfer = 
        Min(Eddington_rate*taum,
            -RLOF_orbit->dM_RLOF_lose);

    stardata->model.supedd =
        Boolean_(RLOF_orbit->dM_RLOF_transfer > Eddington_rate*taum);

    /* accrete the material */
    accretor->mass += RLOF_orbit->dM_RLOF_transfer;

    Dprint("Dynamic transfer dM_RLOF_transfer=%12.12e dM_RLOF_lose=%12.12e\n",
           RLOF_orbit->dM_RLOF_transfer,
           RLOF_orbit->dM_RLOF_lose);

    if(donor->stellar_type==HeWD && 
       accretor->stellar_type==HeWD)
    {
#ifdef ALLOW_HeWD_SUPERNOVAE
    
        /*
         * HeWD+HeWD -> Explosion 
         * 
         * Assume the energy released by ignition of the triple-alpha reaction
         * is enough to destroy the star.
         */
        accretor->SN_type = SN_IA_He_Coal;
        struct star_t * news = 
            new_supernova(stardata,accretor,donor,NULL);
        if(news)
        {
            news->stellar_type = MASSLESS_REMNANT;
            stellar_structure_make_massless_remnant(news);
        }
#else
        /* 
         * HeWD + HeWD -> HeWD
         *
         * Just make a more massive HeWD, unless mass > 0.7
         * then convert to a HeMS star.
         */
        accretor->stellar_type  = HeWD;
        if(accretor->mass > HeWD_HeWD_IGNITION_MASS)
        {
            accretor->stellar_type=HeMS;
            /* reset the initial mass and age */
            accretor->phase_start_mass=accretor->mass;
            accretor->age=0.0;
        }
        accretor->core_mass = 0.0;
#ifdef NUCSYN
        accretor->dmacc=0.0;
#endif//NUCSYN
#endif//ALLOW_HeWD_SUPERNOVAE
    }
    else if(donor->stellar_type==HeWD || accretor->stellar_type==HeWD)
    {
        /*
         * Should be helium overflowing onto a CO or ONe core in which case the
         * helium swells up to form a giant envelope so a HeGB star is formed.
         * Allowance for the rare case of CO or ONe flowing onto He is made.
         */
        Stellar_type kst = HeGB;
        if(accretor->stellar_type==HeWD)
        {
            accretor->core_mass = RLOF_orbit->dM_RLOF_transfer;
        }
#ifdef BSE
        giant_age(&accretor->core_mass,
                  accretor->mass,
                  &kst,
                  &(accretor->phase_start_mass),
                  &(accretor->age),
                  stardata,
                  &(stardata->star[naccretor]));
#endif // BSE
        
#ifdef NUCSYN
        /*
         * Update surface abundances prior to call of nucsyn_WR function
         */
        Copy_abundances(donor->Xenv,accretor->Xenv);
#endif
        accretor->stellar_type = kst;
        accretor->epoch = stardata->model.time - accretor->age;
    }
    else if(accretor->stellar_type<=ONeWD)
    {
        accretor->phase_start_mass = accretor->mass;
        if(donor->stellar_type==ONeWD &&
           accretor->stellar_type==COWD)
        {
            /*
             * Mixture of ONe and CO will result in an ONe product.
             */
            accretor->stellar_type = ONeWD;
        }
    }

    /*
     * Might be a supernova that destroys the system 
     * if the chandrasekhar mass is reached
     */
    if(accretor->stellar_type<=COWD &&
       accretor->mass>stardata->preferences->chandrasekhar_mass)
    { 
        accretor->SN_type = SN_IA_CHAND_Coal;
        struct star_t * news = 
            new_supernova(stardata,accretor,donor,NULL);
        if(news)
        {
            stellar_structure_make_massless_remnant(news);
        }
    }

    donor->stellar_type = MASSLESS_REMNANT;
    donor->mass = 0.0;

    stardata->model.coalesce = TRUE;
    stardata->common.orbit.separation = 0.0;
    stardata->common.orbit.period = 0.0;
    stardata->common.orbit.angular_momentum = 0.0;
    stardata->common.orbit.eccentricity = 0.0;
        
    update_orbital_variables(stardata,
                             &stardata->common.orbit,
                             donor,
                             accretor);
        
    Dprint("force ndonor (star %d), star ndonor (%d) has mass %12.12e\n",
           donor->starnum,accretor->starnum,accretor->mass);
    Dprint("put nucsyn yields here? 4\n");

}

