#include "../binary_c.h"

#ifdef BINT
#include "BINT.h"

/*
 * BINT function to calculate main sequence 
 * luminosity and radius as a function of mass
 * and central hydrogen abundance.
 */


Stellar_type BINT_stellar_structure_MS(struct star_t * const newstar,
                                       struct stardata_t * RESTRICT const stardata)
{
    /************************/
    /** Main sequence star **/
    /************************/
    Dprint("MS star : BINT algorithm\n");

    /* MS stars have no He, CO or GB core */
    newstar->core_mass=0.0;
    newstar->CO_core_mass = 0.0;
    newstar->GB_core_mass = 0.0;

    newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_GB_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_CO_CORE_MASS] = 0.0;
    
    /*
     * Calculate the maximum convective core
     * mass on the MS. 
     * In low-mass stars this is zero.
     */
    double fconvcore = 
        effective_core_mass_fraction_of_MS_stars(newstar->mass,
                                                 stardata->common.metallicity);
    
    newstar->max_MS_core_mass = Max(newstar->max_MS_core_mass,
                                    newstar->mass * fconvcore);


#if defined NUCSYN && defined NUCSYN_FIRST_DREDGE_UP_ACCRETION_CORRECTION
    /* save the main sequence mass, which will remain at the terminal MS value */
    newstar->MS_mass = newstar->phase_start_mass;
#endif
    
    /* 
     * TODO : convert from BSE to BINT
     * convert to "low mass main sequence" star if mass is low enough 
     */
//    newstar->stellar_type = ...
//        LOW_MASS_MAIN_SEQUENCE : MAIN_SEQUENCE;

    /*
     * Use table to look up dXH/dt,L,R as a function of 
     * metallicity, phase start mass and central hydrogen abundance
     */
    {
        double params[BINT_MS_NPARAMS] MAYBE_UNUSED;

        params[BINT_MS_METALLICITY] = log10(stardata->common.metallicity);
        params[BINT_MS_MASS] = newstar->mass;
        params[BINT_MS_CENTRAL_HYDROGEN] = newstar->bint.XHc;
    
        double result[BINT_MS_NDATA];
                
        /*
         * Interpolate to find stellar structure
         */
        Interpolate(stardata->store->BINT_MS,
                    params,
                    result,
                    FALSE);

        /*
         * Hence set the stellar properties
         */

        newstar->luminosity = result[BINT_MS_LUMINOSITY];
        newstar->radius = result[BINT_MS_RADIUS];
        newstar->k2 = result[BINT_MS_K2];
        
        newstar->derivative[DERIVATIVE_STELLAR_CENTRAL_HYDROGEN] = result[BINT_MS_DERITATIVE_CENTRAL_HYDROGEN];
        newstar->core_mass = result[BINT_MS_CORE_MASS]; // convective core mass
        newstar->core_radius = result[BINT_MS_CORE_RADIUS]; // convective core radius
        newstar->menv = result[BINT_MS_MENV];
        newstar->renv = result[BINT_MS_RENV];
        newstar->max_MS_core_mass = Max(newstar->max_MS_core_mass,
                                        newstar->core_mass);

        /*
         * To calculate the "age" we use Xc as a (linear) proxy
         */
        newstar->tms = BINT_MS_lifetime(stardata,newstar);
        const double Xc0 = BINT_initial_XHc(stardata);        
        newstar->age = newstar->tms * (1.0 - newstar->bint.XHc / Xc0);

        newstar->E2 = E2(stardata,newstar); // TODO : E2 in the table
    }
    
    if(1)printf("BINT%d(%g,%g,%g) = L=%g R=%g dXHc/dt=%g\n",
                newstar->starnum,
                log10(stardata->common.metallicity),
                newstar->phase_start_mass,
                newstar->bint.XHc,
                newstar->luminosity,
                newstar->radius,
                newstar->derivative[DERIVATIVE_STELLAR_CENTRAL_HYDROGEN]);

#ifdef PRE_MAIN_SEQUENCE
    if(stardata->preferences->pre_main_sequence == TRUE)
    {
        double f = preMS_radius_factor(newstar->mass, 1e6*newstar->age); 
        newstar->radius *= f;
        newstar->PMS = Boolean_(f>1.00001);
	Dprint("preMS f=%g r=%g roche_radius=%g roche_radius_at_periastron=%g PMS? %d\n",
	       f,
               newstar->radius,
               newstar->roche_radius,
               newstar->roche_radius_at_periastron,
               newstar->PMS);
    }
#endif

    Dprint("MS radius %g\n",newstar->radius);

    return newstar->stellar_type;
}


#endif // BINT
