#include "../binary_c.h"

#ifdef BINT
#include "BINT.h"

int BINT_load_grid(struct stardata_t * const stardata)
{

    /*
     * Load BINT stellar evolution data.
     *
     * The data is in stardata->preferences->BINT_dir
     */


    /*
     * Check the BINT directory has been set
     */
    Dprint("BINT_dir = %s\n",
           stardata->preferences->BINT_dir);
    if(stardata->preferences->BINT_dir == NULL)
    {
        Exit_binary_c(BINARY_C_ENVVAR_EMPTY,
                      "BINT_dir is empty : probably you need to set the BINT_DIR environment variable\n");
    }

    /*
     * Load main sequence data
     */
    Dprint("BINT load MS? (BINT_MS == %p)\n",
           stardata->store->BINT_MS);
    if(stardata->store->BINT_MS == NULL)
    {
        char * MS_file;
        if(asprintf(&MS_file,
                    "%s/MS.dat",
                    stardata->preferences->BINT_dir)!=-1)
        {
            Dprint("Main sequence data should be at \"%s\"\n",
                   MS_file);

            FILE * data = fopen(MS_file,"r");
            Dprint("opened data, file pointer = %p\n",data);
        
            if(data != NULL)
            {
                Dprint("MS file is open at %p\n",data);
                size_t i;
                long int nl = file_linecount(MS_file,'#');
                Dprint("has %ld lines of data excluding comments\n",nl);


                /*
                 * First line is a header line : read the header contents
                 */
                char ** headers;
                size_t nheaders;
                file_splitline(data,
                               '#', // comment char
                               ",", // delimiter
                               &headers,
                               &nheaders);
                    
                for(i=0;i<nheaders;i++)
                {
                    Dprint("Header %ld = \"%s\"\n",i,headers[i]);
                }
                for(i=0;i<nheaders;i++)
                {
                    Safe_free(headers[i]);
                }
                Safe_free(headers);

                
                /*
                 * TODO : map columns to variables
                 */


                /*
                 * allocate memory for the table
                 */
                const int nparam = BINT_MS_NPARAMS;
                const int ndata = BINT_MS_NDATA;
                double * table = Malloc(sizeof(double) *
                                        nl * (nparam + ndata));

                /*
                 * Read in the table table
                 */ 
                for(i=0;i<nl;i++)
                {
                    char ** strings;
                    double * doubles;
                    size_t nstrings;
                    file_splitline(data,
                                   0,
                                   " ",
                                   &strings,
                                   &nstrings);
                    Dprint("got nstrings = %ld : %s, %s, %s...\n",
                           nstrings,
                           strings[0],
                           strings[1],
                           strings[2]);
                    
                    if(nstrings >= 0)
                    {
                        string_array_to_double_array(strings,
                                                     &doubles,
                                                     nstrings);
                        Dprint("doubles : %g,%g,%g...\n",
                               doubles[0],
                               doubles[1],
                               doubles[2]);
                        
                        /* strings are no longer needed: free them */
                        unsigned int j;
                        for(j=0;j<nstrings;j++)
                        {
                            Safe_free(strings[j]);
                        }
                        Safe_free(strings);
                        
                        /* copy doubles to table */
                        double * line = table + i * (nparam + ndata); 
                        Dprint("line = %p (offset %ld)\n",
                               line,
                               i*(nparam+ndata));
                               
                        /* parameters */
                        line[0] = log10(doubles[0]); // log10 metallicity
                        line[1] = doubles[1]; // mass
                        line[2] = doubles[2]; // Xc
                        Dprint("set in table at line : 0=%g 1=%g 2=%g\n",
                               line[0],
                               line[1],
                               line[2]);
                        
                        /* data */
                        unsigned int k;
                        for(k=nparam; k<nparam+ndata; k++)
                        {
                            line[k] = doubles[k];
                            Dprint("Set line[%d] = %g\n",k,line[k]);
                        }
                        
                        /* free memory */
                        Safe_free(doubles);
                    }
                }
                Dprint("data table has been set at %p is %g,%g,%g\n",
                       table,
                       table[0],
                       table[1],
                       table[2]
                    );
                
                NewDataTable_from_Pointer(table,
                                          stardata->store->BINT_MS,
                                          nparam,
                                          ndata,
                                          nl);
                Dprint("MS data has been loaded\n");
            }
            else
            {
                Exit_binary_c(BINARY_C_FILE_OPEN_ERROR,
                              "Could not open file \"%s\" for main-sequence BINT data\n",
                              MS_file);
            }
            fclose(data);
            Safe_free(MS_file);


            /*
             * ... load other tables here
             */ 
        }
        else
        {
            Exit_binary_c(BINARY_C_MALLOC_FAILED,
                          "asprintf failed (memory allocation failed?)");
        }

    }

    return 0;
}
#endif
