package rob_misc;

############################################################
# Rob's misc functions module
#
# Many useful functions are in here, MAX, MIN, slurp etc.
#
# Where possible, I have mapped these functions to Perl 
# modules which you probably want to install using (say) 
# cpanminus.
#
# You will need a recent Perl, and certainly at 
# least 5.10.0 and/or a Perl with the "state" functionality.
#
############################################################ 
#
# This module depends on many other modules.
#
# The best way to install all the dependencies is
#
# 1) Install the latest version of perl with perlbrew
# http://perlbrew.pl/ 
#
# 2) Install cpanm
# http://perlbrew.pl/Perlbrew-and-Friends.html 
#
# 3) install modules on the command line, e.g. with
# 'cpanm <module_name>'
#
# Note that many of the required modules are part of the
# standard perl installation.
#
# You can also install tar.gz files with cpanm, e.g.
#
# cpanm xyz.tar.gz
#
# or aim cpanm directly at the appropriate module directory
#
# cpanm xyz/ 
# 
############################################################

use 5.16.0;
use common::sense;
use feature qw(state);
require Exporter;

use Term::ANSIColor qw/color/;
use Carp qw(confess);
use Time::HiRes qw(tv_interval gettimeofday);

use File::Find qw/find/;
use File::Path qw(make_path);
use File::Temp qw/tempfile tmpnam/;

use Sys::Info;
use Sys::Info::Device;
use Sys::Info::Device::CPU; # in Sys::Info
use Sys::Info::Constants qw( :device_cpu ); # in Sys::Info
use Fcntl qw(O_NONBLOCK O_RDONLY);
use IO::File;
use strict;

$|=1;

############################################################
# module stuff : export all by default because we are (very) lazy
our @ISA = qw(Exporter);
my @subs=qw '&sum_array &MAX &numericMAX &MIN &numericMIN &conv_time_units &trem &trem2 &log10 &slurp &safeslurp_to_array_ref &slurp_to_array_ref &safeslurp &safedumpfile &mapsplit &string_to_array &array_to_string &mem_usage &proc_mem_usage &sum_array_with_pointers &gnuplot_prettytitle &convert_isotopes &numerical_list_sorter &is_numeric &max_from_array &min_from_array &mean_from_array &f_number_to_c_number &ok_directory &gnuplot_ps_to_png &gnuplot_ps_to_png2 &touch &session_id &flip &certain_kill &certain_kill_and_children &remove_pango_markup &ok_file &listfiles &listfiles_path &empty_dir &toggle_state &dumpfile &pmrand &poisson_error &get_from_C_header &determine_if_we_have_procps &in_numerical_order &ssleep &binslurp &file_linecount &ynstring &non_zero_array &non_zero_array_from_pointer &ncpus &splitcomma  &scalars_the_same &coloured_diff_strings &check_lines_not_equal &ps_to_png &thread_log &thread_string &thetime &join_arrays_as_hash &mkdirhier &find_media_location &username &bin_data &random_string &heximage &extract_gnuplot_range &operating_system &nfscheck &versionofmodule &gpshow &list_modules &renice_me &port_use &remove_ANSI &remove_all_escape_codes &be_sudo &hash_count &clamp &numeric_clamp &du &runcmd &listdirs &listdirs_path &nsubdirs &listdirs_with_no_subdirs &is_file_bzip2_archive &parse_cmdline_args_to_perlvars &bytes_pending &swap &compactnumber &colourhash &gnuplot_string &hashcount &reverse_lookup_hash remws checklib float_to_latex &full_command_line';
our %EXPORT_TAGS = ( 'all' => [ @subs ] );
our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );
our @EXPORT = @subs;
our $VERSION = '0.16'; 

############################################################
#
# some kind of vague changelog:
#
# V0.06 : add remove_all_escape_codes
#
# V0.07 : use CPAN modules for ncpus and mkdirhier instead of my functions
#         (works on more paltforms)
#
# V0.08 : remove arithmetically_add_hashes (now in RobMerge)
#
# V0.09 : removed df_mb (never used)
#
# V0.10 : added bytes_pending
#
# V0.11 : Use Posix for is_numeric_scalar where possible
#
# V0.12 : Re-order subroutines in alphabetical order, improve documentation.
#
# V0.13 : Up version for WTTS2 to differentiate previous releases.
#
# V0.14 : Add checklib to export list
#
# V0.15 : Add test_for_header_file
#
# V0.16 : Add full_command_line()
#
############################################################
#
# Subroutines follow...
#
############################################################

sub be_sudo
{
    # Usage: be_sudo(@<command line arg list>)
    # Run command list passed in as arguments as sudo

    if(!defined($ENV{SUDO_USER}))
    {
	print "You are not root, running as sudo:\n";
	my $cmd=join(' ','sudo',$^X,'-I',join(' -I',@INC), $0, @ARGV ); 
	exec($cmd) or confess "Can't execute '$cmd' so killing self: $!\n";
    }
}

sub full_command_line
{
    # return full command line (up to 1MB) :
    # (tested only on Linux and really a hack!)
    #
    # Process ID is passed in, defaults to the current process
    my $pid = $_[0] // $$;
    my $procfile = "/proc/$pid/cmdline";
    my $cmd = `cat $procfile | xargs -0`;
    chomp $cmd;
    return $cmd;
}

sub bin_data
{
    # Usage: bin_data($x,$dx)
    # bin value $x to nearest bin of width $dx

    my ($x,$dx)=@_;
    confess("Either x=$x or dx=$dx is not defined in bin_data\n") 
	if((!defined($x)) || (!defined($dx)));
    return (($x>0.0?1.0:-1.0)*0.5+int($x/$dx))*$dx;
}

sub binslurp
{
    # Usage: binslurp(<filename>)
    # Returns the contents of file given by <filename> using Perl's binmode

    open(my $fh,'<',$_[0])||confess("cannot open $_[0] in slurp");
    binmode $fh;
    return (do { local( $/ ) ; <$fh> } );
}

sub bytes_pending
{
    # Usage: bytes_pending($handle)
    # Returns the number of pending bytes on the pipe given by $handle

    # use the perl cookbook recipe to find the number
    # of pending bytes on a pipe.
    #
    # Requires gcc, File::Temp
    
    my $handle = $_[0];
    state $FIONREAD;

    if(!defined $FIONREAD)
    {
	# need FIONREAD
	my ($code_fh, $code_filename) = File::Temp::tempfile();
	
    # output the code
	print {$code_fh} <<'BYTES_PENDING_CODE'
#include <stdio.h>
#include <sys/ioctl.h>
	    main() {
		printf("%#08x\n", FIONREAD);
	}    
BYTES_PENDING_CODE
    ;
	close $code_fh;

	#print "CODE in $code_filename = ",slurp($code_filename),"\n";

	# exectuable
	my $exec_filename = File::Temp::tmpnam();
	
	# compile and run
	$FIONREAD = `gcc -xc $code_filename -o $exec_filename ; $exec_filename`;
	chomp $FIONREAD;
	
	#print "RAW FIONREAD $FIONREAD\n";

	eval "\$FIONREAD = $FIONREAD";
    }
#    print "Set FIONREAD $FIONREAD\n";
    
    my $size = pack("L", 0);
    ioctl($handle, $FIONREAD, $size) or return undef;
    $size = unpack("L", $size);
    
 #   print "Size of $handle = $size\n";
    return $size;
}

sub certain_kill
{
    # Usage: certain_kill($pid)
    # Kills the process given by $pid by sending a signal 9, then a 15 to make sure

    my $pid=$_[0];
    # kill a process $pid, making certain to kill it
    kill 9,$pid;
    ssleep(1);
    # force a kill
    kill 15,$pid;
    ssleep(1);
}

sub certain_kill_and_children
{
    # as certain_kill for a process and its children
    my $pid=$_[0];
    kill -9,$pid;
    ssleep(1);
    kill -15,$pid;
    ssleep(1);
}

sub check_lines_not_equal
{
    # Usage: check_lines_not_equal($l1,$l2,$threshold)
    # Checks if two lines of numbers (split by spaces or colons), $l1 and $l2, are numerically equals to within error $threshold, returns two list elements: 0 = the maximum difference between any of the two numbers, 1 = an error string (or undef if there is no error).

    # check if two lines of data are equal: return 0 if equal, the fractional 
    # maximum error on any of the variables (or, in pathological cases
    # 1 which is clearly a bad fractional error!)
    # also return a warning string, which is undef if there is no error
    
    # get the two data lines to compare
    my ($l1,$l2,$threshold)=@_;

    if(!defined($l1))
    {
	if(!defined($l2))
	{
	    return (0,undef);
	}
	else
	{
	    return(1,"First string ($l1) is defined, second is not!"); 
	}
    }
    elsif(!defined($l2))
    {
	print (1,"Second arg is not defined!");
    }

    if($l1 eq $l2)
    {
	# the most obvious test for equality
	return (0,undef);
    }
    else
    {
	# now the more complicated test
	my $splitregexp='[\s|\:]+';
	my @l1=split(/$splitregexp/,$l1);
	my @l2=split(/$splitregexp/,$l2);
	if($#l1 != $#l2)
	{
	    # problem : arrays are of different length!
	    return (1,sprintf"Arrays in comparison are of different length! 1=%d, 2=%d\n",$#l1,$#l2);
	}
	else
	{
	    # the threshold to within which numbers should be equal

	    my $err=0.0;
	    my $warn;
	    foreach my $i (0..$#l1)
	    {
		$err=MAX($err,scalars_the_same($l1[$i],$l2[$i],$threshold));
		$warn.= sprintf "Array element %d mismatch (numeric vs string) : in 1 = 
'%s', in 2 = '%s'\n",
		$i,$l1[$i],$l2[$i] if($err!=0.0);
	    }
	    return ($err,$warn);
 	}
    }
}


sub clamp
{
    # Usage: clamp($x,$low,$high)
    # Clamps $x to given range $low to $high

    return MAX($_[1],MIN($_[2],$_[0]));
}

sub coloured_diff_strings
{
    # Usage: coloured_diff_strings($x,$y)
    # Returns a string which highlights the difference between $x and $y in colour

    # given two scalars (strings) split them on spaces
    # and return a string which highlights the differences
    # in colour (magenta)
    my ($x,$y)=@_;
    chomp $x;
    chomp $y;
    $y=~s/^\s+//o;
    $x=~s/^\s+//o;
    my @x=split(/\s+/,$x);
    my @y=split(/\s+/,$y);
    my $n=MAX($#x,$#y);
    $x[0]='Line missing!' if($#x==-1);
    $y[0]='Line missing!' if($#y==-1);

    my $colour_magenta = color('magenta bold');
    my $colour_reset = color('reset');

    foreach my $i (0..$n)
    {
	if(scalars_the_same($x[$i],$y[$i])!=0.0)
	{
	    my $m=0;
	    my $dx=defined($x[$i]);
	    my $dy=defined($y[$i]);
	    if($dx or $dy)
	    {
		$m=MAX($m,length($x[$i])) if($dx);
		$m=MAX($m,length($y[$i])) if($dy);
		my $format='% '.$m.'s';
		#print "FORMAT $i: $format from '$x[$i]' '$y[$i]', lengths ",length($x[$i])," ",length($y[$i]),"\n";
		$x[$i]=$colour_magenta.sprintf $format,$x[$i].$colour_reset if($dx);
		$y[$i]=$colour_magenta.sprintf $format,$y[$i].$colour_reset if($dy);
	    }
	}
    }
    return (join(' ',@x,"\n").join(' ',@y,"\n"));
}

sub conv_time_units
{
    # Usage: conv_time_units($t)
    # Converts $t (in seconds) to seconds, minutes or hours depending on its magnitude

    # convert time $eta in seconds to minutes or hours
    my $eta=$_[0];
    my $units='s'; # default to seconds
    if($eta>60)
    {
        $eta/=60;
        $units='m';
    }
    if($eta>60)
    {
        $eta/=60;
        $units='h';
    }
    return($eta,$units);
}

sub convert_isotopes
{ 
    # Usage: convert_isotopes($i)
    # Convert isotope strings from AB123 to ^{123}AB for gnuplot or LaTeX

    my ($t,$fnsize)=@_;
    $fnsize eq '' ?
	$t=~s/(\W?)([A-Z][a-z]*)(\d+)(\W?)/$1\^\{$3\}\{$2\}$4/g :
	$t=~s/(\W?)([A-Z][a-z]*)(\d+)(\W?)/$1\^\{\/=$fnsize $3\}\{$2\}$4/g;
    return $t;
}

sub determine_if_we_have_procps
{
    # Usage: determine_if_we_have_procps()
    # Returns 1 if the system has procps, 0 otherwise.

    my $u=`uname -a`;
    if(defined($u))
    {
	my $s=`ps --version`;
  	return ((defined($s))&&(($s=~/procps/o)||($u=~/macintosh/io))) ? 1 : 0;
    }
    else
    {
	# probably windoze
	return 0;
    }
}

sub du
{
    # Usage: du($dir)
    # Returns the size of the contents of the directory $dir
   
    my $size = 0;             
    find(sub { $size += -s if -f $_ }, $_[0]);
    return $size;
}

sub swap
{
    # swap $_[0] and $_[1], in place
    my $x = $_[1];
    $_[1] = $_[0];
    $_[0] = $x;
}

sub safedumpfile
{
    # Usage: dumpfile($filename,$data)
    # Puts the string $data into the file given by $filename
    open(my $fp,'>'.$_[0])||return undef;
    binmode $fp;
    print {$fp} $_[1];
    close $fp;
}

sub dumpfile
{
    # Usage: dumpfile($filename,$data)
    # Puts the string $data into the file given by $filename
    open(my $fp,'>'.$_[0])||confess("cannot open $_[0] for output : $!");
    binmode $fp;
    print {$fp} $_[1];
    close $fp;
}

sub empty_dir
{
    # Usage: empty_dir($directory)
    # Removes all the files $directory (NB ignores errors!)
    
    my $dir=$_[0];
    if(-d $_[0])
    {
	my @filelist=listfiles($dir);
	map
	{
	    unlink $dir.'/'.$_;
	}@filelist;
    }
}

sub extract_gnuplot_range
{
    # Usage: extract_gnuplot_range($logstring)
    # Extracts "show range" commands from a the gnuplot output given in $logstring
    
    my %range;
    my $data=$_[0]; # pointer to data (from stderr)
    my $err=$data; # copy so we can return
    my $vb=0;
    my $ismap=0;
    $ismap=1 if($$data=~/view is map/);

    foreach my $ax ('x','x2','y','y2','z')
    {
	my $ax2=$ax.'range';
	if($$data=~s/(set $ax2.*)//)
	{
	    my $r=$1;
	    print "RANGE REGEXP $1\n"if($vb);
	    my $reverse=($r=~/ reverse/o);
	    
	    my $rlow;
	    my $rhigh;

	    # default to the given low and high, whatever they are
	    if($r=~/set $ax2 \[ (\S+) \: (\S+) \]/)
	    {
		$rlow=$1;
		$rhigh=$2;		
	    }
	    else
	    {
		print "Error: Failed to extract \"set $ax2 ...\" from gnuplot range\n";
		exit;
	    }

	    print "REVERSE=$reverse : ismap? $ismap\n"if($vb);
	    if($r=~/currently\s*\[\s*(\S*)\s*\:\s*(\S*)\s*\]/)
	    {
		my $newlow=$1;
		my $newhigh=$2;

		print "\"currently\" regexp got \[$newlow:$newhigh\]\n"if($vb);

		$rlow=$newlow if(is_numeric($newlow));
		$rhigh=$newhigh if(is_numeric($newhigh));

		$range{$ax}= $reverse ? [$rhigh,$rlow] : [$rlow,$rhigh];

	    }
	    else
	    {
		$range{$ax}= ($reverse && (!$ismap)) ? [$rhigh,$rlow] : [$rlow,$rhigh];
	    }
	    print "Extracted range $ax: @{$range{$ax}}\n" if((defined($range{$ax})) && $vb);
	}
    }
    return (\%range,$err);
}

sub f_number_to_c_number
{
    # Usage: f_number_to_c_number($f_number)
    # Converts fortran number $f_number (with 'D' or 'd') to a C-style floating-point number with an 'e'. There is no error checking.

    my $s=$_[0];
    $s=~s/[dED]/e/o;
    return $s;
}

sub file_linecount
{
    # Usage: file_linecount($filename)
    # Count the number of lines in a file given by $filename

    my $f=$_[0];
    my $count=0;
    my $n = 2**16;
    open(my $fp,'<'.$f)||return -1;
    $count += tr/\n/\n/ while sysread($fp, $_, $n);
    close $fp;
    return $count;
}



sub find_media_location
{
    # Usage: find_media_location($id)
    # Finds Rob's media disk given by the name $id

    # find one of rob's media by given id:
    # checks all directories in /media and looks for the 
    # file named $id in those directories.
    # Returns undef on failure
    my $id=$_[0];
    my $mediadir='/media/';
    my $ret=undef;
    #print "Looking for $id in $mediadir/*\n";
    
    opendir(my $dp,$mediadir)||return undef;

    # use both ls and readdir to read the directory, if possible, because
    # sometimes (due to HAL?) they give different results :-|
    my @ls=`ls $mediadir`;
    my @dirs=readdir ($dp);

    foreach my $d (@dirs,@ls)
    {
	chomp $d;
	#print "Readdir \"$d\"\n";
	next if($d=~/^\./); # assume .whatever are not to be read
	
	my $d1=$mediadir.'/'.$d;
	
	#print "Check for dir $d1\n";
	if(defined $d1)
	{
	    my $f1=$d1.'/'.$id;
	    print "Check for file $f1\n";
	    { 
		# verbosity
		my $d_d1=-d $d1;
		my $f_f1=-f $f1;
		if(defined($d_d1) && defined($f_f1))
		{
		    print "Check dir d=$d : d1=$d1 : f1=$f1 : -d:",1*($d_d1),' -f',1*($f_f1),"\n";
		}
	    }

	    if(
		(-d $d1)&& # must be a directory 
		(-f $f1) # must have id file
		)
	    {
		# found it! return directory
		closedir $dp;
		return $d1;
	    }
	}

    }
    # nothing found : return undef
    closedir $dp;
    return undef;
}

sub flip
{
    # Usage: flip($x,$y)
    # Returns ($y,$x) i.e. flips the numbers in the array
    return ($_[1],$_[0]);
}

sub get_from_C_header
{
    # Usage: get_from_C_header($macro,$headerfile)
    # Uses gcc and/or Convert::Binary::C to get the value of the macro $macro in the C-header file $headerfile

    my ($macro,$headerfile)=@_;

    # first, try with gcc
    my @res=`gcc -O0 -g3 -E $headerfile`;
    chomp @res;
    map
    {
	if(/^\#define $macro (.*)/)
	{
	    return $1;
	}
	elsif(/^\#define $macro/)
	{
	    return 'DEFINED';
	}
    }@res;

    # using gcc clearly failed :(

    # try Convert::Binary::C, if y
    eval 'use Convert::Binary::C';

    if(! $@)
    {
        # try to use Convert::Binary::C
        my $c = Convert::Binary::C->new;
        
        # use va args
        $c->configure(HasMacroVAARGS=>1);
        
        # fix failure with builtin_va_list (who cares?)
        $c->KeywordMap({ 
            __builtin_va_list => undef,
                       });
        
        # build includes
        my @include ;
        
        # gcc gives (gcc -v -E - < /dev/null > /dev/null)
        #
        # /usr/lib/gcc/x86_64-linux-gnu/4.6/include
        # /usr/local/include
        # /usr/lib/gcc/x86_64-linux-gnu/4.6/include-fixed
        # /usr/include/x86_64-linux-gnu
        # /usr/include

        # use rob_misc::listdirs to avoid the version number problem
        
        my @gccdirs=('/usr/lib/gcc/x86_64-linux-gnu/');
        foreach my $d (@gccdirs)
        {
            push(@include,map {$d.$_.'/include'} listdirs($d));
        }
        push(@include,'/usr/local/include');
        foreach my $d  (@gccdirs)
        {
            push(@include,map {$d.$_.'/include-fixed'} listdirs($d));
        }
        push(@include,@gccdirs,'/usr/include','/include');

        # do the include
        $c->Include(@include);

        # parse
        #print "Parse $headerfile\n";
        eval { $c->parse_file($headerfile) };

        # check for error
        if($@)
        {
            print "Error parsing $headerfile with Convert::Binary::C : $@\n";
            return undef;
        }

        # loop over macros, see if required macro is defined or not
        map
        {
            if($_ eq $macro)
            {
                return 'DEFINED';
            }
            elsif(/^$macro (.*)/)
            {
                return $1;
            }
        }$c->macro;
    }
    return undef;
}

sub gnuplot_ps_to_png
{
    # Usage: gnuplot_ps_to_png($pltfile,$pngfile,$dpi,$colourblind,$invert,$extra)
    # Given a gnuplot .plt file (list of gnuplot commands) and generates a png file from it via a postscript file. The PNG file is given in $pngfile, and the plt file MUST make a .ps file with the same name as the .png file (except for the file extension). DPI is the conversion resolution. If $colourblind is true, then the default red and green are converted. If $invert is true then black and white are flipped. $extra contains extra args sent to the ps_to_png() function call (see below). Returns an array containing the graph range and error string.

    # universal gnuplot -> ps -> png function
    my $pltfile=$_[0]; # .plt gnuplot file
    my $pngfile=$_[1]; # output file (xyz.png, assumes gnuplot dumps xyz.ps)
    my $dpi=$_[2]; # conversion resolution
    my $colourblind=$_[3]; # gnuplot green -> other colour
    my $invert=$_[4];  # black > white?
    my $extra=$_[5]; # extra args

    # save psfile too
    my $psfile=$pngfile;
    $psfile=~s/\.png$/\.ps/o;
    
    # get the result of gnuplot, send (small) stderr to file.err
    my $ps=`gnuplot $pltfile 2> $psfile.err`;
   
    # convert to EPSF
    $ps=~s/Adobe-2.0/Adobe-2.0 EPSF/;

    # save gnuplot error data
    my $err=slurp($psfile.'.err');

    # extract range information
    my $rangedata = $err;
    unlink($psfile.'.err'); # remove file.err
    #print "ERR $err\n";
    my ($range,$null) = extract_gnuplot_range(\$rangedata);
   
    # colourblind/invert colour options
    $ps=~s/0 1 0/0.8 0.2 0.8/go if($colourblind);
    $ps=~s/0 0 0/0.7 0.8 0.9/go if($invert);
    
    # save to file
    dumpfile($psfile,$ps);

    # convert to PNG
    ps_to_png($psfile,$pngfile,$dpi,$extra);

    # return the range and gnuplot error 
    # (only $range is used in scalar context)
    return ($range,$err);
}

sub gnuplot_ps_to_png2
{
    # Usage: gnuplot_ps_to_png2($pltfile,$pngfile,$dpi,$colourblind,$invert,$extra)
    # Given a gnuplot .plt file (list of gnuplot commands) and generates a png file from it via a postscript file. The PNG file is given in $pngfile, and the plt file MUST make a .ps file with the same name as the .png file (except for the file extension). DPI is the conversion resolution. If $colourblind is true, then the default red and green are converted. If $invert is true then black and white are flipped. $extra contains extra args sent to the ps_to_png() function call (see below). This version of the subroutine returns the graph range and error string as a hash.

    # universal gnuplot -> ps -> png
    #
    # this version takes a hash pointer as input and 
    # returns a hash pointer
    my $opts=$_[0];

    my $pltfile=$$opts{'input file'}; # .plt gnuplot file
    my $pngfile=$$opts{'png output file'}; # output file (xyz.png, assumes gnuplot dumps xyz.ps)
    my $dpi=$$opts{'dpi'} || 100; # conversion resolution
    my $colourblind=$$opts{'colourblind'} || 0; # gnuplot green -> other colour
    my $invert=$$opts{'invert'}||0;  # black > white?
    my $extra=$$opts{'extra'}; # extra args
   
    # save psfile too
    my $psfile=$pngfile;
    $psfile=~s/\.png$/\.ps/o;
    
    my $ps;
    my $err;
    my $plt=slurp($pltfile);

    if($invert)
    {
	# black background please
	my $bg='#0c0c0c'; # almost, but not quite, black
	my $invstring="
set obj 1 rectangle behind from screen 0,0 to screen 1,1
set obj 1 fillstyle solid 1.0 fillcolor rgbcolor \"$bg\"
";
	$plt=~s/(s?plot)/\n$invstring\n$1/g;
    }

    ($ps,$err)=gp_send_and_return($opts,$plt);

    # convert to EPSF
    $ps=~s/Adobe-2.0/Adobe-2.0 EPSF/;

    #print "PS $ps\nERR $err\n";

    # extract range information
    my $rangedata = $err;

    my ($range,$null) = extract_gnuplot_range(\$rangedata);
   
    # colourblind/invert colour options
    $ps=~s/^(\/LC[\w]+ )\{0 1 0\}( def)$/$1\{0.8 0.2 0.8\}$2/mgo if($colourblind);
    if($invert)
    {
	my $c='0.7 0.8 0.9';
	$ps=~s/^(\/LC[\w]+ )\{0 0 0\}( def)$/$1\{$c\}$2/mgo; 
	$ps=~s/^0\.00 0\.00 0\.00 C/$c C/mgo;
    }

    # remove lines starting in "Warning" : these are almost
    # certainly produced by gnuplot because any text in the 
    # plot is wrapped in parentheses
    $ps=~s/^Warning.*//mgo;

    # save to file
    dumpfile($psfile,$ps);

    # convert to PNG
    ps_to_png($psfile,$pngfile,$dpi,$extra);

    # return the range and gnuplot error 
    # (only $range is used in scalar context)
    return {'graph range'=>$range,
	    'err'=>$err};
}

sub gnuplot_prettytitle
{
    # Usage: gnuplot_prettytitle($title)
    # Cleans up $title for gnuplot enhanced mode postscript, converting alpha, lambda and sigma to their symbols.

    my $title = $_[0];
    $title=~s/([aA])lpha(_)(\S+)/\{\/Symbol $1\}$2\{$3\}/g;
    $title=~s/([lL])ambda(_)(\S+)/\{\/Symbol $1\}$2\{$3\}/g;  
    $title=~s/([sS])igma(_)(\S+)/\{\/Symbol $1\}$2\{$3\}/g;  
    return $title;
}

sub gp_send_and_return
{
    # Usage: send commands to a gnuplot stream
    # Description: this is an experimental function!

    # arg 0 = opts hash
    # arg 1 = commands (string) to send to gnuplot
    # returned: gnuplot's output, range string
    my $opts=$_[0];
    
    # send commands
    print {$$opts{'gnuplot stream in'}} "reset\n",$_[1],
    "\nprint \"gnuplot finished\"\nset output\n";

    # get output
    my $brk=0;
    my $start=0;
    my @out;
    while(($brk==0) && ($_=$$opts{'gnuplot stream out'}->getline))
    {
	if($_ eq "gnuplot finished\n")
	{
	    $brk=1;
	}
	else
	{
	    # ignore everything before the start of the postscript
	    # (e.g. gnuplot warnings)
	    $start=1 if(/\%\!PS-Adobe-2\.0/);
	    push(@out,$_) if($start);
	}
    }

    # get range info
    my @range;
    $brk=0;
    print {$$opts{'gnuplot stream in'}} gpshow(),"\nprint \"gnuplot finished\"\nset output\n";
    while(($brk==0) && ($_=$$opts{'gnuplot stream out'}->getline))
    {
	if($_ eq "gnuplot finished\n")
	{
	    $brk=1;
	}
	else
	{
	    push(@range,$_);
	}
    }

    # return
    return join('',@out),join('',@range);
}

sub gpshow
{
    # Usage: gpshow()
    # Returns the command to output the range information in gnuplot 

    # final command to gnuplot to output the axis ranges
    my $s="\n\n";
    foreach ('x','y','y2')
    {
	$s .= 'show '.$_."range\n";
    }
    $s.="show view\n\n";
    return $s;
}

sub hash_count
{
    # Usage: hash_count($hash_pointer,$offset)
    # count the number of elements in a (nested) hash at $hash_pointer, offset by $offset (0 if not given).

    my ($h,$n)=($_[0],$_[1] // 0);

    #printf "Hash count $h (nkeys %d)\n",scalar keys %$h;
    
    # add up the keys in this hash
    $n += scalar keys %$h;
    
    # add up the keys in sub hashes
    #
    # I have two algorithms, one with map and one, which should
    # use less memory, using each. 'each' is faster... 

    # 40.51user 2.00system 1:46.68elapsed 39%CPU (0avgtext+0avgdata 1813200maxresident)k
    #map
    #{
	#printf "Check element $_ (is element hash ref? %d)\n",ref $$h{$_};
#	$n = hash_count($$h{$_},$n);
 #   }grep {ref($$h{$_}) eq 'HASH'} keys %$h;

    # 36.04user 2.19system 1:34.61elapsed 40%CPU (0avgtext+0avgdata 1812784maxresident)k
    while(my ($key, $value) = each (%$h))
    {
	$n = hash_count($value,$n) if (ref($value));
    } 

    return $n;
}


sub heximage
{
    # Usage: heximage($filename)
    # Returns the contents of $filename in hexadecimal

    return join('',unpack('h*',binslurp($_[0])));
}

sub in_numerical_order
{
    # Usage: sort in_numerical_order ( ... )
    # A crude numerical sorter. You would be better off using nsort from Sort::Key

    return($a<=>$b);
}


sub is_numeric
{
    # Usage: is_numeric( ... )
    # Returns 1 if ALL arguments are numeric, otherwise 0.
    return 0 if(!defined $_[0]);
    map
    {
	return 0 if(is_numeric_scalar($_)==0);
    }@_;
    return 1;
}

sub is_numeric_scalar
{
    # Usage: is_numeric_scalar($n)
    # Returns true if $n is a numeric scalar. Uses the Scalar::Util::looks_like_number function, but also converts a FORTRAN number to C-like floating point first.

    use Scalar::Util qw(looks_like_number);
    my $s=$_[0];
    $s=f_number_to_c_number($s) if(defined $s);
    return defined($s) ? Scalar::Util::looks_like_number($s) : 0;


### Rob's old code : mostly worked, but don't use it any more!

#    # test if $s is numeric: return 1 if it is
 #   return 0 if ((!defined($s))||
#		   ($s eq '')||
#		   ($s=~/[^0-9defDEF\.\+\-]/o))
#		   ; # 
#
 #   # if we start with a . prepend a zero
  #  if($s=~/^(-?)\./o)
   ## {
#	$s=$1.'0'.$s
 #   }
#
 #   return
  #  # x or +x or -x or x.y or +x.y or -x.y
   #    ($s=~/^[\+\-]?\d+\.?(?:\d+)?$/o)||
    #   # x.yez etc.
     ##  ($s=~/^[\+\-]?\d+\.?(?:\d+)?[eEdD][\+\-]?\d+$/o)
#	;
}

sub is_vnc_port
{
    # Usage: is_vnc_port($port)
    # Returns true if a port is an open VNC port 

    my ($port,$host)=($_[0],$_[1]//'127.0.0.1');
    # return 1 if it's an in-use VNC port 
    my @r=port_use($port,$host);
    return $r[0] ? $r[1]=~/open/ && $r[1]=~/vnc/i : undef;
}


sub join_arrays_as_hash
{
    # Usage: join_arrays_as_hash($array_pointer1,$array_pointer2)
    # Given two arrays (as pointers) join into a hash and return its reference

    my ($p1,$p2)=@_;
    my $n=1+MAX($#{$p1},$#{$p2});
    my %h;
    for(my $i=0;$i<$n;$i++)
    {
	$h{$$p1[$i]}=$$p2[$i];
    }
    return \%h;
}





sub listdirs
{
    # Usage: listdirs($paths...)
    # Returns an array of directories in $path
    my $dir = shift @_;
    $dir //= '.';
    return grep {-d $dir.'/'.$_} (listfiles($dir,@_));
}

sub listdirs_path
{
    # Usage: listdirs_path($path)
    # Returns an array of directories in $path with their full paths
    my $dir = $_[0] // '.';
    return grep {-d $_} map {$dir.'/'.$_} listfiles($dir);
}

sub listfiles
{
    # Usage: listfiles($dir,$failmode)
    # Returns a list of files in $dir. If there is an error and $failmode is 'confess' then an error is reported through Carp::Confess (and your code stops), otherwise nothing happens.

    # like UNIX ls, lists files in a directory, returns then in an array
    
    # arg 0 is the directory name
    # arg 1 is the failure mode: if 'confess' then stop (and confess)
    # on failure, otherwise return empty, as when 

    # NB ignores . and ..
    if(-d $_[0])
    {
	if(opendir (my $fp,$_[0]))
	{

	    my @x= grep {$_ ne '.' && $_ ne '..'} (readdir $fp); 
	    #print "LISTFILES $_[0] gives @x\n";
	    return @x;
	    # nb $fp closes when out of scope
	}
	else
	{
	    if(defined $_[1] && $_[1] eq 'confess')
	    {
		confess("Cannot open $_[0] for listing : $!");
	    }
	    else
	    {
		return(); # return empty list
	    }
	}
    }
    else
    {
	if(defined $_[1] && $_[1] eq 'confess')
	{
	    confess("Cannot open $_[0] for listing : is not a directory or does not exist\n");
	}
	else
	{
	    return(); # return empty list
	}
    }
}


sub listfiles_path
{
    # Usage: listfiles_path($path)
    # Returns an array of the contents of $path with full paths

    my @l=listfiles(@_);
    map{$_=$_[0].'/'.$_;}@l;
    return(@l);
}

sub listdirs_with_no_subdirs
{
    # Usage: listdirs_with_no_subdirs($path)
    # Returns a list of directories that contain only files, recursively checking subdirectories
    
    my @subdirs=listdirs_path($_[0]);
    #print "SUBDIRS of $_[0] are @subdirs\n";

    my @x1=grep {-d $_ && nsubdirs($_)==0} @subdirs;

    #print "SUBDIRS with no subdirs are @x1\n";

    my @x2=map{ listdirs_with_no_subdirs($_) } grep {-d $_ && nsubdirs($_)>0} @subdirs;

    #print "SUBDIRS with subdirs are @x2\n";

    #print "RETURN @x1,@x2\n";

    return (@x1,@x2);

} 

sub list_modules
{
    # Usage: list_modules();
    # Returns a list of loaded modules

    my @x;
    no warnings;
    map
    { 
	if(my $m=($_=~/(.*)\.pm$/)[0])
	{
	    my $v;
	    if(defined($m))
	    {
		$v = versionofmodule($m);
		$v = defined($v) ? 'version '.$v : 'unknown version';
	    }
	    else
	    {
		$v='unknown version';
	    }
	    s/\//\:\:/go;
	    push @x, (sprintf '%s : % 80s : %s',$_,$INC{$_},$v);
	}
    }sort keys %INC;
    use warnings;
    return @x;
}


sub log10
{
    # Usage: log10($x)
    # Returns log to base 10 of $x.

    # log10 function with number checks : NB use POSIX' log10
    # function, not my own as suggested by perl, POSIX is faster
    confess("cannot take log of '$_[0]' (zero, nan or inf?)\n")
	if($_[0]==0.0 || $_[0]=~/(?:nan|inf)/oi);
    return POSIX::log10($_[0]);
}

sub MAX
{
    # Usage: MAX($x,$y)
    # Returns the maximum of $x and $y, first checking if the values are numeric.
    if(!is_numeric($_[0]))
    {
	return $_[1];
    }
    elsif(!is_numeric($_[1]))
    {
	return $_[0];
    }
    else
    {  
	return $_[0] > $_[1] ? $_[0] : $_[1];
    }
}

sub max_from_array
{
    # Usage: max_from_array(@array)
    # Returns the maximum value in @array (using List::Util::max)

    use List::Util qw/max/;
    return scalar(max(@_));
}

sub mean_from_array
{
    # Usage: mean_from_array(@array)
    # Returns the numerical mean of the @array using List::Util::sum

    return undef if ($#_==-1);
    use List::Util qw/sum/;
    return (1.0*scalar(sum(@_)))/(1.0*$#_+1.0);
}

sub mem_usage
{
    # Usage: mem_usage($procps,$pid)
    # Tries to find the memory usage of process $pid (or the current process if no $pid is given). If $procps is true and scanning of /proc/ fails, tries to use the ps command.
    #
    # beware : this is (for some reason) not thread safe
    #
    
    my ($procps,$pid,$children)=($_[0],$_[1]//$$,$_[2]);
    
    # crude way to determine memory usage in perl
    # NB procps should be 1 on a linux system, but this is
    # compatible with Macs (Darwin) as well

    # try to do it through the proc file rather than the ps command
    my $procfile='/proc/'.$pid.'/status';
    if(-f $procfile &&
       #open(MEM_USAGE,'<'.$procfile)
       sysopen(MEM_USAGE,$procfile,O_NONBLOCK|O_RDONLY)
        )
    {
        my $m=0;
	while(<MEM_USAGE>)
	{
	    if(/^vmrss:\s+(\d+) kB/oi)
	    {
		$m += $1;
                #print "Parent $m\n";
                last;
            }
	}
	close MEM_USAGE;

        if($children)
        {
            # we require all processes which have the process $pid
            # as a parent
            opendir(my $procdir, '/proc/');
            foreach my $dir (readdir($procdir))
            {
                next if($dir eq '.' || $dir eq '..' || $dir!~/^\d+$/o);
                next if(!-d '/proc/'.$dir);
                next if($dir == $pid);

                #open(my $status,'<','/proc/'.$dir.'/status');
                if(sysopen(my $status,'/proc/'.$dir.'/status',O_NONBLOCK|O_RDONLY))
                {
                    my $this_m=0; # this process memory use
                    my $this_ppid=0; # parent process id
                    while(readline($status))
                    {
                        if(/^vmrss:\s+(\d+) kB/oi)
                        {
                            $this_m = $1;
                        }
                        elsif(/^ppid\:\s+(\d+)/oi)
                        {
                            $this_ppid = $1;
                        }
                        if($this_m && $this_ppid)
                        {
                            last;
                        }
                    }
                    close $status;
                    
                    if($this_ppid == $pid)
                    {
                        $m += $this_m;
                        #print "Child (pid $dir, parent $this_ppid) memory is $this_m (total $m)\n";
                    }
                }
            }
            close $procdir;
        }

        return sprintf('%2.2f',$m*0.0009765625); # return Mbytes
    }

    # otherwise, try using the ps command
    if($procps==1)
    {
	my @s=`ps v -p $pid`;
	my $head=shift @s;
	if(defined($head))
	{
	    my $data=shift @s;
	    if(defined($data))
	    {
		$head=~s/^\s+//o;
		my @head=split(/\s+/o,$head);
		for(my $i=0;$i<=$#head;$i++)
		{
		    if($head[$i]=~/rss/io)
		    {
			#print "MATCH RSS col $i hence $data[$i]\n";
			$data=~s/^\s+//o;
                        no warnings;
			return sprintf('%2.2f',(split(/\s+/o,$data,$i+1))[$i]/1024.0); # return Mbytes
                        use warnings;
		    }
		}
	    }
	}
    }
    return '?'; # what to do?
}

sub MIN
{
    # Usage: MIN($x,$y)
    # Returns the minimum of $x and $y, first checking if the values are numeric.

    if(!is_numeric($_[0]))
    {
	return($_[1]);
    }
    elsif(!is_numeric($_[1]))
    {
	return($_[0]);
    }
    else
    {
	return $_[0] < $_[1] ? $_[0] : $_[1];
    }
}

sub min_from_array
{
    # Usage: min_from_array(@array)
    # Returns the minimum value in @array (using List::Util::max)
    
    use List::Util qw/min/;
    return scalar(min(@_));
}

sub mkdirhier
{
    # Usage: mkdirhier($dir)
    # Makes the directory $dir and parents if required (cf. Linux commands 'mkdir -p' or 'mkdirhier')

    # mkdir on $dir and parents (like mkdirhier or mkdir -p)
    my $err;
    my $dir = $_[0];
    make_path($dir,{error=>$err});
    if (defined $err && @$err) {
	print "mkdirhier called make_path on '$dir' and found an error\n";
	for my $diag (@$err) {
	    my ($file, $message) = %$diag;
	    if ($file eq '') {
		print "general error: $message\n";
	    }
	    else {
		print "problem making $file: $message\n";
	    }
	}
    }
}

sub mkdirhier_legacy
{
    # Usage: mkdirhier_legacy($dir)
    # Makes the directory $dir and parents if required (cf. Linux commands 'mkdir -p' or 'mkdirhier'). This is the inferior, legacy version, which may fail.

    # mkdir on $dir and parents (like mkdirhier or mkdir -p)
    my $dir=$_[0];
    $dir=~s/\/+/\//g;
    my $root='';
    while($dir=~s/^(\/*[^\/]+)//)
    {
	my $d=$root.$1;
	$root.=$1;
	next if(-d $d);
	mkdir $d;
    }
}

sub ncpus
{
    # Usage: ncpus()
    # Returns the number of CPUs in your machine using Sys::Info.

    # count cpus using Sys::Info module
    my $info = Sys::Info->new;
    $info->device('CPU')->count;
}

sub ncpus_old
{
    # Usage: ncpus()
    # Returns the number of CPUs in your machine using /proc/cpuinfo, i.e. this will nto work on machines without /proc/ (windows, mac?)

    open(CPU_COUNT,'</proc/cpuinfo')||
	confess('cannot open /proc/cpuinfo in ncpus() in rob_misc');
    my $n=0;
    while(<CPU_COUNT>)
    {
	$n++ if(/processor/o);
    }
    close CPU_COUNT;
    return $n;
}

sub nfscheck
{
    # Usage: nfscheck($path)
    # Return 1 if $path is on an NFS mounted partition

    `stat -f -L -c %T "$_[0]"`=~/^nfs/io ? 1 : 0;
}

sub nonblockGetLines {
    # Usage: nonblockGetlines($filehandle,$timeout)
    # A non-blocking filehandle read that returns an array of lines read. Returns:  ($eof,@lines)

    CORE::state %nonblockGetLines_last;
    my ($fh,$timeout) = @_;

    $timeout = 0 unless defined $timeout;
    my $rfd = '';
    $nonblockGetLines_last{$fh} = ''
        unless defined $nonblockGetLines_last{$fh};

    my $fno = fileno($fh);
    return if($fno==-1);
    vec($rfd,$fno,1) = 1;
    return unless select($rfd, undef, undef, $timeout)>=0;
    # I'm not sure the following is necessary?
    return unless fileno($fh) && vec($rfd,$fno,1);
    my $buf = '';
    my $n = sysread($fh,$buf,1024*1024);
    # If we're done, make sure to send the last unfinished line
    return (1,$nonblockGetLines_last{$fh}) unless $n;
    # Prepend the last unfinished line
    $buf = $nonblockGetLines_last{$fh}.$buf;
    # And save any newly unfinished lines
    $nonblockGetLines_last{$fh} =
        (substr($buf,-1) !~ /[\r\n]/ && $buf =~ s/([^\r\n]*)$//)
	? $1 : '';
    $buf ? (0,split(/\n/,$buf)) : (0);
}


sub non_zero_array
{
    # Usage: non_zero_array(@array)
    # Return 1 if the array has a non-zero element

    map
    {
	return 1 if(defined $_ && $_);
    }@_;
    return 0;
}

sub non_zero_array_from_pointer
{
    # Usage: non_zero_array($array_reference)
    # Return 1 if the array has a non-zero element

    map
    {
	return 1 if(defined $_ && $_);
    }(@{$_[0]});
    return 0;
}

sub nsubdirs
{
    # Usage: nsubdirs($paths...)
    # Returns the number of subdirs in the given paths (ignores hidden directories, i.e. those that start with a dot)

    my @subdirs = grep {!/^\./} listdirs(@_);
    return $#subdirs+1;
}

sub numeric_clamp
{
    # Usage: numeric_clamp($x,$min,$max)
    # clamp $x to given range $min,$max using (unsafe but faster) numeric_* functions

    return numeric_MAX($_[1],numeric_MIN($_[2],$_[0]));
}

sub numerical_list_sorter($$)
{
    # Usage: numerical_list_sorter(...)
    # numerical list sorter: almost an nsort but splits first

    if($_[0] eq '')
    {
	return -1;
    }
    elsif($_[1] eq '')
    {
	return 1;
    }
    else
    {
	my @s1=split(/[\s,]+/o,$_[0]);
	my @s2=split(/[\s,]+/o,$_[1]);
	my $n1=$#s1+1;
	for(my $i=0;$i<$n1;$i++)
	{
	    if($s1[$i] ne $s2[$i])
	    {
		no warnings;
		return scalar (($s1[$i] <=> $s2[$i]) || ($s1[$i] cmp $s2[$i])) 
		    if($s1[$i] ne $s2[$i]);
		use warnings;

	    }
	}
	# just in case
	return $_[0] cmp $_[1];
    }
}


sub numericMAX
{
    # Usage: numericMAX($x,$y)
    # Same as MAX but we are sure they exist and are numbers i.e. rather unsafe, but faster

    return $_[0] > $_[1] ? $_[0] : $_[1];
}


sub numericMIN
{
    # Usage: numericMIN($x,$y)
    # Same as MIN but we are sure they exist and are numbers i.e. rather unsafe, but faster

    return $_[0] < $_[1] ? $_[0] : $_[1];
}

sub ok_file
{
    # Usage: ok_file($filename)
    # Check if a file exists at $filename and is of significant size (>100 bytes)
    
    no warnings;
    return 1*((-f $_[0])&&(-s $_[0] > 100));
    use warnings;
}

sub ok_directory
{
    # Usage: ok_directory($dirname)
    # Check if a directory exists at $dirname

    my $d=$_[0];
    no warnings;
    return 1*((defined($d)&&($d ne '')&&(-d $d)&&(-r $d)));
    use warnings;
}

sub operating_system
{
    # Usage: operating_system()
    # Returns either "windows" or "unix" depending on the type of operating system (defaults to unix if windows is not detected)

    return ((defined($ENV{'OS'})&&($ENV{'OS'}=~/windows/io))||
	    (defined($ENV{'WINDIR'})&&($ENV{'WINDIR'}=~/windows/io)))?
	    'windows' : 'unix';
}

sub parse_cmdline_args_to_perlvars
{
    # Usage: parse_cmdline_args_to_perlvars($h)
    # Parse command line args, setting perl variables as required. e.g. if the arg is x=y then Perl evaluates $x="$y"; If $h is defined, instead sets $h{$x}=$y.

    if(defined $_[0])
    {
	my $h=$_[0];
	# set hash ref
 	foreach my $arg (@ARGV)
	{
	    if($arg=~/-?-?(\S+)=(.*)/)
	    {
		$$h{$1}=$2;
	    }
	    elsif(defined $$h{$arg})
	    {
		$$h{$arg}=1;
	    }
	}
    }
    else
    {
	foreach my $arg (@ARGV)
	{
	    if($arg=~/(\S+)=(.*)/)
	    {
		eval "\$main::$1 = \"$2\"";
	    }
	}
    }
}

sub pmrand
{
    # Usage: pmrand()
    # Returns a random floating-point number between -1.0 and +1.0

    return(rand()*2.0-1.0);
}

sub poisson_error
{
    # Usage: poisson_error($p,$n)
    # Given a 'result' $p and the number of measurements which went into it $n, calculate the Poisson error

    my ($p,$n)=@_;
    return($p/MAX(1.0,sqrt($n)));
}

sub port_use
{
    # Usage: port_use($port,$host,$regexp)
    # Examine port $port (on $host or, if undef, 127.0.0.1) and return true if it is in use, otherwise undef (false). if $regexp is given as the third arg, this is forced to match the port description, otherwise undef is returned.

    my ($port,$host,$regexp)=($_[0],$_[1]//'127.0.0.1',$_[2]);
    my $r=`nmap -p $port $host 2>&1`;
    return !defined($r)?undef:(!($r=~/closed/) && ((!defined $regexp)||($r=~/$regexp/)),$r);
}

sub proc_mem_usage
{
    # Usage: proc_mem_usage($pid)
    # Uses Proc::ProcessTable to find the memory use of process given by ID $pid or, if no argument is given, the current process. Returns in Mbytes.

    # rewrite to use Proc::ProcessTable to find memory use
    use Proc::ProcessTable;
    my $t = new Proc::ProcessTable;
    return undef if (!defined($t));
    my $pid=$_[0] // $$;
    foreach my $got ( @{$t->table} ) 
    {
	next if((!defined($got)) || (not $got->pid eq $$));
	return(sprintf('%2.2f',$got->rss*0.0000009765625)); # return Mbytes
    }   
    return undef;
}

sub ps_to_png
{
    # Usage: ps_to_png($psfile,$pngfile,$dpi,$extra,$regexp)
    # Converts postscript file $psfile to PNG file $pngfile using ghostscript at a resolution given by $dpi. $extra contains extra arguments for the ghostscript command (is ignored if undefined). $regexp is applied to the command line string (or ignored if undefined).

    my ($psfile,$pngfile,$dpi,$extra,$regexp)=@_;
    $extra=' '  if(!defined($extra));
    my $cmd="gs -q -dBATCH -dSAFER -dNOPAUSE -sDEVICE=png16m -dEPSCrop -dGraphicsAlphaBits=1 -dTextAlphaBits=4 -r$dpi $extra -sOutputFile=$pngfile $psfile";
    $cmd=~s/$regexp// if(defined $regexp);
    `$cmd`;
}

sub random_string
{
    # Usage: random_string($len)
    # Returns a random string of $len characters

    my $len=$_[0];
    my $r;
    my @chars=('a'..'z','A'..'Z','0'..'9');
    foreach (1..$len)
    {
	$r.=$chars[rand @chars];
    }
    return $r;
}

no warnings;
sub remove_ANSI
{
    # Usage: remove_ANSI($string)
    # Removes ANSI codes from $string

    my $x=$_[0];
    $x=~s/\e\[\d+(?>(;\d+)*)m//g; # remove ANSI codes
    $x=~s/\e\[[Km]//g;

    return $x;
}

sub remove_all_escape_codes
{    
    # Usage: remove_all_escape_codes($string)
    # Removes all escape codes from $string

    my $x=$_[0];
    $x=~s/\x1b[[()=][;?0-9]*[0-9A-Za-z]?//g;s/\r//g;
    $x=~s/\007//g;
    return $x;
}
use warnings;

sub remove_pango_markup
{
    # Usage: remove_pango_markup($string)
    # Removes pango <span> markup from given $string

    my $s=$_[0];
    $s=~s/<\\?span[^>]*>//o;
    return($s);
}


sub renice_me
{
    # Usage: renice_me($prio)
    # Renices the current process with priority $prio (or 10 if $prio not defined), and sets the ionice (disk I/O) priority to minimum.

    my $nice = $_[0] // 10; # default to nice +10
    `ionice -c3 -p $$ ; renice -n +$nice -p $$; chrt -i -p 0 $$`;
}

sub runcmd
{
    # Usage: runcmd($cmd,$where)
    # Runs the command given by $cmd and sends output to either the screen or an array to be returned or does nothing with it. $where can be: 0 or undef or 'none': output is lost, 1 or 'screen' : output is printed to the screen, 2 or 'array' : output is returned as an array, 3 or 'screen and array' or 'both' : 1 and 2

    my ($cmd,$where)=($_[0],$_[1] // 1);
 
    if($where eq 'screen')
    {
	$where=1;
    }
    elsif($where eq 'array')
    {
	$where=2;
    }
    elsif($where eq 'screen and array' || $where eq 'both')
    {
	$where=3;
    }
    elsif($where eq 'none')
    {
	$where=0;
    }

    my $ret = ($where==2 || $where==3);
    my $screen = ($where==1 || $where==3);
    $|=1;
    my $Fwas = $^F; 
    $^F = 1;
    print 'Run ',color('red'),$cmd,color('reset'),"\n" if($screen);
    my @out;
    {
        open(STDERR , '>&', \*STDOUT) or die("cannot redirect STDERR to STDOUT");
        open(my $f,$cmd.' |')||confess('cannot run '.$cmd);
        autoflush $f 1;
        while(<$f>)
        {
            print $_ if($screen);
            push(@out,$_) if($ret);
        }
        # don't die here : if the command
        # fails or the pipe closes, then trying to close()
        # will also fail. You probably don't want that, you
        # just want to return.
        close $f; 
    }
    $^F = $Fwas;
    return $ret ? @out : undef;
}

sub safeslurp
{
    # Usage: safeslurp(<filename>)
    # Returns the contents of file given by <filename> and on failure returns undef

    open(my $fh,'<'.$_[0])||return undef;
    return (do { local( $/ ) ; <$fh> } );
}

sub scalars_the_same
{
    # Usage: scalars_the_same($x,$y)
    # Checks if $x and $y are numerically the same, returns the fractional error, or 0.0 if they are the same, or 1.0 if one is numeric and the other not.

    my ($x,$y)=@_;
    my $err;
    if((!defined($x)) || (!defined($y)))
    {
	$err=1.0;
    }
    elsif($x eq $y)
    {
	$err=0.0
    }
    else
    { 
 	$err = (is_numeric($x) && is_numeric($y)) ?
	
	    (($x==$y) ? 
	     # x and y are zero: no error
	     0.0 :
	     
	     # if x==0 and y==0 cannot get here (caught above)
	     # so either x==0 (y!=0) and we divide by $y, 
	     # or x!=0 (y==0) and we divide by $x
	     abs(($x-$y)/ ($x==0.0 ? $y : $x))) :
	     
	    # otherwise ... what to do ?
 	    1.0;
    }
    return $err;
}

sub session_id
{
    # Usage: session_id($length)
    # Make a "unique" session id string, of length $length or 16 characters if $length is not given.

    my $sessionId  ='';
    my $length=$_[0] // 16; # default to 16 characters
    for(my $i=0;$i<$length;)
    {
	my $j = chr(int(rand(127)));	
	if($j =~ /[a-zA-Z0-9]/o)
	{
	    $sessionId .= $j;
	    $i++;
	}
    }
    return ($sessionId);
}

sub slurp
{
    # Usage: slurp(<filename>)
    # Returns the contents of file given by <filename> as a string, on failure dies with Carp::confess

    open(my $fh,'<'.$_[0])||confess("cannot open $_[0] in slurp");
    return (do { local( $/ ) ; <$fh> } );
}

sub slurp_filehandle
{
    # Usage: slurp_filehandle($fh)
    # Returns the contents of the file conntected to $fh

    return (do { local( $/ ) ; <$_[0]> } );
}

sub slurp_to_array_ref
{
    # Usage: slurp(<filename>)
    # Returns the contents of file given by <filename> in an array reference, on failure dies with Carp::confess

    my @x;
    open(my $fh,'<'.$_[0])||confess("cannot open $_[0] in slurp");
    do { local( $/ ) ; push(@x,split(/\n/o,<$fh>)) };
    return \@x;
}

sub safeslurp_to_array_ref
{
    # Usage: slurp(<filename>)
    # Returns the contents of file given by <filename> in an array reference, on failure dies with Carp::confess
    # safe version : Returns empty array ref on failure
    my @x;
    open(my $fh,'<'.$_[0])||return [];
    do { local( $/ ) ; push(@x,split(/\n/o,<$fh>)) };
    return \@x;
}

sub splitcomma
{
    # Usage: splitcomma($string)
    # Splits $string on outermost commas

    my $x=$_[0];
    my @x=split(//o,$x);
    my @l;
    my $bcount;
    my $s;
    no warnings;
    foreach my $i (0..$#x)
    {
	if($x[$i] eq '(')
	{
	    $bcount++;
	    $s.=$x[$i];
	}
	elsif($x[$i] eq ')')
	{
	    $bcount--;
	    $s.=$x[$i];
	}
	elsif(($x[$i] eq ',')&&($bcount==0))
	{
	    push(@l,$s);
	    $s='';
	}
	else
	{
	    $s.=$x[$i];
	}
    }
    return (@l,$s);
    use warnings;
}

sub ssleep
{
    # Usage: ssleep($nsecs)
    # Sleep function : stop doing anything for $nsecs seconds. NB this uses the "select" function call which works on both windows (also with fork) and Linux.

    select(undef,undef,undef,$_[0]);
}


sub sum_array
{
    # Usage: sum_array(@array)
    # sum up @array using List::Util::sum and return the result
    use List::Util qw/sum/;
    return scalar(sum(@_));
}

sub sum_array_with_pointers
{
    # Usage: sum_array($array)
    # sum up $array (a reference to an array) using List::Util::sum and return the result
    use List::Util qw/sum/;
    return scalar(sum(@{$_[0]}));
}

sub thetime
{
    # Usage: thetime($offset)
    # Output the time either now (if no $offset is given), or $offset seconds later, as a human-readable string.

    my $t=time();
    $t+=$_[0] if(defined($_[0]));
    my @lt=localtime($t); # get localtime array and return string
    return(sprintf'%02d:%02d:%02d',$lt[2],$lt[1],$lt[0]);
}


sub thread_log
{
    # Usage: thread_log(@array)
    # Print to the screen using thread_string on the @array

    print thread_string(@_);
}

sub thread_string
{
    # Usage: thread_string($l,$n)
    # Writes $l to line $n of the terminal.

    # for thread number $n output a log line $l and move back to the origin
    # using the ANSI up character
    my ($l,$n)=@_;

    # add newline if there isn't one already
    chomp $l;
    $l.="\n";

    if(defined($n))
    {
	my $up_char="\e[1A";
	my $post_string=$up_char.($up_char x $n);
	my $pre_string=' '.("\n " x $n);
    	$l=$pre_string.$l.$post_string;
    }
    return $l; 
}


sub toggle_state
{
    # Usage: toggle_state($state)
    # toggles integer $state, i.e. sets 0 to 1, and vice versa

    no warnings;
    return 1*($_[0]==0);
    use warnings;
}

sub touch
{
    # Usage: touch($filename)
    # Equivalent to UNIX 'touch' command on $filename: opens the file and sends no data to it. Does not destroy an existing file.

    open(my $fh,">>$_[0]")||confess("cannot touch $_[0]");
    close $fh;
}

sub trem
{
    # Usage: trem($tstart,$count,$n)
    # Estimate time remaining (seconds) given a starting time and a count (i.e. progress = $count/$n). $tstart is the start time, $count is the current progress count, and $n is the total number required.

    my ($tstart,$count,$n)=@_;
    my $etasecs;
    my $tpr=((time()-$tstart)/MAX(1,$count)); # time per run (seconds)
    $etasecs=$tpr*($n-$count);
    my ($eta,$units)=conv_time_units($etasecs);
    return($eta,$units,$tpr,$etasecs);
}


sub trem2
{
    # Usage: trem2($dt,$count,$dn,$n)
    # Estimate time remaining (seconds) given a differential time and count (i.e. progress = $count/$n). $dt is the time since the last call, $count is the current progress count, $dn is the number run since the last call, and $n is the total number required.

    # estimate time remaining  (Version 2)
    my ($dt,$count,$dn,$n)=@_; 
    # dt= time since previously called
    # count = number run in total so far
    # dn = number run recently
    # n = total grid number 
    my $tpr=$dt/MAX(1,$dn); # time per run (seconds)
    my $etasecs=$tpr*($n-$count);
    my ($eta,$units)=conv_time_units($etasecs);
    return($eta,$units,$tpr,$etasecs);
}

sub username
{ 
    # Usage: username()
    # Return user name of person running the script

    return getlogin || (getpwuid($<))[0];
}

sub versionofmodule
{
    # Usage: versionofmodule($module)
    # Return version of a Perl module given by $module 

    return(eval '$'.$_[0].'::VERSION');
}

sub ynstring
{
    # Usage: ynstring($x)
    # Return 'Y' if $x is true, 'N' otherwise

    return $_[0]?'Y':'N';
}

sub string_to_array
{
    # given a string, convert to a 2d array of elements
    return [map {[split(/\s+/,$_)]}  split(/\n/,$_[0])];
}

sub array_to_string
{
    # given a 2D array, convert to a string
    my $string;
    return join("\n",
                map
                {
                    join(' ',@$_);
                }@{$_[0]}
        );
}

sub mapsplit
{
    # given an array ref in $data, split each line into sub arrays
    # after removing excessive whitespace
    my $data = $_[0];
    map
    {
        s/^\s+//; s/\s+$//;
	$_ = [split(/\s+/,$_)];
    }@$data;
    return $data;
}



sub compactnumber
{
    # return most compact form of a number

    my $original = $_[0]; # original number

    # return if not a number
    return $original if(!is_numeric($original));

    # copy into the return value
    my $r = $original; 

    my $vb=$_[1]//0;
    my $n=0;

    foreach my $format ('%g','%e')
    {
        my $y = sprintf $format, $original;

        # clean

        # reformat number before exponential 
        $y=~s/(\d\.\d+)+e/sprintf'%ge',$1/e;
        print "$n : $y\n"if($vb);$n++;
        
        # remove . before the exponential without digits
        $y=~s/\.e/e/;
        print "$n : $y\n"if($vb);$n++;
        
        # remove 0 exponential (just 1)
        $y=~s/e[\+-]0+$//;  
        print "$n : $y\n"if($vb);$n++;

        # remove 0s after the exponential
        $y=~s/(e[\+-])0+(\d+)$/$1$2/; 
        print "$n : $y\n"if($vb);$n++;

        # remove + before an exponential
        $y=~s/e\+/e/g;
        print "$n : $y\n"if($vb);$n++;

        #print "test $y len ",length $y," vs ",length $original;
        $r = $y if(length $y < length $r);
        print "$n : $y\n"if($vb);$n++;
    }
    return $r;
}

sub colourhash
{
    # return hash of commonly used colours
    my %colour;
    foreach my $c ('red','green','blue','cyan',
                   'magenta','reset','yellow',
                   'red bold','blue on_white',
                   'red on_white','blue bold',
                   'yellow bold on_black',
                   'blue on_white','bold red on_black')
    {
        $colour{$c} = color($c);
    }
    return %colour;
}

sub gnuplot_string
{
    # convert strings to Symbols
    my $s = $_[0];
    $s=~s!\b([Aa])lpha\b!{/Symbol $1}!g;
    $s=~s!\b([Bb])eta\b!{/Symbol $1}!g;
    $s=~s!\b([Gg])amma\b!{/Symbol $1}!g;
    $s=~s!\b([Dd])elta\b!{/Symbol $1}!g;
    $s=~s!\b([Ee])psilon\b!{/Symbol $1}!g;
    $s=~s!\b([Kk])appa\b!{/Symbol $1}!g;
    $s=~s!\b([Ll])ambda\b!{/Symbol $1}!g;
    return $s;
}

sub mindiff
{
    # given a references to a sorted array of numbers, find the minimum
    # difference between two elements
    my $array = shift; 
    my $mindiff = 1e200;
    for(my $i=1;$i<=$#{$array};$i++)
    {
        $mindiff = MIN(abs($array->[$i] - $array->[$i-1]),$mindiff);
    }
    return $mindiff;
}


sub hashcount
{
    ############################################################
    # count the number of elements in a hash
    #
    # Recurses in the hash, adding up arrays and subhashes
    ############################################################ 
    
    my $s = 0;
    my ($hash) = @_;
    __hashcount2($hash,\$s);
    return $s;

    sub __hashcount2
    {
        my ($hash,$s) = @_;
        $s //= 0;
        foreach my $k (keys %$hash)
        {
            my $r = ref $hash->{$k};
            if($r eq 'HASH')
            {
                # hash of hashes : recursive call
                $$s++;
                __hashcount2($hash->{$k},$s);
            }
            elsif($r eq 'ARRAY')
            {
                # array : just use the count
                $$s += scalar @{$hash->{$k}};
            }
            else
            {
                $$s++;
            }
        }
    }
}


sub checklib
{
    # simple library checking function :
    # on error return '' (empty string)
    # on success returns the library path (or a list of paths)
 
    # get the library name
    my $lib = shift;
    my $striplib = ($lib=~/lib(.*)/)[0];
    my $path;
    my $os = operating_system();
    my $vb = 0;
    my $cc = $ENV{CC} // 'gcc';
        
    print "  Check for library $striplib ...\n"if($vb);
    
    # check we have ld : doesn't have to be GNU ld
    # but should return a version number
    if(`ld -v -o /dev/null`=~/\d\.\d/)
    {
        # check if the library is found with ld
        # because this works on multiple architectures
        if(`ld -l$striplib -o /dev/null 2>\&1`!~/(cannot find|not found for) -l$striplib/)
        {
            # find location using ld, return if found.
            # NB will not work on OSX because "--verbose" is not supported
            if($os ne 'Darwin')
            {
                $path = (`ld --verbose -l$striplib -o /dev/null 2>\&1`
                         =~
                         /attempt to open (\S+lib$striplib.so) succeeded/)[0];
            }


            # find location using $cc, gcc or cc, return if found
            if(! $path)
            {
                foreach my $c ($cc,'gcc','cc')
                {
                    my $cmd = "$c -l$striplib -Xlinker --verbose -o /dev/null 2>\&1";
                    print "  try $cmd\n"if($vb);
                    $path = (``
                             =~
                             /attempt to open (\S+lib$striplib.so) succeeded/)[0];
                
                    if($vb && defined $path && $path)
                    {
                        print "  Found $path (1)\n";
                    }
                }
            }
        }
    }
    

    if(!$path && `ldconfig --version 2>\&1`=~/\d\.\d/)
    {
        # ld failed, try ldconfig
        my $r = `ldconfig -p | grep $lib`;
        remws($r);
        $path = $r if($r ne '');

        if($vb && defined $path && $path)
        {
            print "  Found $path (2)\n";
        }
    }
    
    if(!$path)
    {
        # check in LIBRARY_PATH and LD_LIBRARY_PATH
        my @libs;
        foreach my $dir (grep {defined $_ && -d $_} (
            $ENV{LIBRARY_PATH},
            $ENV{LD_LIBRARY_PATH}
            ))
        {
            my $sofile = $dir.'/'.$lib.'.so';
            push(@libs, $sofile) if(-f $sofile);
        }
        
        $path = scalar(@libs)==0 ? '' : join("\n", @libs);
        if($vb && defined $path && $path)
        {
            print "  Found $path (3)\n";
        }

    }

    $path = '' if(!$path);
    
    print "Found $striplib at $path\n" if($vb);
    
    return $path;
}



sub test_for_header_file
{ 
    # use the compiler to test whether we have a header file
    my $cc = shift;
    my $hfile = shift;
    my $flags = shift // '';
    my $cmd = "echo  \"\#include <$hfile> ; int test\{ return 0; \}\" | $cc $flags -E -c - 2>\&1";
    
    my $r = (`$cmd`=~/No such file/) ? 0 : 1;
    #print "Check for header file $hfile with command $cmd : ".($flags ? "(flags $flags) " : '')." : ".($r ? '' : 'not ')."found\n";
    return $r;
}

sub remws
{ 
    # remove leading and ending whitespace
    # in string passed in: work on the string itself,
    # not a copy
    $_[0] =~ s/^\s+//;
    $_[0] =~ s/\s+$//;
    # and return it
    return $_[0];
}

sub reverse_lookup_hash
{
    my ($array) = @_;
    # given an array $array make a reverse lookup hash
    # and return a reference to it
    my $hash={};
    for(my $i=0;$i<=$#{$array};$i++)
    {
        $hash->{$array->[$i]} = $i;
    }
    return $hash;
}

sub float_to_latex
{
    my ($l) = @_;
    
    # change * to \times
    $l=~s/\*/ \\times /g;

    # change +- or -+ to -
    $l=~s/\+\-/\-/g;
    $l=~s/\-\+/\-/g;

    # change ++ to +
    while($l=~/\+\+/)
    {
        $l=~s/\+\+/\+/g;
    }
    
    # convert exponentials
    $l=~s/(\d(?:\.\d+)?)0*[edED]\+?0*(\d+)/$1\\times 10\^\{$2\} /g;
    $l=~s/(\d(?:\.\d+)?)0*[edED]\-0*(\d+)/$1\\times 10\^\{\-$2\} /g;
    $l=~s/^\s*1\s*\\times//;
    #$l=~s/(\d)[edED]\+?0*(\d+)/$1\\times 10\^\{$2\} /g;
    #$l=~s/(\d)0*[edED]\-0*(\d+)/$1\\times 10\^\{\-$2\} /g;
    $l=~s/\\times 10\^\{0\}//g;
    $l=~s/(\d)\.(\d)(\d+)\\times 10\^\{1\}/$1$2.$3/g;
    $l=~s/(\d)\.(\d)(\d+)\\times 10\^\{-1\}/0.$1$2$3/g;
    
    return $l;
}

############################################################

### Generate documentation for the following with the following command
### from within the modules directory
#
# grep -A3 ^sub ./misc/rob_misc/lib/rob_misc.pm  | sed s/^sub/Subroutine\:/ | sed s/^\{// | sed s/\#//| sed s/--//
#
#


1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

rob_misc - Robert Izzard's often-used (probably very dodgy and poorly coded) functions.

=head1 SYNOPSIS

To use all the functions:

  use rob_misc;

But you probably want to 
    
  use rob_misc qw/func1 func2/;

=head1 VERSION

Version 0.12

=head1 DESCRIPTION

Brief descriptions of the various routines are given below, and are subject to change!

Subroutine: be_sudo

     Usage: be_sudo(@<command line arg list>)
     Run command list passed in as arguments as sudo

Subroutine: bin_data

     Usage: bin_data($x,$dx)
     bin value $x to nearest bin of width $dx

Subroutine: binslurp

     Usage: binslurp(<filename>)
     Returns the contents of file given by <filename> using Perl's binmode

Subroutine: bytes_pending

     Usage: bytes_pending($handle)
     Returns the number of pending bytes on the pipe given by $handle

Subroutine: certain_kill

     Usage: certain_kill($pid)
     Kills the process given by $pid by sending a signal 9, then a 15 to make sure

Subroutine: check_lines_not_equal

     Usage: check_lines_not_equal($l1,$l2,$threshold)
     Checks if two lines of numbers (split by spaces or colons), $l1 and $l2, are numerically equals to within error $threshold, returns two list elements: 0 = the maximum difference between any of the two numbers, 1 = an error string (or undef if there is no error).

Subroutine: clamp

     Usage: clamp($x,$low,$high)
     Clamps $x to given range $low to $high

Subroutine: coloured_diff_strings

     Usage: coloured_diff_strings($x,$y)
     Returns a string which highlights the difference between $x and $y in colour

Subroutine: conv_time_units

     Usage: conv_time_units($t)
     Converts $t (in seconds) to seconds, minutes or hours depending on its magnitude

Subroutine: convert_isotopes
 
     Usage: convert_isotopes($i)
     Convert isotope strings from AB123 to ^{123}AB for gnuplot or LaTeX

Subroutine: determine_if_we_have_procps

     Usage: determine_if_we_have_procps()
     Returns 1 if the system has procps, 0 otherwise.

Subroutine: du

     Usage: du($dir)
     Returns the size of the contents of the directory $dir

Subroutine: dumpfile

     Usage: dumpfile($filename,$data)
     Puts the string $data into the file given by $filename

Subroutine: empty_dir

     Usage: empty_dir($directory)
     Removes all the files $directory (NB ignores errors!)

Subroutine: extract_gnuplot_range

     Usage: extract_gnuplot_range($logstring)
     Extracts "show range" commands from a the gnuplot output given in $logstring

Subroutine: f_number_to_c_number

     Usage: f_number_to_c_number($f_number)
     Converts fortran number $f_number (with 'D' or 'd') to a C-style floating-point number with an 'e'. There is no error checking.

Subroutine: file_linecount

     Usage: file_linecount($filename)
     Count the number of lines in a file given by $filename

Subroutine: find_media_location

     Usage: find_media_location($id)
     Finds Rob's media disk given by the name $id

Subroutine: flip

     Usage: flip($x,$y)
     Returns ($y,$x) i.e. flips the numbers in the array

Subroutine: get_from_C_header

     Usage: get_from_C_header($macro,$headerfile)
     Uses gcc and/or Convert::Binary::C to get the value of the macro $macro in the C-header file $headerfile

Subroutine: gnuplot_ps_to_png

     Usage: gnuplot_ps_to_png($pltfile,$pngfile,$dpi,$colourblind,$invert,$extra)
     Given a gnuplot .plt file (list of gnuplot commands) and generates a png file from it via a postscript file. The PNG file is given in $pngfile, and the plt file MUST make a .ps file with the same name as the .png file (except for the file extension). DPI is the conversion resolution. If $colourblind is true, then the default red and green are converted. If $invert is true then black and white are flipped. $extra contains extra args sent to the ps_to_png() function call (see below). Returns an array containing the graph range and error string.

Subroutine: gnuplot_ps_to_png2

     Usage: gnuplot_ps_to_png2($pltfile,$pngfile,$dpi,$colourblind,$invert,$extra)
     Given a gnuplot .plt file (list of gnuplot commands) and generates a png file from it via a postscript file. The PNG file is given in $pngfile, and the plt file MUST make a .ps file with the same name as the .png file (except for the file extension). DPI is the conversion resolution. If $colourblind is true, then the default red and green are converted. If $invert is true then black and white are flipped. $extra contains extra args sent to the ps_to_png() function call (see below). This version of the subroutine returns the graph range and error string as a hash.

Subroutine: gnuplot_prettytitle

     Usage: gnuplot_prettytitle($title)
     Cleans up $title for gnuplot enhanced mode postscript, converting alpha, lambda and sigma to their symbols.

Subroutine: gp_send_and_return

     Usage: send commands to a gnuplot stream
     Description: this is an experimental function!

Subroutine: gpshow

     Usage: gpshow()
     Returns the command to output the range information in gnuplot 

Subroutine: hash_count

     Usage: hash_count($hash_pointer,$offset)
     count the number of elements in a (nested) hash at $hash_pointer, offset by $offset (0 if not given).

Subroutine: heximage

     Usage: heximage($filename)
     Returns the contents of $filename in hexadecimal

Subroutine: in_numerical_order

     Usage: sort in_numerical_order ( ... )
     A crude numerical sorter. You would be better off using nsort from Sort::Key

Subroutine: is_numeric

     Usage: is_numeric( ... )
     Returns 1 if ALL arguments are numeric, otherwise 0.

Subroutine: is_numeric_scalar

     Usage: is_numeric_scalar($n)
     Returns true if $n is a numeric scalar. Uses the Scalar::Util::looks_like_number function, but also converts a FORTRAN number to C-like floating point first.

Subroutine: is_vnc_port

     Usage: is_vnc_port($port)
     Returns true if a port is an open VNC port 

Subroutine: join_arrays_as_hash

     Usage: join_arrays_as_hash($array_pointer1,$array_pointer2)
     Given two arrays (as pointers) join into a hash and return its reference

Subroutine: listdirs

     Usage: listdirs($paths...)
     Returns an array of directories in $path

Subroutine: listdirs_path

     Usage: listdirs_path($path)
     Returns an array of directories in $path with their full paths

Subroutine: listfiles

     Usage: listfiles($dir,$failmode)
     Returns a list of files in $dir. If there is an error and $failmode is 'confess' then an error is reported through Carp::Confess (and your code stops), otherwise nothing happens.

Subroutine: listfiles_path

     Usage: listfiles_path($path)
     Returns an array of the contents of $path with full paths

Subroutine: listdirs_with_no_subdirs

     Usage: listdirs_with_no_subdirs($path)
     Returns a list of directories that contain only files, recursively checking subdirectories

Subroutine: list_modules

     Usage: list_modules();
     Returns a list of loaded modules

Subroutine: log10

     Usage: log10($x)
     Returns log to base 10 of $x.

Subroutine: MAX

     Usage: MAX($x,$y)
     Returns the maximum of $x and $y, first checking if the values are numeric.

Subroutine: max_from_array

     Usage: max_from_array(@array)
     Returns the maximum value in @array (using List::Util::max)

Subroutine: mean_from_array

     Usage: mean_from_array(@array)
     Returns the numerical mean of the @array using List::Util::sum

Subroutine: mem_usage

     Usage: mem_usage($procps,$pid)
     Tries to find the memory usage of process $pid (or the current process if no $pid is given). If $procps is true and scanning of /proc/ fails, tries to use the ps command.

Subroutine: MIN

     Usage: MIN($x,$y)
     Returns the minimum of $x and $y, first checking if the values are numeric.

Subroutine: min_from_array

     Usage: min_from_array(@array)
     Returns the minimum value in @array (using List::Util::max)

Subroutine: mkdirhier

     Usage: mkdirhier($dir)
     Makes the directory $dir and parents if required (cf. Linux commands 'mkdir -p' or 'mkdirhier')

Subroutine: mkdirhier_legacy

     Usage: mkdirhier_legacy($dir)
     Makes the directory $dir and parents if required (cf. Linux commands 'mkdir -p' or 'mkdirhier'). This is the inferior, legacy version, which may fail.

Subroutine: ncpus

     Usage: ncpus()
     Returns the number of CPUs in your machine using Sys::Info.

Subroutine: ncpus_old

     Usage: ncpus()
     Returns the number of CPUs in your machine using /proc/cpuinfo, i.e. this will nto work on machines without /proc/ (windows, mac?)

Subroutine: nfscheck

     Usage: nfscheck($path)
     Return 1 if $path is on an NFS mounted partition

Subroutine: nonblockGetLines {
     Usage: nonblockGetlines($filehandle,$timeout)
     A non-blocking filehandle read that returns an array of lines read. Returns:  ($eof,@lines)


Subroutine: non_zero_array

     Usage: non_zero_array(@array)
     Return 1 if the array has a non-zero element

Subroutine: non_zero_array_from_pointer

     Usage: non_zero_array($array_reference)
     Return 1 if the array has a non-zero element

Subroutine: nsubdirs

     Usage: nsubdirs($paths...)
     Returns the number of subdirs in the given paths (ignores hidden directories, i.e. those that start with a dot)

Subroutine: numeric_clamp

     Usage: numeric_clamp($x,$min,$max)
     clamp $x to given range $min,$max using (unsafe but faster) numeric_* functions

Subroutine: numerical_list_sorter($$)

     Usage: numerical_list_sorter(...)
     numerical list sorter: almost an nsort but splits first

Subroutine: numericMAX

     Usage: numericMAX($x,$y)
     Same as MAX but we are sure they exist and are numbers i.e. rather unsafe, but faster

Subroutine: numericMIN

     Usage: numericMIN($x,$y)
     Same as MIN but we are sure they exist and are numbers i.e. rather unsafe, but faster

Subroutine: ok_file

     Usage: ok_file($filename)
     Check if a file exists at $filename and is of significant size (>100 bytes)

Subroutine: ok_directory

     Usage: ok_directory($dirname)
     Check if a directory exists at $dirname

Subroutine: operating_system

     Usage: operating_system()
     Returns either "windows" or "unix" depending on the type of operating system (defaults to unix if windows is not detected)

Subroutine: parse_cmdline_args_to_perlvars

     Usage: parse_cmdline_args_to_perlvars($h)
     Parse command line args, setting perl variables as required. e.g. if the arg is x=y then Perl evaluates $x="$y"; If $h is defined, instead sets $h{$x}=$y.

Subroutine: pmrand

     Usage: pmrand()
     Returns a random floating-point number between -1.0 and +1.0

Subroutine: poisson_error

     Usage: poisson_error($p,$n)
     Given a 'result' $p and the number of measurements which went into it $n, calculate the Poisson error

Subroutine: port_use

     Usage: port_use($port,$host,$regexp)
     Examine port $port (on $host or, if undef, 127.0.0.1) and return true if it is in use, otherwise undef (false). if $regexp is given as the third arg, this is forced to match the port description, otherwise undef is returned.

Subroutine: proc_mem_usage

     Usage: proc_mem_usage($pid)
     Uses Proc::ProcessTable to find the memory use of process given by ID $pid or, if no argument is given, the current process. Returns in Mbytes.

Subroutine: ps_to_png

     Usage: ps_to_png($psfile,$pngfile,$dpi,$extra,$regexp)
     Converts postscript file $psfile to PNG file $pngfile using ghostscript at a resolution given by $dpi. $extra contains extra arguments for the ghostscript command (is ignored if undefined). $regexp is applied to the command line string (or ignored if undefined).

Subroutine: random_string

     Usage: random_string($len)
     Returns a random string of $len characters

Subroutine: remove_ANSI

     Usage: remove_ANSI($string)
     Removes ANSI codes from $string

Subroutine: remove_all_escape_codes
    
     Usage: remove_all_escape_codes($string)
     Removes all escape codes from $string

Subroutine: remove_pango_markup

     Usage: remove_pango_markup($string)
     Removes pango <span> markup from given $string

Subroutine: renice_me

     Usage: renice_me($prio)
     Renices the current process with priority $prio (or 10 if $prio not defined), and sets the ionice (disk I/O) priority to minimum.

Subroutine: runcmd

     Usage: runcmd($cmd,$where)
     Runs the command given by $cmd and sends output to either the screen or an array to be returned or does nothing with it. $where can be: 0 or undef or 'none': output is lost, 1 or 'screen' : output is printed to the screen, 2 or 'array' : output is returned as an array, 3 or 'screen and array' or 'both' : 1 and 2

Subroutine: safeslurp

     Usage: safeslurp(<filename>)
     Returns the contents of file given by <filename> and on failure returns undef

Subroutine: scalars_the_same

     Usage: scalars_the_same($x,$y)
     Checks if $x and $y are numerically the same, returns the fractional error, or 0.0 if they are the same, or 1.0 if one is numeric and the other not.

Subroutine: session_id

     Usage: session_id($length)
     Make a "unique" session id string, of length $length or 16 characters if $length is not given.

Subroutine: slurp

     Usage: slurp(<filename>)
     Returns the contents of file given by <filename> as a string, on failure dies with Carp::confess

Subroutine: slurp_filehandle

     Usage: slurp_filehandle($fh)
     Returns the contents of the file conntected to $fh

Subroutine: slurp_to_array_ref

     Usage: slurp(<filename>)
     Returns the contents of file given by <filename> in an array reference, on failure dies with Carp::confess

Subroutine: splitcomma

     Usage: splitcomma($string)
     Splits $string on outermost commas

Subroutine: ssleep

     Usage: ssleep($nsecs)
     Sleep function : stop doing anything for $nsecs seconds. NB this uses the "select" function call which works on both windows (also with fork) and Linux.

Subroutine: sum_array

     Usage: sum_array(@array)
     sum up @array using List::Util::sum and return the result

Subroutine: sum_array_with_pointers

     Usage: sum_array($array)
     sum up $array (a reference to an array) using List::Util::sum and return the result

Subroutine: thetime

     Usage: thetime($offset)
     Output the time either now (if no $offset is given), or $offset seconds later, as a human-readable string.

Subroutine: thread_log

     Usage: thread_log(@array)
     Print to the screen using thread_string on the @array

Subroutine: thread_string

     Usage: thread_string($l,$n)
     Writes $l to line $n of the terminal.

Subroutine: toggle_state

     Usage: toggle_state($state)
     toggles integer $state, i.e. sets 0 to 1, and vice versa

Subroutine: touch

     Usage: touch($filename)
     Equivalent to UNIX 'touch' command on $filename: opens the file and sends no data to it. Does not destroy an existing file.

Subroutine: trem

     Usage: trem($tstart,$count,$n)
     Estimate time remaining (seconds) given a starting time and a count (i.e. progress = $count/$n). $tstart is the start time, $count is the current progress count, and $n is the total number required.

Subroutine: trem2

     Usage: trem2($dt,$count,$dn,$n)
     Estimate time remaining (seconds) given a differential time and count (i.e. progress = $count/$n). $dt is the time since the last call, $count is the current progress count, $dn is the number run since the last call, and $n is the total number required.

Subroutine: username
 
     Usage: username()
     Return user name of person running the script

Subroutine: versionofmodule

     Usage: versionofmodule($module)
     Return version of a Perl module given by $module 

Subroutine: ynstring

     Usage: ynstring($x)
     Return 'Y' if $x is true, 'N' otherwise

Subroutine: compactnumber

     Usage: compactnumber($x)
     Returns the most compact string that represents $x.
     Warning: because sprintf %g and %e are used, the number
     may lose some precision. If this matters to you, do not use
     this function.


=head1 SEE ALSO


http://en.wikipedia.org/wiki/Heavy_metal_umlaut

=head1 AUTHOR

Robert Izzard (R.G. Izzard) based at www.surrey.ac.uk
<< <r.izzard at surrey.ac.uk > >>
alternatively rob.izzard@gmail.com

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2005-2018 by Robert Izzard 

Do what you want with the software, but you must cite the appropriate author and/or papers if you make money from it or use it in your own (scientific or otherwise) work!

Note that I take NO RESPONSIBILITY for what this code may do, or what you may do with it. I provide it to you in good faith, and you are (of course!) allowed to view and hack all or any of it as you see fit. I do ask that if you improve something, you send your new code to be included in the module for the benefit of all.  

=cut
