package cosmology;

use 5.008005;
use strict;

use vars qw(%cosmology $cosmology_is_setup @hubble_parameter_table %hubble_parameter_hash   %redshift_time_table  @redshift_time_keys @time_redshift_keys %time_redshift_table );
use warnings;

require Exporter;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use cosmology ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw( &galaxy_mass_metallicity_relation &age_of_the_universe &setup_cosmology &galaxy_mass_metallicity_relation &hubble_time &cosmological_sfr &redshift_to_time &time_to_redshift
	 
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw( &galaxy_mass_metallicity_relation &age_of_the_universe &setup_cosmology &galaxy_mass_metallicity_relation &hubble_time &cosmological_sfr &redshift_to_time &time_to_redshift
	
);

our $VERSION = '0.01';


#########
# TODO : replace our integrator with GSL to make the redshift table
# TODO : use Robinterpolate to interpolate log(z) and log(t)
use RobInterpolation;

# Preloaded methods go here.

sub setup_cosmology
{ 
    $cosmology{'H0'}=70.0; # hubble parameter in km/s
    $cosmology{'H0_per_second'}=($cosmology{'H0'}/3.08568025e19); # in cgs
    $cosmology{'h'}=0.7;#$cosmology{'H0'}/100.0;
    $cosmology{'Omega_M'}=0.3;
    $cosmology{'Omega_Lambda'}=0.7;
    $cosmology{'Omega_k'}=0.0;
    # Mstar,Phistar,alpha from the Panter et al 2004 fit
    $cosmology{'Mstar'}=7.64e10*($cosmology{'h'}**-2.0);
    $cosmology{'Phistar'}=7.8e-3*($cosmology{'h'}**3.0);
    $cosmology{'alpha'}=-1.159;
    $cosmology{'Zsolar'}=0.02;
}

sub setup_check
{
    if(!defined($cosmology_is_setup))
    {
	setup_cosmology();
	$cosmology_is_setup=1;
    }
}

sub galaxy_mass_metallicity_relation
{
    setup_check();

    my $mgal=shift; # galactic mass in Mstar
    my $th=shift; # hubble time (Myr)
    my $K=1.0;
    my $beta=2.0;

    # either use the fit of Norman and Langer 2005
    #return ($cosmology{'Zsolar'} * ( (1.0/$K) * $mgal )**(1.0/$beta));

    # or the fit from Savaglio et al 2005 (sec 9.1 eq 11)
    #print "Mgal $mgal , TH $th\n";

    # we should 
    my $l=log10($mgal*$cosmology{'Mstar'}); # convert to solar masses
    $th=log10($th)-3.0; # convert from Myr to Gyr

    # this is (O/H) for the given galaxy
    my $oh=10.0**(-7.5903+2.5315*$l-0.09649*$l*$l+5.1733*$th-0.3944*$th*$th-0.4030*$th*$l-12.0);

    # this is (O/H) for the sun (AG89)
    my $oh_solar=8.495e-4; 
    my $z=$cosmology{'Zsolar'}*$oh/$oh_solar;
    #print "Hence Z=$z\n";
    return($z);
}
sub age_of_the_universe
{
    setup_check();
    # find the age of the universe as a function of redshift
    my $redshift=shift;
    my $n=100.0;
    my $dz=0.001;
    my $th=0.0;
    my $omegaM=$cosmology{'Omega_M'};
    my $omegaL=$cosmology{'Omega_Lambda'};
#    print "Omega M $omegaM L $omegaL\n";

    # most of the contribution to the integral is in the range z<100:
    # this is good enough!
    
    my $z=1e2;
    my $t1=1.0/(1.0+$z)*((1+$z)**2.0*
			 (1+$omegaM*$z) - $z*(2.0+$z)*$omegaL)**-0.5;
    for($z=1e2;$z>=$redshift;)
    {
	$z-=$dz;
	my $t2=(1.0/(1.0+$z))*((1.0+$z)**2.0*(1.0+$omegaM*$z) - $z*(2.0+$z)*$omegaL)**-0.5;
	# trapezium rule
	$th+=$dz*($t1+($t2-$t1)*0.5);
	$t1=$t2;
    }
    # remember to convert H0 to seconds
    $th/=$cosmology{'H0_per_second'};
    $th/=(3600.0*24.0*365.25); # convert to per year
    return $th;
}

sub hubble_time
{
    setup_check();
    if($#hubble_parameter_table<0)
    {
	# just in case the table has not been made!
	my $age=age_of_the_universe(0.0);
	printf "Age of the universe %g\n",$age;
	$age=redshift_to_time(0.0);
	printf "Age of the universe %g\n",$age;
	make_hubble_parameter_table($age);
    }
    
    # calculate hubble time as a function of time
    my $t=shift;
    my $vb=1;
    $t=int($t+0.1);
    print "At time $t we have Ht=",$hubble_parameter_table[$t],"\n" if $vb;
    return($hubble_parameter_table[$t]);
}

sub make_hubble_parameter_table
{
     setup_check();
     print "Make hubble table\n";
     # calculate the hubble parameter as a function of time
     my $a=1; # scale factor now
     my $t; # time
     my $maxtt=shift; # the age of the universe, when a=1
     my $dt=1; # steps of 1Myr
     my $dadt;
     my $H0=$cosmology{'H0_per_second'}; # work in cgs!
     
     my $omegaM=$cosmology{'Omega_M'};
     my $omegaL=$cosmology{'Omega_Lambda'};
     my $Ht; # hubble parameter(t)
     my $tH; # hubble time(t)
     my $c=(1e6*365.25*24.0*3600); # Myr conversion factor

     # k1 is da/dt at time t=t0 (now)
     my $k1=$H0*(1.0+$omegaM*(1.0/$a-1.0)+$omegaL*($a*$a-1.0))**0.5;
     $Ht=($k1/$a)*3.08568025e19;
     $tH=($a/$k1)/$c;
     $dadt=$k1; # best guess for now...

     for($t=$maxtt;$t>=10;)
     {
	 # calculate Ht and tH at time $t
	 $dadt=$k1;
	 $Ht=($dadt/$a)*3.08568025e19;
	 $tH=($a/$dadt)/$c; # convert to Myr
	 # save in our table
	 $hubble_parameter_table[$t]=$tH;

	 # perhaps output for Carolina :)
	 #printf "%g %g %g %g %g\n",$t,$a,$dadt,$Ht,$tH;
	 
	 # calculate next timestep values in k2
	 $t-=$dt;
	 my $k2=$H0*(1.0+$omegaM*(1.0/$a-1.0)+$omegaL*($a*$a-1.0))**0.5;
	 
	 # integrate with trapezium rule
	 $a-=$c*$dt*($k1+($k2-$k1)*0.5);

	 # and save k2 as k1 for the next timestep
	 $k1=$k2;
	 
     }
}


sub cosmological_sfr
{
    setup_check();
    # fit to the SFR of the universe
    my $redshift=shift;
    #my $sfr;#=-1.81820e+00+4.43220e-01*$redshift+2.51220e-01*$redshift*$redshift;
#    $sfr=10.0**$sfr;
     # norbert's polynomial fit
    #my @c=(-2.19532585,0.543708801,-0.176081672,0.0183224473,-0.000655161508);
    #$sfr=$c[0]+$redshift*$c[1]+$redshift*$redshift*$c[2]+$redshift**3.0*$c[3]+$redshift**4.0*$c[4];

    # Madau's Eq15 https://arxiv.org/pdf/1403.0007.pdf
    # Msun / year / Mpc     
    my $sfr = 0.015 * (1+$redshift)**2.7 / (1.0 + ((1+$redshift)/2.9)**5.6);
    
    #print "At redshift $redshift have sfr=$sfr\n";
    
    return $sfr;
}

sub redshift_to_time
{
    setup_check();
    make_redshift_time_table() if($#redshift_time_keys<0);
    my ($redshift) = @_;    

    # interpolate to find the redshift given the time
    my $x = generic_interpolation_wrapper($redshift,
                                          \@redshift_time_keys,
                                          \%redshift_time_table);
    return $x;

    
    # EdS
#    my $eds=((1.0/($redshift+1.0))**1.5*13700);

    my $bestkey='';
    my $i;
    my $save;
    for($i=0;$i<=$#redshift_time_keys;$i++)
    {
	#print "Compare redshift $redshift vs $redshift_time_keys[$i]\n";
	if($redshift_time_keys[$i] > $redshift)
	{
	    $save=$i;
	    $i=$#redshift_time_keys+10;
	}
    }
 #   print "Break at $save which is redshift $redshift_time_keys[$save] time $redshift_time_table{$redshift_time_keys[$save]}\n";
  #  exit;

    return $redshift_time_table{$redshift_time_keys[$save]}; # returns value in Myr
}

sub log10
{
    return(log($_[0])/log(10.0));
}

sub time_to_redshift
{
    setup_check();
    make_redshift_time_table() if($#redshift_time_keys<0);
    
    my ($t) = @_;
    
    # at time zero return something large and finite
    return 1e10 if($t == 0.0);

    # interpolate to find the redshift given the time
    my $x = generic_interpolation_wrapper($t,
                                          \@time_redshift_keys,
                                          \%time_redshift_table);
    return $x;
    
    # EdS approximation
    # my $eds=(($t/13700)**(-2.0/3.0)-1.0);

    # old, poor method
    my $redshift;
    my $bestkey='';
    my $i;
    my $save;
    for($i=0;$i<=$#redshift_time_keys;$i++)
    {
	#print "Compare time $t vs $redshift_time_table{$redshift_time_keys[$i]}\n";
	if($redshift_time_table{$redshift_time_keys[$i]} < $t)
	{
	    $save=$i;
	    $i=$#redshift_time_keys+10;
	}
    }
#    print "Break at $save which is redshift $redshift_time_keys[$save]\n";

    if(!defined $save)
    {
        print "Save undefined\n";
        exit;
    }

    return $redshift_time_keys[$save]; # returns value in Myr
}

sub make_redshift_time_table
{
    setup_check();
   
    # integrate to make redshift(time) table
    my $n=100.0;
    my $dz=0.01;
    my $th=0.0;
    my $omegaM=$cosmology::cosmology{'Omega_M'};
    my $omegaL=$cosmology::cosmology{'Omega_Lambda'};
    my $c=1e6*$cosmology::cosmology{'H0_per_second'}*(3600.0*24.0*365.25);
    my $z=1e2;
    my $t1=1.0/(1.0+$z)  * ((1+$z)**2.0*(1+$omegaM*$z) - $z*(2.0+$z)*$omegaL)**-0.5;
    for($z=1e2;$z>=0.0;)
    {
	$z-=$dz;
        my $t2=(1.0/(1.0+$z))*((1.0+$z)**2.0*(1.0+$omegaM*$z) - $z*(2.0+$z)*$omegaL)**-0.5;
	# trapezium rule
	$th += ($t1*$dz+($t2-$t1)*0.5*$dz)/$c;
	$t1 = $t2;

        $z = sprintf('%g',$z);
        $th = sprintf('%g',$th);
        #print "At redshift $z we have time $th \n";
	$redshift_time_table{$z} = $th;
        $time_redshift_table{$th} = $z;
    }
    @redshift_time_keys = sort {$a<=>$b} keys (%redshift_time_table);
    @time_redshift_keys = sort {$a<=>$b} values (%redshift_time_table);
}


1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

cosmology - A Perl module to calculate useful cosmological quantities.

=head1 SYNOPSIS

  use cosmology;
  setup_cosmology();
  # .. your own cosmology ..
  my $t=redshift_to_time(2.5);
  my $t_H=hubble_time($t);
  # etc...

=head1 DESCRIPTION

How often do you want to know time as a function of redshift, or the hubble time at redshift 2.5? Not often perhaps, but sometimes! This module calculates such quantities for you.

NB This only works for a flat universe. At the moment...

To start using the module, make a call to the setup_cosmology() function:

    use cosmology;
    setup_cosmology();

This sets some default parameters in the %cosmology::cosmology hash. These are

The Hubble parameter now (aka the Hubble constant) in km /s /Mpc
    $cosmology::cosmology::cosmology{'H0'}=70.0
The Hubble constant in cgs units
    $cosmology::cosmology{'H0_per_second'}
The parameter h = Hubble constant / 100
    $cosmology::cosmology{'h'}=0.7;
The fraction of mass in matter
    $cosmology::cosmology{'Omega_M'}=0.3;
The fraction of mass in a cosmological constant (dark energy)
    $cosmology::cosmology{'Omega_Lambda'}=0.7;
The curvature term: throughout we assume a flat universe.
    $cosmology::cosmology{'Omega_k'}=0.0;
Values for M*, phi* and alpha from the Panter et al 2004 fit to the galaxy mass function.
    $cosmology::cosmology{'Mstar'}=7.64e10;
    $cosmology::cosmology{'Phistar'}=7.8e-3;
    $cosmology::cosmology{'alpha'}=-1.159;
A proxy for solar metallicity
    $cosmology::cosmology{'Zsolar'}=0.02;

You can (and probably should) change these to the values you require after calling setup_cosmology().

Next there are the functions:

To find the age of the universe call
    age_of_the_universe()

To convert redshift to time call 
    redshift_to_time(<redshift>)
and to convert time to redshift call
    time_to_redshift(<time>)
where time is in MYr.

To calculate the Hubble time at a given time, call
    hubble_time(<time>)
and of course if you want to calculate the Hubble time as a function of redshift,
    hubble_time(redshift_to_time(<redshift>))

To calculate the probability density function of galaxy masses, based on the 
fit of Savaglio et al 2005 (sec 9.1 eq 11, which is a function of mass and 
the hubble time) call
    galaxy_mass_metallicity_relation(<galaxy mass>,<hubble time>)

To calculate the cosmological star formation rate as a function of redshift call
    cosmological_sfr(<redshift>)


=head1 AUTHOR

Rob Izzard rgi somewhereat carolune.net

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2005 by Rob Izzard

Use at your own risk!

=cut
