package distribution_functions;
$|=1;
#  distribution_functions module
#
# simple functions used to calculate probability distributions
# for binary_grid (part of binary_c)
#

our $VERSION = '0.05';

# version 0.01 : initial version, simple const and power laws
# version 0.02 : add gaussian distribution, some code reduction, better auto
#                calc of constants
# version 0.03 : add monte carlo variables
# version 0.04 : new Izzard2012 period interpolation function
# version 0.05 : change Izzard2012 to use more appropriate period minimum

use common::sense;
use 5.16.0;
use feature qw(state);
use warnings;

# rob's modules
use rob_misc;
use binary_stars;
use Math::Trig;
#use binary_grid;
use Sort::Key qw(nsort);
use Data::Dumper;
#use Memoize; # not thread safe
#memoize('const');
use Sub::Identify ':all';

use Carp qw/confess/;

use Devel::Size qw/size total_size/;

# global variables
use vars qw( %gauss_consts %powerlaw_consts %threepart_powerlaw_consts );

# export no functions by default: binary_grid will explictly
# call them as distribution_functions::function
my @funcs = qw(  );

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ @funcs ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = @funcs;

# constants
use constant PI => acos(0.0)*2.0;
use constant GAUSSIAN_PREFACTOR => (1.0/sqrt(2.0*PI));
use constant logmin => 1e-30;
use constant value_of_log10 => log(10.0);
use constant log_ln_converter => (1.0/log(10.0));

# hash used to store normalization constants
my %distdata;

sub const
{
    # a constant distribution function between min=$_[0] and max=$_[1]
    # $_[2] is optional (if given, perform a range check)
    
    my $C= (defined($_[2]) && (($_[2]<$_[0]) || ($_[2]>$_[1]))) ? 0.0 : (1.0/($_[1]-$_[0]));
    #printf "CONST $_[0] to $_[1] ($_[2]) $C\n";
    return $C;
}

sub number
{
    # return just the number passed in, used as a dummy function
    return $_[0];
}
    
sub powerlaw
{
    # single power law of index k at x from min to max 
    # (NOT k=-1)
    my ($min,$max,$k,$x) = @_;
    
    if($x<$min || $x>$max)
    {
	return 0.0;
    }
    else
    {
        # Normalizer
        my $C = $powerlaw_consts{$min}{$max}{$k} // 
            initialize_powerlaw_consts($min,$max,$k);
	
        # power law
	my $y = $C * $x**$k;
	#print "PLAW $min to $max : C = $C : $x ** $k = $y\n"; 
	return $y;
    }
}

sub initialize_powerlaw_consts
{
    local $SIG{__DIE__} = sub { Carp::confess @_ };
    my ($min,$max,$k) = @_;
    my $k1=$k+1.0;

    #print "Powerlaw consts from $min, $max, $k where k1=$k\n";
    $powerlaw_consts{$min}{$max}{$k} = $k1/($max**$k1 - $min**$k1); 
    return $powerlaw_consts{$min}{$max}{$k};
}

sub initialize_three_part_power_law_consts
{
    local $SIG{__DIE__} = sub { Carp::confess @_ };
    # calculate constants for this power law
    my ($m0,$m1,$m2,$mmax,$p1,$p2,$p3)=@_;
    #print "INIT three-part power law m0=$m0 m1=$m1 m2=$m2 mmax=$mmax p1=$p1 p2=$p2 p3=$p3\n";
    my $array=[];

    $$array[1]=(($m1**$p2)*($m1**(-$p1)))*
	(1.0/(1.0+$p1))*
	($m1**(1.0+$p1)-$m0**(1.0+$p1))+
	(($m2**(1.0+$p2)-$m1**(1.0+$p2)))*
	(1.0/(1.0+$p2))+
	(($m2**$p2)*($m2**(-$p3)))*
	(1.0/(1.0+$p3))*
	($mmax**(1.0+$p3)-$m2**(1.0+$p3));
    $$array[1]=1.0/($$array[1]+1e-50);
    $$array[0]=$$array[1]*$m1**$p2*$m1**(-$p1);
    $$array[2]=$$array[1]*$m2**$p2*$m2**(-$p3);

    #print "ARRAY SET @_ => @$array\n";
    $threepart_powerlaw_consts{"@_"}=[@$array];
}

sub ktg93_lnspace
{
    # wrapper for KTG93 on a ln(m) grid
    my $m=$_[0];
    return ktg93(@_) * $m; 
}

sub setopts
{
    # wrapper to take a hash (reference) of options,
    # override with $newopts (where appropriate) and
    # return the hash of options
    my $opts=$_[0];
    my $newopts=$_[1];
    if(defined $newopts)
    { 
	# newopts is a hash of new options 
	foreach my $opt (keys %$newopts)
	{
	    # overwrite opt
	    $$opts{$opt} = $$newopts{$opt};
	}
    }
    return $opts;
}

sub Kroupa2001
{
    my $m=$_[0];
    my $newopts=$_[1];

    # default parameters
    my $opts=setopts({m0=>0.1,
		      m1=>0.5,
		      m2=>1.0,
		      mmax=>100.0,
		      p1=>-1.3,
		      p2=>-2.3,
		      p3=>-2.3},
		     $newopts);

    return three_part_power_law($m,
				$$opts{m0},
				$$opts{m1},
				$$opts{m2},
				$$opts{mmax},
				$$opts{p1},
				$$opts{p2},
				$$opts{p3});
}

sub ktg93
{
    # wrapper for the mass distribution of KTG93
    my ($m,$newopts)=@_;

    if($m eq 'uncertainties')
    {
	# return (pointer to) the uncertainties hash
	return {
	    m0=>{default=>0.1,
		 fixed=>1},
	    m1=>{default=>0.5,
		 fixed=>1},
	    m2=>{default=>1.0,
		 fixed=>1},
	    mmax=>{default=>80.0,
		   fixed=>1},
	    p1=>{default=>-1.3,
		 low=>-1.3,
		 high=>-1.3},
	    p2=>{default=>-2.2,
		 low=>-2.2,
		 high=>-2.2},
	    p3=>{default=>-2.7,
		 low=>-2.7,
		 high=>-2.7}
	};
    }

    # set options
    my $opts=setopts({m0=>0.1,
		      m1=>0.5,
		      m2=>1.0,
		      mmax=>80.0,
		      p1=>-1.3,
		      p2=>-2.2,
		      p3=>-2.7},
		     $newopts);

    return three_part_power_law($m,
				$$opts{m0},
				$$opts{m1},
				$$opts{m2},
				$$opts{mmax},
				$$opts{p1},
				$$opts{p2},
				$$opts{p3});
}

sub three_part_power_law
{
    # generalized three part power law, usually used for mass distributions
    # args are M, M0, M1, M2, MMAX, P1, P2, P3 (powers)
    my $m=shift;

    # use pre-calculated consts if possible, otherwise calculate them
    my $consts=$threepart_powerlaw_consts{"@_"} //
	initialize_three_part_power_law_consts(@_);
   
    my $p;
    if($m<$_[1])
    { 
	$p = $m<$_[0]
	   ? 0.0
	   : $$consts[0]*($m**$_[4]);
    }
    elsif($m<$_[3])
    {
	$p = $m<$_[2]
	    ? $$consts[1]*($m**$_[5])
	    : $$consts[2]*($m**$_[6]);
    }
    else
    {
	$p=0.0;
    }
    return $p;
}

sub gaussian
{
    # Gaussian distribution function e.g. for Duquennoy + Mayor 1991

    # location (X value), mean and sigma, min and max range
    my ($x,$mean,$sigma,$gmin,$gmax) = @_;
 
    my $p;
    if($x<$gmin || $x>$gmax)
    {
	$p=0.0;
    }
    else
    {
	# normalize over given range
	my $mult= $gauss_consts{$mean}{$sigma} //
	    gaussian_normalizing_const($mean,$sigma,$gmin,$gmax);
	$p = $mult * gaussian_func($x,$mean,$sigma);
    }
    return $p;
}

sub gaussian_normalizing_const
{
    # first time: calculate multiplier for given $mean, $sigma
    my ($mean,$sigma,$gmin,$gmax) = @_;
    my $ptot=0.0;
    my $d=($gmax-$gmin)/1000.0;
    for(my $y=$gmin;$y<=$gmax;$y+=$d)
    {
	$ptot += $d * gaussian_func($y,$mean,$sigma);
    }
    $gauss_consts{$mean}{$sigma}=$ptot; # save for later
    return $gauss_consts{$mean}{$sigma};
}

sub gaussian_func
{
    # local function
    # three args: x, mean, sigma
    my $r=1.0/$_[2]; # 1/sigma
    my $y=($_[0]-$_[1])*$r;
    return(GAUSSIAN_PREFACTOR*$r*exp(-0.5*$y*$y));
}

sub Arenou2010_binary_fraction
{
    
    # Arenou 2010 function for the binary fraction as f(M1)
    #
    # GAIA-C2-SP-OPM-FA-054
    # www.rssd.esa.int/doc_fetch.php?id=2969346
    my ($m1) = @_;
    return 0.8388 * tanh(0.688*$m1+0.079);
}

sub raghavan2010_binary_fraction
{
    #
    # Fit to the Raghavan 2010 binary fraction as a function of   
    # spectral type (Fig 12). Valid for local stars (Z=Zsolar).
    #
    # I converted spectral type to mass by use of the ZAMS 
    # effective temperatures from binary_c/BSE (at Z=0.02)
    # and the new "long_spectral_type" function of binary_c
    # (based on Jaschek+Jaschek's Teff-spectral type table).
    # 
    # I then fitted the result.
    #

    my $M=$_[0]; # input mass
    return MIN(1.0,MAX(($M**0.1)*
		       (5.12310e-01)+(-1.02070e-01),(1.10450e+00)*
		       ($M**(4.93670e-01))+(-6.95630e-01)));
}

sub Izzard2012_period_distribution
{
    # period distribution which interpolates between 
    # Duquennoy and Mayor 1991 at low mass (G/K spectral type <~1.15Msun)
    # and Sana et al 2012 at high mass (O spectral type >~16.3Msun)

    # This gives dN/dlogP, i.e. DM/Raghavan's Gaussian in log10P at low mass
    # and Sana's power law (as a function of logP) at high mass

    my $P=$_[0]; # period
    my $M=$_[1]; # primary star mass
    my $log10Pmin = $_[2]; # minimum period (optional)

    $log10Pmin //= -1.0;
    $log10Pmin = MAX(-1.0,$log10Pmin);

    # save mass input and limit mass used ($M from now on) to fitted range
    my $Mwas=$M;
    $M=MAX(1.15,MIN(16.3,$M));
  
#    print "Izzard2012 called for M=$Mwas (trunc'd to $M), P=$P\n";

    if(!defined($distdata{Izzard2012}{$M}{$log10Pmin}))
    {
	# need to normalize the distribution for this mass 
	# (and perhaps secondary mass)
	$distdata{Izzard2012}{$M}{$log10Pmin}=1.0; # prevent infinite recursion
	my $N=1000.0; # resolution for normalization: I hope 1000 is enough...
	my $dlP = (10.0-$log10Pmin)/$N; # log10P from minimum to 10
	my $C=0.0;
	for(my $lP=$log10Pmin;$lP<=10.0;$lP+=$dlP)
	{
	    $C += $dlP * __SUB__->(10.0**$lP,$M,$log10Pmin);
	}
	$distdata{Izzard2012}{$M}{$log10Pmin} = 1.0/$C;
	#print "Normalization constant for Izzard2012 M=$M (log10Pmin=$log10Pmin) is $distdata{Izzard2012}{$M}{$log10Pmin}\n";
    }

    my $lP = log10($P); # log period

    # fits
    my $mu = interpolate_in_mass($M,-17.8,5.03);
    my $sigma= interpolate_in_mass($M,9.18,2.28);
    my $K=interpolate_in_mass($M,6.93e-2,0.0);
    my $nu=interpolate_in_mass($M,0.3,-1);
    my $g=1.0/(1.0+1e-30**($lP-$nu));
    
sub interpolate_in_mass
{
    my $M=$_[0];
    my $high=$_[1]; # at M=16.3
    my $low=$_[2]; # at M=1.15
    my $log_interpolation=0;

    return 
	$log_interpolation ?
	($high-$low)/(log10(16.3)-log10(1.15)) * (log10($M)-log10(1.15)) + $low :
	($high-$low)/(16.3-1.15) * ($M-1.15) + $low;

}

    my $lPmu=$lP-$mu;

    #printf "M=%g (%g) P=%g : mu=%g sigma=%g K=%g nu=%g norm=%g\n",$Mwas,$M,$P,$mu,$sigma,$K,$nu,$distdata{Interpolation}{normconsts}{$M}{$log10Pmin};

    #print "FUNC $distdata{Izzard2012}{$M}{$log10Pmin} * (exp(- (x-$mu)**2/(2.0*$sigma*$sigma) ) + $K/MAX(0.1,$lP)) * $g;\n";

# TODO fix $lPmin to avoid RLOF

    return ($lP < $log10Pmin || $lP > 10.0) ? 0.0 : 
	$distdata{Izzard2012}{$M}{$log10Pmin} * 
    (exp(- $lPmu*$lPmu/(2.0*$sigma*$sigma) ) +
     $K/MAX(0.1,$lP)) * $g;
}

sub flatsections
{
    # distribution of flat sections
    my $opts=$_[1];
    my $x=$_[0]; # location to calculate the y value

    # normalize
    my $C=0.0;
    my $y=0.0;
    foreach my $opt (@$opts)
    {
	
	my $dC = ($$opt{max}-$$opt{min})*$$opt{height};
	#printf "add flatsection (%g-%g)*%g = %g\n",$$opt{max},$$opt{min},$$opt{height},$dC;
	$C+=$dC;
	if($x>=$$opt{min} && $x<=$$opt{max})
	{
	    $y=$$opt{height};
	    #print "Use this\n";
	}
    }
    $C=1.0/$C;
    $y*=$C;
    #printf "flatsections gives C=%g : y=%g\n",$C,$y;
    return $y;
}


sub cumulate
{
    # calculate cumulative PDF table
    my $var=shift;
    my $pdffunc=$var->{'probdist'};
    my @range=@{$var->{'range'}};
    my $n=$var->{'sample resolution'}||1000; # resolution
    my $v="\$$var->{'name'}";
    print "Make cumulated table for $v\n";

    print "Dist func $pdffunc\n";
    my %h; # hash to store table (pointer to this is returned)
    my $C=0.0; # cumulative number : should be ~1 at the end

#    $h{$range[0]}=0.0; # fix starting point: required?
# NB the code is a LOT slower with this in here, probably because most
# stars go into the "zero" bin. Leaving it out seems to make no difference,
# because stars < the first bin automatically just get dumped in the 
# first bin.

    $|=1;
    my $loop=
$binary_grid::grid_variables{'flexigrid'}{'default_vars'}.
'
    my $dv = (('.$range[1].')-('.$range[0].'))/$n;
    my $dv2=$dv*0.5;
    print "VAR \$$var->{name} ; from $range[0] to $range[1] ; dv2=$dv2\n";
    for(my '.$v.'=('.$range[0].'); '.$v.'<('.$range[1].')-$dv2*0.5;)
    {
	'.$v.'+=$dv2; # get to bin midpoint
        '.$var->{precode}.';       
        my $dpdv = '.$pdffunc.';
        #printf "PDFCALC pdffunc '.$pdffunc.' = %g where var \$$var->{name}=%g\n",$dpdv, '.$v.';
	$C += $dpdv*$dv; # cumulative variable (0 to 1)
	$h{$C}='.$v.'; # set the cumulative table
	'.$v.'+=$dv2; # next
    }
';
    print "EVAL loop =\n$loop\n";

    eval $loop;

    if($@)
    {
	print "EVAL failed: $@\n";
	kill 9,$$;
    }

    print STDOUT "PDF normalized to $C\n";

    # renormalize to *exactly 1.0* otherwise the last bin 
    # will overfill (do we need to do this at the low end?)
    map
    {
	$h{$_/$C}=delete $h{$_};
    }keys %h;
    
    # return pointer to hash and normalization constant
    print "Return hash pointer\n";
    return \%h,$C;
}

sub bastard_distribution
{
    # the bastard child of distribution functions : 
    # designed as a drop-in replacement for the flexigrid
    
    # based on Duchene and Kraus 2013 (arXiv1303.3028)
    # but with Rob's hybrid period distribution function
     
    # first arg is the options
    my $options=shift;

    if(ref $options eq 'binary_grid2')
    {
        # use OO version
        return bastard_distribution2($options,@_);
    }

    # options are: (default in brackets)
    # binary_fraction_shift : shifts binary fraction (0.0)
    # mmin : minimum mass in the grid (0.1)
    # mmax : maximum mass in the grid (80.0)
    # m2min : minimum mass of secondary (0.1)
    # nm : number of single stars (ignored if time_adaptive is set)
    # nm1 : number of primary stars (ignored if time_adaptive is set)
    # nm2 : number of secondary stars
    # nper : number of stars in period direction
    # necc : number of stars in eccentricity direction
    # qmin : minimum allowed q on the grid
    # qmax : maximum allowed q on the grid
    # time_adaptive : use time adaptive grid (undef)
    # useecc : if true then use eccentricity (undef)
    # agbzoom : if 0 ignored, 
    #           if 1 zoom in on AGB HBB region (auto settings)
    # Kroupa2001_opts : options for Kroupa2001 call

    my $nvar=0;
    
    # set defaults
    my %defaults = (
	binary_fraction_shift=>0.0,
	mmin=>0.1,
	mmax=>80.0,
	m2min=>0.1,
	nm=>100,
	nm1=>10,
	nm2=>10,
	nper=>10,
	necc=>10,
	qmin=>0.0,
	qmax=>1.0,
	time_adaptive=>undef,
	useecc=>undef,
        agbzoom=>0,
	Kroupa2001_opts=>undef,
        metallicity=>$binary_grid::bse_options{metallicity},
	);
    
    # set options with defaults if not already set
    map
    {
	$$options{$_} //= $defaults{$_};
    }keys %defaults;


    # duplicity : takes values 0 and 1
    $binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar++}=
    {
	'name'=>'duplicity',
	'longname'=>'Duplicity',
	'range'=>[0,1],
	'resolution'=>1,
	'spacingfunc'=>'number(1.0)',
	'precode'=>'$binary_grid::grid_options{binary}=$duplicity;binary_grid::flexigrid_grid_resolution_shift();',
	'gridtype'=>'edge',
	'noprobdist'=>1,
    };
    
    ############################################################
    # Mass 1 : Kroupa 2001 IMF
    #
    # choose resolution
    # if $$options{time_adaptive} is defined, it is a hash pointer
    # with appropriate options for the time adaptive grid

    my $m1_precode=  '$m1=exp($lnm1);  $eccentricity=0.0; my $binary_fraction=rob_misc::clamp(distribution_functions::raghavan2010_binary_fraction($m1)+'.$$options{binary_fraction_shift}.',0.0,1.0);  $binary_grid::grid_options{weight}= $binary_grid::grid_options{binary}==0 ? 1.0-$binary_fraction : $binary_fraction;   ';

    if(defined($$options{time_adaptive}))
    {
	my $o=$$options{time_adaptive}; # annoying to type a lot!
	my %odefaults=(
	    mass_grid_log10_time=>1,
	    mass_grid_log10_step=>0.05,
	    mass_grid_step=>100,
	    extra_flash_resolution=>0,
	    mass_grid_nlow_mass_stars=>10,
	    debugging_output_directory=>undef,
	    max_delta_m=>2.0,
	    savegrid=>undef,
	    vb=>0,
	    mmin=>$$options{mmin},
	    mmax=>$$options{mmax},
	    max_evolution_time=>20000,
	    stellar_lifetime_table_nm=>400,
	    nthreads=>1,
	    thread_sleep=>1,
            metallicity=>$binary_grid::bse_options{metallicity},
	    );

	# enforce defaults if nothing given for a hash key
	map
	{
	    print "time_adaptive key check $_\n";
	    print "Set time_adaptive_mass_grid option $_ to $odefaults{$_}\n" if(!defined $$o{$_} && defined $odefaults{$_});
	    $$o{$_} //= $odefaults{$_};
	}keys %odefaults;


	no warnings;
	$binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar}=
	{
	    # const_dt spacing function options
	    'preloopcode'=>"
my \$const_dt_opts=\{
    max_evolution_time=>$$o{max_evolution_time},
    stellar_lifetime_table_nm=>$$o{stellar_lifetime_table_nm},
    nthreads=>$$o{nthreads},
    thread_sleep=>$$o{thread_sleep},
    mmin=>$$o{mmin},
    mmax=>$$o{mmax},
    use_grid_mass_limits_in_lifetime_table=>0,
    time_adaptive_mass_grid_log10_time=>$$o{mass_grid_log10_time},
    time_adaptive_mass_grid_log10_step=>$$o{mass_grid_log10_step},
    time_adaptive_mass_grid_step=>$$o{mass_grid_step},
    extra_flash_resolution=>$$o{extra_flash_resolution},
    time_adaptive_mass_grid_nlow_mass_stars=>$$o{mass_grid_nlow_mass_stars},
    debugging_output_directory=>\"$$o{debugging_output_directory}\",
    max_delta_m=>$$o{max_delta_m},
    savegrid=>$$o{savegrid},
    vb=>$$o{vb},
    metallicity=>$$o{metallicity},
\};
spacing_functions::const_dt(\$const_dt_opts,'reset');
",
	    # use const_dt function
	    'spacingfunc'=>"const_dt(\$const_dt_opts,'next')",

	    # and its resolution
	    'resolution'=> "spacing_functions::const_dt(\$const_dt_opts,'resolution');",
	    'precode'=>$m1_precode
	};
	use warnings;
    }
    else
    {
	# log-spaced m1
	$binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar}=
	{ 
	    'resolution'=> "$$options{nm1}", # ignore single stars in approximate resolution calculation
	    'spacingfunc'=>"const(log($$options{mmin}),log($$options{mmax}),\$binary_grid::grid_options{binary} ? $$options{nm1} : $$options{nm})",
	    'precode'=>$m1_precode
	};
    }   

    # options common to both primary mass distributions
    $binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar}{name} = 'lnm1';
    $binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar}{longname} = 'Primary mass';
    $binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar}{range}=["log($$options{mmin})","log($$options{mmax})"];
    $binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar}{dphasevol} = '$dlnm1';

	# probdist is an array:
	# first is the subroutine name, second is the opts
    
    if(defined($$options{mass_dist_opts}))
    {
	my $mass_dist_opts=Dumper($$options{mass_dist_opts});
	$mass_dist_opts=~s/\$VAR1\s+=\s+//o; 
	$mass_dist_opts=~s/\n//og;
	$mass_dist_opts=~s/;$//go;
	$binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar}{probdist} = "Kroupa2001(\$m1,$mass_dist_opts)*\$m1";
    }
    else
    {
	$binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar}{probdist} = "Kroupa2001(\$m1)*\$m1";
    }
    
    #'probdist'=>'powerlaw(0.8,100.0,-2.35,$m1)*$m1', # salpeter (0.8-100)
    #'probdist'=>"ktg93(\$m1)*\$m1", # KTG93 (all masses)
    $nvar++;

############################################################
# Binary stars: Mass 2
# Mass 2 follows a q=M2/M1<1 distribution
# with a given power law, taken from tabular interpolation

    my %qtable = ();

# use a q powerlaw with slope $qgamma
    $binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar++}=
    {
	'condition'=>'$binary_grid::grid_options{binary}==1',
	'name'=>'q',
	'longname'=>'Mass ratio',
	'range'=>[$$options{m2min}.'/$m1',1.0],
	'resolution'=>$$options{nm2},
	'spacingfunc'=>'const('.$$options{m2min}.'/$m1,1.0,'.$$options{nm2}.')',
	'probdist'=>"powerlaw($$options{m2min}/\$m1,1.0,\$qgamma,\$q)",
# calculate qgamma from lookup table
	'precode'=>'
my %qhash=(0.1=>0.4,
           0.3=>0.4,
           1.0=>0.3,
           3.25=>-0.5,
           16.0=>-0.1);
my @qtable=sort {$a<=>$b} keys %qhash;
$m2=$q*$m1; 
my $qgamma = generic_interpolation_wrapper($q,\@qtable,\%qhash);',
	'dphasevol'=>'$dq',
    };

    my $perspacing;
    if(defined $$options{agbzoom})
    {
        no warnings;
        if($$options{agbzoom}==0)
        {
            # no agbzoom
            $perspacing = "const(\$log10permin,10.0,$$options{nper})";
        }
        elsif($$options{agbzoom}==1)
        {
            # default agbzoom
            $perspacing = 'agbzoom({
vb=>0,
logperiod=>$log10per//$log10permin,
logpermin=>$log10permin, # set by RLOF minimum
logpermax=>12.0, # DM91 log Period maximum
dlogperiod=>1.0,
m1=>$m1, 
m2=>$m2,
metallicity=>$binary_grid::bse_options{metallicity},
resolution=>'.$$options{nper}.',
m1hbbmin=>3.0,
m1hbbmax=>8.0,
zoomfactor=>0.1
})';
        }
        elsif($$options{agbzoom}!=0)
        {
            # custom agbzoom
            $perspacing = $$options{agbzoom};
        }
        use warnings;
    }
    else
    {
        $perspacing = "const(\$log10permin,10.0,$$options{nper})";
    }
    
    #print "PERSPACING $perspacing\n";exit;

    # period distribution : Rob's bastard
    $binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar++}=
    {
	'name'=>'log10per',
	'longname'=>'log10(Orbital_Period)',
	'range'=>['$log10permin','10.0'],
	'resolution'=>$$options{nper},
	'spacingfunc'=>$perspacing,
	'preloopcode'=>'my $log10permin = POSIX::log10(binary_grid::minimum_period_for_RLOF($m1,$m2));',
	'precode'=>"my \$per=10.0**\$log10per;my\$sep=calc_sep_from_period(\$m1,\$m2,\$per);",
	'probdist'=>'Izzard2012_period_distribution($per,$m1,$log10permin)',
	'dphasevol'=>'('.value_of_log10.'*$dlog10per)',
    };

    if($$options{useecc})
    {
	# eccentricity: flat distribution
	$binary_grid::grid_options{'flexigrid'}{'grid variable '.$nvar++}=
	{
	    'name'=>'ecc',
	    'longname'=>'log10(Eccentricity)',
	    'range'=>[0,0.999],
	    'resolution',$$options{necc},
	    'spacingfunc',"const(0.0,0.999,$$options{necc})",
	    'precode','$eccentricity=$ecc;',
	    # gaussian as in Meiborn & Mathieu 2005
	    'probdist'=>'gaussian($ecc,0.38,0.23,0.0,0.999)',
	    # or flat (more likely)
	    #'probdist'=>'const(0.0,0.999)',
	    'dphasevol'=>'$decc'
	};
    }

    return $nvar;

}


sub bastard_distribution2
{
    my $self = shift;

    # the bastard child of distribution functions : 
    # designed as a drop-in replacement for the flexigrid
    
    # based on Duchene and Kraus 2013 (arXiv1303.3028)
    # but with Rob's hybrid period distribution function
    
    # Object-oriented version for binary_grid2

    # first arg is the options
    my $options = shift;

    # options are: (default in brackets)
    # binary_fraction_shift : shifts binary fraction (0.0)
    # mmin : minimum mass in the grid (0.1)
    # mmax : maximum mass in the grid (80.0)
    # m2min : minimum mass of secondary (0.1)
    # nm : number of single stars (ignored if time_adaptive is set)
    # nm1 : number of primary stars (ignored if time_adaptive is set)
    # nm2 : number of secondary stars
    # nper : number of stars in period direction
    # necc : number of stars in eccentricity direction
    # qmin : minimum allowed q on the grid
    # qmax : maximum allowed q on the grid
    # time_adaptive : use time adaptive grid (undef)
    # useecc : if true then use eccentricity (undef)
    # agbzoom : if 0 ignored, 
    #           if 1 zoom in on AGB HBB region (auto settings)
    # Kroupa2001_opts : options for Kroupa2001 call

    # set defaults
    my %defaults = (
	binary_fraction_shift=>0.0,
	mmin=>0.1,
	mmax=>80.0,
	m2min=>0.1,
	nm=>100,
	nm1=>10,
	nm2=>10,
	nper=>10,
	necc=>10,
	qmin=>0.0,
	qmax=>1.0,
	time_adaptive=>undef,
	useecc=>undef,
        agbzoom=>0,
	Kroupa2001_opts=>undef,
	);
    
    # set options with defaults if not already set
    map
    {
	$$options{$_} //= $defaults{$_};
    }keys %defaults;


    # duplicity : takes values 0 and 1
    $self->add_grid_variable(
	'name'=>'duplicity',
	'longname'=>'Duplicity',
	'range'=>[0,1],
	'resolution'=>1,
	'spacingfunc'=>'number(1.0)',
	'precode'=>'$self->{_grid_options}{binary}=$duplicity;$self->flexigrid_grid_resolution_shift();',
	'gridtype'=>'edge',
	'noprobdist'=>1,
        );
    
    ############################################################
    # Mass 1 : Kroupa 2001 IMF
    #
    # choose resolution
    # if $$options{time_adaptive} is defined, it is a hash pointer
    # with appropriate options for the time adaptive grid

    my $m1_precode=  '$m1=exp($lnm1);  $eccentricity=0.0; my $binary_fraction=rob_misc::clamp(distribution_functions::raghavan2010_binary_fraction($m1)+'.$$options{binary_fraction_shift}.',0.0,1.0);  $self->{_grid_options}{weight}= $self->{_grid_options}{binary}==0 ? 1.0-$binary_fraction : $binary_fraction;   ';

    my %m1opts;
    if(defined($$options{time_adaptive}))
    {
	my $o=$$options{time_adaptive}; # annoying to type a lot!
	my %odefaults=(
            metallicity=>$self->{_bse_options}->{metallicity},
	    mass_grid_log10_time=>1,
	    mass_grid_log10_step=>0.05,
	    mass_grid_step=>100,
	    extra_flash_resolution=>0,
	    mass_grid_nlow_mass_stars=>10,
	    debugging_output_directory=>undef,
	    max_delta_m=>2.0,
	    savegrid=>undef,
	    vb=>0,
	    mmin=>$$options{mmin},
	    mmax=>$$options{mmax},
	    max_evolution_time=>20000,
	    stellar_lifetime_table_nm=>400,
	    nthreads=>1,
	    thread_sleep=>1
	    );

	# enforce defaults if nothing given for a hash key
	map
	{
	    print "time_adaptive key check $_\n";
	    print "Set time_adaptive_mass_grid option $_ to $odefaults{$_}\n" if(!defined $$o{$_} && defined $odefaults{$_});
	    $$o{$_} //= $odefaults{$_};
	}keys %odefaults;


	no warnings;

        %m1opts = (%m1opts ,
                   # const_dt spacing function options
                   preloopcode => "
my \$const_dt_opts=\{
    max_evolution_time=>$$o{max_evolution_time},
    stellar_lifetime_table_nm=>$$o{stellar_lifetime_table_nm},
    nthreads=>$$o{nthreads},
    thread_sleep=>$$o{thread_sleep},
    mmin=>$$o{mmin},
    mmax=>$$o{mmax},
    use_grid_mass_limits_in_lifetime_table=>0,
    time_adaptive_mass_grid_log10_time=>$$o{mass_grid_log10_time},
    time_adaptive_mass_grid_log10_step=>$$o{mass_grid_log10_step},
    time_adaptive_mass_grid_step=>$$o{mass_grid_step},
    extra_flash_resolution=>$$o{extra_flash_resolution},
    time_adaptive_mass_grid_nlow_mass_stars=>$$o{mass_grid_nlow_mass_stars},
    debugging_output_directory=>\"$$o{debugging_output_directory}\",
    max_delta_m=>$$o{max_delta_m},
    savegrid=>$$o{savegrid},
    vb=>$$o{vb},
    metallicity=>$$o{metallicity},
\};
spacing_functions::const_dt(\$const_dt_opts,'reset');
",
                   # use const_dt function
                   'spacingfunc' => "const_dt(\$const_dt_opts,'next')",
                   
                   # and its resolution
                   'resolution'=> "spacing_functions::const_dt(\$const_dt_opts,'resolution');",
                   'precode'=>$m1_precode
            );
	use warnings;
    }
    else
    {
	# log-spaced m1
	%m1opts = (%m1opts,
	    'resolution'=> "$$options{nm1}", # ignore single stars in approximate resolution calculation
	    'spacingfunc'=>"const(log($$options{mmin}),log($$options{mmax}),\$self->{_grid_options}{binary} ? $$options{nm1} : $$options{nm})",
	    'precode'=>$m1_precode
            );
    }   

    # options common to both primary mass distributions
    %m1opts = (%m1opts,
        name => 'lnm1',
        longname => 'Primary mass',
        range => ["log($$options{mmin})","log($$options{mmax})"],
        dphasevol => '$dlnm1'
        );

    # probdist is an array:
    # first is the subroutine name, second is the opts
    
    if(defined($$options{mass_dist_opts}))
    {
	my $mass_dist_opts=Dumper($$options{mass_dist_opts});
	$mass_dist_opts=~s/\$VAR1\s+=\s+//o; 
	$mass_dist_opts=~s/\n//og;
	$mass_dist_opts=~s/;$//go;
	$m1opts{probdist} = "Kroupa2001(\$m1,$mass_dist_opts)*\$m1";
    }
    else
    {
	$m1opts{probdist} = "Kroupa2001(\$m1)*\$m1";
    }
    
    #'probdist'=>'powerlaw(0.8,100.0,-2.35,$m1)*$m1', # salpeter (0.8-100)
    #'probdist'=>"ktg93(\$m1)*\$m1", # KTG93 (all masses)

    $self->add_grid_variable(%m1opts);

############################################################
# Binary stars: Mass 2
# Mass 2 follows a q=M2/M1<1 distribution
# with a given power law, taken from tabular interpolation

    my %qtable = ();

# use a q powerlaw with slope $qgamma
    $self->add_grid_variable(
    	'condition'=>'$self->{_grid_options}{binary}==1',
	'name'=>'q',
	'longname'=>'Mass ratio',
	'range'=>[$$options{m2min}.'/$m1',1.0],
	'resolution'=>$$options{nm2},
	'spacingfunc'=>'const('.$$options{m2min}.'/$m1,1.0,'.$$options{nm2}.')',
	'probdist'=>"powerlaw($$options{m2min}/\$m1,1.0,\$qgamma,\$q)",
# calculate qgamma from lookup table
	'precode'=>'
my %qhash=(0.1=>0.4,
           0.3=>0.4,
           1.0=>0.3,
           3.25=>-0.5,
           16.0=>-0.1);
my @qtable=sort {$a<=>$b} keys %qhash;
$m2=$q*$m1; 
my $qgamma = generic_interpolation_wrapper($q,\@qtable,\%qhash);',
	'dphasevol'=>'$dq',
        );

    my $perspacing;
    if(defined $$options{agbzoom})
    {
        no warnings;
        if($$options{agbzoom}==0)
        {
            # no agbzoom
            $perspacing = "const(\$log10permin,10.0,$$options{nper})";
        }
        elsif($$options{agbzoom}==1)
        {
            # default agbzoom
            $perspacing = 'agbzoom({
vb=>0,
logperiod=>$log10per//$log10permin,
logpermin=>$log10permin, # set by RLOF minimum
logpermax=>12.0, # DM91 log Period maximum
dlogperiod=>1.0,
m1=>$m1, 
m2=>$m2,
metallicity=>$self->{_bse_options}{metallicity},
resolution=>'.$$options{nper}.',
m1hbbmin=>3.0,
m1hbbmax=>8.0,
zoomfactor=>0.1
})';
        }
        elsif($$options{agbzoom}!=0)
        {
            # custom agbzoom
            $perspacing = $$options{agbzoom};
        }
        use warnings;
    }
    else
    {
        $perspacing = "const(\$log10permin,10.0,$$options{nper})";
    }
    
    #print "PERSPACING $perspacing\n";exit;

    # period distribution : Rob's bastard
    $self->add_grid_variable(
	'name'=>'log10per',
	'longname'=>'log10(Orbital_Period)',
	'range'=>['$log10permin','10.0'],
	'resolution'=>$$options{nper},
	'spacingfunc'=>$perspacing,
	'preloopcode'=>'my $log10permin = POSIX::log10($self->minimum_period_for_RLOF($m1,$m2));',
	'precode'=>"my \$per=10.0**\$log10per;my\$sep=calc_sep_from_period(\$m1,\$m2,\$per);",
	'probdist'=>'Izzard2012_period_distribution($per,$m1,$log10permin)',
	'dphasevol'=>'('.value_of_log10.'*$dlog10per)',
        );

    if($$options{useecc})
    {
	# eccentricity: flat distribution
	$self->add_grid_variable(
	    'name'=>'ecc',
	    'longname'=>'log10(Eccentricity)',
	    'range'=>[0,0.999],
	    'resolution',$$options{necc},
	    'spacingfunc',"const(0.0,0.999,$$options{necc})",
	    'precode','$eccentricity=$ecc;',
	    # gaussian as in Meiborn & Mathieu 2005
	    'probdist'=>'gaussian($ecc,0.38,0.23,0.0,0.999)',
	    # or flat (more likely)
	    #'probdist'=>'const(0.0,0.999)',
	    'dphasevol'=>'$decc'
            );
    }
}




sub dep_log10_minimum_period_without_RLOF
{
    # use evolution code to determine the minimum period for which
    # RLOF does NOT occur. Returns log10 of that period.
    no warnings;

    # NB evcode must have appropriate functionality coded in! 

    my ($m1,$m2) = @_; # input variables are m1,m2
    state %cache;
    if(!defined $cache{$m1}{$m2})
    {
	# The problem is that this is run in the main code context, not the thread
	# context. Thus we have to launch a new binary_c to do the work.
	my $system={
            M_1=>$m1,
            M_2=>$m2,
            metallicity=>$binary_grid::bse_options{metallicity},
            orbital_period=>0.0,
            eccentricity=>0.0,
            probability=>1.0,
            phasevol=>1.0,
            postargs=>{minimum_orbital_period_for_instant_RLOF=>1}
	};

	my $args = binary_grid::make_evcode_arghash($system);
	my $command = binary_grid::evcode_command_string({nobatch=>1}).' '.
	    join(' ', map{$binary_grid::grid_options{arg_prefix}.$_.' '.$$args{$_}}(keys %$args));

        print "Running:\n$command\n" if($binary_grid::grid_options{vb}>1);
        my $result = `$command`;
        my $err=0;
        my $minper;

        # check the command succeeded
        if(defined $result && $result &&
           ($result=~/MINIMUM PERIOD (\S+)/)[0])
        {
            $minper = $1;

            # check for further failure
            if(!defined $minper ||
               $minper=~/(inf|nan)/i ||
               $minper <= 1e-10)
            {
                $err=2;
            }
            else
            {
                $cache{$m1}{$m2} = POSIX::log10($minper);
            }
        }
        else
        {
            $err=1;
        }

        if($err || !defined $cache{$m1}{$m2})
        {
            say STDOUT "MINIMUM PERIOD failed for '$command' (minper=$minper, err=$err)\nResult was \"$result\"";
            say STDERR "MINIMUM PERIOD failed for '$command' (minper=$minper, err=$err)\nResult was \"$result\"";

            foreach my $c ('ps aux','top -n1','free -m')
            {
                my $cc = "$c 2>\&1";
                my $r = `$cc`;
                say STDOUT "$c:\n\"$r\"";
                say STDERR "$c:\n\"$r\"";
            }

            printf "Cache size %s\n",total_size(%cache);
            printf "Results hash size %s\n",total_size($main::results);

            my @threads = @{$binary_grid::flexigrid{threads}};
            printf "Threads: @threads\n";
            sleep 1;

            # KILL the grid
            shutdown_binary_grid(9);
            exit;
        }
    }
    print "Minimum period without RLOF (@_) m1=$m1 m2=$m2 -> $cache{$m1}{$m2}\n" if($binary_grid::grid_options{vb} >1);
    return $cache{$m1}{$m2};
    use warnings;
}


sub dep_log10_minimum_separation_without_RLOF
{
    # use evolution code to determine the minimum separation for which
    # RLOF does NOT occur. Returns log10 of that separation.

    # NB evcode must have appropriate functionality coded in! 

    my ($m1,$m2) = @_; # input variables are m1,m2
    state %cache;
    if(!defined $cache{$m1}{$m2})
    {
	# The problem is that this is run in the main code context, not the thread
	# context. Thus we have to launch a new binary_c to do the work.
	my $system={M_1=>$m1,
		    M_2=>$m2,
		    metallicity=>$binary_grid::bse_options{metallicity},
		    separation=>0.0,
		    eccentricity=>0.0,
		    probability=>1.0,
		    phasevol=>1.0,
		    postargs=>{minimum_separation_for_instant_RLOF=>1}
	};

	my $args = binary_grid::make_evcode_arghash($system);
	no warnings;
	my $command = binary_grid::evcode_command_string({nobatch=>1}).' '.
	    join(' ', map{$binary_grid::grid_options{arg_prefix}.$_.' '.$$args{$_}}(keys %$args));
	use warnings;
	$cache{$m1}{$m2} = POSIX::log10((`$command`=~/MINIMUM SEPARATION (\S+)/)[0]);
    }
    print "Minimum log10 separation without RLOF (@_) m1=$m1 m2=$m2 -> $cache{$m1}{$m2}\n" if($binary_grid::grid_options{vb} >1);
    return $cache{$m1}{$m2};
}


sub dep_ln_minimum_separation_without_RLOF
{
    # use evolution code to determine the minimum separation for which
    # RLOF does NOT occur. Returns ln (log) of that separation.

    # NB evcode must have appropriate functionality coded in! 

    my ($m1,$m2) = @_; # input variables are m1,m2
    state %cache;
    if(!defined $cache{$m1}{$m2})
    {
	# The problem is that this is run in the main code context, not the thread
	# context. Thus we have to launch a new binary_c to do the work.
	my $system={M_1=>$m1,
		    M_2=>$m2,
		    metallicity=>$binary_grid::bse_options{metallicity},
		    separation=>0.0,
		    eccentricity=>0.0,
		    probability=>1.0,
		    phasevol=>1.0,
		    postargs=>{minimum_separation_for_instant_RLOF=>1}
	};

	my $args = binary_grid::make_evcode_arghash($system);
	no warnings;
	my $command = binary_grid::evcode_command_string({nobatch=>1}).' '.
	    join(' ', map{$binary_grid::grid_options{arg_prefix}.$_.' '.$$args{$_}}(keys %$args));
	use warnings;
	$cache{$m1}{$m2} = log((`$command`=~/MINIMUM SEPARATION (\S+)/)[0]);
    }
    print "Minimum ln separation without RLOF (@_) m1=$m1 m2=$m2 -> $cache{$m1}{$m2}\n";# if($binary_grid::grid_options{vb} >1);
    return $cache{$m1}{$m2};
}


sub sana12
{
	# distribution of initial orbital periods as found by Sana et al. (2012)
	# which is a flat distribution in ln(a) and ln(P) respectively for stars 
	# * less massive than 15Msun (no O-stars)
	# * mass ratio q=M2/M1<0.1
	# * log(P)<0.15=x0 and log(P)>3.5=x1
	# and is be given by dp/dlogP ~ (logP)^p for all other binary configurations (default p=-0.55)
	#
	# arguments are M1, M2, a, Period P, amin, amax, x0=log P0, x1=log P1, p

	my $m1 = shift;
	my $m2 = shift;
	my $a = shift;
	my $P = shift;
	my $amin = shift;
	my $amax = shift;
	my $x0 = shift;
	my $x1 = shift;
	my $p = shift;

	my $res = 0.0;

	if ($m1 < 15.0 || $m2/$m1 < 0.1) {
		$res = (1.0/(log($amax)-log($amin)));
	} else {
		my $p1 = 1.0 + $p;

		# For more details see the LyX document for this distribution where the variables and normalizations are given
		# we use the notation x=log(P), xmin=log(Pmin), x0=log(P0), ... to determine the 
		my $x = log_ln_converter*log($P);
		my $xmin = log_ln_converter*log(calc_period_from_sep($m1,$m2,$amin));
		#my $x0 = 0.15;
		#my $x1 = 3.5;
		my $xmax = log_ln_converter*log(calc_period_from_sep($m1,$m2,$amax));

		# normalization coefficients
		my $A1 = 1.0/($x0**$p*($x0-$xmin) + ($x1**$p1-$x0**$p1)/$p1 + $x1**$p*($xmax-$x1));
		my $A0 = $A1*$x0**$p;
		my $A2 = $A1*$x1**$p;

		# the factor '3.0/2.0*log_ln_converter' converts from dlogP to dlna (see the LyX document for this distribution)
		if ($x < $x0) {
			$res = 3.0/2.0*log_ln_converter*$A0;
		} elsif ($x > $x1) {
			$res = 3.0/2.0*log_ln_converter*$A2;
		} else {
			$res = 3.0/2.0*log_ln_converter*$A1*$x**$p;
		}
	}

	return $res;
}

1;

__END__


