package binary_grid2;

############################################################
#
# binary_grid2 module for population synthesis
#              with binary_c/nucsyn, BONNFIRES and
#              your stellar evolution code of choice
#
############################################################
#
# Written by Robert Izzard (rgi@ast.cam.ac.uk
#                           or via gmail: rob.izzard) 
#
# (c) 2000-2018 and onwards into the 21st century.
#
# This module contains the functions required to run
# population synthesis grids with binary_c/nucsyn, BONNFIRES or
# your stellar evolution code of choice.
#
# Full documentation is found in doc/binary_grid-flexigrid.pdf
# (or source doc/binary_grid-flexigrid.lyx)
#
############################################################
#
# Version 2.00 : Fully object-oriented interface.
#
# Version 2.01 : unchanged
#
# Version 2.02,2.03,2.04 : Updates to C.pm
#
# Version 2.1 : update version to match binary_c, 
#               new 'process_name' grid option
#
# Version v2.1.1 : add use_full_resolution capability
#
# Version v2.1.2 : add overspin_algorithm
#
# Version v2.1.3 : sync to binary_c 2.1.3, remove log compression
#
# Version v2.1.4 : update for Meson build of binary_c
#
# Version v2.1.5 : update with per-variable monte carlo options
#                  clean up the grid generation code somewhat.
#                  Add "say state" features for perl 5.31.6 bug.
#
############################################################
#
our $VERSION = 'v2.1.5';
#
############################################################

$|=1; # autoflush (for log output)

############################################################
# Perl modules:
#
# The best way to install all the dependencies is
#
# 1) Install the latest version of perl with perlbrew
# http://perlbrew.pl/ 
#
# 2) Install cpanm
# http://perlbrew.pl/Perlbrew-and-Friends.html 
#
# 3) install modules on the command line, e.g. with
# 'cpanm <module_name>'
#
# Note that many of the required modules are part of the
# standard perl installation.
#
# You should ALSO install these with cpanm,
# e.g.
#
# cpanm xyz.tar.gz
#
# or, given a URL from the cpan website,
#
# cpanm <URL>
#
# or aim directly at the appropriate module directory
#
# cpanm xyz/
#
############################################################
use 5.16.0;
use common::sense;
use feature qw(state say);
use threads;
use threads::shared;
use Thread::Queue; # thread queue
use AutoLoader;
use Carp qw(confess); # use confess instead of die
use Carp::Always;
use Carp::Always::Color;
$Carp::MaxEvalLen=0; # show all of a failed eval
$Carp::MaxArgLen=0; # show all of a failed arg string
$Carp::MaxArgNums=0; # show all failed args
local $SIG{__DIE__} = sub{ confess @_ }; # force die signal to use confess()
use Clone qw(clone); # for deep copying hashes
use Config; # for signals and C build flags
use IO::File;
use Sort::Key qw(nsort); # fastest perl sorter
use Sys::Hostname qw(hostname); # reliable way to get hostnames
use Time::HiRes qw(sleep gettimeofday tv_interval usleep); # for timers
use Cwd qw(getcwd); # for the current working directory
use POSIX 'setsid'; # for signals
#use Hook::LexWrap; # for pre/post subroutine timers
use Data::Dumper; # for saving hashes
$Data::Dumper::Sortkeys = 1 ;
use Data::Serializer; # for saving hashes
use Data::Serializer::Raw; # for saving hashes
use Data::Serializer::RobJSON;
#use Compress::Zlib; # for saving compressed hashes and decompressing 
                     # compressed binary_c output
use File::Type; # used to detect bzip2 files
use File::Copy;
use Clone qw(clone); # used to copy objects
use Sub::Identify qw/sub_fullname/;
use Term::ANSIColor qw/color/;

############################################################
# import Rob's modules
############################################################
use distribution_functions;
use spacing_functions;
use binary_stars qw(calc_period_from_sep calc_sep_from_period rzams roche_lobe);
use rob_misc qw(slurp dumpfile operating_system MAX MIN get_from_C_header nfscheck random_string thread_log sum_array_with_pointers trem2 thetime mem_usage is_numeric proc_mem_usage versionofmodule gnuplot_prettytitle);
use Hash::RobMerge qw(arithmetically_add_hashes);
use RobInterpolation qw(generic_interpolation_wrapper);
#RobInterpolation::useC(1); # use C extensions if we can : currently broken!

############################################################
# export nothing
############################################################
require Exporter;
our @ISA = qw(Exporter);
my @funcs= qw(  );
our %EXPORT_TAGS = ( 'all' => [ @funcs ] );
our @EXPORT_OK = ( @{ $EXPORT_TAGS{all} } );
our @EXPORT = @funcs;
use vars qw(@ISA @EXPORT @EXPORT_OK $backend $default_vars);
$default_vars = '
    my $m1;
    my $m2;
    my $sep;
    my $per;
    my $eccentricity;
';
############################################################
# Constants : used for logging only. 
#             binary_c has its own constants which you
#             should extract for any scientific work.
############################################################
use constant secs_per_day => (86400.0);
use constant Mega => (1024*1024);

############################################################
# Logging stuff
############################################################
my %colour = rob_misc::colourhash();
my @next_alarm_time;

############################################################

############################################################
#
# binary_grid2 object constructor and its helper functions
#
############################################################

sub new
{
    # make a new binary_grid2 object
    my ($class, @args) = @_;
    $class //= __PACKAGE__;
    my $self = {};
    bless $self,$class;
    $self->initialise(@args);
    return $self;
}

sub initialise
{
    my $self = shift;
    my @args = @_;

    # initialise new binary_grid2 object
    %$self = (
        _grid_options => {},
        _bse_options => {},
        _flexigrid => {},
        _threadinfo => {},
        _results => {},
        _timers => {},
        _is_binary_grid_object => 1,
        _priority_args => [qw(M_1 M_2 eccentricity orbital_period metallicity max_evolution_time probability)],
        );

    # set up _grid_options and _bse_options to default values
    $self->set_defaults();

    # apply given args
    $self->set(@args);
    
    return $self;
}

# destructors (formally required)
sub destroy
{
    my $self = shift;
    $self = undef;
}

sub DESTROY
{
    my $self = shift;
    $self->destroy(@_);
}

sub results
{
    # return the results hash
    my $self = shift;
    return $self->{_results};
}

############################################################
# AUTOLOAD to handle unknown attributes
############################################################
our $AUTOLOAD;
sub AUTOLOAD
{
    my $self = shift;
    # if attribute name is actually a value in the bse or grid
    # options hash, get it (with get) and return it
    # otherwise returns undef
    my $attr = $AUTOLOAD =~ s/.*\:\://r;
    print STDERR "AUTOLOAD get $attr\n";
    print STDERR "caller: ",caller(1),"\n";
    return $self->get($attr);
}


############################################################
#
# Code require (and require_not) commands.
#
# These take a string argument which is interpolated as a
# regexp and matched against the version string.
#
# On failure, exit. 
#
# If required, you can eval the call and hence not exit, 
# but usually you want to exit and fix the problem.
#
# NB you can pass a list of strings and each will be tested.
#
# The require_on and require_off are similar.
#
# require_off requires that the macro is explicitly "off"
# in the version string.
#
# require on will match anything except "is off".
#
############################################################

sub require
{
    my $self = shift;
    my $v = $self->evcode_version_string();
    foreach my $regexp_string (@_)
    {
        if($v !~ /$regexp_string/)
        {
            print STDERR "Evcode failed test '$regexp_string'\n";
            exit;
        }
    }
}

sub require_on
{
    my $self = shift;
    my $v = $self->evcode_version_string();
    foreach my $regexp_string (@_)
    {
        if($v =~ /$regexp_string is off/)
        {
            print STDERR "Evcode failed test for '$regexp_string is <something not off>'\n";
            exit;
        }
    }
}

sub require_not
{
    my $self = shift;
    my $v = $self->evcode_version_string();
    foreach my $regexp_string (@_)
    {
        if($v =~ /$regexp_string/)
        {
            print STDERR "Evcode failed negative test '$regexp_string'\n";
            exit;
        }
    }
}

sub require_off
{
    my $self = shift;
    my $v = $self->evcode_version_string();
    foreach my $regexp_string (@_)
    {
        if($v !~ /$regexp_string is off/)
        {
            print STDERR "Evcode failed test for '$regexp_string is off'\n";
            exit;
        }
    }
}


sub require_not
{
    my $self = shift;
    my $v = $self->evcode_version_string();
    foreach my $regexp_string (@_)
    {
        if($v =~ /$regexp_string/)
        {
            print STDERR "Evcode failed negative test '$regexp_string'\n";
            exit;
        }
    }
}

############################################################
# Logging functions
############################################################


sub vb1out
{
    my $self = shift;
    ############################################################
    # verbose (thread) output for when vb>=1
    ############################################################
    my ($t,$tpr_mult) = @_;
    my $ngrid;
    state $nkeys //= scalar keys %{$self->{_flexigrid}{resolution}{n}};

    # keep the last ~ ten calls in the @stats arrays
    state @statsdn;
    state @statsdt;
    
    # estimate current grid resolution : nb can change during the run
    $ngrid = $self->flexigrid_grid_resolution();

    # set start time (if required)
    $self->{_flexigrid}{tstart} //= [gettimeofday]; 
        
    # estimate time remaining, time of finishing
    if(!defined $self->{_flexigrid}->{tprev_vb1out})
    {
	$self->{_flexigrid}->{tprev_vb1out} = $self->{_flexigrid}->{tstart};
	$self->{_flexigrid}->{nprev_vb1out} = 0 ;
    }

    my $now = [gettimeofday];
    
    # calculate number of stars run since previous call
    my $dn = $self->{_flexigrid}->{count} - $self->{_flexigrid}->{nprev_vb1out};

    # calculate time
    my $dt = (defined $self->{_flexigrid}->{tprev_vb1out} && 
              ref $self->{_flexigrid}->{tprev_vb1out} eq 'ARRAY')  ? 
        tv_interval($self->{_flexigrid}->{tprev_vb1out},$now) : 0.0;
    
    # push the number of stars and the time it took to run them 
    # onto the stats array
    push(@statsdn,$dn);
    push(@statsdt,$dt);
    
    # only keep the last 50 data points for time estimates
    if($#statsdn>50)
    {
        @statsdn = splice(@statsdn,-50);
        @statsdt = splice(@statsdt,-50);
    }

    # calculate time and number for the last 50 calls
    $dt = rob_misc::sum_array_with_pointers(\@statsdt);
    $dn = rob_misc::sum_array_with_pointers(\@statsdn);

    # hence estimate the time remaining
    my ($eta,$units,$tpr,$eta_secs) = 
        rob_misc::trem2($dt,
                        $self->{_flexigrid}->{count},
                        $dn,
                        $ngrid);

    # save time per run
    $self->{_flexigrid}->{time_per_run} = $tpr;

    # save timing stats for next time
    $self->{_flexigrid}->{tprev_vb1out} = $now;
    $self->{_flexigrid}->{nprev_vb1out} = $self->{_flexigrid}->{count};

    $tpr *= $tpr_mult if(defined($tpr_mult));

    my $etf;
    if($eta_secs < secs_per_day)
    {
	$etf = rob_misc::thetime(int($eta_secs));
    }
    else
    {
 	my $d = int($eta_secs/secs_per_day);
	$etf = $d==1 ? 'Tomorrow' : "In $d days";
    }
    chomp $etf;

    state $hostname = hostname();

    my $pstring=' ';
    
    {
	# modulo
	my $modulo = 
	    ($self->{_grid_options}{modulo}!=1 ||
	     $self->{_grid_options}{offset}) ? 
	    (sprintf '%%%d+%d',
	     $self->{_grid_options}{modulo},
	     $self->{_grid_options}{offset}) : '';

	my $vb1string=' % '.(length($ngrid)+1).'d/%d%s %s%s%s %s -> ';
	
	# output to the terminal (stdout)
	my $s=sprintf " %s$vb1string%s%s % 5.1f %% complete%s ETA=% 7.1f $units  tpr=%2.2e %sETF=%s %s %s pid=$$ mem:grd=%d fam=%d (kids:%3.2f%%)  %s %s",
	$colour{"yellow bold on_black"},
	$self->{_flexigrid}->{count},$ngrid,$modulo,
	$colour{green},
        rob_misc::thetime(),
	$colour{yellow},$t,
	$colour{'blue on_white'},$pstring,100.0*$self->{_flexigrid}->{count}/$ngrid,
	$colour{'bold red on_black'},$eta,$tpr,
	$colour{blue},$etf,
	$colour{magenta},
	defined($self->{_grid_options}{evcode_pid}) ? 'pid='.$self->{_grid_options}{evcode_pid}.' ' : ' ',
	$self->binary_grid_mem_usage(),
	$self->binary_grid_family_mem_usage(),
	100.0 * $self->frac_mem_usage(),

        # job id on condor/slurm
	$self->{_grid_options}{condor_jobid} ne '' 
            ? $colour{magenta}."\@$hostname (job=$self->{_grid_options}{condor_jobid}) " 
            : $self->{_grid_options}{slurm_jobid} ne ''
            ? $colour{magenta}."\@$hostname (job=$self->{_grid_options}{slurm_jobid}) " :
            '',
 
	$colour{reset};

	$s=~s/weight=1//; # trivial if true
	$s=~s/\(kids\:0.00\%\)//;
	$s=~s/\s+/ /g; # save space

	$self->thread_log($s,$self->{_threadinfo}{thread_number});
	$self->thread_vb($s) if($self->verbose_output(3));
    }

    $self->{_flexigrid}->{nextlogtime} = time() +
        $self->{_grid_options}{log_dt_secs};
}

sub vbout
{
    my ($self,$level) = (shift,shift);
    
    # verbose output function : outputs a string IF and ONLY IF
    # the 'verbose_output' function returns true.

    # $level is constant throughout, so we can cache the results
    # of calling verbose_output() : this is very fast.
    #
    # In vbcache we store the prefix for the output string, which
    # is always boolean 'true', or 0, which is boolean 'false'.
    state %vbcache;
    $vbcache{$level} //= 
        ($self->verbose_output($level) ?
         (
          ($self->{_threadinfo}->{thread_number} ? 
           'Thread '.$self->{_threadinfo}->{thread_number} : 
           '').': ')  
          : 0);

    if($vbcache{$level})
    {
        state $lockv : shared;
        {
            lock $lockv;
            print @_;
        }
    }
}

sub verbose_output
{
    # return true if we are verbose enough
    my ($self,$level) = @_;
    return $self->{_grid_options}->{vb} >= $level ? 1 : 0;
}

############################################################
# Memory use and accounting
############################################################

sub frac_mem_usage
{
    my $self = shift;
    # return the fractional memory usage of all the threads
    return $self->binary_grid_children_mem_usage()/
        MAX(1e-10,$self->binary_grid_children_max_allowed_mem_usage());
}

sub binary_grid_mem_usage
{ 
    my $self = shift;
    # wrapper to choose which mem_usage function you wish to use:
    # I was using Proc::ProcessTable but had issues
    return rob_misc::mem_usage();
}

sub binary_grid_family_mem_usage
{
    my $self = shift;
    # get the memory usage for binary_grid + child threads 
    return rob_misc::mem_usage() + $self->binary_grid_children_mem_usage();
}

sub binary_grid_children_mem_usage
{ 
    my $self = shift;
    # calculate memory usage of flexigrid children
    my $m=0.0;
    foreach my $pid (@{$self->{_flexigrid}{_evcode_pids}})
    {
	$m += rob_misc::mem_usage($pid);
    }
    return $m;
}

sub binary_grid_children_max_allowed_mem_usage
{
    my $self = shift;
    # estimate maximum allowed memory use of flexigrid children
    return MAX(1.0,(1+$self->{_flexigrid}{nthreads}) * $self->{_grid_options}{threads_stack_size});
}

############################################################
# Resolution estimate functions
############################################################

sub flexigrid_grid_resolution
{
    my $self = shift;
    ############################################################
    # ESTIMATE the grid resolution, allowing for previous subgrids 
    # which have finished and hence should 
    ############################################################
    my $ngrid;
    
    # use full resolution from dummy run if available
    if($self->{_grid_options}->{use_full_resolution})
    {
        $ngrid = $self->{_flexigrid}{full_resolution};
    }

    # if not available, calculate using the resolution variables
    # that are possibly inaccurate
    if(!defined $ngrid)
    {
        $ngrid = 1;
        foreach my $k (keys %{$self->{_flexigrid}{resolution}{n}})
        {
            $ngrid *= $self->{_flexigrid}{resolution}{n}{$k} // 1;
        }
    }

    $self->{_flexigrid}{resolution}{previous}=$ngrid; # save for later shift
    $ngrid += $self->{_flexigrid}{resolution}{shift};
    
    return $ngrid;
}

sub flexigrid_grid_resolution_shift
{
    my $self = shift;
    # shift the resolution by the current resolution
    $self->{_flexigrid}{resolution}{shift} = $self->{_flexigrid}{resolution}{previous};
    return ($self->{_flexigrid}{resolution}{shift});
}


############################################################
# Argument parsing
############################################################

sub parse_args
{
    my $self = shift;
    $self->parse_grid_args(@_);
}

sub parse_grid_args
{
    my $self = shift;

    # parse command line arguments,
    # defaulting to main:: for unknown variables
    foreach my $arg (@ARGV)
    {
        if($arg=~/^-?-v/o)
	{
	    $self->{_grid_options}->{vb}=1;
        }
	elsif($arg=~/([^=]+)\=(.*)/o)
        {
            $self->set({default_to=>'main'},$1,$2);
	}
	# else do nothing
    }

    # rename z -> metallicity for backwards compatibility
    $self->orset('metallicity',$self->{_bse_options}->{z});
}

############################################################
# Option setting
############################################################

sub set
{
    my $self = shift;
    my %opts = (
        default_to => 'bse'
        );
    if(ref $_[0] eq 'HASH')
    {
        %opts = (%opts, %{$_[0]});
        shift @_;
    }

    # Set a grid, bse or flexigrid option given a list of key,value
    # pairs. If unknown, assumes it's a bse option.
    #
    # Returns the number of values set.
    return 0 if($#_==-1); 
    my $n = 0;

    while(my $key = shift @_)
    { 
        my $value = shift @_;
        if($key eq 'vb' && !defined $value)
        {
            # special case : vb from the command line
            $self->{_grid_options}->{vb} = ("@ARGV"=~/vb=(\d+)/)[0];
        }
        else
        {
            if(exists($self->{_bse_options}->{$key}))
            {
                $self->{_bse_options}->{$key} = $value;
                if($key eq 'z')
                {
                    $self->{_bse_options}->{metallicity} = $value;
                }
                elsif($key eq 'metallicity')
                {
                    $self->{_bse_options}->{z} = $value;
                }
                $n++;
            }
            elsif(exists($self->{_grid_options}->{$key}))
            {
                $self->{_grid_options}->{$key} = $value;
                $n++;
            } 
            elsif(exists($self->{_flexigrid}->{$key}))
            {
                $self->{_flexigrid}->{$key} = $value;
                $n++;
            }
            elsif($opts{default_to} eq 'bse')
            {
                $self->{_bse_options}->{$key} = $value;
                if($key eq 'z')
                {
                    $self->{_bse_options}->{metallicity} = $value;
                }
                elsif($key eq 'metallicity')
                {
                    $self->{_bse_options}->{z} = $value;
                }
                $n++;
            }
            elsif($opts{default_to} eq 'grid')
            {
                $self->{_grid_options}->{$key} = $value;
                $n++;
            }
            elsif($opts{default_to} eq 'main')
            {
                # default main::
                eval "\$main::$key = \$value;";
                $n++;
            }
        }
    }
    return $n;
}

sub orset
{
    my $self = shift;
    # Set a grid or bse option given a list of key,value
    # pairs.
    # Returns the number of values set.

    return 0 if($#_==-1); 
    my $n = 0;
    foreach my $key (shift @_)
    {
        my $value = shift @_;
        if(exists($self->{_bse_options}->{$key}))
        {
            $self->{_bse_options}->{$key} //= $value; 
            if($key eq 'z')
            {
                $self->{_bse_options}->{metallicity} //= $value;
            }
            elsif($key eq 'metallicity')
            {
                $self->{_bse_options}->{z} //= $value;
            }
            $n++;
        }
        elsif(exists($self->{_grid_options}->{$key}))
        {
            $self->{_grid_options}->{$key} //= $value;
            $n++;
        }
        else
        {
            print STDERR "Unclear what to do with '$key' (value is '$value') because it does not exist in either grid_options or bse_options. Possible typo or version clash? ",join(' ',caller(1)),"\n"; 
            exit 1;
        }
    }
    return $n;

}

sub get
{
    my $self = shift;
    # return a grid or bse option (list of) scalar
    my @ret;
    foreach my $key (@_)
    {
        push(@ret,
             $self->{_grid_options}->{$key} //
             $self->{_bse_options}->{$key} //
             undef);
    } 
    return @ret;
}

sub gethashval
{
    my $self = shift;
    # return a grid or bse hash value
    my ($key,$key2) = @_;
    return 
        (defined $self->{_grid_options}->{$key} && defined $self->{_grid_options}->{$key}->{$key2}) ?
        $self->{_grid_options}->{$key}->{$key2} :
        (defined $self->{_bse_options}->{$key} && defined $self->{_bse_options}->{$key}->{$key2}) ? 
        $self->{_bse_options}->{$key}->{$key2} :       
        undef;
}


############################################################
# Default setting
############################################################

sub set_defaults
{
    my $self = shift;

    ############################################################
    # Set default grid properties (in %self->{_grid_options}} 
    # and %{$self->{_bse_options}})
    # This is the first thing that should be called by the user!
    ############################################################

    # set signal handlers for timeout
    $self->set_class_signal_handlers();

    # set operating system
    my $os = rob_misc::operating_system();
    
    # set tmp dir
    my $tmp = $self->tmpdir($os);
     
    %{$self->{_grid_options}}=(

        # save operating system
	operating_system=>$os,

        # process name
        process_name => 'binary_grid'.$VERSION,
        
	# temp directory: 
	tmp=>$tmp,

	grid_defaults_set=>1, # so we know the grid_defaults function has been called

	# grid suspend files: assume binary_c by default
	suspend_files=>[$tmp.'/force_binary_c_suspend',
			'./force_binary_c_suspend'],

	snapshot_file=>$tmp.'/binary_c-snapshot',
	

	########################################
	# infomration about the running grid script
	########################################
	working_directory=>getcwd(), # the starting directory
	perlscript=>$0, # the name of the perlscript
	perlscript_arguments=>join(' ',@ARGV), # arguments as a string
	perl_executable=>$^X, # the perl executable
	command_line=>join(' ',$0,@ARGV), # full command line
	process_ID=>$$, # process ID of the main perl script

	########################################
	# GRID
	########################################

        # if undef, generate gridcode, otherwise load the gridcode
        # from this file. useful for debugging
        gridcode_from_file => undef,
        
        # assume binary_grid perl backend by default
        backend => 
        $self->{_grid_options}->{backend} // 
        $binary_grid2::backend //
        'binary_grid::Perl',

        # custom C function for output : this automatically
        # binds if a function is available.
        C_logging_code => undef,
        C_auto_logging => undef,
        custom_output_C_function_pointer => binary_c_function_bind(), 
        
	# control flow
	rungrid=>1, # usually run the grid, but can be 0
	# to skip it (e.g. for condor/slurm runs)
	merge_datafiles=>'',
	merge_datafiles_filelist=>'',

	# parameter space options
	weight=>1.0, # weighting for the probability

	repeat=>1, # number of times to repeat each system (probability is adjusted to be 1/repeat)

	binary=>0, # set to 0 for single stars, 1 for binaries     

        # if use_full_resolution is 1, then run a dummy grid to
        # calculate the resolution. this could be slow...
        use_full_resolution => 1,
        
	# the probability in any distribution must be within
	# this tolerance of 1.0, ignored if undef (if you want
	# to run *part* of the parameter space then this *must* be undef)
	probability_tolerance=>undef,

	# how to deal with a failure of the probability tolerance:
	# 0 = nothing
	# 1 = warning
	# 2 = stop
	probability_tolerance_failmode=>1,

	# add up and log system error count and probability
	add_up_system_errors=>1,
	log_system_errors=>1,

	# codes, paths, executables etc.
	
	# assume binary_c by default, and set its defaults
	code=>'binary_c',
	arg_prefix=>'--',
	prog=>'binary_c', # executable 
	nice=>'nice -n +0', # nice command
	ionice=>'',
        
	# compress output?
	binary_c_compression=>0,

        # get output as array of pre-split array refs
        return_array_refs=>1,

	# environment
	shell_environment=>undef,
	libpath=>undef, # for backwards compatibility

	# where is binary_c? need this to get the values of some counters
	rootpath=>$self->okdir($ENV{BINARY_C_ROOTPATH}) // 
	$self->okdir($ENV{HOME}.'/progs/stars/binary_c') //
	'.' , # last option is a fallback ... will fail if it doesn't exist

	srcpath=>$self->okdir($ENV{BINARY_C_SRCPATH}) // 
	$self->okdir($ENV{BINARY_C_ROOTPATH}.'/src') // 
	$self->okdir($ENV{HOME}.'/progs/stars/binary_c/src') //
	'./src' , # last option is fallback... will fail if it doesn't exist

	# stack size per thread in megabytes
	threads_stack_size=>50, 

	# thread sleep time between starting the evolution code and starting 
	# the grid
	thread_presleep=>0,

	# threads
	# Max time a thread can sit looping (with calls to tbse_line)
	# before a warning is issued : NB this does not catch real freezes,
	# just infinite loops (which still output)
	thread_max_freeze_time_before_warning=>10,
	
	# run all models by default: modulo=1, offset=0
	modulo=>1,
	offset=>0,	    

        # max number of stars on the queue
        maxq_per_thread => 100,
        
	# data dump file : undef by default (do nothing)
	results_hash_dumpfile => '',

	# compress files with bzip2 by default
 	compress_results_hash => 1,

	########################################
	# Condor stuff
	########################################	
	condor=>0, # 1 to use condor, 0 otherwise
        condor_command=>'',# condor command e.g. "run_flexigrid", 
	# "join_datafiles"
	condor_dir=>'', # working directory containing e.g.
	# scripts, output, logs (e.g. should be NFS available to all)
	condor_njobs=>'', # number of scripts
	condor_jobid=>'', # condor job id
	condor_postpone_join=>0, # if 1, data is not joined, e.g. if you
	# want to do it off the condor grid (e.g. with more RAM)
	condor_join_machine=>undef, # if defined then this is the machine on which the join command should be launched (must be sshable and not postponed)
	condor_join_pwd=>undef, # directory the join should be in
	# (defaults to $ENV{PWD} if undef)
	condor_memory=>1024, # in MB, the memory use (ImageSize) of the job
	condor_universe=>'vanilla', # usually vanilla universe
	condor_snapshot_on_kill=>0, # if 1 snapshot on SIGKILL before exit
	condor_load_from_snapshot=>0, # if 1 check for snapshot .sv file and load it if found
	condor_checkpoint_interval=>0, # checkpoint interval (seconds) 
	condor_checkpoint_stamp_times=>0, # if 1 then files are given timestamped names (warning: lots of files!), otherwise just store the lates
	condor_streams=>0, # stream stderr/stdout by default (warning: might cause heavy network load)
	condor_save_joined_file=>0, # if 1 then results/joined contains the results (useful for debugging, otherwise a lot of work)
	condor_requirements=>'', # used?

        # resubmit options : if the status of a condor script is
        # either 'finished','submitted','running' or 'crashed',
        # decide whether to resubmit it.
        # NB Normally the status is empty, e.g. on the first run.
        # These are for restarting runs.
        condor_resubmit_finished=>0,
	condor_resubmit_submitted=>0,
	condor_resubmit_running=>0,
	condor_resubmit_crashed=>0,


	########################################
	# Slurm stuff
	########################################	
        slurm=>0, # don't use slurm by default
	slurm_command=>'',# slurm command e.g. "run_flexigrid", 
	# "join_datafiles"
	slurm_dir=>'', # working directory containing e.g.
	# scripts, output, logs (e.g. should be NFS available to all)
	slurm_njobs=>'', # number of scripts
	slurm_jobid=>'', # slurm job id (%A)
	slurm_jobarrayindex=>'', # slurm job array index (%a)
        slurm_jobname=>'binary_grid', # set to binary_grid
        slurm_postpone_join=>0, # if 1, data is not joined, e.g. if you
	# want to do it off the slurm grid (e.g. with more RAM)
        slurm_postpone_sbatch=>0, # if 1, don't submit, just make the script
	
	# (defaults to $ENV{PWD} if undef)
	slurm_memory=>512, # in MB, the memory use of the job
        slurm_warn_max_memory=>1024, # in MB : warn if mem req. > this
        slurm_partition=>undef,
        slurm_ntasks=>1, # 1 CPU required per array job: usually only need this
        slurm_time=>0, # 0 = infinite time
	slurm_use_all_node_CPUs=>0, # 1 = use all of a node's CPUs (0)
	# you will want to use this if your Slurm SelectType is e.g. linear
	# which means it allocates all the CPUs in a node to the job
	slurm_control_CPUs=>0, # if so, leave this many for Perl control (0)
        slurm_array=>undef,# override for --array, useful for rerunning jobs
        
	########################################
	# CPU 
	########################################
	cpu_cap=>0, # if 1, limits to one CPU
 	cpu_affinity => 0, # do not bind to a CPU by default

	########################################
	# Code, Timeouts, Signals
	########################################
	binary_grid_code_filtering=>1, #  you want this, it's (MUCH!) faster
	pre_filter_file=>undef, # dump pre filtered code to this file
	post_filter_file=>undef,  # dump post filtered code to this file

	timeout=>30, # timeout in seconds
	timeout_vb=>0, # no timeout logging
	tvb=>0, # no thread logging
	nfs_sleep=>1, # time to wait for NFS to catch up with file accesses

	# flexigrid checks the timeouts every 
	# flexigrid_timeout_check_interval seconds
	flexigrid_timeout_check_interval=>0.01, 

	# this is set to 1 when the grid is finished
	flexigrid_finished=>0,

	# allow signals by default
	'no signals'=>0,

	# but perhaps disable specific signals?
	'disable signal'=>{INT=>0,ALRM=>0,CONT=>0,USR1=>0,STOP=>0},

	# dummy variables
	single_star_period=>1e50,  # orbital period of a single star

	#### timers : set timers to 0 (or empty list) to ignore, 
	#### NB these must be given context (e.g. main::xyz) 
	#### for functions not in binary_grid
	timers=>0,
	timer_subroutines=>[
	    # this is a suggested default list
	    'flexigrid',
            'set_next_alarm',
	    'vbout',
            'vbout_fast',
	    'run_flexigrid_thread',
            'thread_vb'
	],

	########################################
	# INPUT/OUTPUT
	########################################
	blocking=>undef, # not yet set
	
	# prepend command with stdbuf to stop buffering (if available)
	stdbuf_command=>`stdbuf --version`=~/stdbuf \(GNU/ ? ' stdbuf -i0 -o0 -e0 ' : undef,

	vb=>("@ARGV"=~/\Wvb=(\d+)\W/)[0] // 0, # set to 1 (or more) for verbose output to the screen
	log_dt_secs=>1, # log output to stdout~every log_dt_secs seconds
	nmod=>10, # every nmod models there is output to the screen,
	# if log_dt_secs has been exceeded also (ignored if 0)

	colour=>1, # set to 1 to use the ANSIColor module for colour output
	log_args=>0, # do not log args in files
	log_fins=>0, # log end of runs too
        sort_args=>0, # do not sort args
	save_args=>0, # do not save args in a string
	log_args_dir=>$tmp, # where to output the args files
	always_reopen_arg_files=>0, # if 1 then arg files are always closed and reopened (may cause a lot of disk I/O)

	lazy_arg_sending=>1, # if 1, the previous args are remembered and
	# only args that changed are sent (except M1, M2 etc. which always
	# need sending)
 
	# force output files to open on a local disk (not an NFS partion)
	# not sure how to do this on another OS
	force_local_hdd_use=>($os eq 'unix'), 

	# for verbose output, define the newline
	# For terminals use "\x0d", for files use "\n", in the 
	# case of multiple threads this will be set to \n
	newline=> "\x0d",

        # use reset_stars_defaults
        reset_stars_defaults=>1,

	# set signal captures: argument determines behaviour when the code locks up
	# 0: exit
	# 1: reset and try the next star (does this work?!)
	alarm_procedure=>1,

	# exit on eval failure?
	exit_on_eval_failure=>1,

	## functions: these should be set by perl lexical name
	## (they are automatically converted to function pointers
	## at runtime)

	# function to be called just before a thread is created
	thread_precreate_function=>undef,
        thread_precreate_function_pointer=>undef,
	
	# function to be called just after a thread is created
	# (from inside the thread just before *grid () call)
	threads_entry_function=>undef,
        threads_entry_function_pointer=>undef,

	# function to be called just after a thread is finished
	# (from inside the thread just after *grid () call)
	threads_flush_function=>undef,
	threads_flush_function_pointer=>undef,

	# function to be called just after a thread is created
	# (but external to the thread)
	thread_postrun_function=>undef,
	thread_postrun_function_pointer=>undef,

	# function to be called just before a thread join
	# (external to the thread)
	thread_prejoin_function=>undef,
	thread_prejoin_function_pointer=>undef,

	# default to using the internal join_flexigrid_thread function
	threads_join_function=>'binary_grid2::join_flexigrid_thread',
	threads_join_function_pointer=>sub{return $self->join_flexigrid_thread(@_)},
	
	# function to be called just after a thread join
	# (external to the thread)
	thread_postjoin_function=>undef,
	thread_postjoin_function_pointer=>undef,

	# usually, parse_bse in the main script is called
	parse_bse_function=>'main::parse_bse',
        parse_bse_function_pointer=>undef,

	# if starting_snapshot_file is defined, load initial
	# values for the grid from the snapshot file rather 
	# than a normal initiation: this enables you to 
	# stop and start a grid
	starting_snapshot_file=>undef,
	
	# flexigrid options
	flexigrid=>{
            # the flexigrid 'grid type' can be either:
            #
            # 'grid' (default) is the traditional N-dimensional grid
            # or
            # 'list' takes a list of systems from 
            #
            # $self->{_grid_options}->{flexigrid}->{'list filename'}
            #
            # (which is a file containing list strings on each line)
            #
            # or from 
            # 
            # $self->{_grid_options}->{flexigrid}->{'list reference'}
            #
            # (which is a reference to a perl list)
	    'grid type'=>'grid',
            'list filename'=>undef,  # undef unless 'grid type' is 'list'
            'list reference'=>undef, # ditto
            'listFP'=>undef, # file pointer : keep undef here (set automatically) 

        },

        # start at this model number: handy during debugging 
        # to skip large parts of the grid
        start_at => 0
	);

    ########################################
    # PHYSICS
    ########################################
    %{$self->{_bse_options}} = (
	# internal buffering and compression level
	internal_buffering=>0,

        # log filename
        log_filename => '/dev/null',

	metallicity=>0.02, # metallicity (default 0.02, solar)
        
	max_evolution_time=>13700.0, # max evol time in Myr (default WMAP result)

        # stellar evolution parameters
        max_tpagb_core_mass=>1.38,
        chandrasekhar_mass=>1.44,
        max_neutron_star_mass=>1.8,
        minimum_mass_for_carbon_ignition=>1.6,
        minimum_mass_for_neon_ignition=>2.85,
        AGB_core_algorithm=>0,
        AGB_radius_algorithm=>0,
        AGB_luminosity_algorithm=>0,
        AGB_3dup_algorithm=>0,

        # dredge up calibration (either automatic, or not)
	delta_mcmin=>0.0,
	lambda_min=>0.0,
	minimum_envelope_mass_for_third_dredgeup=>0.5,

	# minimum timestep (1yr = 1e-6 is the default)
	minimum_timestep=>1e-6,
	# maximum timestep (1 Gyr = 1e3 is the default)
	maximum_timestep=>1e3,
        
	# orbit
	eccentricity=>0.0,

	# tidally induced mass loss
	CRAP_parameter=>0.0,
	
	# tidal strength factor
	tidal_strength_factor=>1.0,

        # E2 tidal prescription. 0 = H02, 1 = Siess+2013
        E2_prescription=>1, 

        # gravitational radiation model
        # 0 = H02 model (Eggleton) for J and e
        # 1 = H02 when R<RL for both stars for J and e
        # 2 = None
        # 3 = Landau and Lifshitz (1951) model for J (e is not changed)
        # 4 = Landau and Lifshitz (1951) model for J when R<RL only (e is not changed)
        gravitational_radiation_model=>0,

        # magnetic braking multiplier
        magnetic_braking_factor=>1.0,

        ############################################################
	### Mass-loss prescriptions
        ############################################################

        # turn wind mass loss on or off
        wind_mass_loss=>1,

	# massive stars
	wr_wind=>0, # default hurley et al wind
	wr_wind_fac=>1.0, # factor applied to WR stars
	
        # TPAGB wind details
	tpagbwind=>0, # default to 0 (Karakas et al 2002)
	superwind_mira_switchon=>500.0,
	tpagb_reimers_eta=>1.0,

	# eta for Reimers-like GB mass loss
	gb_reimers_eta=>0.5,

	# VW93 alterations
	vw93_mira_shift=>0.0,
	vw93_multiplier=>1.0,

	# systemic wind angular momentum loss prescription
	wind_angular_momentum_loss=>0,
	lw=>1.0,

        # how to deal with overspin?
        # 0 = OVERSPIN_BSE = transfer excess angular momentum to the orbit
        #     (no mass loss)
        #
        # 1 = OVERSPIN_MASSLOSS = use the rotationally_enhanced_mass_loss
        #     as set below
        overspin_algorithm => 'OVERSPIN_BSE',

        
        # enhanced mass loss due to rotation
        # 0 = none = ROTATION_ML_NONE
        # 1 = Langer+ formula (in mass-loss rate calculation, 
        #                      warning: can be unstable 
        #                      = ROTATION_ML_FORMULA)
        # 2 = remove material in a decretion disc until J<Jcrit
        #     (ROTATION_ML_ANGMOM)
        # 3 = 1 + 2 (not recommended!)
        rotationally_enhanced_mass_loss=>2,
        rotationally_enhanced_exponent=>1.0,



	# timestep modulator
	timestep_modulator=>1.0,

	# initial rotation rates (0=automatic, >0 = in km/s)
	vrot1=>0.0,
	vrot2=>0.0,


        ########################################
	# Supernovae and kicks
        ########################################

        # Black hole masses: 
        # 0: H02=0
        # 1: Belczynski
        # 2: Spera+ 2015
        # 3: Fryer 2012 (delayed)
        # 4: Fryer 2012 (rapid)
	BH_prescription=>2,
        post_SN_orbit_method=>0,
 
	wd_sigma=>0.0,
	wd_kick_direction=>0,
	wd_kick_pulse_number=>0,
	wd_kick_when=>0,

        # sn_kick_distribution and 
        # sn_kick_dispersion are only defined 
        # for SN types that leave a remnant
        sn_kick_distribution_II=>1,
        sn_kick_dispersion_II=>190.0,
        sn_kick_distribution_IBC=>1,
        sn_kick_dispersion_IBC=>190.0,
        sn_kick_distribution_GRB_COLLAPSAR=>1,
        sn_kick_dispersion_GRB_COLLAPSAR=>190.0,
        sn_kick_distribution_ECAP=>1,
        sn_kick_dispersion_ECAP=>190.0,
        sn_kick_distribution_NS_NS=>0,
        sn_kick_dispersion_NS_NS=>0.0,
        sn_kick_distribution_TZ=>0,
        sn_kick_dispersion_TZ=>0.0,
        sn_kick_distribution_BH_BH=>0,
        sn_kick_dispersion_BH_BH=>0.0,
        sn_kick_distribution_BH_NS=>0,
        sn_kick_dispersion_BH_NS=>0.0,
        sn_kick_distribution_AIC_BH=>0,
        sn_kick_dispersion_AIC_BH=>0.0,

        sn_kick_companion_IA_He=>0,
        sn_kick_companion_IA_ELD=>0,
        sn_kick_companion_IA_CHAND=>0,
        sn_kick_companion_AIC=>0,
        sn_kick_companion_ECAP=>0,
        sn_kick_companion_IA_He_Coal=>0,
        sn_kick_companion_IA_CHAND_Coal=>0,
        sn_kick_companion_NS_NS=>0,
        sn_kick_companion_GRB_COLLAPSAR=>0,
        sn_kick_companion_HeStarIa=>0,
        sn_kick_companion_IBC=>0,
        sn_kick_companion_II=>0,
        sn_kick_companion_IIa=>0,
        sn_kick_companion_WDKICK=>0,
        sn_kick_companion_TZ=>0,
        sn_kick_companion_AIC_BH=>0,
        sn_kick_companion_BH_BH=>0,
        sn_kick_companion_BH_NS=>0,

        # evolution run splitting
        evolution_splitting=>0,
        evolution_splitting_sn_n=>10,
        evolution_splitting_maxdepth=>1,

        ########################################
        #### Mass transfer 
        ########################################

	# critical mass ratio for unstable RLOF 
        #
        # qc = m (donor) / m (accretor) : 
        # if q>qc mass transfer is unstable
        #
        # H02 = Hurley et al. (2002)
        # C14 = Claeys et al. (2014)

	# non-degenerate accretors
        qcrit_LMMS=>0.6944, # de Mink et al 2007 suggests 1.8, C14 suggest 1/1.44 = 0.694
        qcrit_MS=>1.6, # C14 suggest 1.6
        qcrit_HG=>4.0, # H02 sect. 2.6.1 gives 4.0
        qcrit_GB=>-1, # -1 is the H02 prescription for giants
        qcrit_CHeB=>3.0, 
        qcrit_EAGB=>-1, # -1 is the H02 prescription for giants
        qcrit_TPAGB=>-1, # -1 is the H02 prescription for giants
        qcrit_HeMS=>3,
        qcrit_HeHG=>0.784, # as in H02 2.6.1
        qcrit_HeGB=>0.784, # as in H02 2.6.1
        qcrit_HeWD=>3, # H02
        qcrit_COWD=>3, # H02
        qcrit_ONeWD=>3, # H02
        qcrit_NS=>3, # H02
        qcrit_BH=>3, # H02

        # degenerate accretors
        qcrit_degenerate_LMMS=>1.0, # C14
        qcrit_degenerate_MS=>1.0, # C14
        qcrit_degenerate_HG=>4.7619, # C14
        qcrit_degenerate_GB=>1.15, # C14 (based on Hachisu)
        qcrit_degenerate_CHeB=>3, # not used
        qcrit_degenerate_EAGB=>1.15, # as GB
        qcrit_degenerate_TPAGB=>1.15, # as GB
        qcrit_degenerate_HeMS=>3,
        qcrit_degenerate_HeHG=>4.7619, # C14
        qcrit_degenerate_HeGB=>1.15, # C14
        qcrit_degenerate_HeWD=>0.625, # C14
        qcrit_degenerate_COWD=>0.625, # C14
        qcrit_degenerate_ONeWD=>0.625, # C14
        qcrit_degenerate_NS=>0.625, # C14
        qcrit_degenerate_BH=>0.625, # C14

	# disk wind for SNeIa
	hachisu_disk_wind=>0, # 0 
        hachisu_qcrit=>1.15, # 1.15

	# ELD accretion mass
	mass_accretion_for_eld=>0.15, # 0.15, or 100(off)

        # mergers have the critical angular momentum
        # multiplied by this factor
        merger_angular_momentum_factor=>1.0,

        # RLOF rate method : 0=H02, 1=Adaptive R=RL, 3=Claeys et al 2014
        RLOF_method=>3,
        RLOF_mdot_factor=>1.0,
        
        # RLOF time interpolation method
        # 0 = binary_c (forward in time only), 1 = BSE (backwards allowed)
        RLOF_interpolation_method=>0,

        # Angular momentum in RLOF transfer model
        # 0 : H02 (including disk treatment)
        # 1 : Conservative
        RLOF_angular_momentum_transfer_model=>0,

        # ang mom factor for non-conservative mass loss
        # -2 : a wind from the secondary (accretor), i.e. gamma=Md/Ma (default in C14)
        # -1 : donor (alternative in C14), i.e. gamma=Ma/Md
        # >=0 : the specific angular momentum of the orbit multiplied by gamma
        #
        # (NB if Hachisu's disk wind is active, or loss is because of
        # super-Eddington accretion or novae, material lost through the 
        #  disk wind automatically gets gamma=-2 i.e. the Jaccretor) 
        nonconservative_angmom_gamma=>-2,

        # Hachisu's disc wind
        hachisu_disk_wind=>0,
        hachisu_qcrit=>-1.0,

        # donor rate limiters
        donor_limit_thermal_multiplier=>1.0,
        donor_limit_dynamical_multiplier=>1.0,

        # RLOF assumes circular orbit (0) or at periastron (1)
        rlperi=>0,
	
        # general accretion limits
	accretion_limit_eddington_multiplier=>1.0, # eddington limit factor
        accretion_limit_dynamical_multiplier=>1.0, # dynamical limit factor
        accretion_limit_thermal_multiplier=>1.0, # thermal limit on MS,HG and CHeB star factor

        # default to the Claeys et al. (2014) WD accretion limits
        WD_accretion_rate_novae_upper_limit_hydrogen_donor=>'DONOR_RATE_ALGORITHM_CLAEYS2014',
        WD_accretion_rate_new_giant_envelope_lower_limit_hydrogen_donor=>'DONOR_RATE_ALGORITHM_CLAEYS2014',
        WD_accretion_rate_novae_upper_limit_helium_donor=>'DONOR_RATE_ALGORITHM_CLAEYS2014',
        WD_accretion_rate_new_giant_envelope_lower_limit_helium_donor=>'DONOR_RATE_ALGORITHM_CLAEYS2014',
        WD_accretion_rate_novae_upper_limit_other_donor=>'DONOR_RATE_ALGORITHM_CLAEYS2014',
        WD_accretion_rate_new_giant_envelope_lower_limit_other_donor=>'DONOR_RATE_ALGORITHM_CLAEYS2014',
        

	# novae
        nova_retention_method=>0,
	nova_retention_fraction=>1e-3,
        individual_novae=>0,
        beta_reverse_nova=>-1,
        nova_faml_multiplier=>1.0,
        nova_irradiation_multiplier=>0.0,

        ########################################
	# common envelope evolution
        ########################################
	comenv_prescription=>0, # 0=H02, 1=nelemans, 2=Nandez+Ivanova2016
        alpha_ce=>1.0,
	lambda_ce=>-1, # -1 = automatically set
	lambda_ionisation=>0.0,
	lambda_enthalpy=>0.0,
	comenv_splitmass=>0.0,
        comenv_ms_accretion_mass=>0.0,
        nelemans_minq=>0.0, # 0.0 min q for nelemans
	nelemans_max_frac_j_change=>1.0, # 1.0
	nelemans_gamma=>1.0, # 1.0
	nelemans_n_comenvs=>1, # 1
        comenv_merger_spin_method=>2,
        comenv_ejection_spin_method=>1,
        comenv_post_eccentricity=>0.0,
        comenv_splitmass=>1.01,

	# comenv accretion
	comenv_ns_accretion_fraction=>0.0,
	comenv_ns_accretion_mass=>0.0,

        ##################################################
        # circumbinary disc
        ##################################################
#       comenv_disc_mass_fraction => 0.0,
#       comenv_disc_angmom_fraction => 0.0,
#       cbdisc_gamma => 1.6666666666,
#       cbdisc_alpha => 1e-6,
#       cbdisc_kappa => 1e-2,
#       cbdisc_torquef => 1.0,
#       cbdisc_mass_loss_constant_rate => 0.0,
#       cbdisc_mass_loss_inner_viscous_accretion_method => 1,
#       cbdisc_mass_loss_inner_viscous_multiplier => 1.0,
#       cbdisc_mass_loss_inner_L2_cross_multiplier => 0.0,
#       cbdisc_mass_loss_ISM_ram_pressure_multiplier => 0.0,
#       cbdisc_mass_loss_ISM_pressure => 3000.0,
#       cbdisc_mass_loss_FUV_multiplier => 0.0,
#       cbdisc_mass_loss_Xray_multiplier => 1.0,
#       cbdisc_viscous_photoevaporation_coupling => 1,
#       cbdisc_inner_edge_stripping => 1,
#       cbdisc_outer_edge_stripping => 1,
#       cbdisc_minimum_luminosity => 1e-4,
#       cbdisc_minimum_mass => 1e-6,
#       cbdisc_eccentricity_pumping_method => 1,
#       cbdisc_resonance_multiplier => 1.0,
#       comenv_post_eccentricity => 1e-5,

        ##################################################
        # wind accretion
        ##################################################

        # Bondi-Hoyle accretion multiplier
	Bondi_Hoyle_accretion_factor=>1.5,
        # Wind-RLOF method: 0=none, 1=q-dependent, 2=quadratic
        # (See Abate et al. 2012,13,14 series of papers)
        WRLOF_method=>0,

        
        # pre-main sequence evolution
        pre_main_sequence=>0,
        pre_main_sequence_fit_lobes=>0,

        ########################################
        # Nucleosynthesis
        ########################################

	# lithium
	lithium_hbb_multiplier=>1.0,
	lithium_GB_post_1DUP=>0.0,
	lithium_GB_post_Heflash=>0.0,

	########################################
	# Yields vs time (GCE)
	########################################
	
        yields_dt=>100000,
	escape_velocity=>1e9, # if wind v < this, ignore the yield
	# and assume it is lost to the IGM
	escape_fraction=>0.0, # assume all yield is kept in the population

	# minimum sep/per for RLOF on the ZAMS
	minimum_separation_for_instant_RLOF=>0,
	minimum_orbital_period_for_instant_RLOF=>0, 


	# extra options : assume blank but not an empty string
	extra=>' ',
	);

    # if available, use the evcode's defaults
    if(1)
    {
        my $evcode_args = $self->evcode_args_list();
        
        if(defined $evcode_args && 
           $evcode_args &&
           ref $evcode_args eq 'ARRAY' &&
           $#{$evcode_args} > 1)
        {
            foreach my $arg (grep {
                !(
                     # some args should be ignored
                     /=\s*$/ ||
                     /Function$/ ||
                     /NULL$/ ||
                     /\(?null\)?$/i ||
                     /^M_[12]/ ||
                     /^eccentricity/ ||
                     /^orbital_period/ ||
                     /^phasevol/ ||
                     /^separation/ ||
                     /^probability/ ||
                     /^stellar_type/ ||
                     /^_/||
                     /^batchmode/  ||
                     /^initial_abunds_only/ ||
                     /^monte_carlo_kicks/
                    )
                             }@$evcode_args)
            {
                if($arg=~/(\S+) = (\S+)/)
                {
                    if(!defined $self->{_bse_options}->{$1})
                    {
                        #print "NEW set $1 to $2\n";
                    }
                    $self->{_bse_options}->{$1} = 
                        $2 eq 'TRUE' ? 1 :
                        $2 eq 'FALSE' ? 0 :
                        $2;
                    #print "Set $1 -> $2 = $self->{_bse_options}->{$1}\n";
                }
            }
        }
    }
   
    $self->{_flexigrid} = {
	count         => 0,
        error         => 0,
        failed_count  => 0,
        failed_prob   => 0.0,
        global_error_string => undef,

        # random string to ID the flexigrid
      	id            => rob_misc::random_string(8),
        modulo        => 1, # run modulo n
        monitor_files => [],
        nextlogtime   => 0,
        nthreads      => 1, # number of threads
        # start at model offset (0-based, so first model is zero)
        offset        => 0, 
        probtot       => 0.0,
        resolution=>{
            shift   =>0,
            previous=>0,
            n       =>{} # per-variable resolution
        },
        results_hash  => $self->{_results},
        thread_q      => undef,
        threads       => undef, # array of threads objects
        tstart        => [gettimeofday], # flexigrid start time		
        __nvar        => 0, # number of grid variables
        _varstub      => undef,
        _lock         => undef,
        _evcode_pids  => [],
        default_vars  => $default_vars,
    };
}

############################################################
# signal handlers
############################################################
my $class_signal_handlers;
sub set_class_signal_handlers
{
    my $self = shift;
    ############################################################
    # allow us to catch signals (not SIGALRM, which is used as a timeout)
    ############################################################


    # make signame hashes and array
    my $i=0;
    defined $Config{sig_name} || confess "No sigs?";
    
    if(!$self->get('no signals'))
    {
	# send SIGCONT to restart the grid
	$SIG{CONT} = \&restart_grid if(!$self->gethashval('disable signal','CONT'));
    }
}

sub smartINT
{ 
    # experimental CTRL-C handler
    my $self = shift;
    $SIG{INT} = sub{
        print "Caught INT signal : smart handler\n";
        #$SIG{INT} = undef;

        # send the signal to all threads
        foreach my $thread (@{$self->{_flexigrid}{threads}})
        {
            print "STOP THREAD ID ",$thread->tid(),"\n";
            $thread->kill('SIGSTOP');
            $thread->detach();
        }
        
        # wait for threads to die, then exit
        while(scalar threads->list(threads::running))
        {
            sleep 1;
            printf "Still have %d threads\n",
            scalar @{$self->{_flexigrid}{threads}};
        }
        exit;
    }
}

sub setup_class_sig_handlers
{
    my $self = shift;
    if(!$self->get('no signals'))
    {
	# set up the alarm signal and capture Ctrl-C
	# note that in practice, it's best to just let ctrl-c propagate
	# to the binary-grid script and its child processes: this 
	# kills theem off quite nicely (in most cases)
	foreach my $sig ('INT','ALRM','TERM','DIE')
        {
            if(!$self->gethashval('disable signal',$sig))
            {
                $SIG{$sig} = sub{ binary_grid2::class_handle_signal($sig); };
                $class_signal_handlers->{$sig}->{$self} = eval "\$self->handle_$sig";
            }
        }
	# warning: SIG{INT} is used by condor
    }
    else
    {
	say 'Signals (INT, ALRM etc) have been disabled (Timeout will not function)';
    }
}

sub class_handle_signal 
{
    my $sig=shift;
    foreach (keys %{$class_signal_handlers->{$sig}})
    {
        &{$class_signal_handlers->{$sig}->{$_}}();
    }
}

sub handle_ALRM
{
    my $self = shift;
    ############################################################
    # capture alarm signals: usually generated by timeouts
    ############################################################
    
    if(defined($self->{_flexigrid}->{id}))
    {
	$self->flexigrid_timeout_check();
    }
    else
    {
	# NB this will be in the main (monitor) thread so this may not be accurate!
	say STDERR 'Timeout: code has frozen!';
	
	if($self->alarm_procedure)
	{
	    # send back a die so the grid can continue to the next star
	    say STDERR 'Try to restart evolution code';
	    confess('Restart evolution code');
	}
	else
	{
	    say STDERR 'Exit';
	    $self->tbse_kill(); # just to make sure!
	    exit(-1);
	}
    }
    
}

sub flexigrid_timeout_check
{
    my $self = shift;
    # check for thread timeout
    if($self->{_grid_options}->{'flexigrid_timeout_check_interval'} > 0.0)
    {
	$self->check_for_evcode_timeout();
	Time::HiRes::alarm($self->{_grid_options}->{flexigrid_timeout_check_interval});
    }
}

sub check_for_evcode_timeout
{
    my $self = shift;
    
    ############################################################
    # check all threads for timeouts, act if there's a problem
    ############################################################
    my $timeout=0;
    my %timeout_threads;

    say 'timeout_vb in check_for_evcode_timeout ...' if($self->{_grid_options}->{timeout_vb});
    my $now = Time::HiRes::time();
    for(my $i=0; $i<$self->{_flexigrid}->{nthreads}; $i++)
    {
	if($next_alarm_time[$i]>1.0)
	{
	    my $dt = $next_alarm_time[$i] - $now;
	    if($dt < 0.0)
	    {
		# this thread timed out 
		$timeout=1; # set action flag

		# save a list of timeout threads
		$timeout_threads{$i}=1;
	    }
	    printf "timeout_vb thread : CHECK THREAD $i (%.2f s until timeout)\n",$dt
		if($self->verbose_output(3) || $self->{_grid_options}->{timeout_vb});
	}
    }

    if($timeout)
    {
	my @threads = threads->list();
	my @timeout_threads = nsort keys (%timeout_threads);
	say "Timeout detected in threads (@timeout_threads): wait for timed out threads to clean themselves up";

	# disable join function
	$self->{_grid_options}->{thread_postjoin_function_pointer} = undef;

	say 'Sending stop commands';
	for(my $i=0;$i<$self->{_flexigrid}->{nthreads};$i++)
	{
	    # set up thread exit command
	    $self->{_threadinfo}->{cmd} = defined($timeout_threads{$i}) ? 1 : 2; # 2=signal exit	    
	    say "send STOP to thread $i (cmd ",$self->{_threadinfo}->{cmd};

	    # signal thread stop and then join (which will do nothing)
	    $threads[$i]->kill('STOP')->join;
	}
	say 'Exit grid';
	exit(0);
    }
}

sub set_next_alarm
{
    my $self = shift;

    ############################################################
    # the next alarm time because there was no timeout
    ############################################################
    my $n = $_[0] // $self;

    return if ($self->{_grid_options}->{'disable signal'}{ALRM});
    
    # use thread number
    my $nt = $self->{_threadinfo}->{thread_number};
    
    if($n==0)
    {
	# evcode finished
	$self->{_threadinfo}->{state}=0;
	$self->{_threadinfo}->{runcount}++;
	$next_alarm_time[$nt] = 0;
    	say "timeout_vb thread $nt: evcode finished" if($self->{_grid_options}->{timeout_vb});
    }
    else
    {
	# running evcode
	$self->{_threadinfo}->{state}=1;
	say "timeout_vb thread $nt: Set next alarm (now+$n)" if($self->{_grid_options}->{timeout_vb});

	# save the time of the next alarm call
	$next_alarm_time[$nt] = Time::HiRes::time()+$n;
	say "timeout_vb thread $nt: Alarm is set ($next_alarm_time[$nt])" if($self->{_grid_options}->{timeout_vb});
    }
}

sub handle_TERM
{
    my $self = shift;

    # capture condor/slurm signals if required
    if($self->get('condor_jobid'))
    {
	print "Condor : capture TERM for grid restarts\n";
	$self->grid_interrupted();
    }
    if($self->get('slurm_jobid'))
    {
	print "Slurm : capture TERM for grid restarts\n";
	$self->grid_interrupted();
    }

}
sub handle_DIE
{
    my $self = shift;
    $self->shutdown_binary_grid(9);
    Carp::confess("via die handler: ",@_);
    0;
}

sub handle_INT
{
    my $self = shift;
    print "INT handlers\n";
}
    
sub class_pipeHandler 
{
    my $self = shift;
    my $sig = $_[0];

    say "Caught SIGPIPE: sig=$sig \$!=$! : this usually means that the child process (your stellar evolution code) has died. I am process $$, binary_grid's main process is $self->{_grid_options}->{process_ID}\n";

    my $pid = $self->{_threadinfo}->{evcode_pid}; 
    if($pid)
    {
	say "Nominally, your process id is/was $self->{_threadinfo}->{evcode_pid}";
	if($self->operating_system eq 'unix')
	{
	    say "ps gives the following (if you see nothing but the header line, your evcode is dead!):\n******\n",
	    `ps -p $pid u`,
	    "******";
	}
    }
    else
    {
	say "Process id is unknown (\$pid = $pid), death happened before threads launched the evolution code?)";
    }
    say "I can do nothing at this point except stop and hope you know how to fix the bug!\nEnable debugging either with the command line switch vb=n (n=1,2,3 for progessively more debugging)\nor by setting the verbosity flag in your script.";
    exit(1);
}

############################################################
# directory functions
############################################################

sub okdir
{
    my $self = shift;
    # return directory name if it exists
    return -d $_[0] ? $_[0] : undef;
}

sub tmpdir
{
    my $self = shift;
    # guess temporary files location
    my $os = $_[0] // 
        $self->{_grid_options}->{operating_system} //
        rob_misc::operating_system(); 
    return $os eq 'windows'
	# what to do on windoze? according to http://en.wikipedia.org/wiki/Temporary_folder it is %temp%... this is untested.
	? '%temp%' 
	# on Unix, default to /tmp/ (or TMPDIR if defined)
	: $ENV{TMPDIR} // '/tmp';
}

############################################################
# Grid setup functions
############################################################


sub add_grid_variable
{
    my $self = shift;
    # add a variable to the grid
    my %opts = @_;
    $self->vbout(2,sprintf"add_grid_variable %s : @_ \n",$self->{_flexigrid}{__nvar});
    $self->{_grid_options}->{flexigrid}{
        'grid variable '.$self->{_flexigrid}{__nvar}++
    } = \%opts;
}



############################################################
# Flexigrid functions
############################################################

sub run
{
    my $self = shift;
    # run a grid
    $self->flexigrid(@_);
}

sub pre_thread_setup
{
    my $self = shift;

    # setup to be done before threads are launched

    # set process name
    $self->{_grid_options}{original_process_name} = $0;
    if(defined $self->{_grid_options}{process_name})
    {
        $0 = $self->{_grid_options}{process_name};
    }
    $self->{_grid_options}{new_process_name} = $0;
    
    # bind C logging code
    #
    # This must be done here, before threads are set up
    {
        $binding::_code = '';

        # start with autogen'd code
        if(defined $self->{_grid_options}->{C_auto_logging})
        {
            $binding::__code .= 
                $self->autogen_C_logging_code(); 
        }

        # add custom code
        if(defined $self->{_grid_options}->{C_logging_code})
        {
            # put the code in a new (temporary) package variable
            $binding::__code .= 
                $self->{_grid_options}->{C_logging_code};
        }
        
        if(defined $binding::__code && $binding::__code)
        {
            # attach it using binary_c_logging which runs
            # in the main package space
            $self->vbout(1,"Bind logging code: \"$binding::__code\"\n");
            $0 = $self->{_grid_options}{original_process_name};
            package main
            {
                binary_c_logging($binding::__code);
            }
            $0 = $self->{_grid_options}{new_process_name};
        }
        # reset the pointer 
        $self->{_grid_options}->{custom_output_C_function_pointer} = 
            binary_grid2::binary_c_function_bind();
        print "Pointer is $self->{_grid_options}->{custom_output_C_function_pointer} \n";
        
        # undef the temporary variable
        $binding::__code = undef;
    }
}

sub autogen_C_logging_code
{
    # given a hash of arrays of variable names, where the hash
    # key is the header, autogenerate Printf statements
    my ($self) = @_;
    my $code = undef;
    if(defined $self->{_grid_options}->{C_auto_logging} &&
       ref $self->{_grid_options}->{C_auto_logging} eq 'HASH'
        )
    {
        $code = '';

        foreach my $header (keys %{$self->{_grid_options}->{C_auto_logging}})
        {
            if(ref $self->{_grid_options}->{C_auto_logging}->{$header} eq 'ARRAY')
            {
                $code .= 'Printf("'.$header.' ';
                foreach my $x (@{$self->{_grid_options}->{C_auto_logging}->{$header}})
                {
                    $code .= '%g ';
                }
                $code .= '\n"';

                foreach my $x (@{$self->{_grid_options}->{C_auto_logging}->{$header}})
                {
                    $code .= ',((double)stardata->'.$x.')';
                }
                $code .= ');'
            }
        }
    }
    print "Made auto logging code:\n\n************************************************************\n\n$code\n\n************************************************************\n";
    
    return $code;
}

sub evolve
{
    # call the grid evolution algorithm
    my $self = shift;

    # but before launching threads, 
    # do any required setup
    $self->pre_thread_setup();
    
    if($self->{_grid_options}{condor})
    {
        return $self->condor_grid(@_);
    }
    elsif($self->{_grid_options}{slurm})
    {
        return $self->slurm_grid(@_);
    }
    else
    {
        # run a grid
        return $self->flexigrid(@_);
    }
}


sub flexigrid
{
    my $self = shift;

    ############################################################
    #
    # flexigrid : a completely user-configurable stellar grid solution
    #
    # here we
    # 1) make the flexigrid code (i.e. the user-defined loops over M1,M2,a..)
    # 2) for each thread, launch evolution code (e.g. binary_c or BONNFIRES)
    # 3) run the grid which throws stars at the threads
    # 4) join the data together
    # 5) return
    #
    # Takes one argument: the number of threads
    #
    # Note: you should not call the flexigrid method directly,
    # instead you should call $self->evolve().
    #
    ############################################################
    say "In flexigrid : rungrid == $self->{_grid_options}->{rungrid} condor == $self->{_grid_options}{condor} slurm == $self->{_grid_options}{slurm}";

    ############################################################
    # signals
    ############################################################
    $SIG{CHLD}='IGNORE'; 
    $SIG{PIPE}=\&binary_grid2::class_pipeHandler; 

    if($self->{_grid_options}->{rungrid} == 1)
    {
	# if there is no flexigrid hash in existence (it may have 
        # been loaded from a snapshot) then make a new one
	if(!defined($self->{_flexigrid}->{id}))
	{
	    # set up new flexigrid hash (all threads have a copy of this)
	    %{$self->{_flexigrid}} =
                (
                 count=>0,
                 global_error_string=>undef,
                 id=>rob_misc::random_string(8), # random string to ID the flexigrid
                 modulo=>$self->{_grid_options}->{modulo}, # run modulo n
                 monitor_files=>[],
                 nextlogtime=>0,
                 nthreads=>($_[0]//1), # number of threads (passed in)
                 offset=>$self->{_grid_options}->{offset}, # start at model offset (0-based, so first model is zero)
                 probtot=>0.0,
                 resolution=>{
                     shift=>0,
                     previous=>0,
                     n=>{} # per-variable resolution
                 },
                 results_hash=>$self->{_results}, # set in your grid script
                 thread_q=>undef,
                 threads=>undef, # array of threads objects
                 tstart=>[gettimeofday], # flexigrid start time		
		);
	    $self->convert_functions_to_pointers();
        }

        $self->reset_flexigrid();
        
	confess("results_hash is undefined in flexigrid: please make sure \$self->{_results\} points to an anonymous hash")
            if(!defined($self->{_flexigrid}->{results_hash}));

        # erase arrays
	$self->{_flexigrid}->{_evcode_pids} = [];

	my $starcount;
	
	# check/set threads' stack size
	$self->set_threads_stacksize();
	
	# launch threads
	$self->vbout(0,"Launch $self->{_flexigrid}->{nthreads} threads\n");
    
	($self->{_flexigrid}->{threads},$self->{_flexigrid}->{thread_q})=
	    $self->launch_flexigrid_threads($self->{_flexigrid}->{nthreads});

	# set grid option in case external programs want to know the number of threads
	$self->{_grid_options}->{nthreads} = $self->{_flexigrid}->{nthreads};

	$self->vbout(1,"Make flexigrid:\n");
	
	# make flexigrid code on the fly while we wait for the threads to start 
        my $gridcode = $self->make_flexigrid();
        my $gridcode_fast = $self->fast_gridcode({gridcode=>$gridcode});
        
        # if use_full_resolution is 1, run a dummy grid 
        # to find the true resolution
        if($self->{_grid_options}->{use_full_resolution})
        {
            print "Calculating full grid resolution : please wait ...\n";
            my $h = $self->run_gridcode_fast({
                logdt => 1,
                nthreads => $self->{_flexigrid}->{nthreads},
                pre_code => '',
                post_code => '',
                                             });

            
            # reset starcount
            $starcount = 0;
        }
        
        # if using condor, write the join script
	if($self->{_grid_options}->{condor_jobid} ne '')
        {
            my $f= $self->{_grid_options}{condor_dir}.'/scripts/joincmd';
            if(!-f $f)
            {
                open(my $fp,'>'.$f)||confess("cannot open $f for writing");
                say {$fp} $self->condor_rerun_command();
                close $fp;
                chmod 0700, $f;
            }
        }
 	
	$self->vbout(1,"Run flexigrid\n");
        
	# run the grid to send commands to the threads
        my $gridcode_retval = eval $gridcode;

	# check for a return value (undef means error!)
	$self->flexigrid_eval_error($@,$gridcode) if(!defined($gridcode_retval));

	$self->vbout(1, "Grid finished: joining threads... (be patient)\n");
 	$self->{_grid_options}->{flexigrid_finished}=1;

	# join threads to get the data
	my ($local_threadinfo, $failed_count, $failed_prob) =
            $self->join_flexigrid_threads($self->{_flexigrid}->{thread_q},
                                          $self->{_flexigrid}->{threads});

	if(defined $self->{_grid_options}->{results_hash_dumpfile} && 
	   $self->{_grid_options}->{results_hash_dumpfile} ne '')
	{
	    $self->vbout(1, "DUMP to results_hash_dumpfile=$self->{_grid_options}->{results_hash_dumpfile}\n");
	}


	# dump results hash if required
	$self->dump_results_hash()
	    if(defined($self->{_grid_options}->{results_hash_dumpfile})&&
	       ($self->{_grid_options}->{results_hash_dumpfile} ne ''));

	# timing stats
	printf "Flexigrid took %g seconds\n",
            (defined $self->{_flexigrid}->{tstart} &&
             ref $self->{_flexigrid}->{tstart} eq 'ARRAY') ? 
            Time::HiRes::tv_interval(
                $self->{_flexigrid}->{tstart},
                [gettimeofday]
            ) : 0.0; 

	# error stats
	if($failed_count > 1e-50)
	{
	    printf "%d stellar systems failed corresponding to a probability %g\n",
	    $failed_count,
	    $failed_prob;
	}
	else
	{
	    print "No stellar systems failed\n";
	}

	if($self->{_grid_options}->{condor_jobid} ne '')
	{
	    # end of a condor run : call the condor_job_hook function in the condor module
            print "Calling condor job hook\n";
	    $self->condor_job_hook();
	}
    }
    else
    {
	# rungrid is zero: do something else
	$self->vbout(1,"Flexigrid: rungrid=0, do something else\n");

	# merge a set of datafiles, the result goes into results_hash,
	# and is dumped if a results_hash_dumpfile is specified
	if($self->{_grid_options}->{merge_datafiles})
	{
	    # the merge_datafiles variable should be a comma-separated list of filenames
	    # (without any spaces in it!)
	    my @datafiles=split(',',$self->{_grid_options}->{merge_datafiles});
	    
	    say "Merge datafiles: @datafiles";

	    # now do the merge
	    $self->{_results} = $self->merge_results_hash_dumps(@datafiles);

	    say "MERGED : $self->{_results}, dump to $self->{_grid_options}->{results_hash_dumpfile}";

	    # save if dumpfile name is given
	    $self->dump_results_hash() if($self->{_grid_options}->{results_hash_dumpfile} ne '');
	}
	elsif($self->{_grid_options}->{merge_datafiles_filelist})
	{
	    # instead of a list passed in, process a file
	    open(my $fp, '<', $self->{_grid_options}->{merge_datafiles_filelist}) || 
		confess("cannot open $self->{_grid_options}->{merge_datafiles_filelist} to read a list of filenames that need to be joined");
	    my @datafiles=(<$fp>);
	    close $fp;

	    chomp @datafiles;
	    @datafiles = sort hashdumpsorter @datafiles;

	    say "Merge ",scalar @datafiles," datafiles from file: $self->{_grid_options}->{merge_datafiles_filelist}";

	    # now do the merge
	    $self->{_results} = $self->merge_results_hash_dumps(@datafiles);

	    say "MERGED : $self->{_results}\n";

	    # save if dumpfile name is given
	    if($self->{_grid_options}->{results_hash_dumpfile} ne '')
	    {
		say "Dump to to $self->{_grid_options}->{results_hash_dumpfile}";
		$self->dump_results_hash();
	    }
	}

	# load the results_hash from a file (that was previously dumped)
	if(defined($self->{_grid_options}->{load_results_from_file}))
	{
	    # fill the results hash from data in a file : this allows us to process
	    # the data with a single grid script
	    $self->{_results} = 
                $self->load_results_hash_file($self->{_grid_options}->{load_results_from_file});
	}
    }


    return;
}

sub reset_flexigrid
{
    my ($self) = @_;
    $self->{_flexigrid}->{error}=0;
    $self->{_flexigrid}->{count}=0;
    $self->{_flexigrid}->{failed_count}=0;
    $self->{_flexigrid}->{failed_prob}=0.0;
    $self->{_flexigrid}->{probtot}=0.0;
}

sub run_gridcode_fast
{
    # run a fast grid
    my ($self,$opts) = @_;
    my $starcount; # returned as first item in a list
    $opts //= {};
    my $default_opts =
    {
        logdt => 1, # output every logdt seconds,
        nthreads => $self->{_flexigrid}->{nthreads}//1, # number of threads
        keep_probabilities => 0,
        keep_phasevol => 0,
        system_code => '',
        use_increment_count => 0,
        keep_vbout => 0,
        keep_actions => 0,
        pre_code => '',
        post_code => '',
        gridcode => undef,
    };
    # construct options
    $opts = {
        %$default_opts,
            %$opts,
    };
    
    # use passed in gridcode, or make it ourselves
    my $gridcode_fast = $opts->{gridcode} //
        $self->fast_gridcode($opts);
    
    my $g = $gridcode_fast;
        
    my $n = $self->{_flexigrid}->{nthreads};
    $g =~ /(0.5\*\$fvar\{)(\S+)(\}\{d\})/;
    my $firstvar = $2; # save the first var : we'll thread on this

    # increase step
    $g =~ s/(last if\(\$fvar\{$firstvar\}\{d\}==0.0\);)/\$fvar\{$firstvar\}\{value}\+=($n-1)\*\$fvar\{$firstvar\}\{d\};\n$1/;

    # remove original return statement (we return $h below)
    $g =~s/return 1;//;

    # disable memoize if it breaks threading
    #$g =~ s/use Memoize;//;
    #$g =~ s/Memoize\:\:memoize.*//g;

    # remove say statements
    $g =~ s/say.*//g;

    # remove $starcount[n]
    #$g =~ s/\$starcount\[\d+\]\+\+;//g;

    # pre and postcode
    $g = join("\n",
              'my $h;',
              $opts->{pre_code},
              $g,
              '$h->{starcount}=$starcount;',
              $opts->{post_code},
              'return $h;',
        );
    
    # debugging
    #dumpfile('/tmp/gridcode_fast',$g);

    my @cthreads;
    for(my $n=0; $n < $self->{_flexigrid}->{nthreads}; $n++)
    {
        # offset first point by $n * {d}
        my $g2 = $g;
        $g2 =~ s/(\$fvar\{$firstvar\}{value}=.*\n)/$1 \$fvar\{$firstvar\}\{value\} \+= $n \* \$fvar\{$firstvar\}\{d\};\n/;

        if($n==0)
        {
            # % status logging only on thread 0
            $g2 =~ s/(while\(\$fvar\{$firstvar\}.*\n\s*{\s*\n)/my \$__lastlog = time\(\);\n$1\nif\(time\(\)-\$__lastlog>=1)\{ printf\"   \%5\.2f \%\%\x0d\",100\.0\*\(\$fvar\{$firstvar\}\{value\}-\$fvar\{$firstvar\}\{min\}\)\/\(\$fvar\{$firstvar\}\{delta\}\); \$__lastlog=time\(\);\}\n/;
        }

        # debugging
        dumpfile($self->{_grid_options}->{tmp}.'/gridcode_fast.'.$n,
                 $g2);
        
        # start thread
        push(@cthreads,
             threads->create(
                 sub{
                     # use $self in here so it's local to the eval
                     $self;

                     
                     # eval which returns a hash
                     my $h = eval $g2;
                     
                     # on error : Return -1
                     if($@)
                     {
                         print "Thread eval error $@\n";
                         $h->{__thread_error} = $@;
                     }

                     return ($h);
                 }
             ));
    }

    # join threads to count stars
    $starcount = 0;
    my $results_hash = {};
    foreach my $t (@cthreads)
    {
        # the threads return a hash
        my ($h) = $t->join();

        # if the starcount < 0, or __thread_error
        # is set, we had a problem
        if($h->{starcount} < 0 || defined $h->{__thread_error})
        {
            print color('red')."ERROR : nstars < 0! See above for more errors.\n".color('reset');
            exit;
        }
        else
        {
            # use Hash::RobMerge to combine the data
            Hash::RobMerge::arithmetically_add_hashes($results_hash,$h);
        }
    }

    # on error, report the error
    if($@)
    {
        print color('red')."There was an error using the \$gridcode_fast block".color('reset')." : falling back to manual resolution counters by setting full_grid_resolution = 0 : warning! this may be inaccurate!\n";
        $self->{_grid_options}{use_full_resolution} = 0;
    }
    # all ok : report the numebr of stars
    else
    {
        print "Full grid resolution : ".color('blue bold').$results_hash->{starcount}.color('reset')."\n";
        $self->{_flexigrid}{full_resolution} = $results_hash->{starcount};
    }

    return $results_hash;
}

############################################################
# flexigrid queue admin
############################################################

sub flexigrid_timeout_check
{
    my $self = shift;
    # check for thread timeout
    if($self->{_grid_options}->{flexigrid_timeout_check_interval}>0.0)
    {
	$self->check_for_evcode_timeout();
	Time::HiRes::alarm($self->{_grid_options}->{flexigrid_timeout_check_interval});
    }
}

sub flexigrid_thread_number
{
    my $self = shift;
    # return the thread number
    return $self->{_threadinfo}->{thread_number};
}

sub run_flexigrid_thread
{
    my $self = shift;
    
    # run a flexigrid thread
    # input/output variables
    # 0 = queue
    # 1 = thread number
    # 2  = data hash (pointer to) to be filled
    my ($thread_q,$n,$h)=@_;
    print "RUN THREAD $n\n";
    my $thetime = time();

    # set global threadinfo hash
    %{$self->{_threadinfo}}=(
	evcode_pid=>undef, # evolution code process id
	cmd=>0,	# internal thread command (used for stopping)
	h=>$h, # propagate data hash
	lastargs=>undef, # the last args to be fed to evolution code
	lastruntime=>undef, # the last time that args were fed to the ev code  
        runtime=>0.0, # the total runtime (including perl)
	runcount=>0, # the number of stars run
	state=>0, # state: 0=finished, 1=running
	
	thread_number=>$n, # thread number
	thread_queue=>$thread_q, # the thread queue object
	tstart=>[gettimeofday], # start time
	# tvb = thread verbose logging 
	tvb_fp=> $self->{_grid_options}->{tvb} ? $self->open_thread_logfile($n) : undef,
	tvb_repeat=>0,
	tvb_last=>undef,
	tvb_lasttime=>undef,
	thread_prev_alive=>$thetime, # thread timer
	thread_prev_complaint=>$thetime, # warning timer
	);

    say "Starting flexigrid thread $n on q=$thread_q, self $self, threadinfo $self->{_threadinfo}"; 

    # capture STOP signal
    if((!$self->{_grid_options}->{'no signals'}) && 
       (!$self->{_grid_options}->{'disable signal'}{STOP}))
    {
    	$SIG{STOP} = sub
        {
	    # detected abnormal STOP signal
            my $threadn = $self->{_threadinfo}->{thread_number};
	    say "STOP thread $n ($threadn) : cmd=$self->{_threadinfo}->{cmd}";
	    $self->thread_vb("STOP thread $n : cmd=$self->{_threadinfo}->{cmd}\n");

	    # stop evolution code
	    $self->kill_flexigrid_evcode_pid($n);

	    # select error and log it
	    say "$n: write error ($self->{_threadinfo}->{cmd})";
	    no warnings;
	    my $error = ('grid terminated?', 
                         "timeout in  this thread (Time=".time()." : alarm last set at ".$next_alarm_time[$n].")\nargs = ".$self->{_threadinfo}->{lastargs}."\n",
                         "timeout detected in another thread\n\n",
                         "flexigrid_eval_error triggered")
		[$self->{_threadinfo}->{cmd}];
            
	    say "Thread $n terminated : $error";

            if(defined $self->{_threadinfo}->{lastargs})
            {
                my $dsecs = time - $self->{_threadinfo}->{lastruntime};
                say "Last system on thread $n was launched $dsecs seconds ago at ",
                scalar localtime($self->{_threadinfo}->{lastruntime}),
                ' and was :: ',
                $self->{_threadinfo}->{lastargs};
            }

	    $self->thread_vb("Thread $n terminated : $error\n");

	    # terminate!
            {
		no warnings 'threads';
		$SIG{__DIE__} = sub {Carp::confess(@_)};
                $SIG{INT} = undef;
                print "EXIT THREAD $n\n";
                threads->exit();
	    }
	    use warnings;
	};
    }

    $self->vbout(1,"Call evolution code launch for thread $self->{_threadinfo}->{thread_number}\n");
    
    if($self->{_grid_options}->{tvb})
    {
        $self->thread_vb("launching evolution code")
    }

    # launch private evcode
    $self->{_threadinfo}->{evcode_pid} =
        $self->tbse_launch($self->{_threadinfo});
    
    $self->{_flexigrid}->{_evcode_pids}[$n] = 
        $self->{_threadinfo}->{evcode_pid};

    $self->vbout(0,"Flexigrid : thread $n : $self->{_grid_options}->{code} process ID $self->{_threadinfo}->{evcode_pid}\n");

    if($self->{_grid_options}->{tvb})
    {
        $self->thread_vb("evcode $self->{_grid_options}->{code} is pid=$self->{_threadinfo}->{evcode_pid}");
        $self->thread_vb("calling pre-run function");
    }

    # perhaps call special pre-run function
    &{$self->{_grid_options}->{threads_entry_function_pointer}}($self,$h)
	if(defined($self->{_grid_options}->{threads_entry_function_pointer}));

    if($self->{_grid_options}->{tvb})
    {
        $self->thread_vb("pre-run function done")
    }

    # keep monitoring the stack, see if there are commands waiting
    # if so : run them (an 'undef' will break the loop)
    # if not : wait for the next to be queued
    my $stars_run=0;
    
    while(my $args = $thread_q->dequeue())
    { 
	# got a command : run it 
	$self->{_grid_options}->{'running args'}=$args;

	my $t0=[gettimeofday];
	
        # run a star
	my $err = $self->tbse($args,
                              $self->{_threadinfo}->{h},
                              $self->{_threadinfo}->{thread_number}); 
        
        $self->{_threadinfo}->{evcode_runtime} +=  
            Time::HiRes::tv_interval ($t0, [gettimeofday]);
        
        if(defined $err)
        {
            say "evcode error detected : $err";
            $self->{_flexigrid}->{error}="Error in thread $n: $err\n";
            Carp::confess;
        }
       
        $stars_run++;
    }

    # do not stop the evolution code until pending == 0
    # i.e. all the runs have finished. This takes a little while sometimes.
    while($thread_q->pending() > 0)
    {
        sleep 1;
    }

    # close evolution code
    $self->tbse_land($n);

    if($self->{_grid_options}->{tvb})
    {
        $self->thread_vb("call thread flush function")
    }

    # perhaps call special flush function 
    &{$self->{_grid_options}->{threads_flush_function_pointer}}($self,$h)
	if(defined($self->{_grid_options}->{threads_flush_function_pointer}));

    if($self->{_grid_options}->{tvb})
    {
        $self->thread_vb("thread flush function done")
    }
    $self->{_threadinfo}->{runtime} = Time::HiRes::tv_interval ( $self->{_threadinfo}->{tstart}, [gettimeofday]);

    if($self->{_grid_options}->{tvb})
    {
        $self->thread_vb("runtime ".$self->{_threadinfo}->{runtime}."\nThread $n finished\n")
    }

    $self->close_thread_logfile();

    return ($h,$self->{_threadinfo});
}


sub open_thread_logfile
{
    my $self = shift;
    # open thread log file
    my $n=$_[0];
    my $f="$self->{_grid_options}->{tmp}/thread.$n.log";
    my $fp = IO::File->new('>'.$f);
    confess ("Could not open thread log file $f\n")if(!defined $fp);
    $fp->autoflush(1);
    $self->thread_file_head($n,$fp);
    return $fp;
}


sub thread_log
{
    my $self = shift;
    rob_misc::thread_log(@_);
}

sub thread_vb
{
    my $self = shift;
    # verbose thread output with repetitive counteraction
    my $s = $_[0]; 
    chomp $s;
    if(defined $self->{_threadinfo}{thread_number})
    {
	no warnings;
	if(($s ne $self->{_threadinfo}{tvb_last}) ||
	   (time()-$self->{_threadinfo}{tvb_lasttime} > 1))
	{
            if(defined($self->{_threadinfo}{tvb_fp}))
	    {
                my @out = (($self->{_threadinfo}{tvb_repeat}>0) ? 
                           (' x',$self->{_threadinfo}{tvb_repeat},"\n") : "\n");
                my $t = $self->{_threadinfo}{thread_number}.' '.thetime().
                    ' ('.$self->{_threadinfo}{runcount}.')'.' ';
                state $lockv : shared;
                {
                    lock $lockv;
                    print {$self->{_threadinfo}{tvb_fp}} @out,$t,$s if(defined($s) && ($s ne ''));
                    $self->{_threadinfo}{tvb_fp} -> flush;
                }
            }
	    $self->{_threadinfo}{tvb_last} = $s;
	    $self->{_threadinfo}{tvb_repeat} = 0;
	    $self->{_threadinfo}{tvb_lasttime} = time();
	}
	else
	{
	    $self->{_threadinfo}{tvb_repeat}++;
	}
        use warnings;
    }
}



sub close_thread_logfile
{
    my $self = shift;
    # close thread log file
    if(defined($self->{_threadinfo}->{tvb_fp}))
    {
	say {$self->{_threadinfo}->{tvb_fp}} '';
	close $self->{_threadinfo}->{tvb_fp};
    }
}

sub backend
{
    my $self = shift;
    return $self->{_grid_options}->{backend};
}

sub thread_file_head
{
    my $self = shift;
    # output a header for given thread log
    my $n=$_[0];
    my $fp=$_[1];
    print {$fp}  "binary_grid thread log for thread $n\nLog started at ",
    scalar(localtime()),$self->infostring();   
}

sub commented_infostring
{
    my ($self,$prefix) = @_;
    $prefix//='# ';
    $self->infostring($prefix);
}


sub about
{
    my $self = shift;
    # wrapper to return information about a run
    return 
        join("\n",
             $self->evcode_version_string(),
             @{$self->evcode_args_list()},
             $self->infostring()
        );
}

sub infostring
{
    my $self = shift;
    my $prepend = $_[0] // '';
    
    # make string of information about this run
    # prepend each line with $prepend
    my @s;
    push @s,
    "\nHost: ",hostname(),
    "\nOperating system: ",$self->{_grid_options}->{operating_system},
    "\nPerl is $^X version ",(sprintf "%vd",$^V),
    "\nPerl modules loaded:\n";
    
    # list loaded modules
    map
    { 
	my $m=($_=~/(.*)\.pm$/)[0];
	my $v;
	if(defined($m))
	{
	    $v = eval "rob_misc::versionofmodule(\$m);";
	    $v = defined($v) ? 'version '.$v : 'unknown version';
	}
	else
	{
	    $v='unknown version';
	}
	push @s,sprintf "% 50s : % 80s : %s\n",$_,$INC{$_},$v;
    }sort keys %INC;

    push @s, "\n\nEnvironment:\n";
    map
    {
	push @s, "$_ = $ENV{$_} : ";
    }sort keys %ENV;


    foreach my $o (
        ['_grid_options','Grid options'],
        ['_bse_options','Binary star evolution options'],
    )
    {
        my $hashname = $o->[0];
        my $header = $o->[1];
        push @s, "\n\n($hashname $header) $header:\n";
        foreach my $opt (sort keys %{$self->{$hashname}})
        {
            my $x = "$hashname $opt =";
            if(!defined $self->{$hashname}->{$opt})
            {
                push @s, "$x undef\n";
            }
            # expand 1D arrays
            elsif(ref $self->{$hashname}->{$opt} eq 'ARRAY')
            {
                if(1)
                {
                    my $v = Data::Dumper::Dumper($self->{$hashname}->{$opt});
                    $v =~s/\$VAR1\s*=//;
                    $v =~s/\;\s*$//;
                    push @s, $x, $v."\n";
                }
                else
                {
                    # used to be this
                    push @s, $x.' ['.join(', ',@{$self->{$hashname}->{$opt}})."\]\n";
                }
            }
            # expand 1D hashes
            elsif(ref $self->{$hashname}->{$opt} eq 'HASH')
            {
                if(1)
                {
                    my $v = Data::Dumper::Dumper($self->{$hashname}->{$opt});
                    $v =~s/\$VAR1\s*=//;
                    $v =~s/\;\s*$//;
                    push @s, $x, $v."\n";
                }
                else
                {
                    # used to be this
                    push @s, $x.' {'.
                                       join(', ',
                                            map
                                            {
                                                $_.' => '.$self->{$hashname}->{$opt}->{$_}
                                            }keys %{$self->{$hashname}->{$opt}} 
                                       )."\}\n";
                }
            }
            elsif(ref $self->{$hashname}->{$opt} eq 'CODE')
            {
                push @s, "$x CODE\n";
            }
            # defined scalar
            else
            {
                push @s, $x.' '.$self->{$hashname}->{$opt}."\n";
            }
        }
        push @s, "\n\n";
    }
    
    my $s = join ('',@s);
    if(defined $prepend && $prepend ne '')
    { 
        $s=~s/(^|\n)/$1$prepend/g;
        chomp $s;
        $s.="\n";
    }
    return $s;
}

sub check_for_system_error
{
    # check for a system error
    my $self = shift;
    my $x = shift;
    if($self->{_grid_options}{add_up_system_errors} &&
       (($self->{_grid_options}{return_array_refs} && 
         $$x[0] eq 'SYSTEM_ERROR') ||
        $x=~/^SYSTEM_ERROR/o))
    { 
        $self->report_system_error($x);
    }
}

sub report_system_error
{
    # report a system error
    my $self = shift;
    my $x = shift ; # error string

    $self->{_flexigrid}{failed_prob} += 
        $self->{_grid_options}{progenitor_hash}{prob};
    $self->{_flexigrid}{failed_count}++;

    if($self->{_grid_options}{log_system_errors})
    {
        my $file = $self->{_grid_options}{tmp}.'/system_errors';
        printf "Caught system error (so far on this thread (which is %d, count = %d, modulo %d, offset %d) %d failed, prob failed %g) : logged in $file\n",
            $self->{_threadinfo}{thread_number},
            $self->{_flexigrid}->{count},
            $self->{_flexigrid}->{modulo},
            $self->{_flexigrid}->{offset},
            
            $self->{_flexigrid}{failed_count},
            
            $self->{_flexigrid}{failed_prob};

        $self->{_threadinfo}{failed_count} = $self->{_flexigrid}{failed_count};
        $self->{_threadinfo}{failed_prob} = $self->{_flexigrid}{failed_prob};

        state $elock : shared;
        lock $elock;
        if(open(my $errfile,'>>',$file))
        {
            say {$errfile} "Thread ",$self->{_threadinfo}{thread_number},' running ',$self->{_grid_options}{command_line};
            say {$errfile} 'PID ',$self->{_grid_options}{process_ID},' in ',$self->{_grid_options}{working_directory},' at ',scalar(localtime());
            say {$errfile} $self->{_grid_options}{'args'};
            if(ref $x eq 'ARRAY')
            {
                say {$errfile} "\n@$x";
            }
            else
            {
                say {$errfile} "\n$x";
            }
            close $errfile;
        }
        else
        {
            say "Failed to open $file for writing : please check location of grid option 'tmp'";
            say "Thread ",$self->{_threadinfo}{thread_number},' running ',$self->{_grid_options}{command_line};
            say 'PID ',$self->{_grid_options}{process_ID},' in ',$self->{_grid_options}{working_directory},' at ',scalar(localtime());
            say $self->{_grid_options}{'args'};
            if(ref $x eq 'ARRAY')
            {
                say "\n@$x";
            }
            else
            {
                say "\n$x";
            }
            close $errfile;
        }
    }
}

############################################################
# flexigrid creation
############################################################

sub make_flexigrid
{
    my $self = shift;
    
    # make new user-defined flexigrid, put it in $gridcode and return it
    # (it is eval'd elsewhere)
    my $opts = $_[0];

    # if a file is specified, use it (for debugging)
    if(defined $self->{_grid_options}{gridcode_from_file})
    {
        if(-f $self->{_grid_options}{gridcode_from_file})
        {
            return rob_misc::slurp($self->{_grid_options}{gridcode_from_file});
        }
        else
        {
            print STDERR "Wanted to load gridcode from $self->{_grid_options}{gridcode_from_file} but this file does not exist\n";
            exit;
        }
    }                            
                        
    my $gridcode=' '; # variable to hold the final code
   
    $self->vbout(1,"make_flexigrid on self=$self\n");

    # make variable reverse lookup hash
    my %revhash;
    foreach my $k (reverse sort
		   grep {/grid variable/} 
		   keys %{$self->{_grid_options}->{flexigrid}})
    {
	my $s = $self->{_grid_options}->{flexigrid}{$k};
	my $nvar = ($k=~/grid variable (\d+)$/)[0];
	my $name = ref($s) ? $s->{name} : ($s=~/name (\S+)/)[0];
	$self->vbout(1,"Set revhash \$$name > \$v$nvar\n");
	$revhash{'$'.$name}='$v'.$nvar;
    }

    # make the grid code
    my $count = 1;
    my $central_block_done=0;
    my $maxcount;
    my $number_of_stars = 1; # used for monte carlo grid only
    my $monte_carlo_variables;
    $self->{_flexigrid}->{_varstub}='fvar';
        

    my @flexigrid_variables =  (sort grep {/grid variable/o} 
				keys %{$self->{_grid_options}->{flexigrid}});
    my $varlist;

    $varlist .= "\n\nmy \%$self->{_flexigrid}->{_varstub};\n";

    my $condor = 0;
    if($condor)
    {
    $varlist .= "

my \$condor_snapshot_file = binary_grid::condor::check_for_saved_snapshot();

if(defined(\$self->{_grid_options}->\{starting_snapshot_file\})
   ||
   defined(\$condor_snapshot_file)
   )
{
    # call condor pre-load hook
    if(defined \$condor_snapshot_file)
    {
	\$self->{_grid_options}->\{starting_snapshot_file\} = 
	    \$condor_snapshot_file;
    }

    print \"Load snapshot file from \$self->{_grid_options}->\{starting_snapshot_file\}\\n\";
    %fvar = %{\$self->load_snapshot(\$self->{_grid_options}->\{starting_snapshot_file\})};
    print \"Loaded snapshot: initial grid parameters are:\\n\";
    foreach my \$variable (keys \%fvar)
    {
	print \"\$variable :: \$fvar\{\$variable\}\{value\}\\n\";
    }

    # prevent loading twice
    \$self->{_grid_options}->\{starting_snapshot_file\} = undef;

    # call condor post-load hook
    if(defined \$condor_snapshot_file)
    {
	\$self->post_load_snapshot(\$condor_snapshot_file);
    }

}
    \n";
    }

    if($self->{_grid_options}->{flexigrid}{'grid type'} eq 'list')
    {
        # we want to run a list of systems rather than a grid
        if(defined $self->{_grid_options}->{flexigrid}{'list filename'})
        {
            $self->vbout(2,"running from a list from file at ".
                         $self->{_grid_options}->{flexigrid}{'list filename'}."\n");             
            $gridcode .= "
    
    open(\$self->\{_grid_options\}->\{flexigrid\}\{listFP\},
         '<',
         \$self->\{_grid_options\}->\{flexigrid\}\{'list filename'\})
    || 
    die(\"Cannot open list of systems which is supposed to be at \\\$self->\{_grid_options\}->\{flexigrid\}\{'list filename'\} = \\\"\$self->\{_grid_options\}->\{flexigrid\}\{'list filename'\}\\\"\");

            while(my \$stellar_system = readline(\$self->\{_grid_options\}->\{flexigrid\}\{listFP\}))
                \{
            print \"RUN \$stellar_system \\n\";
            \$self->queue_arglist_run(\$self->\{_flexigrid\}->\{thread_q\},\$stellar_system);
            \$starcount++;
            \}

            close(\$self->\{_grid_options\}->\{flexigrid\}\{listFP\});
            ";

        }
        elsif(defined $self->{_grid_options}->{flexigrid}{'list reference'} &&
              ref $self->{_grid_options}->{flexigrid}{'list reference'} eq 'ARRAY')
        {
            # use list of stars given in Perl array
            $self->vbout(2,"running from a list from array at ".
                         $self->{_grid_options}->{flexigrid}{'list reference'}."\n");
            
            $gridcode .= "

                foreach my \$stellar_system (\@\{\$self->\{_grid_options\}->\{flexigrid\}\{'list reference'\}\})
                \{
            if(ref \$stellar_system eq 'HASH')
                \{
            # arg list hash
            \$self->queue_evolution_code_run(
                    \$self->{_flexigrid}->{thread_q},
                    \$stellar_system
                );
            \}
            else
                \{
            # arg list string 
            \$self->queue_arglist_run(\$self->{_flexigrid}->\{thread_q\},\$stellar_system);
            \}
            \$starcount++;
            \}

            ";

        }
        else
        {
            print "WARNING: we are trying to run a list of systems, but I don't know from where I should get the list data\n";
        }

    }
    else
    {
        foreach my $k (reverse @flexigrid_variables)
        {
            # gri variable options
            my $s = $self->{_grid_options}->{flexigrid}->{$k};

            if(!ref($s))
            {
                say 'Old regexp-based flexigrid is deprecated: please use hashes for all the latest functionality!';
                exit;
            }

            # extract variable number
            my $nvar = ($k=~/grid variable\s+(\d+)$/)[0];
                        
            my $longname = $s->{longname};
            my $probdist = $s->{probdist};

            # default to a grid-like, central distribution
            $s->{gridtype} //= 'central'; # edge or central
            $s->{method} //= 'grid'; # grid or 'monte carlo'

            # no verbosity by default
            $s->{vb} //= 0;
            
            # clean up probability distribution code
            $probdist=~s/\n/ /g;
            $probdist=~s/^\s+//g;
            $probdist=~s/^\s+$//g;

            # clean up longname
            $longname=~s/_/ /go;

            # update maximum variable count 
            $maxcount //= $nvar;

            # verbose output
            $self->vbout(1,"Grid variable $nvar : Range @{$s->{range}} : Resolution $s->{resolution}\n");

            # make grid code loop chunk
            my $v = '$'.$self->{_flexigrid}->{_varstub}.'{'.$s->{name}.'}'; # variable string

            my %subgrid;
            
            $subgrid{start} = "\n" if($nvar==0);

            if(defined($s->{condition}))
            { 
                $subgrid{start} .= "
                if($s->{condition})
                \{
                # check for condition
                ";
            }

            # start block
            $subgrid{start} .= "
                \# grid variable v$nvar is $s->{name} ($longname)
                \# range $s->{range}->[0] to $s->{range}->[1] in $s->{resolution} steps 
                $s->{preloopcode}
                \$self->{_flexigrid}{resolution}{n}{'$k'}=eval $s->{resolution};
                $v\{nvar\} = $nvar; # set variable number
                $v\{min\} = $s->{range}->[0]; # minimum value
                $v\{max\} = $s->{range}->[1]; # maximum value 
                $v\{delta\} = $v\{max\} - $v\{min\}; # max - min
                \$starcount\[$nvar\] = 0; 
                \$probsum\[$nvar\] = 0.0; 
            ";
            
            if($s->{method} eq 'grid')
            {
                # standard spaced grid
                $subgrid{start} .=
                    " 

$v\{d\}=spacing_functions::$s->{spacingfunc};
                ";
                if($s->{vb})
                {
                    $subgrid{start} .= 
"
                print \"GRIDN gridcode $k eval $s->{resolution} gave \$self->{_flexigrid}{resolution}{n}{'$k'} (error \$\@) keys \",join(' ',keys \%\{\$self->\{_flexigrid\}\{resolution\}\}),\"\\n\";
                    ";
                }
                
                if(defined $s->{gridtype} && $s->{gridtype} eq 'edge')
                {
                    # edge based grid
                    $subgrid{start} .= 
"
                    $v\{value\}=$v\{min\} if(!defined( $v\{value\} ));
                    $v\{type\}='edge';
                    ";
                    if($s->{vb})
                    {
                        $subgrid{start} .= "
                    print \"WHILE1a start edge var $v ($s->{name}) from $v\{value\} shift is $v\{d\}\\n\"; 
                        ";
                    }

                    $subgrid{start} .= "
                    while($v\{value\} - $v\{max\}  <=  + 1e-6 * $v\{d\} || $v\{d\}==0.0)
                    \{
                    ";
                        
                    if($s->{vb})
                    {
                        $subgrid{start} .= "
                        print \"WHILE1b edge var $v ($s->{name}) is $v\{value\} shift is $v\{d\} binary (2) = \$self->{_grid_options}->{binary}\\n\"; 
                        ";
                    }
                }
                else
                {
                    # centred grid
                    $subgrid{start} .= "
                    $v\{value\}=$v\{min\} if(!defined( $v\{value\} ));
                    $v\{type\}='centred';
                    ";
                    if($s->{vb})
                    {
                        $subgrid{start} .= "
                    print \"WHILE2a start grid var $v ($s->{name}) from $v\{value\} shift is $v\{d\} binary = \$self->{_grid_options}->{binary} (<$v\{max\})\\n\"; 
                    ";
                    }
                    $subgrid{start} .= "
                    while($v\{value\} - $v\{max\} < - 1e-6 * $v\{d\} || $v\{d\}==0.0)
                    \{
                        $v\{value\}+=0.5*$v\{d\};
                    ";
                    
                    if($s->{vb})
                    {
                        $subgrid{start} .= "
                        print \"WHILE2b grid var $v ($s->{name}) is $v\{value\} shift is $v\{d\}\\n\"; 
                        ";
                    }
                   
                }
                
                $subgrid{start} .= "
                      
                    $s->{precode};

                    \# calculate contribution of $longname to the phase volume
                    \# and the probability
                    \$phasevol *= $v\{d\} if($v\{d\}>0.0);

                    my \$dprob$s->{name} = ($v\{d\} > 0.0 ? $v\{d\} : 1.0) * distribution_functions::$probdist;
                    ";


            }
            elsif($s->{'method'} eq 'monte carlo')
            {
                ############################################################
                # Monte carlo
                ############################################################
                $number_of_stars *= $s->{resolution};
                
                say "VAR $nvar  : $number_of_stars";

                # the number of points required in the lookup table
                #
                # Note that we use a very crude integration process:
                #
                # We simply divide the space over x into this many even dx
                # and integrate these. We could do this much better,
                # but (presumably) slower.
                #
                my $MCres = $s->{'monte carlo lookup resolution'} // 1000;
                
                $subgrid{start} .=
                    "
                    \# variable $nvar is $v\{value\} ($longname)
                    
                    $v\{d\} = $v\{delta\} / \$self->{_flexigrid}{resolution}{n}{'$k'};

                # set up monte carlo table
       if(!defined \$monte_carlo_cache\[$nvar\])
       {
        # make cumulative density function (CDF) table
        my \$dx = $v\{delta\} / $MCres;
        my \$dxhalf = 0.5 * \$dx;
        my \$ptot = 0.0;
        \$monte_carlo_cache\[$nvar\]->\{hash\} = \{\};
        \$monte_carlo_cache\[$nvar\]->\{array\} = \[\];

        for($v\{value\} = $v\{min\}; $v\{value\} <= $v\{max\} + 1e-3*\$dx; )
        {
            # use the given precode and distribution function
            # to calculate the probability density function
            $v\{value\} += \$dxhalf;
            $s->{precode};

            my \$dphalf = \$dxhalf * distribution_functions::$probdist;
            \$ptot += \$dphalf;
            
            # put in array and hash for interpolation
            \$monte_carlo_cache\[$nvar\]->\{hash\}->{\$ptot} = $v\{value\};
                        
            $v\{value\} += \$dxhalf;
            \$ptot += \$dphalf;                
        }

         # normalizing factor
         \$monte_carlo_cache\[$nvar\]->\{probtot\} = \$ptot;

         {
                # normalize
                my \$newhash = \{\};
                my \$pwas; my \$xwas;
                foreach my \$x (nsort keys \%\{\$monte_carlo_cache\[$nvar\]->\{hash\}\})
                {
                    \$newhash->\{ \$x / \$ptot  \} = \$monte_carlo_cache\[$nvar\]->\{hash\}->{\$x};
                }
                \$monte_carlo_cache\[$nvar\]->\{hash\} = \$newhash;
                \@\{\$monte_carlo_cache\[$nvar\]->\{array\}\} = nsort keys \%\{\$newhash\};
                
         }
       } # end monte carlo setup for variable \$nvar

                # prob = 1/N : same for all stars so calculate once here
                my \$dprob$s->{name} = (1.0 * (\$monte_carlo_cache\[$nvar\]->\{probtot\})) / 
                                       (1.0 * (__NUMBER_OF_STARS__));

                    while(\$starcount\[$nvar\] < $s->{resolution})
                    \{  
                         \$phasevol *= $v\{d\} if($v\{d\}>0.0);

                         $v\{value\} = 
                             generic_interpolation_wrapper(
                               rand(),
                               \$monte_carlo_cache[$nvar]->{array},
                               \$monte_carlo_cache[$nvar]->{hash}
                         );

                         $s->{precode};
                                   

                ";
            }
            else
            {
                Carp::confess("Unknown grid variable method : can be either 'grid' or 'monte carlo'\n");
            }

            # only multiply by a probability if one is defined
            my $dprob = defined($s->{noprobdist}) ? '1.0' : "\$dprob$s->{name}";

            # start probability at 1.0 if this is the first variable
            # to have a probability
            if($nvar==0)
            {
                $subgrid{start}.= "
                    \$prob\[$nvar\] = $dprob;
                ";
            }
            else
            {
                $subgrid{start}.=  "
                    \$prob\[$nvar\] = \$prob\[".($nvar-1)."\]*$dprob;
                ";

                # save the probability for this variable
                $subgrid{start}.="
                    \$probsum\[$nvar\] += $dprob;
                ";
            }

            
            
            # increment the starcount
            $subgrid{start} .= "
                    \$starcount\[$nvar\]++;
            ";
            
            if($self->{_grid_options}->{flexigrid}{'grid type'} ne 'list')
            {
                if(!$central_block_done)
                {	
                    # calculate probability
                    $subgrid{end} ="
                    my \$prob = \$self->{_grid_options}->{weight} * \$prob\[$nvar\];
            ";
                    
                    
                    $subgrid{end} .= "
                            __CENTRAL_BLOCK__
                            ";
                    $central_block_done=1;
                }
            }
            $subgrid{end} .= $s->{postcode} if(defined($s->{postcode}));

            if($s->{method} eq 'grid')
            {
                # end standard grid
                if($s->{gridtype} eq 'edge')
                {
                    $subgrid{end} .= "
                            \# shift to the edge of the next cell
                            $v\{value\}+=$v\{d\};
                    ";
                    if($s->{vb})
                    {
                        $subgrid{end} .= "
                        print \"WHILE3a edge var $v shift up to $v\{value\}\\n\"; 
                    "; 	  
                    }  
                }
                else
                {
                    $subgrid{end} .= "

                        \# shift to next cell boundary and remove the contribution
                        \# of $longname to the phase volume and probability
                        $v\{value\}+=0.5*$v\{d\};
                    ";
                    if($s->{vb})
                    {
                               $subgrid{end} .= "
                        print \"WHILE3b grid var $v shift up to $v\{value\}\\n\"; 
                               ";
                    }
                }

                $subgrid{end} .= "
                        last if($v\{d\}==0.0);
                        \$phasevol /= $v\{d\} if($v\{d\}>0.0);
                        \# update grid spacing for $s->{name}
                        $v\{d\}=spacing_functions::$s->{spacingfunc};
                ";

            }
            elsif($s->{'method'} eq 'monte carlo')
            {
                # nothing required here
            }

            $subgrid{end} .= "
                      __UPDATE_GRID_SPACING__
                    \}

                            $v\{value\}=undef;
                \$monte_carlo_cache\[$nvar\] = undef;

                # post loop code
                $s->{postloopcode};
            ";
            
            if(defined($s->{condition}))
            { 
                $subgrid{end} .= "
                            # end condition true
                            \}
                else
                            \{
                # condition untrue
                my \$prob = \$self->{_grid_options}->{weight} * \$prob\[".($nvar-1)."\];
                __CENTRAL_BLOCK__;
                # end condition untrue
                \}
                ";
            }

            # build into global gridcode loop 
            $gridcode = ($subgrid{start} // '') . $gridcode . ($subgrid{end} // '');

            # update variable counter
            $count++;
        }
    }

    # prepend standard variable list    
    $gridcode = $varlist.$gridcode;
    
    # make commands that depend on whether we are single or binary
    my $tbse;
    my $progenitor_string;
    my $probability_function;
    
    # real run of the evolution code
    $tbse = <<'TBSE_STAR';
    # if($repeat_prob>0.0)
    {
    my $system;
    if($self->{_grid_options}->{binary})
    {
    # binary star
    $system={
    M_1=>$m1,
    M_2=>$m2,
    metallicity=>$self->metallicity(),
    orbital_period=>$per,
    eccentricity=>$eccentricity,
    probability=>$repeat_prob,
    phasevol=>$phasevol
    };
    }
    else
    {
    # single star
    $system={
    M_1=>$m1,
    M_2=>0.01,
    metallicity=>$self->metallicity(),
    orbital_period=>$self->{_grid_options}->{single_star_period},
    eccentricity=>0.0,
    probability=>$repeat_prob,
    phasevol=>$phasevol
    };
    } 

    $self->queue_evolution_code_run($self->{_flexigrid}->{thread_q},
                                    $system);
    
    }
TBSE_STAR


# make central block
my $central_block= <<'END_CENTRAL_BLOCK';

    # central flexigrid code :
    # call probability function and tbse (perhaps repeatedly)
    
    my $repeat_prob = $prob/$self->{_grid_options}->{repeat};

    $starcount++;

    __REPEAT_LOOP__
    {
        # now run the star
        {
        $self->increment_probtot($prob);
        __TBSE_COMMAND__
        }

        $self->increment_count();
        
        # verbose logging
        if($self->verbose_output(1))
        {
        if(
    time() > $self->{_flexigrid}{nextlogtime} 
        &&
    ($self->{_grid_options}->{nmod}==0 ||
     $self->{_flexigrid}->{count} % $self->{_grid_options}->{nmod} == 0)
            )
	    {	
		if($self->{_grid_options}->{binary})
		{
		    $self->vb1out((
                        sprintf "binary=%d weight=%g M1=%3.3e M2=%3.3e SEP=%3.3e PER=%3.3e%sZ=%g",
                        $self->{_grid_options}->{binary},
                        $self->{_grid_options}->{weight},
                        $m1,
                        $m2,
                        $sep,
                        $per,
                        $eccentricity!=0.0 ? sprintf " e=%g ",$eccentricity : ' ',
                        $self->metallicity()),
                                  $self->{_flexigrid}->{nthreads});
		}
		else
		{
		    $self->vb1out((sprintf "binary=%d weight=%g M=%2.2e Z=%g ",
                                   $self->{_grid_options}->{binary},
                                   $self->{_grid_options}->{weight},
                                   $m1,
                                   $self->metallicity),
                                  $self->{_flexigrid}->{nthreads});
                }
	    }
	}

	# log grid points for gnuplot's xyerrorbars mode
	#printf "XXX %g %g %g %g %g %g\n",$lnm1,$m2/$m1,$lnsep,0.5*$dlnm1,0.5*$dm2/$m1,0.5*$dlnsep;
    }

END_CENTRAL_BLOCK

    $central_block=~s/__TBSE_COMMAND__/$tbse/g;

    # perhaps insert a repeat loop
    my $repeat_loop=' ';
    if($self->{_grid_options}->{repeat}>1)
    {
	$repeat_loop=<<'REPEAT_LOOP';
	    for(my $nrepeat=0;$nrepeat<$self->{_grid_options}->{repeat};$nrepeat++)
REPEAT_LOOP
    }

    # replace block placeholders with actual code
    $central_block=~s/__REPEAT_LOOP__/$repeat_loop/g;
    $gridcode=~s/__CENTRAL_BLOCK__/$central_block/g;
    $gridcode=~s/__NUMBER_OF_STARS__/$number_of_stars/g;
    $gridcode=~s/__UPDATE_GRID_SPACING__/\$self->check_for_required_actions(\\\%fvar);\n\$self->do_required_actions(\\\%fvar);/;
    $gridcode=~s/__UPDATE_GRID_SPACING__//go;
		 
    # convert variable names
    foreach my $kk (@flexigrid_variables)
    {
	my $name=$self->{_grid_options}->{flexigrid}{$kk}->{name};
	$gridcode=~s/\$$name\b/\$fvar\{'$name'\}\{value\}/g;
    }

    # add leading and trailing blocks
        
    my $leading_block = <<'LEADING_BLOCK';
    use Memoize;
    Memoize::memoize('distribution_functions::const');
    Memoize::memoize('distribution_functions::flatsections');
    Memoize::memoize('spacing_functions::const');
    
    my $phasevol=1.0;
    my @prob;
    my @probsum;
    my @monte_carlo_cache;

    my @starcount;
    $starcount=0;

    __DEFAULT_VARS__
LEADING_BLOCK

    my $trailing_block= <<'TRAILING_BLOCK';
    say "Flexigrid loops thought we ran $starcount stars (@starcount)";
    say "Total probability = $self->{_flexigrid}{probtot}";
    
TRAILING_BLOCK

    $leading_block=~s/__DEFAULT_VARS__/$self->{_flexigrid}{default_vars}/g;


    # add leading and trailing block, wrapping brackets
    no warnings;
    $gridcode = "{\n".$self->{_grid_options}->{flexigrid}{precode}."\n".$leading_block.$gridcode.$trailing_block.$self->{_grid_options}->{flexigrid}{postcode}."\n}\nreturn 1;\n";
    use warnings;

    # optimize
    $gridcode = $self->optimize_gridcode($gridcode);
    
    # final indentation
    $gridcode = $self->indent_gridcode($gridcode);

    # finally remove comment lines
    $gridcode=~s/\n\s*\#.*//g;

    if(!$self->{_grid_options}->{no_gridcode_dump})
    {
	# verbose output
	$self->output_gridcode_with_linenumbers($gridcode) ;
	$self->output_gridcode($gridcode);
    }

    # and return
    return $gridcode;
}

############################################################
# flexigrid: threads
############################################################


sub launch_flexigrid_threads
{
    my $self = shift;
    
    # launch the $self->{_flexigrid}->{nthreads} threads for use by the flexigrid
    # and put them in a thread_q object
    my @threads;

    my $thread_q = Thread::Queue->new();
    
    # set max number in queue
    $thread_q->limit = $self->{_grid_options}->{maxq_per_thread};
        
    # loop to launch nthreads threads
    for(my $n=0; $n < $self->{_flexigrid}->{nthreads}; $n++)
    {
	my $h={
            thread_number=>$n
        };

	$self->vbout(1,"launch_flexigrid_threads : launch thread $n/$self->{_flexigrid}->{nthreads}\n");

	keys (%$h) = $self->{_grid_options}->{thread_hash_prealloc}
	    if(defined $self->{_grid_options}->{thread_hash_prealloc});

	# perhaps we have a pre-create function which should be called
	&{$self->{_grid_options}->{thread_precreate_function_pointer}}($self,$h)
	    if(defined($self->{_grid_options}->{thread_precreate_function_pointer}));

	# create the thread, pass it $h
	# note: $h stays alive because it is passed to the thread
	($threads[$n]) = threads->create(
            { 
                context => 'list',
                exit    => 'thread_only',
            },
            sub{
                $self->run_flexigrid_thread($thread_q,$n,$h);
            })
        || $self->thread_create_error($@);

	# double check for errors
	$self->thread_create_error($@) if($@);

	# perhaps we have a post-create function which should be called
	&{$self->{_grid_options}->{thread_postrun_function_pointer}}($self,$h)
	    if(defined($self->{_grid_options}->{thread_postrun_function_pointer}));
    }

    say 'All threads launched';
    
    return (\@threads,$thread_q);
}

sub thread_create_error
{
    my $self = shift;
    # error handler for thread creation
    my $err=$_[0];
    say "Binary_grid : thread creation error \"$err\"\nStack size: ",threads->get_stack_size();
    exit;
}

############################################################
# flexigrid: code output and associated functions
############################################################

sub optimize_gridcode
{
    my ($self, $gridcode) = @_;
    
    # move constant definitions to the top
    my $consts;
    while($gridcode =~ s/(\$fvar\{\S+\}\{\S+\}\s*=\s*-?\d+(\.\d+)?\s*;)/\n/)
    {
        $consts .= $1."\n";
    }
    $gridcode =~ s/(my \%fvar;\s*\n)/$1\n\n$consts\n\n/;

    return $gridcode;
}

sub fast_gridcode
{
    # make a fast, stripped down grid for only counting
    # the number of stars
    my $default_opts =
    {
        keep_probabilities => 0,
        keep_phasevol => 0,
        system_code => '',
        use_increment_count => 0,
        keep_vbout => 0,
        keep_actions => 0,
        gridcode => undef,
    };

    my ($self, $opts) = @_;
    $opts //= {};
    
    # make list of options : use defaults then those
    # passed in
    my $opts = {
        %$default_opts,
            %$opts,
    };
    
    my $gridcode_fast = $opts->{gridcode} //
        $self->make_flexigrid();

    $gridcode_fast =~s/\$self->queue_evolution_code.*\n.*\n//g;
    if(!$opts->{keep_actions})
    {
        $gridcode_fast =~s/\$self->check_for_required_actions.*//g;
        $gridcode_fast =~s/\$self->do_required_actions.*//g;
    }
    if(!$opts->{keep_vbout})
    {
        $gridcode_fast =~s/if\s*\(\s*\$self->verbose_output\s*\(\s*\d+\s*\)\s*\)/if\(0\)/g;
    }
    
    $gridcode_fast =~s/(\{\s*my\s*\$proberr)/if(0)$1/g;
    
    #$gridcode_fast =~s/\$self->\{_flexigrid\}\{resolution\}\{n\}.*//g;
    $gridcode_fast =~s/\$fvar\{\S+\}\{type\}=.*//g;
    if(!$opts->{keep_probabilities})
    {
        $gridcode_fast =~s/\$probsum\[\d+\]\s*\+?=.*//g;
        $gridcode_fast =~s/\$prob\[\d+\]\s*=.*//g;
        $gridcode_fast =~s/my\s*\$dprob.*//g;
    }
    if(!$opts->{keep_phasevol})
    {
        $gridcode_fast =~s/my\s*\$phasevol.*//g;
        $gridcode_fast =~s/\$phasevol\s*[\*\/]=.*//g;
        $gridcode_fast =~s!phasevol=>.*!phasevol=>1.0,!g;
    }
    if(!$opts->{keep_probabilities})
    {
        $gridcode_fast =~s/my\s*\$prob.*//g;
        $gridcode_fast =~s/my\s*\$repeat_prob.*//g;
        $gridcode_fast =~s!\$repeat_prob!\$prob!g;
        
        $gridcode_fast =~s!probability=>.*!probability=>1.0,!g;
        $gridcode_fast =~s!\$proberr!(0.0)!g;
    }

    $gridcode_fast =~s!\$self->increment_probtot.*!!g;
    
    #print "Use system code $opts->{system_code}\n";
    $gridcode_fast =~s!(\{\s*+\{\s*my\s*\$system)!$opts->{system_code};\nif(0)$1!g;
    if(!$opts->{use_increment_count})
    {
        $gridcode_fast =~s!\$self->increment_count\(\);!!g;
    }
    
    return $gridcode_fast;
}

sub indent_gridcode
{
    my ($self,$gridcode) = @_;
    # (very) simple Perl indenter
    my @x;
    my $n=0;
    my $space='    ';
    $gridcode=~s/my\$/my \$/go;
    $gridcode=~s/;my \$/;\nmy \$/go;
    map
    {
	if(!/^\s*#/o)
	{
	    s/^\s*//o;
	    my $indent;
	    if(/^\}/o)
	    {
		$n--;
		$indent=$space x $n;
	    }
	    else
	    {
		$indent=$space x $n;
		$n++ if(/^{/o);
	    }
	    s/^/$indent/;
	}
	else
	{
	    my $indent=$space x $n;
	    s/^\s*/$indent/;
	}
	push(@x,$_);
    }split(/\n/o,$gridcode);
    $gridcode=join("\n",@x);
    $gridcode=~s/\n\s*\n/\n\n/go;
    return $gridcode;
}

sub output_gridcode
{
    my $self = shift;
    my $gridcode;
    my $file = "$self->{_grid_options}->{tmp}/gridcode.clean";
    if(open($gridcode,'>',$file.'.tmp'))
    {
        print {$gridcode} "#!/usr/bin/env perl\n".$_[0];
        close $gridcode;

        state $have_emacs;
        if(!defined $have_emacs)
        {
            $have_emacs = (`emacs --version 2>/dev/null`=~/GNU Emacs/) ? 1 : 0;
        }

        if($have_emacs)
        {
            # if we have emacs, indent the gridcode.clean
            unlink $file;
            `emacs -q -batch $file.tmp --eval '(indent-region (point-min) (point-max) nil)' -f save-buffer 2>/dev/null`;
            copy($file.'.tmp',$file);
        }
    }
    else
    {
        print STDERR "WARNING : unable to open gridcode.clean at $self->{_grid_options}->{tmp}/gridcode.clean : check grid_option 'tmp'\n";
    }
}

sub output_gridcode_with_linenumbers
{
    my $self = shift;
    my $x = $self->gridcode_with_linenumbers($_[0]);
    my $gridcode;
    if(open($gridcode,">$self->{_grid_options}->{tmp}/gridcode"))
    {
        map
        {
            $self->vbout(2,$_);
            print {$gridcode} $_;
        }@$x;
        close $gridcode;
    }
    else
    {
        print STDERR "WARNING : unable to open gridcode at $self->{_grid_options}->{tmp}/gridcode : check grid_option 'tmp'\n";
    }
}

sub gridcode_with_linenumbers
{
    my $self = shift;
    # output grid code with line numbers
    my $n=1; 
    my $gridcode=$_[0];
    my @a;
    if(defined($gridcode))
    {
	my @colours=('white','red','red','green','blue','yellow','magenta');
	my $colour=0;
	my $ncolours=$#colours;
	map
	{
	    my $fn=sprintf"% 4d",$n;
	    if(!/^\s*\#/)
	    {
		if(/^\s*\{/)
		{
		    $colour++;
		    push(@a, 
			 $fn,': ',$_,"\n",
			 $colour{$colours[$colour%$ncolours]});
		}
		elsif(/^\s*\}/)
		{
		    $colour--;
		    push(@a,
			 $colour{$colours[$colour%$ncolours]},
			 $fn,': ',$_,"\n");
		}
		else
		{
		    push(@a,$fn,': ',$_,"\n");
		}
	    }
	    else
	    {
		push(@a,$fn,': ',$_,"\n");
	    }
	    $n++;
	}split(/\n/o,$gridcode);
    }
    else
    {
	say "ERROR \$gridcode=undef";
	exit;
    }
    push(@a, $colour{reset});
    return (\@a);
}

############################################################
# Run stellar evolution code
############################################################

sub queue_evolution_code_run
{
    my ($self,$thread_q,$arghash) = @_;

    # Queue evolution code job with the given arguments

    # must be before the start_at point
    return if($self->{_flexigrid}->{count} < $self->{_grid_options}->{start_at});
    # must have correct modulo/offset combination
    return if(($self->{_flexigrid}->{count}+$self->{_grid_options}->{offset}) 
	      % $self->{_grid_options}->{modulo} != 0);
        
    # make arg hash by joining the system passed in to bse_options 
    my $args = $self->make_evcode_arghash($arghash);
    
    $self->vbout(2,"thread_q: send to enqueue loop $args on thread_q=$thread_q (currently ".$thread_q->pending()." pending, maxq =  $self->{_grid_options}->{maxq_per_thread})\n\n");
  
    # enqueue
    $self->vbout(2,'thread_q: enqueue '.$args."\n\n");
    $thread_q->enqueue($args);
}


sub queue_arglist_run
{
    my $self = shift;
    # Queue evolution code job with an already-constructed
    # list of arguments
    # but do nothing if modulo/offset combo is wrong
    return if(($self->{_flexigrid}->{count}+$self->{_grid_options}->{offset}) 
	      % $self->{_grid_options}->{modulo} != 0);
    
    my ($thread_q,$argstring)=@_;
    return if((!defined $argstring) || $argstring eq '');

    # construct $args hash (pointer) from $argstring 
    my $args={};
    foreach my $chunk (grep {$_ ne ''} split(/--/o,$argstring))
    {
        my @d = split(' ',$chunk);
        my $k = shift @d;
        $args->{$k} = "@d";
        print "Make $k : @d\n";
    }

    $self->vbout(2,"thread_q: send to enqueue loop $args on thread_q=$thread_q (currently ".$thread_q->pending()." pending, maxq= $self->{_grid_options}->{maxq_per_thread}) via queue_arglist_run\n\n");

    # enqueue
    $self->vbout(2,'thread_q: enqueue '.$args."\n\n");
    $thread_q->enqueue($args);
}



sub human_error_string
{
    my $self = shift;

    # add 'human-readable' error message
    my ($err)=(@_);
    my $extra;

    if($err=~/division by zero/i)
    {
	$extra .= "There was a division by zero error. Common causes include\n * setup of the normalization constants for your chosen distribution function - it must be normalizable\n";
    }

    if($extra)
    {
	$err .= $colour{'red on_white'}."\n".$extra."\n".$colour{reset};
    }
    return $err;
}


sub flexigrid_error
{
    my $self = shift;
    # generic shutdown on error
    my $err = $self->human_error_string($_[0]); 
    my $gridcode=$_[1];

    say 'Flexigrid_error';
    
    # output grid code with line numbers
    #$self->output_gridcode_with_linenumbers($gridcode);
    
    say "Errors: ",$colour{'red bold'},$err,$colour{reset};
    if($self->{_grid_options}->{tvb})
    {
        $self->thread_vb("flexigrid_eval_error : $err\n")
    }

    # print surrounding failed code
    if($err=~/line (\d+)/)
    {
	my $n=$1; # line number of the error
	my @g=split(/\n/,$gridcode);
	say;
 	for(my $i=rob_misc::MAX(0,$n-10); $i<rob_misc::MIN($#g,$n+10); $i++)
	{
	    printf "%d: %s\n",$i,$g[$i];
	}
	say;
    }

    say 'Exit at flexigrid_error';    

    $self->shutdown_binary_grid();
    
    # close all threads if they haven't already exited themselves
    {
	no warnings 'threads';
	threads->exit() if threads->can('exit');
	map{$_->exit if($_->can('exit'));}(threads->list(threads::all));
    }

    # exit program
    exit;
}

sub flexigrid_eval_error
{
    my $self = shift;
    my $err=$_[0]; 
    my $gridcode=$_[1];
    say 'EVAL returned undef hence ERROR';
    
    # output grid code with line numbers
    #$self->output_gridcode_with_linenumbers($gridcode);
    
    say "Errors: ",$colour{'red bold'},$err,$colour{reset};

    if($self->{_grid_options}->{tvb})
    {
        $self->thread_vb("flexigrid_eval_error : $err\n")
    }

    # print surrounding failed code
    if($err=~/line (\d+)/)
    {
	my $n=$1; # line number of the error
	my @g=split(/\n/,$gridcode);
	say;
 	for(my $i=rob_misc::MAX(0,$n-10); $i<rob_misc::MIN($#g,$n+10); $i++)
	{
	    printf "%d: %s\n",$i,$g[$i];
	}
	say;
    }

    say 'Exit at flexigrid_eval_error';    

    $self->shutdown_binary_grid();
    
    # close all threads if they haven't already exited themselves
    {
	no warnings 'threads';
	threads->exit() if threads->can('exit');
	map{$_->exit if($_->can('exit'));}(threads->list(threads::all));
    }

    # exit program
    exit;
}


############################################################
# misc other functions
############################################################



sub get_from_header

{    my $self = shift;
     # get a C macro value from a typical C header file
     #
     # new get_from_header, uses get_from_C_header function in rob_misc
     # (requires gcc but most people have it)
     return (
         rob_misc::get_from_C_header(
             $_[0],
             $self->{_grid_options}->{srcpath}.'/'.$_[1])
         );
}

sub metallicity
{
   my $self = shift;
   # old code used z for the metallicity, but really
   # we should use the key 'metallicity'
   return $self->{_bse_options}->{metallicity} // $self->{_bse_options}->{z};
}

sub nuclear_mass_list
{
    my $self = shift;
    my @masses;
    foreach (grep{/^Isotope/}split(/\n+/,$self->evcode_version_string()))
    {
        if(/^Isotope (\d+) is \S+ \(mass=\S+ g \(\/amu=([^\)]+)\), /)
        {
            $masses[$1] = $2;
        }
    }
    return @masses;
}


sub nuclear_mass_hash
{
    my $self = shift;
    my %masses;
    foreach (grep{/^Isotope/}split(/\n+/,$self->evcode_version_string()))
    {
        if(/^Isotope \d+ is (\S+) \(mass=\S+ g \(\/amu=([^\)]+)\), /)
        {
            $masses{$1} = $2;
        }
    }
    return %masses;
}


sub isotope_hash
{
    my $self = shift;
    my %isotopes;
    foreach (grep{/^Isotope/}split(/\n+/,$self->evcode_version_string()))
    {
        if(/^Isotope (\d+) is (\S+)/)
        {
            $isotopes{$2} = $1;
        }
    }
    return %isotopes;
}

sub initial_abundance_hash
{ 
    my $self = shift;

    # return a hash of the initial abundances from binary_c
    my ($mix,$Z) = @_;

    # these should match binary_c
    my %mixes = (
        AG89=>0,
        Karakas2002=>1,
        Lodders2003=>2,
        Asplund2005=>3,
        GarciaBerro=>4,
        GrevesseNoels=>5,
        Asplund2009=>6,
        Kobayashi2011=>7,
        Lodders2010=>8,
        );

    # use alphanumerc key if given
    $mix = $mixes{$mix} if(!is_numeric($mix));

    if(!defined $mix)
    {
	Carp::confess ("initial_abundance_hash wants either an integer or a string corresponding to one of ".join(',',sort {$mixes{$a}<=>$mixes{$b}} keys %mixes),"\n");
    }
    elsif(!defined $Z)
    {
        Carp::confess ("initial_abundance_hash requires a metallicity as its second argument");
    }
    
    my @binc = split(/\n/,$self->initial_abundance_string($mix,$Z));
    my %ab;
    while($_ = shift @binc)
    {
	if(/(\S+)=>(\S+),\s+\#\s+(\d+)/)
	{
	    $ab{$1}=$2;
	}
    }
    return %ab;
}

sub agbperiod
{
    my $self = shift; # might be undef, not really required
    my $m1=$_[0];
    my $m2=$_[1];

    # given m1,m2 calculate the period at which RLOF starts when star1
    # reaches its first TP
    
    # RL/a = $rl
    my $r=binary_stars::ragb($m1);
    my $frl=binary_stars::roche_lobe($m1/$m2);
    
    # hence the separation at which RLOF occurs
    my $aRL = $r/$frl;
    
    # hence the period
    return binary_stars::calc_period_from_sep($m1,$m2,$aRL);
}

sub stellar_type_strings
{
    my $self = shift;
    # return some lists of stellar type 
    my $x = rob_misc::slurp($self->{_grid_options}->{srcpath}.'/binary_c_stellar_types.h');
    return undef if(!defined($x));
    my $y=$x; # backup

    # the stellar type strings
    my @longtypes;
    if($x=~/\#define STELLAR_TYPES (.*)/)
    {
	my $y=$1; $y=~s/\"//go;
	@longtypes=split(/,/o,$y);
    }

    # all the macros
    my %macrotypes;
    my @macrotypes;
    while($x=~s/#define (\S+) (\d+)//)
    {
	$macrotypes{$1}=$2;
	$macrotypes[$2]=$1;
	last if($1 eq 'MASSLESS_REMANT');
    }

    # short macros only (amended slightly!)
    my %shorttypes;
    my @shorttypes;
    $x=$y;
    while($x=~s/#define (\S{2,5}|GIANT_BRANCH|MASSLESS_REMNANT) (\d+)//)
    {
	my $type=$1;
	my $num=$2;
	$type=~s/GIANT_BRANCH/GB/o;
	$type=~s/MASSLESS_REMNANT/MASSLESS/o;

	$shorttypes{$type}=$num;
	$shorttypes[$num]=$type;
	last if($type eq 'MASSLESS_REMNANT');
    }
    $shorttypes[0]='LMMS';

    return \@macrotypes,\%macrotypes,\@longtypes,\@shorttypes,\%shorttypes;
}



sub source_list
{
    my $self = shift;
    my @sources;
    map
    {
        if(/^Nucleosynthesis source (\d+) is (\S+)/)
        {
            $sources[$1] = $2;
        }
    }grep{/^Nucleosynthesis source/}split(/\n+/,$self->evcode_version_string());
    return @sources;
}

sub ensemble_list
{
    my $self = shift;
    my @ensemble;
    map
    {
        if(/^Ensemble (\d+) is (\S*)/)
        {
            $ensemble[$1] = $2;
        }
    }grep{/^Ensemble /}split(/\n+/,$self->evcode_version_string());
    return @ensemble;
}

sub evcode_command
{
    my $self = shift;
    # build evolution code command as a list and return it

    # choose executable for evolution code : usually it is specified, but if not,
    # try some defaults... if this fails, there's not much to be done except complain.
    my $prog = $self->evcode_program();

    my $opts = $_[0] // {}; # options (or empty hash)
    
    # check the executable exists and can be executed
    if(!(-e $prog))
    {
	say "Cannot launch $self->{_grid_options}{prog} (or $prog) because it does not exist!";
	exit;
    }

    return(# first set up the environment
	   'env',
	   $self->{_grid_options}{shell_environment},
	   defined $self->{_grid_options}{libpath} ? 'LD_LIBRARY_PATH='.$self->{_grid_options}{libpath} : '',
	   
	   # taskset to limit to a cpu
	   (defined $self->{_grid_options}{cpu_affinity} && $self->{_grid_options}{cpu_affinity}) ? "taskset -c $self->{_grid_options}{cpu_affinity} " : ' ',
	   
	   # then the nice command
	   $self->{_grid_options}{nice}//'',

	   # then the ionice command
	   $self->{_grid_options}{ionice}//'',

	   # maybe stdbuf to prevent buffering
	   $self->{_grid_options}{stdbuf_command}//'',
	   
	   # then the evolution code
	   $prog,
	   
	   # and set into batchmode
	   $$opts{nobatch} ? '' : $self->{_grid_options}{arg_prefix}.'batch'
	);
}

sub evcode_command_string
{
    my $self = shift;
    no warnings;
    return join(' ',$self->evcode_command(@_));
    use warnings;
}

sub evcode_program
{
    my $self = shift;
    # return out best guess evolution code path
    return -e $self->{_grid_options}->{rootpath}.'/'.$self->{_grid_options}->{prog} ? $self->{_grid_options}->{rootpath}.'/'.$self->{_grid_options}->{prog} : $self->{_grid_options}->{prog};
}

sub isotope_list
{
    my $self = shift;
    my @isotopes;
    foreach (grep {/^Isotope/} split(/\n+/,$self->evcode_version_string()))
    {
	if(/Isotope (\d+) is (\S+)/)
	{
	    $isotopes[$1]=$2;
	}
    }
    return @isotopes;
}


sub gnuplot_title
{
    # make a nice gnuplot title assuming binary_grid variable names

    my $title = $_[0];

    # first, try rob_misc's prettytitle function
    #$title = rob_misc::gnuplot_prettytitle($title);

    # then a map between binary_c variable names and postscript equivalents 
    
    # key is a regexp, value is the replacement
    my %maph =
	('sn_sigma'=>'{\/Symbol s}_{SN}',
	 '([aA])lpha(_)?(\S+)?'=>'\{\/Symbol $1\}$2\{$3\}',
	 '([sS])igma(_)?(\S+)?'=>'{\/Symbol $1\}$2\{$3\}',  
	 '([gG])amma(_)?(\S+)?'=>'{\/Symbol $1\}$2\{$3\}',  
	 '([lL])ambda(_)?(\S+)?'=>'{\/Symbol $1\}$2\{$3\}',  
	 'BH_prescription'=>'NS\/BH prescription',
	 'wr_wind'=>'Massive star wind prescription',
	 'metallicity'=>'Metallicity',
	 'tidal_strength_factor'=>'{\/Italic f}_{tid}',
	);
    
    foreach my $regexp (keys %maph)
    {
	my $c= "\$title=~s/$regexp/{$maph{$regexp}}/g";
	#print "Title was $title, eval $c, title is ";
	eval $c;
	#print $title,"\n";
    }
    
    return $title;
}


sub convert_functions_to_pointers
{
    my $self = shift;
    # convert lexical function names to true function pointers
    map
    {
	
	my $func=$self->{_grid_options}->{$_};
	my $funcwas = $func;
	
	# prepend 'main::' if no package is given
	$func='main::'.$func if($func!~/\:\:/o);
	
	$self->vbout(2,"Convert $_ (function $func) to pointer ...");

	$self->{_grid_options}->{$_.'_pointer'} = $self->function_exists($func);

	if(!defined $self->{_grid_options}->{$_.'_pointer'})
	{
	    print STDERR "ERROR: Failed to make a CODEref (function pointer) from function \"$func\" - please check that it exists!\n";
	    exit;
	}

	$self->vbout(2,"Hence CODEREF ".$self->{_grid_options}->{$_.'_pointer'}."\n");

    }grep {defined $self->{_grid_options}->{$_} && 
	       /_function$/o &&
	       !defined $self->{_grid_options}->{$_.'_pointer'}
    } keys %{$self->{_grid_options}};

}

sub function_exists {
    my $self = shift;
    # return a CODEREF to a function if it exists, or undef
    no strict 'refs';
    my $funcname = shift;
    return \&{$funcname} if defined &{$funcname};
    return;
}


############################################################
# Timer support
############################################################

sub output_binary_grid_timers
{
    my $self = shift;
    # output timer information
    return if(!$self->{_grid_options}->{timers});

    # use hash passed in if one is passed in
    my %t;
    if(defined $_[0])
    {
	%t = %{$_[0]}; 
    }
    else
    {
	%t = %{$self->{_timers}};
    }
    my @timers = sort {$t{$b}<=>$t{$a}} keys %t;

    if($#timers>-1)
    {
	my $red=$colour{'red bold'};
	my $white=$colour{reset};
	say "\nTiming information for binary_grid\n**********************************************************************************";
	foreach my $timer (@timers)
	{
	    if($t{$timer} > 0.01)
	    {
		my $timer2=$timer;
		$timer2=~s/(.*\:\:)(.*)/$1$red$2$white/;
		printf "% 80s : % 8.2f s\n",$timer2,$t{$timer}; 
	    }
	}
	say "**********************************************************************************";
    }
    say "Thread : $self->{_threadinfo}->{thread_number}";
}

sub apply_timer
{ 
    my $self = shift;
    # apply timer to subroutine (s)
    return if(!$self->{_grid_options}->{timers});
    state %timed_sub;
    foreach my $subr (@_)
    {
	# assume $subr is in binary_grid unless context is given
	$subr = 'binary_grid2::'.$subr unless($subr=~/\:\:/);
	
	if(!defined $timed_sub{$subr})
	{
	    $timed_sub{$subr}=1;
 	    {
		my $t0; # subroutine entry time
		wrap $subr, 
		pre=>sub{  # subroutine entry : set t0
		    $t0 = scalar gettimeofday;
		}, 
		post=>sub{  # subroutine exist : calculate time spent in it, and save
		    $self->{_timers}{$subr} += scalar gettimeofday - $t0;
		}
	    }
	}
    }
}




sub set_threads_stacksize
{
    my $self = shift;
    ############################################################
    # set the stack (memory) size of each thread
    ############################################################
    if((threads->get_stack_size==0) &&
       (defined($self->{_grid_options}{threads_stack_size}))&&
       ($self->{_grid_options}{threads_stack_size}>0))
    {
	# number is given in Megabytes
	threads->set_stack_size(Mega*$self->{_grid_options}{threads_stack_size});
    }    

}



############################################################
# Monte carlo 
############################################################

sub increment_probtot
{
    my $self = shift;
    # add to the total probability : NB this MUST be a function in binary_grid
    $self->{_flexigrid}{probtot}+=$_[0]
}

sub increment_count
{
    my $self = shift;
    # increment counter: NB this MUST be a function in binary_grid  
    $self->{_flexigrid}{count}++;
}


############################################################
# grid actions
############################################################

sub check_for_required_actions
{
    my $self = shift;
    # set up action flags e.g. by monitoring files
    my $fvar=$_[0];

    foreach my $f (grep {-e $_} (@{$self->{_grid_options}->{suspend_files}}))
    {
	unlink $f;
	$self->{_flexigrid}{actions}{save_snapshot}=1;
	$self->{_flexigrid}{actions}{quit}=1;
    }

    # TODO condor support
    # on condor, perhaps checkpoint
    if(defined $self->{_grid_options}{condor_jobid} &&
       $self->{_grid_options}{condor_jobid} ne ''
        )
    {
	$self->checkpoint();
    }
}
sub condor_command
{
    my $self = shift;
    return undef;
}
sub slurm_command
{
    my $self = shift;
    return undef;
}

sub do_required_actions
{
    my $self = shift;
    my $fvar=$_[0];
    # check for actions: if they exist, do something

    # NB the ordering is important! snapshots first, then quit
    if($self->{_flexigrid}{actions}{save_snapshot})
    {
	print "do_required_actions : call save_snapshot\n";
	$self->save_snapshot($fvar);
	print "do_required_actions : returned from save_snapshot\n";
    } 


    if($self->{_flexigrid}{actions}{quit})
    {
	print "do_required_actions : call shutdown_binary_grid\n";
	$self->shutdown_binary_grid();
	$self->shutdown_binary_grid(9);
	print "do_required_actions : returned from shutdown_binary_grid\n";
    }

    if($self->{_flexigrid}{actions}{relaunch_threads})
    {
	print "do_required_actions : relaunch threads\n";
	($self->{_flexigrid}{threads},$self->{_flexigrid}{thread_q})=
	    $self->launch_flexigrid_threads($self->{_flexigrid}{nthreads});
	print "do_required_actions : done relaunch threads\n";
	$self->{_flexigrid}{actions}{relaunch_threads} = undef;
    } 
}

sub shutdown_binary_grid
{
    my $self = shift;
    $self->stop_flexigrid_threads();
}

############################################################
# grid snapshots and hash save/loaders
############################################################

sub serialized_fh
{
    my $self = shift;

    # return a filehandle to a serialized object, taking into
    # account of whether it is bzipped or not by piping through 
    # bzip2 when required. Do NOT allow the serializer to do the 
    # compression, the RAM use is unpredictable.

    # first arg is the filename
    # second is 'in' or 'out' depending on whether you want to read or write

    my $filename = $_[0];
    my $mode = $_[1]; # 'in' or 'out'
    my $fh;

    print "Make serialized fh for file $filename\n";

    if($mode eq 'in')
    {
	# input from file : test if bzipped, if so transparently
	# open filehandle through bunzip2 command
	if(File::Type->new->checktype_filename($filename) =~/bzip2/o)
	{
	    # bzip should decompress *to* stdout
	    my $cmd = "bunzip2 -c \"$filename\"";
	    print "$filename is to be decompressed with $cmd\n";
	    open($fh, '-|',$cmd) ||
		Carp::confess("cannot open $filename through bunzip2 for reading");
	    print "Filehandle is open at $fh\n";
	}
	else
	{	  
	    print "$filename is uncompressed\n";
	    open($fh, '<',$filename)||
		Carp::confess("cannot open $filename for reading");
	}
    }
    elsif($mode eq 'out')
    {
	# output to file : bzip if compress_results_hash is given
	if($self->{_grid_options}->{compress_results_hash})
	{
	    my $cmd = "bzip2 -z > \"$filename\"";
	    print "$filename is to be compressed through $cmd\n";
	    open($fh,'|-',$cmd) ||
		Carp::confess("cannot open $filename for writing");
	}
	else
	{
	    print "$filename is to be uncompressed\n";
	    open($fh,'>',$filename)||
		Carp::confess("cannot open $filename for writing");
	}
    }
    else
    {
	Carp::confess ("Bad mode in serialized_fh : second argument should be 'in' or 'out'");
    }

    return $fh;
}

sub inspect_snapshot
{
    my $self = shift;
    # load and print snapshot file
    my $o = $self->serializer_object();
    my $h=$o->retrieve(serialized_fh($_[0],'in')); # load snapshot from file
    say Dumper($h);
}

sub save_snapshot
{
    my $self = shift;
    ############################################################
    # save a snapshot of the binary_grid so we can restart at a later date
    #
    # NB all variables MUST BE EXPLICITLY SCOPED because this function
    # will be called out of context by an eval
    ############################################################
    # use Data::Serializer from http://search.cpan.org/~neely/Data-Serializer-0.59/lib/Data/Serializer.pm
    
    my $fvar=$_[0]; # grid variable values

    # first, stop the threads, this will force them to be joined
    # and the data put into $self->{_results}
    my $local_threadinfo=
	$self->join_flexigrid_threads($self->{_flexigrid}{thread_q},
                                      $self->{_flexigrid}{threads});
    
    my $f=$self->{_grid_options}{snapshot_file};
    my $fh = $self->serialized_fh($self->{_grid_options}{snapshot_file},'out');

    # make data object
    my $o = $self->serializer_object();
    say "SNAP Made data::serializer object $o, save to $f, filehandle $fh";

    map
    {
	say "Save result key $_";
    }keys %{$self->{_results}};

    # ignore this warning
    local $SIG{__WARN__} = sub {
	print STDERR $_[0]
	    if $_[0] !~ /^Encountered CODE ref, using dummy placeholder/;
    };

    #say "RESULTS HASH OUT\n************************************************************\n",Dumper($self->{_results}),"\n********************************************************************************\n";
    
    # dump to snapshot
    $o->store({
	bse_options=>$self->{_bse_options},
	computer=>hostname(),
	flexigrid=>$self->{_flexigrid},
	fvar=>$fvar,
	grid_options=>$self->{_grid_options},
	results=>$self->{_results},
	snapshot_version=>1.0,
	snapshot_time=>scalar localtime(),
	who=>$ENV{USER}//$ENV{USERNAME}//getlogin//getpwuid($<)//`whoami`//'unknown',
	      },$fh)||Carp::confess("cannot write snapshot to file $f, filehandle $fh");

    close $fh;

    say "SNAP Saved to disk at count $self->{_flexigrid}{count} (modulo $self->{_grid_options}->{modulo}, offset $self->{_grid_options}->{offset})";

    $self->{_flexigrid}{actions}{save_snapshot}=0;
}



sub load_snapshot
{
    my $self = shift;

    ############################################################
    # load a snapshot of the binary_grid so we can restart
    # 
    # NB all variables MUST BE EXPLICITLY SCOPED because this function
    # will be called out of context by an eval
    ############################################################
    my $o = $self->serializer_object();

    print "Load snapshot\n";
    my $h=$o->retrieve(serialized_fh($_[0])); # load snapshot from file
    say "Retrieved snapshot in $h (object $o)\nSnapshot was taken on $$h{computer} at $$h{snapshot_time} by $$h{who} (version $$h{snapshot_version}) : time is now ",scalar localtime();
    
    # always copy BSE options directly
    %{$self->{_bse_options}} = %{clone($h->{bse_options})};

    # save some variables from being overwritten
    my $oldh = $self->{_results}; # nb POINTER!

    # copy grid_options (we want most of it)
    %{$self->{_grid_options}} = %{clone ($h->{grid_options})};

    # result results_hash : we need the original pointer
    $self->{_results} = $oldh;

    # copy old data into results hash
    %{$self->{_results}} = %{clone($h->{results})}; # deep copy results hash

    # copy results_hash pointer to flexigrid, and reload function pointers 
    $self->{_flexigrid}{results_hash} = $self->{_results};
    
    $self->convert_functions_to_pointers();

    # new start time, log time etc
    $self->{_flexigrid}{tstart} = [gettimeofday];
    $self->{_flexigrid}{nextlogtime} = 0;

    # do not allow the option of showing all the hash : might get messy ...
    #say 'RESULTS HASH (copy)';
    #say Dumper($self->{_results});
    
    #say "RESULTS HASH IN\n************************************************************\n",Dumper($self->{_results}),"\n********************************************************************************\n";

    # inherit the count and probtot, but not the rest of the flexigrid
    $self->{_flexigrid}{count} = $$h{flexigrid}{count};
    $self->{_flexigrid}{probtot} = $$h{flexigrid}{probtot};
    say "Set initial count to $self->{_flexigrid}{count} from $$h{flexigrid}{count}, probtot to $self->{_flexigrid}{probtot}";

    # set the grid variables
    my %fvar=%{$h->{fvar}};
   
    # find the innermost loop variable, all others must be offset by 0.5*d
    my $innermost_var=(sort {$fvar{$b}{value}<=>$fvar{$a}{value}} keys %fvar)[0];

    # all gridpoints except the innermost loop may be saved centred, so offset back
    # so they can be started again
    map
    {
	$fvar{$_}{value} -= $fvar{$_}{d}*0.5 
	    if($fvar{$_}{type} eq 'centred' && $_ ne $innermost_var);
	say "Set flexigrid variable # $fvar{$_}{nvar} (fvar) $_ starting value to $fvar{$_}{value}";
    }keys %fvar;

    return \%fvar;
}

sub serializer_object
{
    my $self = shift;
    # make and return a Data::Serializer object

    # serializer data:
    # 
    #     Parent is 19042
    # Big hash uses 63.13 MB
    # RAW:Bencode                              :                                                                                  :      60.84 MB :       4.00 s :    96.37 % :    2.37689 MB : Loaded Hash OK
    # Bencode                                  :                                         compress=>0,portable=>0,raw_serialize=>0 :      60.67 MB :       4.00 s :    96.10 % :    2.37628 MB : Loaded Hash OK
    # Bencode                                  :                                         compress=>0,portable=>1,raw_serialize=>0 :      87.74 MB :       5.00 s :   138.98 % :    2.40541 MB : Loaded Hash OK
    # RAW:Convert::Bencode                     :                                                                                  :      30.08 MB :       4.00 s :    47.65 % :    2.37689 MB : Loaded Hash OK
    # Convert::Bencode                         :                                         compress=>0,portable=>0,raw_serialize=>0 :      30.07 MB :       3.00 s :    47.63 % :    2.37626 MB : Loaded Hash OK
    # Convert::Bencode                         :                                         compress=>0,portable=>1,raw_serialize=>0 :      57.14 MB :       5.00 s :    90.51 % :     2.4051 MB : Loaded Hash OK
    # RAW:Data::Dumper                         :                                                                                  :     134.85 MB :       4.00 s :   213.61 % :    2.37883 MB : Loaded Hash OK
    # Data::Dumper                             :                                         compress=>0,portable=>0,raw_serialize=>0 :     142.37 MB :       3.00 s :   225.52 % :    2.37946 MB : Loaded Hash OK
    # Data::Dumper                             :                                         compress=>0,portable=>1,raw_serialize=>0 :     164.98 MB :       5.00 s :   261.33 % :    2.41523 MB : Loaded Hash OK
    # RAW:FreezeThaw                           :                                                                                  :      47.76 MB :       4.00 s :    75.65 % :    2.35543 MB : Loaded Hash OK
    # FreezeThaw                               :                                         compress=>0,portable=>0,raw_serialize=>0 :      47.76 MB :       4.00 s :    75.65 % :    2.35588 MB : Loaded Hash OK
    # FreezeThaw                               :                                         compress=>0,portable=>1,raw_serialize=>0 :      77.12 MB :       7.00 s :   122.16 % :    2.37951 MB : Loaded Hash OK
    # RAW:JSON                                 :                                                                                  :      24.37 MB :       2.00 s :    38.60 % :    2.37998 MB : Loaded Hash OK
    # JSON                                     :                                         compress=>0,portable=>0,raw_serialize=>0 :      24.37 MB :       1.00 s :    38.60 % :    2.37818 MB : Loaded Hash OK
    # JSON                                     :                                         compress=>0,portable=>1,raw_serialize=>0 :      51.06 MB :       4.00 s :    80.88 % :    2.41784 MB : Loaded Hash OK
    # RAW:JSON::Syck                           :                                                                                  :      52.29 MB :       3.00 s :    82.83 % :     2.3821 MB : Loaded Hash OK
    # JSON::Syck                               :                                         compress=>0,portable=>0,raw_serialize=>0 :      52.31 MB :       2.00 s :    82.86 % :    2.37845 MB : Loaded Hash OK
    # JSON::Syck                               :                                         compress=>0,portable=>1,raw_serialize=>0 :      79.00 MB :       3.00 s :   125.14 % :    2.41511 MB : Loaded Hash OK
    # RAW:PHP::Serialization                   :                                                                                  :      24.70 MB :       5.00 s :    39.13 % :    2.41369 MB : Loaded Hash OK
    # PHP::Serialization                       :                                         compress=>0,portable=>0,raw_serialize=>0 :      24.72 MB :       7.00 s :    39.16 % :    2.41127 MB : Loaded Hash OK
    # PHP::Serialization                       :                                         compress=>0,portable=>1,raw_serialize=>0 :      60.94 MB :       8.00 s :    96.53 % :    2.45035 MB : Loaded Hash OK
    # RAW:Storable                             :                                                                                  :      43.89 MB :       2.00 s :    69.52 % :    2.35759 MB : Loaded Hash OK
    # Storable                                 :                                         compress=>0,portable=>0,raw_serialize=>0 :      43.78 MB :       3.00 s :    69.35 % :    2.35728 MB : Loaded Hash OK
    # Storable                                 :                                         compress=>0,portable=>1,raw_serialize=>0 :      74.30 MB :       4.00 s :   117.69 % :     2.4171 MB : Loaded Hash OK
    # RAW:YAML                                 :                                                                                  :      62.17 MB :      28.00 s :    98.48 % :    2.38016 MB : Loaded Hash OK
    # YAML                                     :                                         compress=>0,portable=>0,raw_serialize=>0 :      62.17 MB :      32.00 s :    98.48 % :    2.37912 MB : Loaded Hash OK
    # YAML                                     :                                         compress=>0,portable=>1,raw_serialize=>0 :      88.47 MB :      30.00 s :   140.14 % :    2.40804 MB : Loaded Hash OK
    # RAW:YAML::Syck                           :                                                                                  :      75.35 MB :       3.00 s :   119.36 % :    2.37849 MB : Loaded Hash OK
    # YAML::Syck                               :                                         compress=>0,portable=>0,raw_serialize=>0 :      75.35 MB :       3.00 s :   119.36 % :    2.37807 MB : Loaded Hash OK
    # YAML::Syck                               :                                         compress=>0,portable=>1,raw_serialize=>0 :     104.71 MB :       5.00 s :   165.86 % :    2.40482 MB : Loaded Hash OK
    # 

    # given the above, we use RobJSON which is Rob's hacked version
    # of JSON that allows for Histogram objects

    # later:
    # JSON is not thread safe: using it with threads will cause errors

    # outsource to RobJSON 
    my $serializer = 'RobJSON';

    # this is almost as good
    #$serializer => 'Convert::Bencode', 
    
    return Data::Serializer->new(
        serializer => $serializer,
        portable   => '1', # not sure if we need this? for normal ASCII, probably not 
	compress   => '0', # do not compress : let bzip do this!
	);
}



sub dump_results_hash
{
    my $self = shift;
    ############################################################
    # save the results_hash into a file, to be called after
    # all threads are joined
    ############################################################

    say "dump_results_hash @_";

    # make data object
    my $o = $self->serializer_object();
    my $f = $_[0] // $self->{_grid_options}{results_hash_dumpfile};
    my $fh = $self->serialized_fh($f,'out');
    my $h = $self->{_results};
    say "dump_results_hash: data::serializer object $o, save to $f, filehandle $fh";

    # dump to file : include some information other than simply the results
    # just in case we want it
    #say "STORE : $h";
    #say "Is type ",ref($h);
    #say Data::Dumper->Dump([$h]);

    $o->store($h,$fh)
	||Carp::confess("cannot write results hash $h to $f, filehandle $fh");

    close $fh;
    
    # human-readable version using Data::Dumper
    $f = $self->{_grid_options}{results_hash_clean_dumpfile};
    if(defined($f) && $f ne '')
    {
	say "dump clean version to $f";
	if(open(my $log_fp,'>'.$f)||Carp::confess("cannot open $f.datadump"))
        { 
            print {$log_fp} Dumper($self->{_results});
            close $log_fp;    
        }
    }
}

sub load_results_hash_file
{
    my $self = shift;
    # load results hash file (name passed in), 
    # return only the results_hash part
    # or return undef on failure
    my $o = $self->serializer_object();
    print "Load results hash from $_[0]\n";

    if(-f -s $_[0])
    {
        my $fh = $self->serialized_fh($_[0],'in');
        print "Calling retrieve\n";
        my $x = $o->retrieve($fh);
        print "Got data $x : returning it\n";
        return $x;
    }
    else
    {
        print "WARNING : File \"$_[0]\" does not exist or has zero size ::: skipping. Either something went wrong or your grid resolution is too small.\n";
        return undef;
    }
}
 

sub merge_results_hash_dumps
{
    my $self = shift;

    # merge results hash dumps with filenames
    # given in @_
    
    my $results_hash = $self->{_results} // {}; # returned
    
    # add the data for each file
    my $count_hashes = $self->{_grid_options}{condor_counthashes} // 
        $self->{_grid_options}{slurm_counthashes} // 0;

    foreach my $file (sort hashdumpsorter @_)
    { 
	my $c;
	my $m;

	say "\nMerge data for file $file ";

	printf "Load $file to subhash (mem use %g)\n",rob_misc::proc_mem_usage();
	my $h = $self->load_results_hash_file($file);
 	
        if(defined $h)
        {
            say "Loaded results hash file";
            #print Data::Dumper::Dump($h);
            
            $m = rob_misc::proc_mem_usage();
            $c = $count_hashes ? MAX(1,rob_misc::hash_count($h)) : 1;

            printf "Subhash loaded, mem use now %d, subhash length %d (%g)\n",$m,$c,$m/(1.0*$c);

            #say "add $h to $results_hash";
            Hash::RobMerge::arithmetically_add_hashes($results_hash,$h);

            $m = rob_misc::proc_mem_usage();
            $c = $count_hashes ? MAX(1,rob_misc::hash_count($results_hash)) : 1;

            printf "Hash added, mem use now %d, results_hash length %d (%g)\n",$m,$c,$m/(1.0*$c);
            
            %$h=();
            undef $h;
            printf "done (mem use %g)\n",rob_misc::proc_mem_usage();
        }
        else
        {
            print "Warning : Load of \"$file\" failed\n";
        }
    }

    # return the completed results hash
    return $results_hash;
}


############################################################
# binary_c argument creation
############################################################

sub make_binary_c_argstring
{
    my $self = shift;
    # this subroutine is for backwards compatibility only
    return $self->make_evcode_argstring(@_);
}

sub make_evcode_argstring
{
    my $self = shift;
    confess "make_evcode_argstring is deprecated!\n";
    exit;
}

sub make_evcode_arghash
{
    my ($self,$args) = @_;
    # return a string out of all the evolution code arguments e.g.
    # mass, metallicity, period, eccentricity, probability and
    # the bse_options hash

    # required system parameters
    my @restore = ('z','extra','eccentricity');
    my %store;

    # deal with additional arguments
    my %postargs;
    if(defined $args->{postargs})
    {
	my $postargs = delete $$args{postargs};
	foreach my $k (keys %{$postargs})
	{
	    $postargs{$k} = $$postargs{$k};
	    $store{$k} = delete $self->{_bse_options}->{$k};
	}
	$self->vbout(3,"Set postargs ",join(' ',%postargs),"\n");
    }
    
    # optional system parameters
    $args->{monte_carlo_kicks} = undef if($self->{_grid_options}->{code} eq 'binary_c');
    $args->{init_abunds}=$self->{_grid_options}->{init_abunds} if(defined $self->{_grid_options}->{init_abunds});

    # remove dubious entries in bse_options (they are replaced below)
    foreach my $r (@restore)
    {
	$store{$r} = delete $self->{_bse_options}->{$r};
    }

    # from bse_options (stellar evolution options)
    %$args=(%$args, %{$self->{_bse_options}}, %postargs);

    # restore entries in bse_options
    foreach my $r (keys %store)
    {
	$self->{_bse_options}->{$r} = $store{$r};
    }
    
    # return arg hash pointer
    return $args;
}

sub make_arghash
{
    my $self = shift;
    # make arg hash from arg string
    my $s = $_[0]; # string
    my %h; # hash returned
    my @list = split(/\s+/,$s);
    while(scalar @list)
    {
        my $k = shift @list;
        $k = shift @list if($k=~/binary_c/);
        $k=~s/--//;
        my $v = 
            ($k =~/monte_carlo_kicks/) ? '' : 
            ($k=~/init_abund/) ? ((shift @list).' '.(shift @list)) :
            shift @list;
        $h{$k} = $v;
        #print "SET $k -> $v\n";
    }
    return \%h;
}

sub make_argstring
{
    my $self = shift;

    # make arg string from arg hash pointer : useful for debugging
    
    my $h = $_[0]; # hash pointer
    my @k; # keys, perhaps sorted

    if($self->{_grid_options}->{sort_args})
    {
	# sort priority list : lower numbers come first
	state %prio;
        state $c=0;
        if(!$c)
        {
            # set up priority hash
            map{
                $prio{$_} = $c;
                $c++;
            }@{$self->{_priority_args}};
        }

        # sort by prio, then default to cmp
	@k = sort {	    
	     defined $prio{$a} ? 
		(defined $prio{$b} ? $prio{$a} <=> $prio{$b} : -1) :
		(defined $prio{$b} ? 1 : $a cmp $b);
	}keys %{$_[0]};
    }
    else
    {
	@k = keys %{$_[0]};
    }

    return join(' ',map{
        $self->{_grid_options}->{arg_prefix}.$_.
            ' '.$_[0]->{$_}
                }@k);
}

sub make_extra_evcode_arguments
{
    my $self = shift;
    
    # extra arguments which do not depend on e.g. mass, Z...
    # e.g. logging on a per-thread basis
    my $extra_args={};

    if($self->{_grid_options}->{code} eq 'binary_c')
    {
	# no log if its filename is /dev/null
	if((defined($self->{_bse_options}->{log_filename}))&&
	   ($self->{_bse_options}->{log_filename} eq '/dev/null'))
	{
	    $$extra_args{log_filename}='/dev/null';
	    #$extra_args.=' --log_filename /dev/null ';
	}
	else
	{
	    # if we're using multiple threads, and no log file is defined,
	    # give each thread a separate log file
	    if((!defined($self->{_bse_options}->{log_filename}))&&
	       (defined($self->{_threadinfo}->{thread_number})))
	    {
		my $f=$self->{_grid_options}->{tmp}.'/c_log.'.$self->{_threadinfo}->{thread_number}.'.dat';
		#$extra_args.=' --log_filename '.$f.' ';
		$$extra_args{log_filename}=$f;
		$self->{_grid_options}->{current_log_filename}=$f;
	    }
	}
    }

    # the special "extra" arguments given manually
    # SHOULD BE AFTER THE OTHER ARGUMENTS
    #$extra_args.=' '.$self->{_bse_options}->{extra}.' ' 
#	if(defined($self->{_bse_options}->{extra}));
    $$extra_args{extra} = $self->{_bse_options}->{extra}
    if(defined($self->{_bse_options}->{extra}));

    return $extra_args;
}

sub log_args
{
    my $self = shift;
    my ($argstring) = @_;
    $self->vbout(3,"About to log args\n");

    if($self->{_grid_options}{log_fins})
    {
        # if we want to log fins, insist on always reopening arg files
        $self->{_grid_options}{always_reopen_arg_files} = 1;
    }
    
    my $f = $self->{_grid_options}{log_args_dir}.
        "/$self->{_grid_options}{code}-args.thread".$self->{_threadinfo}{thread_number};

    if(!defined($self->{_threadinfo}{logargs_fp}) ||
       $self->{_grid_options}{always_reopen_arg_files})
    {
        if(defined $self->{_threadinfo}{logargs_fp})
        {
            close $self->{_threadinfo}{logargs_fp};
        }
        open($self->{_threadinfo}{logargs_fp},'>'.$f)||
            confess("Cannot open file to log $self->{_grid_options}{code} args ($f)\n"); 
        autoflush {$self->{_threadinfo}{logargs_fp}} 1;
    }
    else
    {
        seek($self->{_threadinfo}{logargs_fp},0,0);
    }

    my $pid= (defined($self->{_threadinfo}{evcode_pid})) ? 
        "evcode PID $self->{_threadinfo}{evcode_pid}" : 
        'no evcode PID';


    print {$self->{_threadinfo}{logargs_fp}} '# At '.rob_misc::thetime()." evcode ($self->{_grid_options}{code}) will try to run (my PID $$, $pid) program $self->{_grid_options}{rootpath}/$self->{_grid_options}{prog} thread $self->{_threadinfo}{thread_number}\n";
    
    say {$self->{_threadinfo}{logargs_fp}} ")\n",$argstring,"                                           ";
    if($self->{_grid_options}{always_reopen_arg_files})
    {
        close $self->{_threadinfo}{logargs_fp};
    }
    else
    {
        $self->{_threadinfo}{logargs_fp}->flush;
    }
}

sub log_fin
{
    my $self = shift;
    my ($argstring) = @_;
    $self->vbout(3,"About to log fin\n");

    # if we want to log fins, insist on always reopening arg files
    $self->{_grid_options}{always_reopen_arg_files} = 1;
        
    my $f = $self->{_grid_options}{log_args_dir}.
        "/$self->{_grid_options}{code}-args.thread".$self->{_threadinfo}{thread_number};

    open($self->{_threadinfo}{logargs_fp},'>>'.$f)||
        confess("Cannot open file to log $self->{_grid_options}{code} fin\n"); 
    #autoflush {$self->{_threadinfo}{logargs_fp}} 1;
    
    my $pid= (defined($self->{_threadinfo}{evcode_pid})) ? 
        "evcode PID $self->{_threadinfo}{evcode_pid}" : 
        'no evcode PID';

    print {$self->{_threadinfo}{logargs_fp}} '# At '.rob_misc::thetime()." evcode ($self->{_grid_options}{code}) tried to run (my PID $$, $pid) program $self->{_grid_options}{rootpath}/$self->{_grid_options}{prog} thread $self->{_threadinfo}{thread_number}\n";
    
    $self->{_threadinfo}{logargs_fp}->flush;

    if(!$self->{_grid_options}->{log_args})
    {
        say {$self->{_threadinfo}{logargs_fp}} "# ",$argstring;
    }

    close $self->{_threadinfo}{logargs_fp}; 
}



############################################################
# Thread join functions
############################################################

sub join_flexigrid_thread
{
    my $self = shift;
    print "JOIN : self = $self\n";
    # function which is called when a thread is joined:
    # Note that $_[0] is a hash pointer which is used by evcode threads
    # to collect data. It dies when it goes out of scope, probably
    # after this function.
    #
    # We expect that $self->{_results} is a 
    # pointer to a globally available hash into which results are merged.
    #
    # You do NOT have to use this function, however it is recommended
 
    # why is the array passed in not the same as in from line 3302

    my $h = $_[0];
    
    # no add hash debug by default (can be long output)
    $self->{_add_hash_debug} //= 0;
    
    if($self->verbose_output(1))
    {
        my $count_left = rob_misc::hashcount($self->{_results});
        my $count_right = rob_misc::hashcount($h);
        
        $self->vbout(1,"Join hashes: _results=$self->{_results} += h=$h (sizes : _results=$count_left, h=$count_right)\n");

        if($self->{_add_hash_debug})
        {
            print "############################################################\nwas\n";
            print Data::Dumper::Dumper($self->{_results});
            print "############################################################\nadd\n";
            print Data::Dumper::Dumper($h);
        }
    }
    Hash::RobMerge::arithmetically_add_hashes($self->{_results},$h);

    if($self->{_add_hash_debug})
    {
        print "############################################################\nnow\n";
        print Data::Dumper::Dumper($self->{_results});
        print "############################################################\n\n";
    }
    return @_;
}

sub join_flexigrid_threads
{
    ############################################################
    # all commands have been sent to threads: send them the undef
    # signal (to force them to finish and then quit) and 
    # then join their data
    ############################################################
    
    my ($self,$thread_q,$threads) = @_;
    my $failed_count;
    my $failed_prob;
    my @local_threadinfo;

    $self->vbout(1,"Joining flexigrid threads (thread_q=$thread_q, pending=".
                 $thread_q->pending().", threads=$threads)\nSend undefs to all threads...\n");

    # send undefs to finish the evolution in the threads
    $thread_q->enqueue((undef) x ($self->{_flexigrid}->{nthreads}));
    $self->vbout(1,"Undefs sent");
    
    # wait for stellar evolution threads to end
    while($thread_q->pending() > 0)
    {
        my $x;
        for(my $i=0;$i<=$#{$self->{_flexigrid}{threads}};$i++)
        {
            if(defined $self->{_flexigrid}{threads}[$i])
            {
                $x .= $i.' ';
            }
        }
        $self->vbout(1,"Waiting for threads to end : npending=".$thread_q->pending()."\n");#' pending threads: '.$x."\n");
        sleep 1;
    }
    
    # join threads to get the data out of them
    $self->vbout(1,"Threads ended: loop to join all threads\n");

    foreach my $j (@$threads)
    {
	# call the prejoin function
	&{$self->{_grid_options}->{thread_prejoin_function_pointer}}($self)
 	if(defined($self->{_grid_options}->{thread_prejoin_function_pointer}));

	# call the join function with the given arguments
	my $f = $self->{_grid_options}->{threads_join_function_pointer};
	
	if(!defined $f)
	{
	    $self->vbout(1,"Join of thread $j warning: no join function specified, join will do nothing with the data returned\n");
	    $j->join;
	    $self->vbout(1,"Join done\n");
	}
	else
	{
	    $self->vbout(1,"Join thread $j with function $f\n");

            # join returns 
	    my @ret = ($j->join);
	    
            # report mass into stars
            my $H = $ret[0];
	    $self->vbout(1,"Mass into stars from h = ".$H->{mass_into_stars}) if(defined $H->{mass_into_stars});

            # get return data
	    my ($r,$threadinfo) = &$f(@ret);

	    $self->vbout(1,"Thread $j has been joined : returned $r\n");

	    if(!defined $threadinfo)
	    {
		say "Warning: thread join function does not return \@_ as required.";
	    }
	    else
	    {
		printf "Thread %d:\n     %g s in evcode\n     %g s in total\nFailures %d (probabiliy %g)\n",
		$threadinfo->{thread_number},
		$threadinfo->{evcode_runtime},
		$threadinfo->{runtime},
                $threadinfo->{failed_count},
                $threadinfo->{failed_prob};
		
		# save thread information
		$local_threadinfo[$self->{_threadinfo}{thread_number}]
                    = $threadinfo;
                $failed_count += $threadinfo->{failed_count};
                $failed_prob += $threadinfo->{failed_prob};
	    }

	    # call the postjoin function
	    &{$self->{_grid_options}->{thread_postjoin_function_pointer}}($self,$r)
		if(defined($self->{_grid_options}->{thread_postjoin_function_pointer}));
	}
    }
    
    return (\@local_threadinfo,$failed_count,$failed_prob); # return thread information
}

sub progenitor
{
    # return progenitor info in a hash
    my ($self) = @_;
    return $self->{_grid_options}{progenitor_hash};
}

sub set_progenitor_info
{
    my $self = shift;
    my ($args) = @_;

    if(ref $args ne 'HASH')
    {
        # we're passed a string  - what to do?
        return;
    }
    
    # The progenitor_hash contains valuable information on the currently running
    # binary system. We must set it up, but note that the hash keys are named oddly.
    state %hashmap;
    %hashmap=(
        m1=>'M_1',
        m=>'M_1',
        m2=>'M_2',
        ecc=>'eccentricity',
        prob=>'probability',
        per=>'orbital_period',
        phasevol=>'phasevol',
	) if(!defined $hashmap{m});
    
    # map the args hash to progenitors hash
    while (my ($progenitor_key, $argskey) = each %hashmap)
    {
	$self->{_grid_options}{progenitor_hash}{$progenitor_key} = $$args{$argskey};
    }

    if(defined $self->{_grid_options}{progenitor_hash}{m2})
    { 
	# binary-dependent variables and short progenitor string
	$self->{_grid_options}{progenitor_hash}{sep} =
            binary_stars::calc_sep_from_period(
                $args->{M_1}, 
                $args->{M_2}, 
                $args->{orbital_period});
	state $x = [qw(m1 m2 per sep ecc metallicity prob phasevol)];
	$self->{_grid_options}{progenitor}=
            join(' ',1,
                 map{$self->{_grid_options}{progenitor_hash}{$_}} 
                 @$x);
    }
    else
    {
	# single star short progenitor string
	state $x = [qw/m1 metallicity prob phasevol/];
	$self->{_grid_options}{progenitor} =
            join(' ',0,
                 map{$self->{_grid_options}{progenitor_hash}{$_}} 
                 @$x);
    }
}


sub hashdumpsorter
{
    ($a=~/(\d+\.\d+)$/)[0] <=> ($b=~/(\d+\.\d+)$/)[0];
}

sub decompress_yields
{
    # yield lines often contain nx0 (n=integer) to represent
    # many zeros.
    # Given a list reference, replace nx0 (n=integer) 
    # with appropriately expanded lists of zeros
    my $population = shift;
    @{$_[0]} = 
    map
    {
        /(\d+)x0/ ? ((0) x $1) : $_;
    }@{$_[0]};
    return $_[0];
}

sub rebin
{
    my $self = shift;
    # a binning function :
    # rebin $_[0] to the desired accuracy, set in $_[1]
    #
    # Note: $_[1] should never be zero! if it is, return 0
    return $_[1]==0 ? 0 : int($_[0]/$_[1] +($_[0]>0.0 ? 0.5 : -0.5))*$_[1];
}


sub commented_evcode_version_string
{
    # prefix every line of the version string with $prefix
    # and return a prefixed (i.e. commented) string
    my ($self,$prefix) = @_;
    $prefix //= '# ';
    my @v = split(/\n/,$self->evcode_version_string());
    map
    {
        $_ = $prefix.$_;
    }@v;
    return join("\n",@v);
}

############################################################
# backend functions
############################################################

sub suicide
{
    # set by the backend
    backend_undef_sub('suicide',caller(1));
}

sub kill_flexigrid_evcode_pids
{ 
    # set by the backend
    backend_undef_sub('kill_flexigrid_evcode_pids',caller(1));
}

sub kill_flexigrid_evcode_pid
{
    # set by the backend
    backend_undef_sub('kill_flexigrid_evcode_pid',caller(1));
}

sub stop_flexigrid_threads
{
    # set by the backend
    backend_undef_sub('stop_flexigrid_threads',caller(1));
}

sub tbse_land
{
    # set by backend
    backend_undef_sub('tbse_land',caller(1));
}

sub tbse_launch
{
    # set by backend
    backend_undef_sub('tbse_launch',caller(1));
}
sub tbse
{ 
    # set by backend
    backend_undef_sub('tbse',caller(1));
}
sub tbse_restart
{
    # set by backend
    backend_undef_sub('tbse_restart',caller(1));
}

sub tbse_kill
{
    # set by backend
    backend_undef_sub('tbse_kill',caller(1));
}

sub evcode_version_string
{
    # set by backend
    backend_undef_sub('evcode_version_string',caller(1));
}

sub minimum_separation_for_RLOF
{
    # set by backend
    backend_undef_sub('minimum_separation_for_RLOF',caller(1));
}

sub minimum_period_for_RLOF
{
    # set by backend
    backend_undef_sub('minimum_period_for_RLOF',caller(1));
}

sub backend_undef_sub
{
    print STDERR "Subroutine ",$_[0]," should be defined by your backend of choice, but isn't... oops! (called from package $_[1], sub $_[4], line $_[3] in file $_[2]. Please load a backend module\n"; 
    exit 1;
}

############################################################

# condor extension functions

sub condor_grid
{
    print STDERR "condor_grid should be defined by a 'use binary_grid::condor' and should not be called directly.\n";
    exit;
}

sub output_allowed
{
}

sub condor_workingdir
{
}

sub make_condor_script
{
}

sub premake_condor_outfiles
{
}

sub write_condor_script
{
}

sub set_condor_job_status
{
}

sub get_condor_job_status
{
}

sub check_and_merge_condor_jobs
{
}

sub check_condor_jobs_done
{
}

sub merge_condor_jobs
{
}

sub condor_submit
{
}

sub jobid
{
}

sub condor_script_data
{
}

sub datafiles_string_comma
{
}

sub datafiles_string
{
}

sub condor_rerun_command
{
}

sub condor_check_joining_file
{
}

sub condor_job_hook
{
}

sub grid_interrupted
{
}

sub check_for_saved_snapshot
{
}

sub pre_load_snapshot
{
}

sub post_load_snapshot
{
}

sub checkpoint
{
}

sub increment_checkpoint_time
{
}

# Slurm extension functions

sub slurm_grid
{
    print STDERR "slurm_grid should be defined by a 'use binary_grid::slurm' and should not be called directly.\n";
    exit;
}

sub slurm_workingdir
{
}

sub make_slurm_script
{
}

sub premake_slurm_outfiles
{
}

sub write_slurm_script
{
}

sub set_slurm_job_status
{
}

sub get_slurm_job_status
{
}

sub check_and_merge_slurm_jobs
{
}

sub check_slurm_jobs_done
{
}

sub merge_slurm_jobs
{
}

sub slurm_submit
{
}

sub slurm_script_data
{
}

sub slurm_rerun_command
{
}

sub slurm_check_joining_file
{
}

sub slurm_job_hook
{
}


############################################################

# config and C build options

sub binary_c_src_dir
{
     return 
         (defined $ENV{BINARY_C}) ? $ENV{BINARY_C}.'/src' :
         (defined $ENV{BINARY_C_ROOT}) ? $ENV{BINARY_C_ROOT}.'/src' :
         (defined $ENV{BINARY_C_SRC}) ? $ENV{BINARY_C_SRC} :
         (defined $ENV{HOME} && -d $ENV{HOME}.'/progs/stars/binary_c/src') ? $ENV{HOME}.'/progs/stars/binary_c/src' : 
         (-d './src') ? './src' :
         '.';
}

sub binary_c_config
{
    my $srcdir = binary_c_src_dir();
    my $binary_c_config = $srcdir.'/../binary_c-config';
    if(-f -x $binary_c_config)
    {
        state $logged = 0;
        if($logged == 0)
        {
            print "Have binary_c-config at $binary_c_config\n";
            $logged = 1;
        }
        return $binary_c_config;
    }
    else
    {
        undef $binary_c_config;
        print "Failed to find binary_c-config : you need a recent binary_c with the binary_c-config script.\n";
        exit(1);
    }

}

sub from_binary_c_config
{
    # run binary_c-config with given flag and return the result
    my ($flag) = @_;
    my $binary_c_config = binary_c_config();
    my $r = `$binary_c_config $flag`;
    chomp $r;
    return $r;
}

sub binary_c_inline_config
{
    # get build options from binary_c-config
        # build options : you can override most of them
    # with the command line but the defaults are generally
    # a mix of Perl's default config and some GSL and binary_c
    # requirements.
    #

    # locate the binary_c src directory
    my $srcdir = 
        binary_grid2::binary_c_src_dir();

    my $libname = $ENV{BINARY_GRID2_LIB} // 'binary_c';
    my $debug = $ENV{BINARY_GRID2_DEBUG} // 0;

    # C compiler : we used to use Perl's compiler by default,
    #              but really you want a compiler that is compatible
    #              with the binary_c shared library. Perl defaults to 'cc'
    #              which is NOT the same compiler as used by binary_c.
    #
    # One could argue we should just build Perl with the same compiler
    # as binary_c. Then all is easy :)
    my $cc = $ENV{BINARY_GRID2_CC} // 
	binary_grid2::from_binary_c_config('cc') //
	$Config{cc};

    # we now require the libraries on which binary_c depends
    #
    # if we can run binary_c to find these libraries, do it
    if(!-f -x $srcdir.'/binary_c'  &&
       !-f -x $srcdir.'/../binary_c') 
    {
        print color('red'),"We require a binary_c executable : have you built binary_c?\n",color('reset');
        exit(2);
    }

    my $binclibs = binary_grid2::from_binary_c_config('libs');
    my $libdirs = binary_grid2::from_binary_c_config('libdirs').' -L'.$srcdir;
    $binclibs = ' '.$libdirs.' -l'.$libname.' '.$binclibs;
    my $bincflags = binary_grid2::from_binary_c_config('cflags');
    my $bincincdirs = binary_grid2::from_binary_c_config('incdirs');

    # remove -fvisibility=hidden : this breaks
    # the .so files Perl builds
    $bincflags =~ s/-fvisibility=hidden//g;
    
    my %defaults =
        (
         # use the C compiler and flags that Perl was built with
         # except that we require libm and libc
         cc => $Config{cc},
         ccflags => $bincflags,
         # use the linker that perl was compiled with
         ld => $Config{ld},
         # debugging
         debug => $debug,
         # include the defaults plus
         # GSL and binary_c
         inc => ' '.($Config{inc}//' ').' '.$bincincdirs." -I$srcdir ",
         # libname is usually just binary_c corresponding to libbinary_c.so
         libname => $libname,
         # library components are set above, joined here in this order
         # or running a grid will fail
         libs => join(' ',
                      $binclibs,
         ),
        );
    $debug = $ENV{BINARY_GRID2_DEBUG} // $defaults{debug};
    my $ld = $ENV{BINARY_GRID2_LD} // $defaults{ld};
    my $inc  = ($ENV{BINARY_GRID2_INC} // $defaults{inc}).' '.($ENV{BINARY_GRID2_EXTRAINC} // '');
    my $libs = ($ENV{BINARY_GRID2_LIBS} // $defaults{libs}).' '.($ENV{BINARY_GRID2_EXTRALIBS}//'');

    {
        # move library paths to the front of $libs
        my $paths;
        while($libs=~s/(-L\S+)//)
        {
            $paths .= $1.' ';
        }
        if(defined $paths)
        {
            $libs = $paths.$libs;
            $libs =~ s/\s+/ /;
        }
    }
    
    my $ccflags = $ENV{BINARY_GRID2_CCFLAGS} // ($defaults{ccflags}) . ($ENV{BINARY_GRID2_EXTRACCFLAGS} // '');

    # you must define _SEARCH_H to prevent it being loaded twice
    $ccflags .= ' -D_SEARCH_H ';
    
    print "Building binary_grid::C backend with source (binary_c.h) at $srcdir on ".hostname()."\n";
    print "Options:\ncc = $cc\nccflags = $ccflags\nld = $ld\nlibs = $libs\ninc = $inc\n\n";

    return ( 
        cc => $cc,
        ld => $ld,
        ccflags => $ccflags,
        libs => $libs,
        inc => $inc, 
        );
}


sub binary_c_log_code
{
    my ($code) = @_;
    return "
#pragma push_macro(\"MAX\")
#pragma push_macro(\"MIN\")
#undef MAX
#undef MIN
#include \"binary_c.h\"

void custom_output_function(struct stardata_t * stardata);
SV * custom_output_function_pointer(void);

SV * custom_output_function_pointer()
{
    /*
     * use PTR2UV to convert the function pointer 
     * &custom_output_function to an unsigned int,
     * which is then converted to a Perl SV
     */
    return (SV*)newSVuv(PTR2UV(&custom_output_function));
}

void custom_output_function(struct stardata_t * stardata)
{
    $code;
}
#undef MAX 
#undef MIN
#pragma pop_macro(\"MIN\")
#pragma pop_macro(\"MAX\")
";
}


sub binary_c_bindings
{
    my ($code) = @_;
    my %inline_config = binary_grid2::binary_c_inline_config();
    return (C => binary_grid2::binary_c_log_code($code),
            %inline_config);
}


sub binary_c_function_bind
{
    return exists(&{'main::custom_output_function_pointer'}) ? 
        main::custom_output_function_pointer() : undef,
}

 
sub memoize_shared {
    my $name = shift;
    my $glob = do {
        no strict 'refs';
        \*{(caller)."::$name"}
    };
    my $code = \&$glob;
    my $sep  = $;;
    my (%scalar, %list) :shared;

    no warnings 'redefine';
    *$glob = sub {
        my $arg = join $sep => @_;
        if (wantarray) {
            @{$list{$arg} ||= sub {\@_}->(&$code)}
        }
        else {
            exists $scalar{$arg}
                 ? $scalar{$arg}
                 :($scalar{$arg} = &$code)
        }
    }
}

1;


__DATA__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

binary_grid - Perl module for running grids of single and binary stars
with binary_c/nucsyn

=head1 SYNOPSIS

Often you want to run a grid of single or binary stars and calculate 
some values (e.g. chemical yields, number counts) based on the results
of the stellar evolution. With the binary_grid module you can do just that
and let the module take care of starting, processing and finishing the
stellar evolution code binary_c/nucsyn.

=head1 DESCRIPTION

Full documentation is found in doc/binary_grid.pdf (or source doc/binary_grid.lyx)

See the file src/perl/scripts/grid-flexigrid.pl for example code.

=head1 SEE ALSO

    You require binary_c/nucsyn to be installed, usually in the default location $HOME/progs/stars/binary_c/.
    See also the rob_misc, IMF, spacing_functions and binary_stars modules (in src/perl/modules).
    You will need a number of Perl modules which can be found either as packages for your operating system, or at CPAN http://www.cpan.org/.

=head1 AUTHOR

Robert Izzard, r.izzard@surrey.ac.uk, also at gmail: rob.izzard

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2005-14 Robert Izzard

Please see the file LICENCE : this module is provided as part of binary_c/nucsyn and comes under the same LICENCE (and hence required citations).


=cut

__END__

