package binary_grid::Perl;

############################################################
#
# Backend to binary_grid, Perl pipe interface.
#
############################################################
my $version = '2.1.5';
our $VERSION = $version;
#
# Version 2.00 : Start, version matched to binary_grid.pm
#
#
# Version 2.1.5 : sync to latest version, add "say state" feature
#
# Note:
# There are two versions of each function, one for binary_grid
# and one for binary_grid2.
#
############################################################
$|=1;
use 5.16.0;
use strict;
use common::sense;
use feature qw(say state);
use warnings;
no warnings qw(redefine);

############################################################
# Perl modules
use Carp qw(confess);
use Carp::Always;
use Carp::Always::Color;
$Carp::MaxEvalLen=0; # show all of a failed eval
$Carp::MaxArgLen=0; # show all of a failed arg string
$Carp::MaxArgNums=0; # show all failed args
local $SIG{__DIE__}=sub{confess @_}; # force die signal to use confess()
use Sub::Identify qw/sub_fullname/;
use Module::Load;
############################################################
# Rob's modules
use binary_stars qw/calc_sep_from_period/;
############################################################
# export functions : none should be required
require Exporter;
our @ISA = qw(Exporter);
my @funcs= qw();
our %EXPORT_TAGS = ( 'all' => [ @funcs ] );
our @EXPORT_OK = ( @{ $EXPORT_TAGS{all} } );
our @EXPORT = @funcs;
use vars qw(@ISA @EXPORT @EXPORT_OK);

use threads::shared;
use Fcntl; # for file (un)blocking
use Fcntl qw(:flock SEEK_END); # import LOCK_* and SEEK_END constants
use IPC::Open3; # used to communication with your evolution code

############################################################
my %colour=rob_misc::colourhash();

# set the code backend identifier
$binary_grid::grid_options{backend} = 'binary_grid::Perl';
$binary_grid2::backend='binary_grid::C';

# list of binary_grid functions we override
my @binary_c_funcs_overridden = (
    'evcode_version_string',
    'evcode_args_list',
    'pipeaction',
    'tbse',
    'tbse_kill',
    'tbse_land',
    'tbse_launch',
    'tbse_line',
    'tbse_restart',
    'kill_flexigrid_evcode_pid',
    'kill_flexigrid_evcode_pids',
    'suicide',
    'stop_flexigrid_threads',
    );

# delete the subroutines we will override
foreach my $subroutine (@binary_c_funcs_overridden)
{
    #print "Redefine $subroutine to use C\n";
    eval "undef \&binary_grid::$subroutine; "; 
    eval "undef \&binary_grid2::$subroutine; "; 
    if($@)
    {
        print $@;
        exit;
    }
}

# provide new subroutines where required

sub print_binary_in
{
    # send data to evolution code
    binary_grid::vbout(3,"Send ".slashn(join(' ',@_))." to BINARY_IN (blocking: stdout=$binary_grid::grid_options{blocking}{stdout})\n");
    print BINARY_IN @_;
    binary_grid::vbout(3,"Sent ".slashn(join(' ',@_))." to BINARY_IN\n");
};

*binary_grid2::print_binary_in = sub
{
    my $self = shift;
    # send data to evolution code
    $self->vbout(3,"Send ".slashn(join(' ',@_))." to BINARY_IN (blocking: stdout=$self->{_grid_options}{blocking}{stdout})\n");
    print BINARY_IN @_;
    $self->vbout(3,"Sent ".slashn(join(' ',@_))." to BINARY_IN\n");
};

sub slashn
{
    # convert newlines to \n
    my $x=$_[0];
    $x=~s/\n/\\n/go;
    return $x;
}

sub binary_out
{
    # check that the evcode is running ok
    test_evcode_pipe();

    binary_grid::vbout(3,"Wait for BINARY_OUT (blocking: stdout=$binary_grid::grid_options{blocking}{stdout}) called by ".(caller(2))[3]."\n");

    my $d = scalar(<BINARY_OUT>); # return next line of data (must be scalar!)
    
    binary_grid::vbout(3,"Got BINARY_OUT = $colour{green}$colour{reset}\n");
    
    # Now for error trapping:
    #
    # If the string is empty then the process MIGHT have died.
    # So do a kill 0 to check.
    if((defined $d && $d eq '') && 
       defined $binary_grid::threadinfo{evcode_pid} &&
       ((kill 0,$binary_grid::threadinfo{evcode_pid})==0))
    { 
	# failure : $d must at least have a new line. 
	# if it is '' then the child process is probably dead.
	# Simulate this with a SIGPIPE error, just like 
	# a BINARY_IN failure.
	print $colour{red};
	my $s;
	$s="Thread $binary_grid::threadinfo{thread_number} "if(defined $binary_grid::threadinfo{thread_number});
	$s.="Received empty string on BINARY_OUT combined with a kill 0 fail : process has died? send SIGPIPE to self (PID $$; binary_grid is $binary_grid::grid_options{process_ID})\n";
	print $s,$colour{reset};

        if($binary_grid::grid_options{threadcomms})
        {
            # set shared failure string
            $binary_grid::threadcomms{failure} .= $s;
        }

	kill 'SIGPIPE', $$;
	sleep 1;
	kill 'SIGPIPE', $$;
	sleep 1;

	# sleep to prevent more <BINARY_OUT> attempts 
	# and allow SIGPIPE kill to work : should exit from
	# signal handler in this case. (5 should be enough
	# except on very heavily loaded machines: don't
	# want to sleep indefinitely though!)
	#sleep 5;
	
	# exit this thread : SIGPIPE should kill the others
	threads->exit;
    }

    # save the last stdout line just in case of a crash
    if($binary_grid::grid_options{threadcomms})
    {
        $binary_grid::threadcomms{"last_stdout_$binary_grid::threadinfo{thread_number}"}=$d  ;
    }

    return $d;   
};

*binary_grid2::binary_out = sub
{     
    my $self = shift;
    # check that the evcode is running ok
    $self->test_evcode_pipe();

    $self->vbout(3,"Wait for BINARY_OUT (blocking: stdout=$self->{_grid_options}{blocking}{stdout}) called by ".(caller(2))[3]."\n");

    my $d = scalar(<BINARY_OUT>); # return next line of data (must be scalar!)
    
    $self->vbout(3,"Got BINARY_OUT = $colour{green}$colour{reset}\n");
    
    # Now for error trapping:
    #
    # If the string is empty then the process MIGHT have died.
    # So do a kill 0 to check.
    if((defined $d && $d eq '') && 
       defined $self->{_threadinfo}{evcode_pid} &&
       ((kill 0,$self->{_threadinfo}{evcode_pid})==0))
    { 
	# failure : $d must at least have a new line. 
	# if it is '' then the child process is probably dead.
	# Simulate this with a SIGPIPE error, just like 
	# a BINARY_IN failure.
	print $colour{red};
	my $s;
	$s="Thread $self->{_threadinfo}{thread_number} "if(defined $self->{_threadinfo}{thread_number});
	$s.="Received empty string on BINARY_OUT combined with a kill 0 fail : process has died? send SIGPIPE to self (PID $$; binary_grid is $self->{_grid_options}{process_ID})\n";
	print $s,$colour{reset};

	kill 'SIGPIPE', $$;
	sleep 1;
	kill 'SIGPIPE', $$;
	sleep 1;

	# sleep to prevent more <BINARY_OUT> attempts 
	# and allow SIGPIPE kill to work : should exit from
	# signal handler in this case. (5 should be enough
	# except on very heavily loaded machines: don't
	# want to sleep indefinitely though!)
	#sleep 5;
	
	# exit this thread : SIGPIPE should kill the others
	threads->exit;
    }
    return $d;   
};


sub blocking
{
    # set BINARY_* file pointers to block
    pipe_action(\*BINARY_OUT,1);
    $binary_grid::grid_options{blocking}{STDOUT} = 1;
};

*binary_grid2::blocking = sub
{
    my $self = shift;
    # set BINARY_* file pointers to block
    pipe_action(\*BINARY_OUT,1);
    $self->{_grid_options}{blocking}{STDOUT} = 1;
};

sub nonblocking
{
    # non-blocking BINARY_* file pointers
    pipe_action(\*BINARY_OUT,0);
    $binary_grid::grid_options{blocking}{STDOUT} = 0; 
};

*binary_grid2::nonblocking = sub
{
    my $self = shift;
    # non-blocking BINARY_* file pointers
    pipe_action(\*BINARY_OUT,0);
    $self->{_grid_options}{blocking}{STDOUT} = 0;     
};

sub pipe_action
{
    # set pipe $_[0] to non-blocking ($_[1]==0) or blocking ($_[1]==1)
    # return undef and do nothing on failure
    my $old_flags = fcntl($_[0], F_GETFL, 0) or return undef;
    fcntl($_[0], F_SETFL, 
	  $_[1]==0 ? ($old_flags | O_NONBLOCK) : ($old_flags & !O_NONBLOCK)) 
	or confess "can't set $_[0] ".($_[1]==0 ? 'non-':'')."blocking: $!";
    return $_[1];
};

sub flush_binary_out
{  
    # flush BINARY_OUT
    pipe_action(\*BINARY_OUT,0); # non-blocking
    test_evcode_pipe();
    while(my $x = binary_out()){
	print "flush_binary_out: found string \"$x\"\n";
	if($x=~/fin/)
	{
	    print "Found a fin in flush data : this is an error\n"; 
	    exit;
	}
    };
    pipe_action(\*BINARY_OUT,1); # blocking
};

*binary_grid2::flush_binary_out = sub
{  
    my $self = shift;
    # flush BINARY_OUT
    pipe_action(\*BINARY_OUT,0); # non-blocking
    $self->test_evcode_pipe();
    while(my $x = $self->binary_out()){
	print "flush_binary_out: found string \"$x\"\n";
	if($x=~/fin/)
	{
	    print "Found a fin in flush data : this is an error\n"; 
	    exit;
	}
    };
    pipe_action(\*BINARY_OUT,1); # blocking
};

sub test_evcode_pipe
{
    # test that the evolution code is still running
    # by sending an empty string to BINARY_IN
    #
    # if the evolution code has died, then this will cause 
    # a SIGPIPE failure and hence a crash. You want this.
    print BINARY_IN '';
};

*binary_grid2::test_evcode_pipe = sub
{
    my $self = shift;
    # test that the evolution code is still running
    # by sending an empty string to BINARY_IN
    #
    # if the evolution code has died, then this will cause 
    # a SIGPIPE failure and hence a crash. You want this.
    print BINARY_IN '';
};

*binary_grid::kill_flexigrid_evcode_pid = sub
{    
    # relatively safe evolution code killer
    my $n = $_[0]; # thread number
    my $pid = $binary_grid::flexigrid_evcode_pids[$n];

    print "Kill pid $pid thread $n\n";

    if(defined($pid) && $pid=~/^\d+$/o && (kill 0,$pid))
    {       
 	# try kill 15 first
	say {$binary_grid::threadinfo{tvb_fp}} "$n: kill 15"if(binary_grid::verbose_output(1) && defined($binary_grid::threadinfo{tvb_fp}));

	if((kill 15, $pid)==0)
	{
	    # kill 15 failed : try kill 9
	    if(binary_grid::verbose_output(1))
	    {
		say {$binary_grid::threadinfo{tvb_fp}} "$n: kill 9"if(defined($binary_grid::threadinfo{tvb_fp}));
		say "Kill evcode (pid $pid) in thread $n with signal 9 (almost-certain death)";
	    }

	    kill 9, $pid;
            
	    # this will fail because the thread will be dead
	    say {$binary_grid::threadinfo{tvb_fp}} "$n: evcode (pid $pid) should be dead" if($binary_grid::grid_options{tvb}) && defined($binary_grid::threadinfo{tvb_fp});
	}

	say "evcode from thread $n (pid $pid) should be dead"
	    if(binary_grid::verbose_output(1));
    }
};


*binary_grid2::kill_flexigrid_evcode_pid = sub
{    
    my $self = shift;

    # relatively safe evolution code killer
    my $n = $_[0]; # thread number
    my $pid = $self->{_flexigrid_evcode_pids}[$n];

    print "Kill pid $pid thread $n\n";

    if(defined($pid) && $pid=~/^\d+$/o && (kill 0,$pid))
    {       
 	# try kill 15 first
	say {$self->{_threadinfo}{tvb_fp}} "$n: kill 15"
            if($self->verbose_output(1) && defined($self->{_threadinfo}{tvb_fp}));

	if((kill 15, $pid)==0)
	{
	    # kill 15 failed : try kill 9
	    if($self->verbose_output(1))
	    {
		say {$self->{_threadinfo}{tvb_fp}} "$n: kill 9"
                    if(defined $self->{_threadinfo}{tvb_fp});
		say "Kill evcode (pid $pid) in thread $n with signal 9 (almost-certain death)";
	    }

	    kill 9, $pid;
            
	    # this will fail because the thread will be dead
	    say {$self->{_threadinfo}{tvb_fp}} "$n: evcode (pid $pid) should be dead" 
                if($self->{_grid_options}{tvb}) && defined($self->{_threadinfo}{tvb_fp});
	}

	say "evcode from thread $n (pid $pid) should be dead"
	    if($self->verbose_output(1));
    }
};


*binary_grid::kill_flexigrid_evcode_pids = sub
{
    # kill all evolution code pids
    for(my $i=0;$i<$binary_grid::flexigrid{nthreads};$i++)
    {
	kill_flexigrid_evcode_pid($i);
    }
};


*binary_grid2::kill_flexigrid_evcode_pids = sub
{
    my $self = shift;
    # kill all evolution code pids
    for(my $i=0;$i<$self->{_flexigrid}{nthreads};$i++)
    {
	$self->kill_flexigrid_evcode_pid($i);
    }
};

*binary_grid::suicide = sub
{
    # kill myself
    kill 15,$$;
    kill 9,$$;
    exit(1);
};

*binary_grid2::suicide = sub
{
    # kill myself
    my $self = shift;
    kill 15,$$;
    kill 9,$$;
    exit(1);
};

*binary_grid::tbse_kill = sub
{
    ############################################################
    # KILL evolution code
    ############################################################

    # NB we cannot print to BINARY_IN or close it because these commands 
    # might wait for some response. We just want the evolution code to be dead!
    my $quiet=$_[0]; # no messages if =1

    # if we have no process ID, we can't do anything
    return if(!($quiet eq 'force') || !defined($binary_grid::threadinfo{evcode_pid}));

    # evolution code should now have stopped... but just in case:
    # make sure binary_c is dead: try to kill with SIGTERM

    say STDERR "tbse_kill: try to kill process $binary_grid::threadinfo{evcode_pid}" if(!$quiet);

    # kill should return 1 because there is just one process to kill
    my $r=kill 'SIGTERM',$binary_grid::threadinfo{evcode_pid};

    if(($r!=1)||(kill 0,$binary_grid::threadinfo{evcode_pid} == 1 ))
    {
	# oops, SIGTERM failed, try with SIGKILL instead 
	if(!$quiet)
	{
	    say STDERR 'tbse_kill(): kill with SIGTERM failed... try SIGKILL';
	}
	# again, kill should return 1 because there is just one process to kill
	$r=kill 'SIGKILL',$binary_grid::threadinfo{evcode_pid}; 
	if($r!= 1)
	{
	    if(!$quiet)
	    {
		say STDERR 'tbse_kill(): SIGKILL failed as well! OOPS! try manual call to shell kill';
	    }
	    `kill -9 $binary_grid::threadinfo{evcode_pid}`;
	    if((kill 0,$binary_grid::threadinfo{evcode_pid} == 1 )&&(!$quiet))
	    {
		say STDERR 'tbse_kill(): Manual call to shell kill failed as well!';
	    }
	    else
	    {
		say STDERR 'tbse_kill(): Manual call to shell kill successful';
	    }
	}
	elsif(!$quiet)
	{
	    say STDERR 'tbse_kill(): SIGKILL successful';
	}
    }
};

*binary_grid2::tbse_kill = sub
{
    my $self = shift;

    ############################################################
    # KILL evolution code
    ############################################################

    # NB we cannot print to BINARY_IN or close it because these commands 
    # might wait for some response. We just want the evolution code to be dead!
    my $quiet=$_[0]; # no messages if =1

    # if we have no process ID, we can't do anything
    return if(!(defined $quiet && $quiet eq 'force') || !defined($self->{_threadinfo}{evcode_pid}));

    # evolution code should now have stopped... but just in case:
    # make sure binary_c is dead: try to kill with SIGTERM

    say STDERR "tbse_kill: try to kill process $self->{_threadinfo}{evcode_pid}" if(!$quiet);

    # kill should return 1 because there is just one process to kill
    my $r=kill 'SIGTERM',$self->{_threadinfo}{evcode_pid};

    if($r!=1 ||(kill 0,$self->{_threadinfo}{evcode_pid} == 1 ))
    {
	# oops, SIGTERM failed, try with SIGKILL instead 
	if(!$quiet)
	{
	    say STDERR 'tbse_kill(): kill with SIGTERM failed... try SIGKILL';
	}
	# again, kill should return 1 because there is just one process to kill
	$r=kill 'SIGKILL',$self->{_threadinfo}{evcode_pid}; 
	if($r!= 1)
	{
	    if(!$quiet)
	    {
		say STDERR 'tbse_kill(): SIGKILL failed as well! OOPS! try manual call to shell kill';
	    }
	    `kill -9 $self->{_threadinfo}{evcode_pid}`;
	    if((kill 0,$self->{_threadinfo}{evcode_pid} == 1 )&&(!$quiet))
	    {
		say STDERR 'tbse_kill(): Manual call to shell kill failed as well!';
	    }
	    else
	    {
		say STDERR 'tbse_kill(): Manual call to shell kill successful';
	    }
	}
	elsif(!$quiet)
	{
	    say STDERR 'tbse_kill(): SIGKILL successful';
	}
    }
};

*binary_grid::tbse_restart = sub
{
    ############################################################
    # restart evolution code
    ############################################################
    say STDERR 'Trying to restart evolution code and then continue';
    tbse_land();
    tbse_kill();
    sleep 1  if($binary_grid::grid_options{condor_jobid} eq '');
    tbse_launch();
    say STDERR 'Restart complete';
};

*binary_grid2::tbse_restart = sub
{
    my $self = shift;
    ############################################################
    # restart evolution code
    ############################################################
    say STDERR 'Trying to restart evolution code and then continue';
    $self->tbse_land();
    $self->tbse_kill();
    sleep 1  if($self->{_grid_options}{condor_jobid} eq '');
    $self->tbse_launch();
    say STDERR 'Restart complete';
};

*binary_grid::tbse_land = sub
{
    ############################################################
    # sent evolution code a "bye" command, which quits the program
    # This sometimes causes errors when Ctrl-C is caught, but these
    # don't seem to be fatal and can be ignored.
    ############################################################
    my ($thread,$pid,$ignore_warning) = @_;
    $pid //= $binary_grid::threadinfo{evcode_pid}; # default to global pid

    binary_grid::vbout(1,"Land tbse for thread '$thread' pid $pid\n");

    if(defined fileno BINARY_IN && -e BINARY_IN)
    {
	print "Send \"bye\" to BINARY_IN of pid $pid\n";
	print_binary_in "bye\n";
    }
    elsif($thread ne 'master shutdown')
    {
	print "BINARY_IN is not defined: cannot send a bye?\n";
    }

    # then close the input and output streams
    binary_grid::vbout(1,"Close BINARY_IN\n");
    close BINARY_IN;

    binary_grid::vbout(1,"Close BINARY_OUT\n");
    close BINARY_OUT;

    binary_grid::vbout(1,"Waitpid for evolution code (pid=$pid, thread=$thread)\n");
    waitpid $pid,0;
    binary_grid::vbout(1,"Pid ($pid) closed (thread $thread)\n");
};

*binary_grid2::tbse_land = sub
{
    my $self = shift;
    ############################################################
    # sent evolution code a "bye" command, which quits the program
    # This sometimes causes errors when Ctrl-C is caught, but these
    # don't seem to be fatal and can be ignored.
    ############################################################
    my ($thread,$pid,$ignore_warning) = @_;
    $pid //= $self->{_threadinfo}{evcode_pid}; # default to global pid

    $self->vbout(1,"Land tbse for thread '$thread' pid $pid\n");

    if(defined fileno BINARY_IN && -e BINARY_IN)
    {
	print "Send \"bye\" to BINARY_IN of pid $pid\n";
	$self->print_binary_in("bye\n");
    }
    elsif($thread ne 'master shutdown')
    {
	print "BINARY_IN is not defined: cannot send a bye?\n";
    }

    # then close the input and output streams
    $self->vbout(1,"Close BINARY_IN\n");
    close BINARY_IN;

    $self->vbout(1,"Close BINARY_OUT\n");
    close BINARY_OUT;

    $self->vbout(1,"Waitpid for evolution code (pid=$pid, thread=$thread)\n");
    waitpid $pid,0;
    $self->vbout(1,"Pid ($pid) closed (thread $thread)\n");
};


*binary_grid::stop_flexigrid_threads = sub
{
    ############################################################
    # stop flexigrid threads by sending enough undefs 
    ############################################################
    if(defined($binary_grid::flexigrid{thread_q}))
    {
	# flush pipes? might be necessary
 	say 'Flush evcode stdout at ',scalar localtime();
	flush_binary_out();

	say "Polite stop of $binary_grid::flexigrid{nthreads} flexigrid threads at ",scalar localtime();
	$binary_grid::flexigrid{thread_q}->enqueue((undef)x($binary_grid::flexigrid{nthreads}+1));
	say 'Threads have been asked to stop, please wait... at ',scalar localtime();;
    }
};


*binary_grid2::stop_flexigrid_threads = sub
{
    my $self = shift;
    ############################################################
    # stop flexigrid threads by sending enough undefs 
    ############################################################
    if(defined($self->{_flexigrid}{thread_q}))
    {
	# flush pipes? might be necessary
 	say 'Flush evcode stdout at ',scalar localtime();
	$self->flush_binary_out();

	say "Polite stop of $binary_grid::flexigrid{nthreads} flexigrid threads at ",scalar localtime();
	$self->{_flexigrid}{thread_q}->enqueue((undef)x($self->{_flexigrid}{nthreads}+1));
	say 'Threads have been asked to stop, please wait... at ',scalar localtime();;
    }
};


*binary_grid::tbse_line = sub
{
    ############################################################
    # get a line from the evolution code process, check
    # for some special cases and then return it.
    # Note that the line returned is guaranteed to be chomped.
    ############################################################

    # if there is a cache of lines, use it instead
    state @cache;
    
    # if there is a cache (from previously compressed data), use it first
    if(scalar @cache)
    {
	return shift @cache;
    }
    else
    {
	# get a data line from the stdout of the evolution code

	my $x=binary_out;
	
	#
	# special case starts with : BUF([UC]) (\d+)
	# $1 stores:
	# U : uncompressed data
	# C : buffered data
	#
	# there are $2 bytes of data
	#

	if($x=~/BUF([UC]) (\d+)/o)
	{
	    my $compressed = $1 eq 'C' ? 1 : 0; 
	    my $nbytes=$2;
	    
	    # the next $nbytes bytes of data contain the information: 
	    # load them, decompress if
	    # required, put in @cache, set $x to the first 
	    # line of the cache, and go

	    my $data;
	    read BINARY_OUT,$data,$nbytes;
	    $data = Compress::Zlib::uncompress($data) if($compressed);
	    @cache = split(/\n/o,$data);
	    $x=shift @cache;
            undef $data;
	}
	elsif($binary_grid::grid_options{add_up_system_errors} &&
	      $x=~/SYSTEM_ERROR/)
	{
	    # system failed : what to do?

	    # add up the probability of the failed systems
	    $binary_grid::flexigrid_shared{failed_prob} += $binary_grid::grid_options{progenitor_hash}{prob};
	    
	    # add up the count of the failed systems
	    $binary_grid::flexigrid_shared{failed_count}++;

	    if($binary_grid::grid_options{log_system_errors})
	    {
		printf "Caught system error (so far %d failed, prob failed %g)\n",
		$binary_grid::flexigrid_shared{failed_count},
		
		$binary_grid::flexigrid_shared{failed_prob};

                state $elock : shared;
                lock $elock;
                if(open(my $errfile,'>>',
                        $binary_grid::grid_options{tmp}.'/system_errors'))
                {
                    say {$errfile} scalar(localtime());
                    say {$errfile} $binary_grid::grid_options{'args'};
                    say {$errfile} "\n$x";
                    close $errfile;
                }
                
	    }
	}
	
	chomp $x;
	
	if(defined($x))
	{	
	    # detect if the thread has become stuck in an 
	    # infinite loop of doom
	    my $t=time();
	    my $dt=$t-$binary_grid::threadinfo{thread_prev_alive};
	    if($dt > $binary_grid::grid_options{thread_max_freeze_time_before_warning})
	    {
		if(($t > $binary_grid::threadinfo{prev_thread_complaint}+2) && 
		   ($binary_grid::grid_options{condor_command} ne 'join_datafiles'))
		{
		    my $s="Thread $binary_grid::threadinfo{thread_number} frozen? (Inactive for $dt seconds.)\n";
		    binary_grid::thread_log($colour{'red bold'}.$s.$colour{reset},$binary_grid::threadinfo{thread_number});
		    $binary_grid::threadinfo{prev_thread_complaint} = $t;
		    
		    if(defined($binary_grid::grid_options{current_log_filename}))
		    {
			state $lockv : shared;
			lock $lockv;
			if(open(my $fp,'>>'.$binary_grid::grid_options{current_log_filename}))
                        {
                            say {$fp} $s;
                            say {$fp} $binary_grid::grid_options{'current args'} if($binary_grid::grid_options{frozen}==1);
                            close $fp;
                        }
                        else
                        {
                            print STDERR "WARNING : Unable to open \$binary_grid\:\:grid_options\{current_log_filename\}= $binary_grid::grid_options{current_log_filename} - proceeding anyway.\n";
                        }
                    }
		}
		$binary_grid::grid_options{frozen}++;
	    }
	    else
	    {
		$binary_grid::grid_options{frozen}=0;
	    }
	}
	else
	{
	    # nothing : pipe died ? EOF ? not sure...
	    $x='#';
	}

	return $x;
    }
};


*binary_grid2::tbse_line = sub
{
    my $self = shift;

    ############################################################
    # get a line from the evolution code process, check
    # for some special cases and then return it.
    # Note that the line returned is guaranteed to be chomped.
    ############################################################

    # if there is a cache of lines, use it instead
    state @cache;
    
    # if there is a cache (from previously compressed data), use it first
    if(scalar @cache)
    {
	return shift @cache;
    }
    else
    {
	# get a data line from the stdout of the evolution code

	my $x = $self->binary_out();
	
	#
	# special case starts with : BUF([UC]) (\d+)
	# $1 stores:
	# U : uncompressed data
	# C : buffered data
	#
	# there are $2 bytes of data
	#

	if($x=~/BUF([UC]) (\d+)/o)
	{
	    my $compressed = $1 eq 'C' ? 1 : 0; 
	    my $nbytes=$2;
	    
	    # the next $nbytes bytes of data contain the information: 
	    # load them, decompress if
	    # required, put in @cache, set $x to the first 
	    # line of the cache, and go

	    my $data;
	    read BINARY_OUT,$data,$nbytes;
	    $data = Compress::Zlib::uncompress($data) if($compressed);
	    @cache = split(/\n/o,$data);
	    $x=shift @cache;
            undef $data;
	}
	elsif($self->{_grid_options}{add_up_system_errors} &&
	      $x=~/SYSTEM_ERROR/)
	{
	    # system failed : what to do?

	    # add up the probability of the failed systems
	    $self->{_flexigrid}{failed_prob} += $binary_grid::grid_options{progenitor_hash}{prob};
	    
	    # add up the count of the failed systems
	    $self->{_flexigrid}{failed_count}++;

	    if($self->{_grid_options}{log_system_errors})
	    {
		printf "Caught system error (so far %d failed, prob failed %g)\n",
		$self->{_flexigrid}{failed_count},
		
		$self->{_flexigrid}{failed_prob};

                state $elock : shared;
                lock $elock;
                if(open(my $errfile,'>>',
                        $self->{_grid_options}{tmp}.'/system_errors'))
                {
                    say {$errfile} scalar(localtime());
                    say {$errfile} $self->{_grid_options}{'args'};
                    say {$errfile} "\n$x";
                    close $errfile;
                }
                
	    }
	}
	
	chomp $x;
	
	if(defined $x)
	{	
	    # detect if the thread has become stuck in an 
	    # infinite loop of doom
	    my $t  = time();
	    my $dt = $t - $self->{_threadinfo}{thread_prev_alive};
	    if($dt > $self->{_grid_options}{thread_max_freeze_time_before_warning})
	    {
		if(($t > $self->{_threadinfo}{prev_thread_complaint}+2) && 
		   ($self->{_grid_options}{condor_command} ne 'join_datafiles'))
		{
		    my $s="Thread $self->{_threadinfo}{thread_number} frozen? (Inactive for $dt seconds.)\n";
		    $self->thread_log($colour{'red bold'}.$s.$colour{reset},
                                      $self->{_threadinfo}{thread_number});
		    $self->{_threadinfo}{prev_thread_complaint} = $t;
		    
		    if(defined $self->{_grid_options}{current_log_filename})
		    {
			state $lockv : shared;
			lock $lockv;
			if(open(my $fp,'>>'.$self->{_grid_options}{current_log_filename}))
                        {
                            say {$fp} $s;
                            say {$fp} $self->{_grid_options}{'current args'} 
                            if($self->{_grid_options}{frozen}==1);
                            close $fp;
                        }
                        else
                        {
                            print STDERR "WARNING : Unable to open grid_options\{current_log_filename\}= $self->{_grid_options}{current_log_filename} - proceeding anyway.\n";
                        }
                    }
		}
		$self->{_grid_options}{frozen}++;
	    }
	    else
	    {
		$self->{_grid_options}{frozen}=0;
	    }
	}
	else
	{
	    # nothing : pipe died ? EOF ? not sure...
	    $x='#';
	}

        if($self->{_grid_options}{return_array_refs})
        {
            return [split(' ',$x)];
        }
        else
        {
            return $x;
        }
    }
};


*binary_grid::tbse_launch = sub
{
    ############################################################
    # Launch evolution code using a tri-directional Open3 pipe #
    ############################################################
    my $threadinfo = $_[0];
    
    binary_grid::check_grid_defaults();
    binary_grid::setup_colours();
    
    # make command string (see functions above)
    my $command = binary_grid::evcode_command_string();
    
    binary_grid::thread_log(
        sprintf "Thread %d : Launch evolution code '$binary_grid::grid_options{code}' with command: '$command' (threadinfo = $threadinfo)\n",
        $threadinfo->{thread_number}
        )
        if(binary_grid::verbose_output(1));
    

    # open a bi-directional pipe to the evolution code process
    #
    # BINARY_OUT combined the standard output (stdout) 
    # and standard error (stderr) from the ev code : 
    # It is up to YOU to differentiate error messages from 
    # data.
    #
    # BINARY_IN is the input for the ev code
    
    $threadinfo->{evcode_pid} = open3 (\*BINARY_IN,
                                       \*BINARY_OUT,
                                       undef, # stderr=stdout
                                       $command)
        ||
        confess("cannot open $command for bi-directional pipe\n");

    print STDERR "BINARY_OUT at launch is ",\*BINARY_OUT," ",*BINARY_OUT,"\n";
    
    # always autoflush
    autoflush BINARY_IN 1;
    autoflush BINARY_OUT 1;
    
    if(defined $binary_grid::grid_options{'post exec wait'})
    {
	sleep $binary_grid::grid_options{'post exec wait'};
    }

    if(defined($binary_grid::grid_options{'gdb wait'}))
    {
	# wait for gdb to attach
	say "$binary_grid::grid_options{code} pid is $threadinfo->{evcode_pid}\nAttach with\n\ngdb attach $threadinfo->{evcode_pid}\n";
	$|=1;
	my $t=time()+$binary_grid::grid_options{'gdb wait'};
	while(time()<$t)
	{
	    printf "%d seconds to go...\n",$t-time();
	    sleep 1;
	}
    }

    if((defined fileno BINARY_IN)&&(-e BINARY_IN))
    { 
	# check BINARY_IN is ok
	binary_grid::vbout(0,"BINARY_IN seems ok (pid $threadinfo->{evcode_pid})\n");
    }
    
    # save the pid somewhere accessible to everyone
    $binary_grid::grid_options{evcode_pid}=$threadinfo->{evcode_pid};

    # logging
    my $s=sprintf "Thread %d : code pid %d \n",
    $threadinfo->{thread_number},$threadinfo->{evcode_pid};

    binary_grid::thread_log($s,$threadinfo->{thread_number});
    
    # set to blocking output by default (see blocking()/nonblocking())
    # NB everywhere you use the nonblocking subroutine, you should
    # call blocking() AS SOON AS POSSIBLE AFTERWARDS
    $binary_grid::grid_options{blocking}{stdout}=0; # avoid undef variable errors

    blocking(); # block both stderr and stdout by default

    binary_grid::vbout(1,'# Bi-directional binary_grid<>'.$binary_grid::grid_options{code}.' pipe open for business'."\n");

    if($binary_grid::grid_options{code} eq 'bonnfires')
    {
	# BONNFIRES is often slow to start up: wait for code to say we can go
	while(my $l = binary_out())
	{
	    if($l eq "Batchmode waiting\n")
	    {
		say "Successfully negotiated batchmode : set dataset $binary_grid::grid_options::bse_options{dataset}";
		
		# always have to send the dataset
		print_binary_in("dataset $binary_grid::bse_options{dataset}\n");
		last;
	    }
	}
    }
    else
    {
	# loop until told by batchmode that we're ready
	while(my $l=binary_out())
	{
	    if($l eq "set batchmode to 1\n")
	    {
		last;
	    }
	    else
	    {
		say "GOT '$l'";
	    }
	}
    }
    
    # sleep to allow threads to kick in
    if(defined($binary_grid::grid_options{thread_presleep}) && 
       ($binary_grid::grid_options{condor_jobid} eq ''))
    {
	sleep $binary_grid::grid_options{thread_presleep};
    }
    
    return $threadinfo->{evcode_pid};
};



*binary_grid2::tbse_launch = sub
{
    my $self = shift;

    ############################################################
    # Launch evolution code using a tri-directional Open3 pipe #
    ############################################################
    my $threadinfo = $_[0];
    
    # make command string (see functions above)
    my $command = $self->evcode_command_string();

    $self->thread_log(
        sprintf "Thread %d : Launch evolution code '%s' with command: '$command' (threadinfo = $threadinfo)\n",
        $self->{_threadinfo}->{thread_number},
        ($binary_grid::grid_options{code}//'unknown')
        )
        if($self->verbose_output(1));
    

    # open a bi-directional pipe to the evolution code process
    #
    # BINARY_OUT combined the standard output (stdout) 
    # and standard error (stderr) from the ev code : 
    # It is up to YOU to differentiate error messages from 
    # data.
    #
    # BINARY_IN is the input for the ev code

    $threadinfo->{evcode_pid} = open3 (\*BINARY_IN,
                                       \*BINARY_OUT,
                                       undef, # stderr=stdout
                                       $command)
        ||
        confess("cannot open $command for bi-directional pipe\n");

    print STDERR "BINARY_OUT at launch is ",\*BINARY_OUT," ",*BINARY_OUT,"\n";
    
    # always autoflush
    autoflush BINARY_IN 1;
    autoflush BINARY_OUT 1;
    
    if(defined $self->{_grid_options}{'post exec wait'})
    {
	sleep $self->{_grid_options}{'post exec wait'};
    }

    if(defined($self->{_grid_options}{'gdb wait'}))
    {
	# wait for gdb to attach
	say "$self->{_grid_options}{code} pid is $threadinfo->{evcode_pid}\nAttach with\n\ngdb attach $threadinfo->{evcode_pid}\n";
	$|=1;
	my $t = time() + $self->{_grid_options}{'gdb wait'};
	while(time() < $t)
	{
	    printf "%d seconds to go...\n",$t-time();
	    sleep 1;
	}
    }

    if((defined fileno BINARY_IN)&&(-e BINARY_IN))
    { 
	# check BINARY_IN is ok
	$self->vbout(0,"BINARY_IN seems ok (pid $threadinfo->{evcode_pid})\n");
    }
    
    # save the pid somewhere accessible to everyone
    $self->{_grid_options}{evcode_pid} = $threadinfo->{evcode_pid};

    # logging
    my $s=sprintf "Thread %d : code pid %d \n",
    $threadinfo->{thread_number},$threadinfo->{evcode_pid};

    $self->thread_log($s,$threadinfo->{thread_number});
    
    # set to blocking output by default (see blocking()/nonblocking())
    # NB everywhere you use the nonblocking subroutine, you should
    # call blocking() AS SOON AS POSSIBLE AFTERWARDS
    $self->{_grid_options}{blocking}{stdout}=0; # avoid undef variable errors

    $self->blocking(); # block both stderr and stdout by default

    $self->vbout(1,'# Bi-directional binary_grid<>'.$self->{_grid_options}{code}.' pipe open for business'."\n");

    if($self->{_grid_options}{code} eq 'bonnfires')
    {
	# BONNFIRES is often slow to start up: wait for code to say we can go
	while(my $l = $self->binary_out())
	{
	    if($l eq "Batchmode waiting\n")
	    {
		say "Successfully negotiated batchmode";
		
		# always have to send the dataset
                $self->print_binary_in("dataset $self->{_bse_options}{dataset}\n");
		last;
	    }
	}
    }
    else
    {
	# loop until told by batchmode that we're ready
	while(my $l = $self->binary_out())
	{
	    if($l eq "set batchmode to 1\n")
	    {
		last;
	    }
	    else
	    {
		say "GOT '$l'";
	    }
	}
    }
    
    # sleep to allow threads to kick in
    if(defined($self->{_grid_options}{thread_presleep}) && 
       ($self->{_grid_options}{condor_jobid} eq ''))
    {
	sleep $self->{_grid_options}{thread_presleep};
    }
    
    return $threadinfo->{evcode_pid};
};

*binary_grid::tbse = sub
{
    ############################################################
    # This function runs the evolution code with the masses, 
    # metallicity, eccentricity and probability passed
    # in as arguments.
    # It sends output the given parser function.
    ############################################################
    
    # tbse must be called with arguments
    if(!defined $_[0])
    {
        confess('binary_grid: tbse() called with no args!');
    }
    # check for failure string : if there is one, stop even
    # if it's in another thread
    elsif($binary_grid::grid_options{threadcomms} &&
          $binary_grid::threadcomms{failure})
    {
        say $colour{red},'Thread : ',$binary_grid::threadinfo{thread_number},' detected threadcomms{failure} : exit',$colour{reset};
        threads->exit();
    }
    # set timeout in case of failure/pause/lockup
    elsif($binary_grid::grid_options{timeout} &&
          (!$binary_grid::grid_options{'disable signal'}{ALRM}))
    {
        # set timeout
        binary_grid::set_next_alarm($binary_grid::grid_options{timeout});	
    }
    
    ### todo : add 'extra' evcode arguments
    my $args = $_[0];
    my $datahash = $_[1];
    my $nthread = $_[2] // $binary_grid::threadinfo{thread_number};

    # if args are scalar, treat as a string, make a hash out of it
    if(ref \$args eq 'SCALAR')
    {
        $args = binary_grid::make_arghash($args);
    }
    
    # set up the progenitor hash and string
    binary_grid::set_progenitor_info($args);

    # convert to string for logging and save for later
    # (in case of failure) NB SLOW!
    my $argstring;

    state $logargs = $binary_grid::grid_options{log_args} || 
        $binary_grid::grid_options{save_args} ||
        $binary_grid::grid_options{vb}>=2 ||
        $binary_grid::grid_options{log_system_errors};

    if($logargs)
    {
        # only make the argstring if required e.g. vb>=2, or if args are to be logged
        $argstring= binary_grid::make_argstring($args);
        $binary_grid::grid_options{args}=$argstring;
        $binary_grid::threadinfo{lastargs}=$argstring;
    }

    # reset the stars and apply defaults
    if($binary_grid::grid_options{reset_stars_defaults})
    {
        print_binary_in "reset_stars_defaults\n"; 
    }
    else
    {
        print_binary_in "reset_stars\ndefaults\n"; 
    }

    # if we want to be very verbose, output info about each run
    $nthread //= -1;
    binary_grid::vbout(2,"\nThread $nthread\nProgenitor: $binary_grid::grid_options{progenitor}\n".$colour{'blue bold'}."The star in blue is running ***right now*** on thread $nthread (and if it crashes, these are the arguments you should test):\n\n ".binary_grid::make_argstring($args).$colour{reset}."\n\n\n");
    
    # save the previous alive time for each thread
    $binary_grid::threadinfo{thread_prev_alive}=time();

    # log arguments to a file
    if($binary_grid::grid_options{log_args})
    {
        binary_grid::log_args($argstring);
    }  # end log_args check

    blocking(); # enforce blocking of stdout 
    
    if($binary_grid::grid_options{arg_checking})
    {
        confess ("arg_checking is broken");
        confess ("arg_checking only works with binary_c!") if ($binary_grid::grid_options{code} ne 'binary_c');
        
        # send arguments one by one, check that they work 
        map
        {
            my $arg='--'.$_;
            binary_grid::vbout(3,"Send arg $arg .. \n");

            # send arg
            print_binary_in $arg."\n";
            binary_grid::vbout(3,"sent $arg : checking response\n");
            
            # get response, was it ok?
            my ($x1,$x2,$x);

            # 3 attempts to get responses: each time we sleep for a second
            # to let the process catch up. That should be enough time, although
            # of course on some machines perhaps not... but then the binary_out
            # call should block. Why it doesn't (sometimes) I don't know.
            my $failcount=0;
            while($failcount<3)
            {
                # try to get a line of output
                $x1=binary_out;
                
                # ignore 'set batchmode to 1'
                $x2=binary_out if(defined($x1) && ($x1 eq "set batchmode to 1\n"));

                # choose which to use: x2 if we have it (is newer)
                $x= $x2 // $x1;
                
                if(defined($x))
                {
                    last;
                }
                else
                {
                    # sleep and try again (warn after first failure, let that slip by into the ether)
                    say STDERR "warning: thread $nthread sleeping (awaiting confirmation of argument '$arg' from evcode: x1='$x1', x2='$x2')" if($failcount>0);
                    $failcount++;
                    sleep 1  if($binary_grid::grid_options{condor_jobid} eq '');
                }
            }

            if((!(defined($x) && $x=~/arg ok/o)) || ($failcount>=3))
            {
                # error detected : close the pipes before damage is done
                say STDERR "ARG $arg failed (output $x)";

                close BINARY_IN;
                close BINARY_OUT;

                waitpid $binary_grid::threadinfo{evcode_pid},0;

                return("Argument to evcode '$arg' failed (received x='$x', x1='$x1', x2='$x2' from evcode instead) : check that it is correct in your evcode or in your grid script, (\%bse_options) and also check that evcode ($binary_grid::grid_options{code}) is running ok. (failcount=$failcount)\n");
            }	
            binary_grid::vbout(3,"response to '$arg' checked and ok\n");
            
        }grep {$_ ne ''} split(/--/o,$args);

    }
    else
    {
        # no arg checking: just send them
        binary_grid::vbout(3,"Send args ".slashn($args)." .. \n");

        # flush PRIOR to sending args : this seems sufficient to prevent lockup
        flush_binary_out();
        
        # need to tell the code not to do arg checking
        state $told;
        state @priority_args;
        if(!$told)
        {
            # only need to do this once
            print_binary_in $binary_grid::grid_options{arg_prefix}."no_arg_checking\n";
            $told=1;  
            @priority_args = binary_grid::priority_args();             
            
        }


        # some args must be sent first : but also must be removed fromt he $arg hash
        foreach my $priority_arg (@priority_args)
        {
            my $x = delete $$args{$priority_arg};
            print_binary_in $binary_grid::grid_options{arg_prefix}.
                $priority_arg.' '.$x."\n";
        }

        if($binary_grid::grid_options{lazy_arg_sending})
        {
            # laziness: send only what we have to
            no warnings; # ignore undefined variable warnings
            state %prev;
            while (my ($arg, $val) = each %$args)
            {
                if($prev{$arg} ne $val)
                {
                    print_binary_in $binary_grid::grid_options{arg_prefix}.$arg.' '.$val."\n"; 
                    $prev{$arg} = $val;
                }		
            }
            use warnings;
        }
        else
        {
            # no laziness : send everything in detail
            while (my ($arg, $val) = each %$args)
            {
                print_binary_in $binary_grid::grid_options{arg_prefix}.$arg.' '.$val."\n";
            }
        }
    }
    
    binary_grid::vbout(3,"evcode go\n");

    # run command, hope we don't fill the buffer (never seems to be a problem)
    print_binary_in "go\n";

    # set parser function : this function should be defined in your code!
    state $parse_bse=$binary_grid::grid_options{parse_bse_function_pointer};

    binary_grid::vbout(3,sprintf"calling parse_bse (thread %d, func pointer $parse_bse, is %s)\n",$nthread,sub_fullname($parse_bse));
    
    # call the output parser
    $parse_bse->($datahash);
    
    binary_grid::vbout(3,"parse_bse returned, calling non-blocking flush_binary_out\n");
    
    # just in case, flush all remaining output
    flush_binary_out();
    
    # reset alarm so the timeout is not hit
    binary_grid::set_next_alarm(0)
        if($binary_grid::grid_options{timeout} && (!$binary_grid::grid_options{'disable signal'}{ALRM}));

    binary_grid::vbout(3,"tbse end : return undef (no error)\n");

    return undef; # undef = no error
};


*binary_grid2::tbse = sub
{
    my $self = shift;
    ############################################################
    # This function runs the evolution code with the masses, 
    # metallicity, eccentricity and probability passed
    # in as arguments.
    # It sends output the given parser function.
    ############################################################
    
    # tbse must be called with arguments
    if(!defined $_[0])
    {
        confess('binary_grid: tbse() called with no args!');
    }
    # set timeout in case of failure/pause/lockup
    elsif($self->{_grid_options}->{timeout} &&
          (!$self->{_grid_options}->{'disable signal'}{ALRM}))
    {
        # set timeout
        $self->set_next_alarm($self->{_grid_options}->{timeout});	
    }
    
    ### todo : add 'extra' evcode arguments
    my $args = $_[0];
    my $datahash = $_[1];
    my $nthread = $_[2] // $self->{_threadinfo}{thread_number};

    # if args are scalar, treat as a string, make a hash out of it
    if(ref \$args eq 'SCALAR')
    {
        $args = $self->make_arghash($args);
    }
    
    # set up the progenitor hash and string
    $self->set_progenitor_info($args);

    # convert to string for logging and save for later
    # (in case of failure) NB SLOW!
    my $argstring;

    state $logargs = $self->{_grid_options}->{log_args} || 
        $self->{_grid_options}->{save_args} ||
        $self->{_grid_options}->{vb}>=2 ||
        $self->{_grid_options}->{log_system_errors};

    if($logargs)
    {
        # only make the argstring if required e.g. vb>=2, or if args are to be logged
        $argstring= $self->make_argstring($args);
        $self->{_grid_options}->{args}=$argstring;
        $self->{_threadinfo}{lastargs}=$argstring;
    }

    # reset the stars and apply defaults
    if($self->{_grid_options}->{reset_stars_defaults})
    {
        $self->print_binary_in("reset_stars_defaults\n"); 
    }
    else
    {
        $self->print_binary_in("reset_stars\ndefaults\n"); 
    }

    # if we want to be very verbose, output info about each run
    $nthread //= -1;
    $self->vbout(2,"\nThread $nthread\nProgenitor: $self->{_grid_options}->{progenitor}\n".$colour{'blue bold'}."The star in blue is running ***right now*** on thread $nthread (and if it crashes, these are the arguments you should test):\n\n ".$self->make_argstring($args).$colour{reset}."\n\n\n");
    
    # save the previous alive time for each thread
    $self->{_threadinfo}{thread_prev_alive}=time();

    # log arguments to a file
    if($self->{_grid_options}->{log_args})
    {
        $self->log_args($argstring);
    }  # end log_args check

    blocking(); # enforce blocking of stdout 
    
    if($self->{_grid_options}->{arg_checking})
    {
        confess ("arg_checking is broken");
        confess ("arg_checking only works with binary_c!") if ($self->{_grid_options}->{code} ne 'binary_c');
        
        # send arguments one by one, check that they work 
        map
        {
            my $arg='--'.$_;
            $self->vbout(3,"Send arg $arg .. \n");

            # send arg
            $self->print_binary_in($arg."\n");
            $self->vbout(3,"sent $arg : checking response\n");
            
            # get response, was it ok?
            my ($x1,$x2,$x);

            # 3 attempts to get responses: each time we sleep for a second
            # to let the process catch up. That should be enough time, although
            # of course on some machines perhaps not... but then the binary_out
            # call should block. Why it doesn't (sometimes) I don't know.
            my $failcount=0;
            while($failcount<3)
            {
                # try to get a line of output
                $x1 = $self->binary_out();
                
                # ignore 'set batchmode to 1'
                $x2 = $self->binary_out() if(defined($x1) && ($x1 eq "set batchmode to 1\n"));

                # choose which to use: x2 if we have it (is newer)
                $x= $x2 // $x1;
                
                if(defined($x))
                {
                    last;
                }
                else
                {
                    # sleep and try again (warn after first failure, let that slip by into the ether)
                    say STDERR "warning: thread $nthread sleeping (awaiting confirmation of argument '$arg' from evcode: x1='$x1', x2='$x2')" if($failcount>0);
                    $failcount++;
                    sleep 1  if($self->{_grid_options}->{condor_jobid} eq '');
                }
            }

            if((!(defined($x) && $x=~/arg ok/o)) || ($failcount>=3))
            {
                # error detected : close the pipes before damage is done
                say STDERR "ARG $arg failed (output $x)";

                close BINARY_IN;
                close BINARY_OUT;

                waitpid $self->{_threadinfo}{evcode_pid},0;

                return("Argument to evcode '$arg' failed (received x='$x', x1='$x1', x2='$x2' from evcode instead) : check that it is correct in your evcode or in your grid script, (\%bse_options) and also check that evcode ($self->{_grid_options}->{code}) is running ok. (failcount=$failcount)\n");
            }	
            $self->vbout(3,"response to '$arg' checked and ok\n");
            
        }grep {$_ ne ''} split(/--/o,$args);

    }
    else
    {
        # no arg checking: just send them
        $self->vbout(3,"Send args ".slashn($args)." .. \n");

        # flush PRIOR to sending args : this seems sufficient to prevent lockup
        $self->flush_binary_out();
        
        # need to tell the code not to do arg checking
        state $told;
        state @priority_args;
        if(!$told)
        {
            # only need to do this once
            $self->print_binary_in($self->{_grid_options}->{arg_prefix}."no_arg_checking\n");
            $told=1;  
            @priority_args = @{$self->{_priority_args}};
        }

        # some args must be sent first : but also must be removed fromt he $arg hash
        foreach my $priority_arg (@priority_args)
        {
            my $x = delete $$args{$priority_arg};
            $self->print_binary_in($self->{_grid_options}->{arg_prefix}.
                                   $priority_arg.' '.$x."\n");
        }

        if($self->{_grid_options}->{lazy_arg_sending})
        {
            # laziness: send only what we have to
            no warnings; # ignore undefined variable warnings
            state %prev;
            while (my ($arg, $val) = each %$args)
            {
                if($prev{$arg} ne $val)
                {
                    $self->print_binary_in($self->{_grid_options}->{arg_prefix}.
                                           $arg.' '.$val."\n"); 
                    $prev{$arg} = $val;
                }		
            }
            use warnings;
        }
        else
        {
            # no laziness : send everything in detail
            while (my ($arg, $val) = each %$args)
            {
                $self->print_binary_in($self->{_grid_options}->{arg_prefix}.$arg.
                                       ' '.$val."\n");
            }
        }
    }
    
    $self->vbout(3,"evcode go\n");

    # run command, hope we don't fill the buffer (never seems to be a problem)
    $self->print_binary_in("go\n");

    # set parser function : this function should be defined in your code!
    state $parse_bse = $self->{_grid_options}->{parse_bse_function_pointer};

    $self->vbout(3,sprintf"calling parse_bse (thread %d, func pointer $parse_bse, is %s)\n",$nthread,sub_fullname($parse_bse));
    
    # call the output parser
    $parse_bse->($self,$datahash,$nthread);
    
    $self->vbout(3,"parse_bse returned, calling non-blocking flush_binary_out\n");
    
    # just in case, flush all remaining output
    $self->flush_binary_out();
    
    # reset alarm so the timeout is not hit
    $self->set_next_alarm(0)
        if($self->{_grid_options}->{timeout} && 
           !$self->{_grid_options}->{'disable signal'}{ALRM});

    $self->vbout(3,"tbse end : return undef (no error)\n");

    return undef; # undef = no error
};

*binary_grid::evcode_args_list = sub
{
    # run evolution code and return version string
    my $evcode = binary_grid::evcode_program();
    my $cmd = $evcode.' --list_args 2>&1';
    
    state $args_list; # cache result
    state $prev;

    if(!defined $args_list || $prev ne $evcode)
    {
	$args_list = [ split(/\n/, scalar `$cmd`) ];
	$prev = $evcode;
    }
    return $args_list;
};

*binary_grid::evcode_version_string = sub
{ 
    my $self = shift;

    # run evolution code and return version string
    my $evcode = binary_grid::evcode_program();
    my $cmd = $evcode.' --version 2>&1';
    
    state $version; # cache result
    state $prev;

    if(!defined $version || $prev ne $evcode)
    {
	$version = scalar `$cmd`;
	$prev = $evcode;
    }
    return $version;
};

*binary_grid2::evcode_args_list = sub
{
    my $self = shift;

    # run evolution code and return version string
    my $evcode = $self->evcode_program();
    my $cmd = $evcode.' --list_args 2>&1';
    
    state $args_list; # cache result
    state $prev;

    if(!defined $args_list || $prev ne $evcode)
    {
	$args_list = [ split(/\n/, scalar `$cmd`) ];
	$prev = $evcode;
    }
    return $args_list;
};

*binary_grid2::evcode_version_string = sub
{
    my $self = shift;

    # run evolution code and return version string
    my $evcode = $self->evcode_program();
    my $cmd = $evcode.' --version 2>&1';
    
    state $version; # cache result
    state $prev;

    if(!defined $version || $prev ne $evcode)
    {
	$version = scalar `$cmd`;
	$prev = $evcode;
    }
    return $version;
};

############################################################
# RLOF orbits

*binary_grid::minimum_period_for_RLOF = sub
{
    my @orbit = binary_grid::minimum_orbit_for_RLOF(@_); 
    return $orbit[1];
};

*binary_grid2::minimum_separation_for_RLOF = sub
{
    my @orbit = binary_grid::minimum_orbit_for_RLOF(@_); 
    return $orbit[0];
};

*binary_grid::minimum_orbit_for_RLOF = sub
{
    my ($m1,$m2) = @_; # input variables are m1,m2
    
    # make argstring
    my $args = binary_grid::make_evcode_arghash({
        M_1=>$m1,
        M_2=>$m2,
        metallicity=>$binary_grid::bse_options{metallicity},
        separation=>0.0,
        eccentricity=>0.0,
        probability=>1.0,
        phasevol=>1.0,
        orbital_period=>1.0,
                                          });

    my $command = binary_grid::evcode_command_string({nobatch=>1}).' '.
        join(' ', map{$binary_grid::grid_options{arg_prefix}.$_.' '.$$args{$_}}(keys %$args));
    my $result = `$command minimum_orbital_period_for_instant_RLOF 1 minimum_separation_for_instant_RLOF 1`;

    # extract and return data
    my $minsep = ($result=~/MINIMUM SEPARATION (\S+)/)[0];
    my $minper = ($result=~/MINIMUM PERIOD (\S+)/)[0];
    
    return ($minsep,$minper);
};


*binary_grid2::minimum_period_for_RLOF = sub
{
    my $self = shift;
    my @orbit = $self->minimum_orbit_for_RLOF(@_); 
    return $orbit[1];
};

*binary_grid2::minimum_separation_for_RLOF = sub
{
    my $self = shift;
    my @orbit = $self->minimum_orbit_for_RLOF(@_); 
    return $orbit[0];
};

*binary_grid2::minimum_orbit_for_RLOF = sub
{
    my $self = shift;
    my ($m1,$m2) = @_; # input variables are m1,m2

    # static data
    state @priority_args;
    state $pre;
    if(!defined $pre)
    {
        $pre = ' '.$self->{_grid_options}->{arg_prefix};
        @priority_args = @{$self->{_priority_args}};
    }
    
    # make argstring
    my $args = $self->make_evcode_arghash({
        M_1=>$m1,
        M_2=>$m2,
        metallicity=>$self->{_bse_options}{metallicity},
        separation=>0.0,
        eccentricity=>0.0,
        probability=>1.0,
        phasevol=>1.0,
        orbital_period=>1.0,
                                          });
    my $argstring='binary_c ';
    foreach my $priority_arg (@priority_args)
    {
        $argstring .= $pre.$priority_arg.' '.delete $$args{$priority_arg};
    }
    while (my ($arg, $val) = each %$args)
    {
        $argstring .= $pre.$arg.' '.($val//'');
    }

    # get binary_c stack and join into an array
    my $command = $self->evcode_command({nobatch=>1}).' '.$argstring;
    my $result = `$command minimum_orbital_period_for_instant_RLOF 1 minimum_separation_for_instant_RLOF 1`;
    # extract and return data
    my $minsep = ($result=~/MINIMUM SEPARATION (\S+)/)[0];
    my $minper = ($result=~/MINIMUM PERIOD (\S+)/)[0];
    
    return ($minsep,$minper);
};

############################################################
# initial abundances
############################################################

*binary_grid::initial_abundance_string = sub
{ 
    # return a string describing the initial abundances
    my ($mix,$Z) = @_;
    my $cmd = binary_grid::evcode_program()." --init_abunds_only --metallicity $Z --initial_abundance_mix $mix"; 
    return `$cmd`;
};


*binary_grid2::initial_abundance_string = sub
{
    # return a string describing the initial abundances
    my $self = shift;
    my ($mix,$Z) = @_;
    my $cmd = $self->evcode_program()." --init_abunds_only --metallicity $Z --initial_abundance_mix $mix"; 
    return `$cmd`;
};




1;
