package binary_grid::C;

############################################################
#
# Backend to binary_grid, C shared library interface.
#
# This requires Inline::C to be working, and you to have
# built the binary_c shared library. You will have to
# rebuild binary_c, libbinary_c.so and then reinstall this
# module each time you change binary_c (otherwise you'll
# get random numbers and segfaults).
#
# You should set the environment variable BINARY_C_SRC
# to point to your binary_c/src directory, or set BINARY_C
# to point to your binary_c directory, prior to building.
#
# See also documentation in the binary_c/doc directory
# for more information.
#
# Note:
# There are two versions of each function, one for binary_grid
# and one for binary_grid2.
#
############################################################
my $version = 'v2.1.5';
our $VERSION = $version;
#
# CHANGES
#
# Version 2.00 : Start, version matched to binary_grid.pm
#
# Version 2.01 : unchanged
#
# Version 2.02 : Add checks for header files using 
#                rob_misc::test_for_header
#
# Version 2.03 : Check binary_c directly for libraries
#
# Version 2.04 : Use binary_c-config properly,
#                allow __THREAD__ in log filename
#                and don't automatically set log_filename to /dev/null
#
# Version 2.1  : sync with binary_c version number
#
# Version 2.1.1: sync with binary_c version number
#
# Version 2.1.2: sync with binary_c version number
#
# Version 2.1.3: handle binary_c error_buffer, sync version with binary_c
#
# Version 2.1.4: sync with binary_c version number 
############################################################
$|=1;
use 5.16.0;
use strict;
use common::sense;
use feature qw(state say);
use warnings;
no warnings qw(redefine);

############################################################
# Perl modules
use Carp qw(confess);
use Carp::Always;
use Carp::Always::Color;
$Carp::MaxEvalLen=0; # show all of a failed eval
$Carp::MaxArgLen=0; # show all of a failed arg string
$Carp::MaxArgNums=0; # show all failed args
local $SIG{__DIE__}=sub{confess @_}; # force die signal to use confess()
use Sub::Identify qw/sub_fullname/;
use Module::Load;
use Config;
use Sys::Hostname;
use Term::ANSIColor;
############################################################
# Rob's modules
use binary_stars qw/calc_sep_from_period/;
############################################################
# export functions : none should be required
require Exporter;
our @ISA = qw(Exporter);
my @funcs= qw();
our %EXPORT_TAGS = ( 'all' => [ @funcs ] );
our @EXPORT_OK = ( @{ $EXPORT_TAGS{all} } );
our @EXPORT = @funcs;
use vars qw(@ISA @EXPORT @EXPORT_OK %colour);
%colour = rob_misc::colourhash();
use binary_grid2;

############################################################
# C inline module and compiler information

# search for the source directory containing the library and header files

my $build = 0;
$binary_grid::grid_options{backend} = 'binary_grid::C';
$binary_grid2::backend='binary_grid::C';

my $binary_c_postargs = ' --internal_buffering INTERNAL_BUFERING_STORE ';
      
package main {
    # You want to do the binding in the main package, not
    # in binary_grid2, but then again you don't want this
    # code in main, but in here. So use package() to fix this.
    sub main::binary_c_logging
    {
        print "Binding binary_c logging code:\n******\n\n$_[0]\n\n******\n\n";
        Inline->bind(binary_grid2::binary_c_bindings($_[0]));
    }
}
                              
BEGIN 
{
    # code source not found : Warn, but just skip compilation, do not error
    # because perhaps we just want to use the native Perl backend
    my $srcdir = binary_grid2::binary_c_src_dir();
    print STDERR "Checking whether to build binary_grid::C (host ",hostname(),") ...\n";
    if(!(-f -e $srcdir.'/binary_c.h'))
    {
        print STDERR "Warning : not building the binary_grid::C because I cannot find binary_c.h. Have you set the BINARY_C_SRC environment variable to point to binary_c/src? Have you set the name of the shared library in BINARY_GRID2_LIB?\n";
        $build = 0;
    }
    else
    {
        # can build binary_grid::C backend
        print STDERR "Binary_c source and shared library found: building C backend\n";  
        $build = 1;

        # get Inline's config
        my %inline_config = binary_grid2::binary_c_inline_config();

        # warn if this failed!
        if(!defined $inline_config{cc} ||
           $inline_config{cc} =~/^\s*$/)
        {
            print "\n\n",color('red'),"************************************************************\n\n WARNING : failed to get cflags from binary_c-config : \nplease check that the binary_c executable is built\nand that the environment variable BINARY_C points to\nits parent directory.\n\n************************************************************\n\n",color('reset');
            exit;
        }

        
	# clean lddlflags
	my $lddlflags = $Config{lddlflags};
	$lddlflags =~ s/-O\d+//;
	$lddlflags =~ s/-m(?:tune|arch)=\S+//g;
	
        # use require, not use, to prevent errors when $build==0
        require Inline;
        Inline->import(
            C => Config => (
                %inline_config,
		lddlflags => $lddlflags,
		#cc => 'gcc',
		optimize => '',
		OPTIMIZE => '',
                name => 'binary_grid::C',
                version => 'v2.1.5',
                force_build => 1,
                FORCE_BUILD => 1,
                clean_after_build=>0,
                build_noisy => 1,
                rewrite_config_file => 1,
            ));
        Inline->init();
	
        # C code header : include binary_c.h and
        # make sure there are no macro clashes
        my $code_head = <<'BINARY_C_CODE_HEAD';

#undef TRUE
#undef FALSE

#include "binary_c.h"
BINARY_C_CODE_HEAD

        # code base: restore macros
        my $code_base = <<'BINARY_C_CODE_BASE';

BINARY_C_CODE_BASE

    # C code body: these functions are the binary_c::C API
    my $code_body = <<'BINARY_C_CODE_BODY';        

void C_shutdown_binary_c(int mode,
                         SV * store_p)
{
    //fprintf(stderr, "C::PM free_memory\n");
    if(mode == 1)
    {
        struct libbinary_c_store_t * store = 
               INT2PTR(struct libbinary_c_store_t *, SvUV(store_p));
        if(store)
        {
            binary_c_free_store_contents(store);
            Safe_free(store);
        }
    }
    else
    {
        binary_c_free_memory(NULL,TRUE,TRUE,TRUE,TRUE);
    }
}

void C_binary_c_version_string(AV * array)
{
    /*
     * Return the binary_c version
     */
    struct libbinary_c_stardata_t * stardata = NULL;
    struct libbinary_c_store_t * store = NULL;
    char * argstring = "binary_c";
    
    binary_c_new_system(&stardata,
                        NULL,
                        NULL,
                        &store,
                        &argstring,
                        -1);  
    
    snprintf(stardata->preferences->log_filename,
             STRING_LENGTH-1,"%s","/dev/null");
    snprintf(stardata->preferences->api_log_filename_prefix,
             STRING_LENGTH-1,"%s","/dev/null");
    stardata->preferences->internal_buffering = INTERNAL_BUFFERING_STORE;
    stardata->preferences->batchmode = BATCHMODE_LIBRARY;

    binary_c_version(stardata);
   
    char * buffer = NULL;
    size_t nbytes = 0; 
    binary_c_buffer_info(stardata,&buffer,&nbytes);

    char * line, * saveptr = NULL;
    size_t nlines = 0;
    line = strtok_r(buffer,"\r\n",&saveptr);
    const char * const maxbuffer = buffer + nbytes;
    while(line != NULL && saveptr <= maxbuffer)
    {
       nlines++;
       av_push(array,newSVpv(line,0));
       line = strtok_r(NULL,"\r\n",&saveptr);
    }

    binary_c_buffer_empty_buffer(stardata);
    binary_c_free_memory(&stardata,TRUE,TRUE,TRUE,TRUE);
}

void C_binary_c_args_list(AV * array)
{
    /*
     * Return a string containing the arguments list
     */
    struct libbinary_c_stardata_t * stardata = NULL;
    struct libbinary_c_store_t * store = NULL;
    char * argstring = "binary_c";

    binary_c_new_system(&stardata,
                        NULL,
                        NULL,
                        &store,
                        &argstring,
                        -1);  

    snprintf(stardata->preferences->log_filename,
             STRING_LENGTH-1,"%s","/dev/null");
    snprintf(stardata->preferences->api_log_filename_prefix, 
             STRING_LENGTH-1,"%s","/dev/null");
    stardata->preferences->internal_buffering = INTERNAL_BUFFERING_STORE;
    stardata->preferences->batchmode = BATCHMODE_LIBRARY;

    binary_c_list_args(stardata);

    char * buffer = NULL;
    size_t nbytes = 0; 
    binary_c_buffer_info(stardata,&buffer,&nbytes);

    char * line, * saveptr = NULL;
    size_t nlines = 0;
    line = strtok_r(buffer,"\r\n",&saveptr);
    const char * const maxbuffer = buffer + nbytes;
    while(line != NULL && saveptr <= maxbuffer)
    {
       nlines++;
       av_push(array,newSVpv(line,0));
       line = strtok_r(NULL,"\r\n",&saveptr);
    }

    binary_c_buffer_empty_buffer(stardata);
    binary_c_free_memory(&stardata,TRUE,TRUE,TRUE,TRUE);
}

void C_binary_c_minimum_orbit_for_RLOF(AV * array,
                                       char * argstring)
{
    /*
     * Return the binary_c version
     */
    struct libbinary_c_stardata_t * stardata = NULL;
    struct libbinary_c_store_t * store = NULL;
    
    binary_c_new_system(&stardata,
                        NULL,
                        NULL,
                        &store,
                        &argstring,
                        -1);
    stardata->preferences->show_minimum_separation_for_instant_RLOF = TRUE;
    stardata->preferences->show_minimum_orbital_period_for_instant_RLOF = TRUE;
    snprintf(stardata->preferences->log_filename,
             STRING_LENGTH-1,"%s","/dev/null");
    snprintf(stardata->preferences->api_log_filename_prefix,
             STRING_LENGTH-1,"%s","/dev/null");
    stardata->preferences->internal_buffering = INTERNAL_BUFFERING_STORE;
    stardata->preferences->batchmode = BATCHMODE_LIBRARY;
    binary_c_show_instant_RLOF_period_or_separation(stardata);

    char * buffer = NULL;
    size_t nbytes = 0; 
    binary_c_buffer_info(stardata,&buffer,&nbytes);

    char * line, * saveptr = NULL;
    size_t nlines=0;
    line = strtok_r(buffer,"\r\n",&saveptr);
    const char * const maxbuffer = buffer + nbytes;
    while(line != NULL && saveptr <= maxbuffer)
    {
       nlines++;
       av_push(array,newSVpv(line,0));
       line = strtok_r(NULL,"\r\n",&saveptr);
    }

    binary_c_buffer_empty_buffer(stardata);
    binary_c_free_memory(&stardata,TRUE,TRUE,TRUE,TRUE);
}



void C_binary_c_initial_abundance_mix(double metallicity, 
                                      int mix,
                                      AV * array,
                                      char * argstring)
{
    //fprintf(stderr,"Init abunds\n");
#ifdef NUCSYN
    //fprintf(stderr,"Init abunds ok\n");
    /*
     * Return the binary_c version
     */
    struct libbinary_c_stardata_t * stardata = NULL;
    struct libbinary_c_store_t * store = NULL;
    binary_c_new_system(&stardata,
                        NULL,
                        NULL,
                        &store,
                        &argstring,
                        -1);
    stardata->preferences->initial_abunds_only = TRUE;
    stardata->preferences->initial_abundance_mix = mix;
    stardata->common.metallicity = metallicity;
    snprintf(stardata->preferences->log_filename,
             STRING_LENGTH-1,"%s","/dev/null");
    snprintf(stardata->preferences->api_log_filename_prefix,
             STRING_LENGTH-1,"%s","/dev/null");
    stardata->preferences->internal_buffering = INTERNAL_BUFFERING_STORE;
    stardata->preferences->batchmode = BATCHMODE_LIBRARY;
    stardata->model.max_evolution_time = 0.0;

    /* find init abunds from initialize_parameters function */
    binary_c_initialize_parameters(stardata);
    
    char * buffer = NULL;
    size_t nbytes = 0; 
    binary_c_buffer_info(stardata,&buffer,&nbytes);
  
    char * line, * saveptr = NULL;
    size_t nlines=0;
    line = strtok_r(buffer,"\r\n",&saveptr);
    
    const char * const maxbuffer = buffer + nbytes;
    while(line != NULL && saveptr <= maxbuffer)
    {
       nlines++;
       av_push(array,newSVpv(line,0));
       line = strtok_r(NULL,"\r\n",&saveptr);
    }

    binary_c_buffer_empty_buffer(stardata);
    binary_c_free_memory(&stardata,TRUE,TRUE,TRUE,TRUE);
#endif
    //printf("init abunds return\n");
}

void C_run_binary_c(int mode,
                    int return_array_refs,
                    SV * store_p,
                    AV * array,
                    char * argstring, 
                    int nthread,
                    SV * custom_output_function_ptr)
{
    const int vb = 0;
    if(vb)
    {
        fprintf(stderr,"nthread=%d RUN BINARY C\n",nthread);
        fflush(NULL);
    }

    struct libbinary_c_stardata_t * stardata = NULL;
    struct libbinary_c_store_t * store = NULL;


    if(vb)
    {
        fprintf(stderr,"nthread=%d Start store = %p, mode = %d\n",
                nthread,store,mode);
        fflush(NULL);
    }

    if(mode == 1)
    {
        /*
         * Use already built store 
         */
        if(vb)
        {
           fprintf(stderr,"nthread=%d Using Passed in store from perl : %p\n",
                   nthread,store_p);
           fflush(NULL);
        }
        store = INT2PTR(struct libbinary_c_store_t *, SvUV(store_p));
        if(vb)
        {
           fprintf(stderr,"nthread=%d Using Converted to pointer : %p\n",
                   nthread,store);
           fflush(NULL);
        }
    }


    /*
     * Make a new binary system.
     * The structs are automatically allocated memory in the process,
     * if this is required.
     */
     if(vb)
    {
        fprintf(stderr,
                   "nthread=%d Using store = %p (%p) : built? %d\n",
                   nthread,
                   store,
                   &store,
                   store?store->built:0);
        fflush(NULL);
    }
    char s[100];
    snprintf(s,99,"thread %d pre-new-system ",nthread);
        
    binary_c_new_system(&stardata,
                        NULL,
                        NULL,
                        &store,
                        &argstring,
                        -1);
    
    snprintf(s,99,"thread %d post-new-system",nthread);
    if(vb)
    {
        fprintf(stderr,"nthread=%d : stardata is at %p\n",
                nthread,stardata);
        fflush(NULL);
    }

    /*
     * Set thread number, disable logging, 
     * force buffering 
     */ 
    stardata->model.id_number = nthread;

    snprintf(stardata->preferences->log_filename,
             STRING_LENGTH-1,"%s","/dev/null");   
    snprintf(stardata->preferences->api_log_filename_prefix,
             STRING_LENGTH-1,"%s","/dev/null");
    
    stardata->preferences->internal_buffering = INTERNAL_BUFFERING_STORE;
    stardata->preferences->batchmode = BATCHMODE_LIBRARY;

    if(custom_output_function_ptr != NULL)
    {
        /* 
         * The function pointer is provided as an 
         * unsigned int inside an SV* : use SvUV to 
         * convert back to unsigned int, then cast to the
         * appropriate pointer. 
         */
        void (*ptr)(struct stardata_t *) =  
            INT2PTR(void (*)(struct stardata_t *),
                    SvUV(custom_output_function_ptr));
        
        /*
         * Save in stardata to force appropriate output
         */
        stardata->preferences->custom_output_function = ptr;
    }
    else
    {
        stardata->preferences->custom_output_function = NULL;
    }

    /*
     * evolve the system
     */
    if(vb)
    {
        fprintf(stderr,"nthread=%d Call evolve_for_dt : stardata=%p store=%p \n",
                nthread,stardata,store);
        fflush(NULL);
    }

    if(Is_not_zero(stardata->model.max_evolution_time))
    {
        binary_c_evolve_for_dt(stardata,stardata->model.max_evolution_time);
    }

    if(vb)
    {
        fprintf(stderr,"thread %d : assault buffer (size %zu)\n",nthread,stardata->tmpstore->raw_buffer_size);
        fflush(NULL);
    }
    char * buffer = NULL;
    char * error_buffer = NULL;
    size_t nbytes = 0;

    /*
     * Get the binary_c output in *buffer.
     */
    binary_c_buffer_info(stardata,&buffer,&nbytes);
    binary_c_error_buffer(stardata,&error_buffer);

    if(vb)
    {
        fprintf(stderr,"thread %d : raw bufferd %p size %zu\n",nthread,buffer,nbytes);
        fflush(NULL);
    }

    if(buffer!=NULL && nbytes>0)
    {
    /*
     * split binary_c output into lines, put in Perl array 
     */
    char * line, * saveptr = NULL;
    size_t nlines=0;
    line = strtok_r(buffer,"\r\n",&saveptr);
    const char * const maxbuffer = buffer + nbytes;
       
    while(line != NULL && saveptr <= maxbuffer)
    {
       nlines++;

       if(return_array_refs)
       {
           /*
            * return array of arrays
            */ 
           AV * linearray = newAV();
           char * savelineptr = NULL;
           char * chunk = strtok_r(line," ",&savelineptr);
           while(chunk != NULL && savelineptr)
           {
               av_push(linearray,newSVpv(chunk,0));
               chunk = strtok_r(NULL," ",&savelineptr);
           }
           av_push(array,newRV_noinc((SV*)linearray));
       }
       else
       {
           /*
            * return array of lines
            */
           av_push(array,newSVpv(line,0));
       }

       line = strtok_r(NULL,"\r\n",&saveptr);
    }


    if(vb)
    {  
        fprintf(stderr,"thread %d : returning %zu lines of data from buffer = %p\n",nthread,nlines,buffer);
        fprintf(stderr,"thread %d : done C.pm run_binary_c (buffer = %p, size = %zu, stardata = %p, store = %p)\n",
             nthread,buffer,nbytes,stardata,store);
        fflush(NULL);
    }
    }
    else
    {
       /* buffer is NULL : return empty array */ 
    }

    /* append error buffer contents to array */
    if(error_buffer != NULL)
    {
        if(return_array_refs)
        {
            char * saveptr = NULL;
            char * line = strtok_r(error_buffer,"\r\n",&saveptr);
            AV * linearray = newAV();
            if(line != NULL)
            {
               char * savelineptr = NULL;
               char * chunk = strtok_r(line," ",&savelineptr);
               while(chunk != NULL && savelineptr)
               {
                   av_push(linearray,newSVpv(chunk,0));
                   chunk = strtok_r(NULL," ",&savelineptr);
               }
            }
            av_push(array,newRV_noinc((SV*)linearray));
        }
        else
        {
            av_push(array,newSVpv(error_buffer,0));
        }
    }

    /*
     * empty binary_c output buffers and free their memory 
     */
    binary_c_buffer_empty_buffer(stardata);

    if(vb)
    {
        fprintf(stderr,"nthread=%d End store = %p : built? %d\n",
                nthread,store,store?store->built:0);
        fflush(NULL);
    }

    /*
     * Put pointer to the store struct in the returned Perl stack
     * so that it can be reused.
     */
    av_push(array,newSVuv(PTR2UV(store)));

    if(vb)
    {
        fprintf(stderr,"call binary_c free on stardata %p\n",stardata);
        fflush(NULL);
    }
    
    /*
     * Free all memory except the store structure.
     */
    binary_c_free_memory(&stardata,TRUE,TRUE,FALSE,TRUE);
    Safe_free(stardata); // just in case

    if(vb)
    {
        fprintf(stderr,
            "nthread=%d RETURNING stardata=%p, store %p\n",
            nthread,stardata,store);
        fflush(NULL);
    }
   return;
}


BINARY_C_CODE_BODY

        # build the code
        my $code =
            $code_head .
            $code_body.
            $code_base;

        # bind the code to Inline::C
        Inline->bind(
            C => $code,
            force_build => 1,
            rewrite_config_file => 1
            );
    }
}


{

    use threads::shared;
    ############################################################

    # list of binary_grid functions we override
    my @binary_grid_funcs_overridden = (
        'evcode_version_string',
        'evcode_args_list',
        'tbse',
        'tbse_kill',
        'tbse_land',
        'tbse_launch',
        'tbse_line',
        'tbse_restart',
        'kill_flexigrid_evcode_pid',
        'kill_flexigrid_evcode_pids',
        'suicide',
        'stop_flexigrid_threads',
        'minimum_period_for_RLOF',
        );

        
    # delete the subroutines we will override
    my @redef_list;
    foreach my $subroutine (@binary_grid_funcs_overridden)
    {
        push(@redef_list,$subroutine);
        eval "undef \&binary_grid::$subroutine; "; 
        eval "undef \&binary_grid2::$subroutine; "; 
        if($@)
        {
            print $@;
            exit;
        }
    }
    print STDERR 'Redefine ',color('yellow');
    foreach my $subroutine (@redef_list)
    {
        print STDERR $subroutine,' ';
    }
    print STDERR color('reset'),"to use C\n";
    print STDERR "binary_grid API subroutines claimed by binary_grid::C\n";
    
    # provide new subroutines where required

        *binary_grid::kill_flexigrid_evcode_pid = sub
    {
        # not required
        1;
    };
    *binary_grid2::kill_flexigrid_evcode_pid = sub
    {
        # not required
        1;
    };

    *binary_grid::kill_flexigrid_evcode_pids = sub
    {
        # not required
        1;
    };
    *binary_grid2::kill_flexigrid_evcode_pids = sub
    {
        # not required
        1;
    };

    *binary_grid::suicide = sub
    {
        # not required
        1;
    };
    *binary_grid2::suicide = sub
    {
        # not required
        1;
    };

    *binary_grid::stop_flexigrid_threads = sub
    {
        # not required
        1;
    };
    *binary_grid2::stop_flexigrid_threads = sub
    {
        # not required
        1;
    };


    *binary_grid::tbse_kill = sub
    {
        # not required
        1;
    };
    *binary_grid2::tbse_kill = sub
    {
        # not required
        1;
    };


    *binary_grid::tbse_restart = sub
    {
        # not yet implemented
        1;
    };

    *binary_grid2::tbse_restart = sub
    {
        # not yet implemented
        1;
    };


    {
        # Data stack and structures are local
        # to each flexigrid thread.
        #
        # Each run of tbse filled @binary_c_data_stack
        #
        # %structures contains information on memory
        # structure locations which are passed to subsequent calls
        # to tbse, saving the re-allocation each time.
        my @binary_c_data_stack;
        my %structures;
        
        
        *binary_grid2::tbse = sub
        {
            my ($self,$args,$datahash,$nthread) = @_;

            # tbse must be called with arguments
            if(!defined $_[1])
            {
                confess('binary_grid: tbse () called with no args!');
            }
            # set timeout in case of failure/pause/lockup
            elsif($self->{_grid_options}->{timeout} &&
                  (!$self->{_grid_options}->{'disable signal'}{ALRM}))
            {
                # set timeout
                $self->set_next_alarm($self->{_grid_options}->{timeout});	
            }
 # default log_filename to /dev/null
            $args->{log_filename}//='/dev/null';

            if(defined $nthread)
            {
                # replace thread number in filename
                $args->{log_filename} =~ s!__THREAD__!$nthread!;
            }
            else
            {
                # if thread number is unknown, set to -1
                $nthread = -1;
            }
            
            # set up the progenitor hash
            $self->set_progenitor_info($args);
            
            # if $args is a hash, construct the argstring from it
            if(ref($args) eq 'HASH')
            {
                state @priority_args;
                state $pre;
                if(!defined $pre)
                {
                    $pre = ' '.$self->{_grid_options}->{arg_prefix};
                    @priority_args = @{$self->{_priority_args}};
                }

                my $argstring='';
                foreach my $priority_arg (@priority_args)
                {
                    no warnings;
                    $argstring .= $pre.$priority_arg.' '.
                        delete $$args{$priority_arg};
                    use warnings;
                }
                if($self->{_grid_options}->{sort_args})
                {
                    foreach my $k (sort keys %$args)
                    {
                        $argstring .= $pre.$k.' '.($args->{$k}//'');
                    }
                }
                else
                {
                    while (my ($arg, $val) = each %$args)
                    {
                        $argstring .= $pre.$arg.' '.($val//'');
                    }
                }
                $args = 'binary_c '.$argstring;
            }
            
            # force internal buffering
            $args .= ' --internal_buffering INTERNAL_BUFFERING_STORE ';

            state $logargs =
                $self->{_grid_options}->{log_args} || 
                $self->{_grid_options}->{save_args} ||
                $self->{_grid_options}->{vb}>=2 ||
                $self->{_grid_options}->{log_system_errors};
            
            if($logargs)
            {
                # only make the argstring if required e.g. vb>=2, or if args are to be logged
                $self->{_grid_options}->{args}=$args;
                $self->{_threadinfo}->{lastargs}=$args;
                $self->{_threadinfo}->{lastruntime}=time;
            }

            # save the previous alive time for each thread
            $self->{_threadinfo}->{thread_prev_alive} = time();
            
            $self->vbout(2,"\nThread $nthread\nProgenitor: $self->{_grid_options}->{progenitor}\n".$colour{'blue bold'}."The star in blue is running ***right now*** on thread $nthread (and if it crashes, these are the arguments you should test):\n\n ".(($args=~/^binary_c  ?(.*)/)[0]).$colour{reset}."\n\n\n");

            # log arguments to a file
            if($self->{_grid_options}->{log_args})
            {
                $self->log_args($args);
            }  # end log_args check

            $self->vbout(3,"evcode go\n");
            
            # if $mode is 1, we have an existing store structure
            # which can be used, otherwise it is 0
            my $mode = defined $structures{store} ? 1 : 0;
            $self->vbout(3,"mode $mode\n");
            
            # run the evolution code, putting the result 
            # on the binary_c_data_stack array
            my $binary_c_data_stack = [];
            $self->vbout(3,"Call C_run_binary_c\n");
            binary_grid::C::C_run_binary_c(
                $mode,
                $self->{_grid_options}->{return_array_refs}//0,
                $structures{store},
                $binary_c_data_stack,
                $args,
                $nthread,
                $self->{_grid_options}->{custom_output_C_function_pointer}//0
                );
            $self->vbout(3,"Returned from C_run_binary_c\n");
            
            # take a copy of the stack, it will be cleaned up!
            if($binary_c_data_stack)
            {
                @binary_c_data_stack = @$binary_c_data_stack;
                $binary_c_data_stack = undef;
            }
            else
            {
                @binary_c_data_stack = ();
            }
            
            # final line points to the store structure, get it
            $structures{store} = pop @binary_c_data_stack;
            
            # push fin to mark the end 
            if($self->{_grid_options}->{return_array_refs})
            {
                push(@binary_c_data_stack,['fin']);
            }
            else
            {
                push(@binary_c_data_stack,'fin');
            }
            
            # parse the output
            state $parse_bse = $self->{_grid_options}->{parse_bse_function_pointer};
            $self->vbout(3,
                         sprintf"calling parse_bse (thread %d, func pointer $parse_bse, is %s)\n",
                         $self->{_threadinfo}->{thread_number}//-1,
                         sub_fullname($parse_bse));
            

            my $eval_return =
                eval{
                   $parse_bse->($self,$datahash,$nthread);
            };

            if(!$eval_return && $@)
            {
                # eval failed : report a system error
                $self->report_system_error($@);
            }

            # reset alarm so the timeout is not hit
            $self->set_next_alarm(0)
                if($self->{_grid_options}->{timeout} && 
                   !$self->{_grid_options}->{'disable signal'}{ALRM});

            # log fin to a file
            if($self->{_grid_options}->{log_fins})
            {
                $self->log_fin($args);
            }  # end log_fin check
            
            return undef;
        };

 
        *binary_grid2::tbse_land = sub
        {
            # get memory back
            my ($self,$nthread) = @_; 
            my $mode = defined $structures{store} ? 1 : 0;
            binary_grid::C::C_shutdown_binary_c($mode,
                                                $structures{store});
        };
    
   
        *binary_grid2::tbse_line = sub
        {
            my ($self) = @_;

            # return a line of the data stack
            my $x = shift @binary_c_data_stack;
            
            # but check for SYSTEM_ERROR 
            $self->check_for_system_error($x);
            
            return $x;
        };


        *binary_grid2::tbse_array_ref = sub
        {
            my ($self) = @_;
            # non-API routine to return the entire
            # stack as a reference, useful for debugging
            return \@binary_c_data_stack;
        };

        # wrapper for tbse to make running individual
        # stars easier.
        # Takes one argument, the args for the system (as a hash pointer)
        # and optionally then:
        # results hash pointer (defaults to _results)
        # thread number (defaults to -1)
        #
        
        *binary_grid2::run_system = sub
        {
            my ($self,$system_args,$results,$nthread) = @_;
            $results //= $self->{_results};
            $nthread //= -1;
            my $args = $self->make_evcode_arghash($system_args);
            $self->tbse($args,$results,$nthread);
        }

    }


    *binary_grid2::tbse_launch = sub
    {
        my ($self) = @_;
        no warnings;
        my $pid = "Thread $self->{_threadinfo}->{thread_number} running libbinary_c.so";
        use warnings;
        $self->{_threadinfo}->{evcode_pid} = 'libbinary_c.so';
        $self->{_grid_options}->{evcode_pid} = 'libbinary_c.so';
        $self->thread_log($pid);
        print "LAUNCH binary_grid::C with parser subroutine = $self->{_grid_options}->{parse_bse_function_pointer}\n";
        return $pid;
    };

    
    *binary_grid2::evcode_version_string = sub
    {
        my $binary_c_data_stack = [];
        binary_grid::C::C_binary_c_version_string($binary_c_data_stack);
        my $version = join("\n",@$binary_c_data_stack);
        return $version;
    };

    *binary_grid2::evcode_args_list = sub
    {
        my $binary_c_data_stack = [];
        binary_grid::C::C_binary_c_args_list($binary_c_data_stack);
        return $binary_c_data_stack;
    };


############################################################
# minimum orbital period and separation for RLOF
############################################################


    *binary_grid2::minimum_period_for_RLOF = sub
    {
        my $self = shift;
        my @orbit = $self->minimum_orbit_for_RLOF(@_); 
        return $orbit[1];
    };
    
    *binary_grid2::minimum_separation_for_RLOF = sub
    {
        my $self = shift;
        my @orbit = $self->minimum_orbit_for_RLOF(@_); 
        return $orbit[0];
    };

    *binary_grid2::minimum_orbit_for_RLOF = sub
    {
        my $self = shift;
        my ($m1,$m2) = @_; # input variables are m1,m2
        
        # static data
        state @priority_args;
        state $pre;
        if(!defined $pre)
        {
            $pre = ' '.$self->{_grid_options}->{arg_prefix};
            @priority_args = @{$self->{_priority_args}};
        }
    
        # make argstring
 	my $args = $self->make_evcode_arghash({
            M_1=>$m1,
            M_2=>$m2,
            metallicity=>$self->{_bse_options}{metallicity},
            separation=>0.0,
            eccentricity=>0.0,
            probability=>1.0,
            phasevol=>1.0,
            orbital_period=>1.0,
                                              });
        my $argstring='binary_c ';
        foreach my $priority_arg (@priority_args)
        {
            $argstring .= $pre.$priority_arg.' '.delete $$args{$priority_arg};
        }
        while (my ($arg, $val) = each %$args)
        {
            $argstring .= $pre.$arg.' '.($val//'');
        }
        
        # get binary_c stack and join into an array
        my $binary_c_data_stack = [];
        binary_grid::C::C_binary_c_minimum_orbit_for_RLOF(
            $binary_c_data_stack,
            $argstring
            );
        my $result = join("\n",@$binary_c_data_stack);
        
        # extract and return data
        my $minsep = ($result=~/MINIMUM SEPARATION (\S+)/)[0];
        my $minper = ($result=~/MINIMUM PERIOD (\S+)/)[0];
        
        return ($minsep,$minper);
    };

############################################################
# initial abundances
############################################################



    *binary_grid2::initial_abundance_string = sub
    {
        # return a string describing the initial abundances
        my $self = shift;
        my ($mix,$Z) = @_;
        my $array = [];
        my $argstring = 'init_abunds_only';
        print "INIT ABUNDS Z=$Z mix=$mix, args $argstring\n";
        C_binary_c_initial_abundance_mix($Z,$mix,$array,$argstring);
        return join("\n",@$array);
    };




}

sub gsl_error
{
    no warnings;
    print "\n\n\n************************************************************\n\nGSL and its header files not found : you need these\nI have looked in the  environment variable GSL_DIR (set to \"$ENV{GSL_DIR}\"), but cannot find them there. 'gsl-config --libs' gave me nothing also.\nPlease make sure GSL is installed and that either GSL_DIR is set or gsl-config gives the appropriate result.\n\n************************************************************\n\n\n";
    exit(1);
    use warnings;
}



1;
