package RobInterpolation;

###################################################
#                                                 #
# Rob's module to do simple interpolation of data #
#                                                 #
###################################################

use 5.10.0;
use warnings;
use strict;
use Data::Dumper;
use Module::Load;
require Exporter;
#use Maths_Double;
use Math::Trig ':pi';

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use RobInterpolation ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.

use subs qw(useC vb);
use vars qw($useC $VERSION);

my @subs = qw'useC generic_interpolation_wrapper generic_binary_search generic_interpolation';
our %EXPORT_TAGS = ( 'all' => [ @subs ] );
our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );
our @EXPORT = @subs;
our $VERSION = '0.04';


# changelog:
# V0.01 initial version
# V0.02 interpolate arrays
# V0.03 use Maths_Double and $useC
# V0.04 add Lanczos experimental code (not well tested!)

$useC=0; # default to not using C

###### methods and subroutines follow

sub vb
{
    state $vbhead = "RobInterpolation$VERSION: ";
    print $vbhead,@_;
}

sub useC
{
    # use C or pure Perl ? 
    # call useC(1) to turn C extensions ON (these are probably faster)
    if(defined $_[0])
    {
        $useC = $_[0];
        vb "Setting RobInterpolation::useC to $useC\n";
        
        if($useC)
        {
            # test that we have the Inline module
            eval {
                require Inline;
                Inline->import(
                    C=>Config=>BUILD_NOISY=>1,
                    );
            };

            # on Inline failure, disable C functions
            if($@ || !defined $Inline::VERSION) 
            {
                vb "Could not load Inline (\$\@=$@, Inline::VERSION $Inline::VERSION) -> setting \$useC = 0 to use pure Perl implementation\n";
                $useC = 0;
            }
            else
            {
                # try to load Maths_Double
                Module::Load::autoload Maths_Double;
                vb "Loaded Maths_Double ($VERSION $Maths_Double::VERSION)\n";
                
                # report an error: this should have worked!
                if($@)
                {
                    vb "Loading Maths_Double gave an error \"$@\" -> setting \$useC = 0 to use pure Perl implementation\n";
                    $useC = 0;
                }
            }
        }
    }
    
    # will search for binary_grid options (since this was written 
    # for binary_grid) or main::code_flavour
    return $useC;
}

sub pure_perl_generic_interpolation_wrapper
{
    # no longer do we have a pure Perl version of the 
    # generic_interpolation_wrapper: whether to use C
    # or not is controlled by $useC.
    return generic_interpolation_wrapper(@_);
}

sub generic_interpolation_wrapper
{
    # interpolate in an array given its values in a hash
    my ($value, # variable value
        $array, # the array
        $hash) # the hash
        =@_;
    
    #print "Interpolate $value in array = $array, hash = $hash\n";
    my $n = $#{$array};
    #print "CF VALUE = $value to low=$$array[0] ($hash->{$$array[0]}), high = $$array[$n] ($hash->{$$array[$n]})\n";
    
    return 
        # lower bound
        $value<=$$array[0] ? $hash->{$array->[0]} :
        # upper bound
        $value>=$$array[$n] ? $hash->{$array->[$n]} :
        # interpolate
	return generic_interpolation($value,
				     (generic_binary_search($value,$array)),
				     $array,$hash);
}

sub generic_binary_search
{
    # generic binary search function:
    #
    # given an array performs the binary search
    # and returns the bounding indices
    my ($value, # value
        $array) # array to search    
        =@_;

    my ($sa,$sb);
    if($useC)
    {
        ($sa,$sb) = Maths_Double::binary_search($value,$array);
    }
    else
    {
        # pure perl binary search
        $sa = 0;
        $sb = $#{$array};
        my $sc;
        while($sb > $sa+1)
        {
            $sc = int(($sa+$sb)/2);
            ($value > $$array[$sc] ? $sa : $sb) = $sc;
        }
    }
    return($sa,$sb);
}


sub generic_interpolation
{
    # interpolate in an array given its values in a hash

    # NB there is no bounds checking 

    my ($value, # variable value
        $index_low, # spanning array indices
        $index_high,
        $array, # the array (reference)
        $hash) # the hash (reference)
        =@_;
    
    my $low  = $$array[$index_low];
    my $high = $$array[$index_high];
 
    # interpolation factor
    my $f = ($value-$low)/($high-$low);

    #print "Interpolate: value=$value : f=$f : indices=$index_low,$index_high : values=$low,$high\n";

    if((ref $hash->{$high}) eq 'ARRAY')
    {
        # interpolate all values in an array
        my @x;
        my $f1 = 1.0-$f;
        if($useC)
        {
            Maths_Double::C_add_two_arrays($f,$f1,\@x,$hash->{$high},$hash->{$low});
        }
        else
        {
            for(my $i=0;$i<=$#{$hash->{$high}};$i++)
            {
                $x[$i] = $f * $hash->{$high}->[$i] + $f1 * $hash->{$low}->[$i];
            }
        }
        return \@x;
    }
    else
    {
        # assume scalar : quicker in pure perl
        return ($f*$hash->{$high} + (1.0-$f)*$hash->{$low});
    }
}


sub Lanczos
{
    my ($coords,$widths,$table) = @_;
    # given a table of data (as [ [], [], ... ])
    # and some coordinates in it (as []) 
    # perform Lanczos interpolation to find the values
    # at the coords givens the widths (as []) of the 
    # smoothing kernel
    #
    # returns $S = [], the resulting interpolation point
    # combined with the interpolated data
    my $nd = scalar @$coords;
    print "Table nd=$nd\n";
    
    # interpolated data
    my $S = [];
    
    my $pisq = pi * pi;
    foreach my $line (@$table)
    {
        # find the vector $x to the interpolation point
        my $x = [];
        print "Line $line (@$line)\n";
        
        # calculate the Lanczos function at this point
        my $L = 1.0;
        for(my $i=0;$i<$nd;$i++)
        {
            $x->[$i] = ($line->[$i] - $coords->[$i]);
            my $absx = abs($x->[$i]);
                        
            # Lanczos function
            my $fL = 
                # must be inside the window
                 $absx > $widths->[$i] ? 0.0 : 
                 
                 # prevent 1/0
                 $absx<1e-20 ? 1.0 :
                 
                 # sincs
                 $widths->[$i] * 
                 sin(pi * $x->[$i]) * 
                 sin(pi * $x->[$i]/$widths->[$i])
                 /
                 ($pisq * $absx*$absx)
                 ;
            $L *= $fL;
            
            print "X $i = $line->[$i] - $coords->[$i] = $x->[$i], abs $absx cf width = $widths->[$i] : fL=$fL (L=$L)\n";
        }
        print "Hence L = $L\n";
        
        # loop over the data, finding the contributions to the
        # interpolated data
        if($L>1e-30)
        {
            for(my $i=$nd; $i<=$#{$line}; $i++)
            {
                $S->[$i] += $line->[$i] * $L;
            }
        }
    }

    # fill the rest of the array
    for(my $i=0;$i<$nd;$i++)
    {
        $S->[$i] = $coords->[$i];
    }

    return ($S);
}


1;
