package robqueue;
$|=1;
use 5.019000;
use strict;
use threads;
use Thread::Queue;
use rob_misc;
use Sys::Info;
use Sys::Info::Device;
use Sys::Info::Device::CPU;
use Sys::Info::Constants qw( :device_cpu );
use Hash::RobMerge;
use Data::Dumper;

#
# robqueue : a simple task queuer for perl
#
# Can either run commands as shell commands directly, or send them to a subroutine
#
# Usage:
#
#
# # make new queue
# my $q=robqueue->new(
#	    nthreads=>4, # set number of threads
#	    command_queue=>0, # send commands for the shell (0) 
#                             # or subroutine reference 
#	    subr=>\&my_subroutine, # sub ref
#	    vb=>1, # verbosity
#	    prepend_thread_number_to_subroutine=>1, # send thread number (0) ?
#           sumhash=>undef, # if defined, returned results are added to this using Hash::RobMerge::arithmetically_add_hashes
#	);
#
#
#
#  while(1)
#  {
#     # send args to the queue
#     $q->q(...);
#    
#     $q->qwait(); # don't let the queue get too large (memory leak!)
#  }
#
#  $q->end; # when you are finished, sync and end
#
#  # subroutine to do the work
#  sub my_subroutine
#  {
#      my $nthread=$_[0];
#      my $arg = $_[1];
#      
#      ...
#      ... do stuff with $arg ...
#      ...
#  }
#
#
#

require Exporter;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use robqueue ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw(
) ] );
our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );
our @EXPORT = qw(
);
our $VERSION = '0.05';

#
# Change log
#
# V0.01-0.04 initial development
# V0.05 better documentation for perldoc
#

# Preloaded methods go here.

sub new
{
    # make a new robqueue object
    my ($class,@args) = @_;
    my $self={};
    bless $self,$class;
    $self->initialise(@args);
    return $self;
}

sub initialise
{
    # initialise robqueue object
    my ($self,@args)=@_;

    # set defaults and apply args if given
    %$self =(

	# options : can ALL be left at the default
	# EXCEPT subr if you want to use subroutines.
	#
	# otherwise, defaults to the number of CPUs
	# (linux) or 1 thread

	vb => 0, # verbosity
	nthreads => MAX(1,robqueue::ncpus()), # number of threads
	command_queue => 0, # use as command queue 
	subr => undef,  # use as subroutine queue
	arguments=>[], # arguments to be passed after the command
	prepend_thread_number_to_subroutine=>0, # prepend thread number to subroutine?
	sumhash=>undef,

	#### internal variables - do not change!
	q => undef, # internal queue object
	threads => [], # internal thread list

	@args # apply args
	);

    if($self->{vb})
    {
	print "robqueue object:\n";
	map
	{
	    print "$_ = $$self{$_}\n";
	}keys %$self;
	print "Calling init\n";
    }
 
    $self->init();

    return $self;
}

sub command_queue
{
    # set or return command queue
    $_[0]->{command_queue}=$_[1] if(defined($_[1]));
    return $_[0]->{command_queue};
}

sub init
{
    # function to make the command queue by spawning threads
    my $self=$_[0];
    
    my $q=Thread::Queue->new();
    $self->{q}=$q;
    
    print "Made robqueue's queue: $self->{q}\n" if($self->{vb});
    
    # thread number is 0 ... nthreads-1 (total of nthreads of them)
    foreach my $thread_number (0..$self->{nthreads}-1)
    { 
	if($self->{command_queue})
	{
	    # command queue
	    push(@{$self->{threads}}, threads->create(
		     sub{
			 while(defined(my $i=$q->dequeue()))
			 {
			     #print "RUN $i\n" if(0);
			     `$i`;
			 }
		     }));
	}
	else
	{
	    # subroutine queue
            print "Make subroutine thread\n" if($self->{vb});

	    if(!defined($self->{subr}))
	    {
		print "You must define a subroutine if you're not using a command queue\n";
		exit;
	    }

	    push(@{$self->{threads}}, threads->create(
		     sub{
                         print "loop sub on subr=$self->{subr}, q=$q\n"if($self->{vb});
			 my $sumhash={};
			 while(defined(my $i=$q->dequeue()))
			 {
			     my @args=($i);
 			     # add arguments if required
			     push(@args,@{$self->{arguments}}) if(defined $self->{arguments});
			     # add thread number if required
			     unshift(@args,$thread_number) 
				 if($self->{prepend_thread_number_to_subroutine});			     
			     
			     print "Add queue job '@args'\n" if($self->{vb});
			     
			     my $x = &{$self->{subr}}(@args);
			     
			     # if given a sumhash, and data to add, add to it
			     if(defined $self->{sumhash} && defined $x)
			     {
#				 print "merge new results=$x with sumhash=$sumhash\n";
#				 print "x           = ",Dumper($x),"\n";
#				 print "sumhash was = ",Dumper($sumhash),"\n";
				 Hash::RobMerge::arithmetically_add_hashes($sumhash,$x);
#				 print "sumhash now = ",Dumper($sumhash),"\n";
			     }
			 }
#			 print "return \$sumhash=",Dumper($sumhash),"\n";
			 return $sumhash;
		     }));
	}
    }
}

sub enqueue
{
    # send job strings to queue
    my $self=shift @_;
    $_[0]//="";

    $self->{q}->enqueue(@_);    
}

sub q
{
    # send job strings to queue
    my $self=shift @_;
    $_[0]//="";
    print "Queue args \"@_\" onto $self->{q}\n" if($self->{vb});
    $self->{q}->enqueue(@_); 
}

sub pending
{
    $_[0]->{q}->pending();
}

sub which_pending
{
    my $self = shift @_;
    return (grep {defined $_}@{$self->{threads}});
}

sub running
{
    $_[0]->{q}->running();
}

sub qwait
{
    # wait for the queue to empty to at least $_[1] items, then return:
    # use this to not allow the queue array to become huge on long jobs
    #
    # if no number is given, assume the queue can be 10* nthreads
    #
    # you may want to sleep for a shorter time than one second, 
    # but of course this kind of wait will never be required for 
    # short-running jobs.
    #
    my $lim = $_[1] // (10*$_[0]->{nthreads});
    $lim = MAX($_[0]->{nthreads},$lim);
    while((scalar $_[0]->pending()) > $lim)
    {
	sleep 1;
    } 
}

sub end
{
    # end the queue
    my $self=$_[0];
    #print "Ending robqueue $_[0]\n" if(0);
    $self->{q}->enqueue((undef)x($self->{nthreads}+1));
    map
    { 
 	my $h = $_->join;
	# if hash is returned, add it to the global hash
	if(defined $h)
	{
#	    print "thread returned hash $h : add to $self->{sumhash}\n";
	    Hash::RobMerge::arithmetically_add_hashes($self->{sumhash},$h);
	}
    }@{$self->{threads}};
    $self->{q}->enqueue((undef)x($self->{nthreads}+1));
    
    # try to free memory : might have no effect :-/
    map{
        $_ = undef;
    }@{$self->{threads}};
    
    return $self->{sumhash};
}


sub ncpus
{
    # count number of cpus
    my $info = Sys::Info->new;
    my $cpu  = $info->device('CPU');
    return $cpu->count;
}


1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

robqueue - Perl extension for setting up a job queue, according to
rob's specs.

=head1 SYNOPSIS

  use robqueue;


 robqueue : a simple task queuer for perl

 Can either run commands as shell commands directly, or send them to a subroutine


=head1 DESCRIPTION

 Usage (defaults in brackets):

  make new queue
 my $q=robqueue->new(
	    nthreads=>4, # set number of threads (ncpus)
	    command_queue=>0, # send commands for the shell (0) 
                              # or subroutine reference (subr below) 
	    subr=>\&my_subroutine, # sub ref (undef)
	    vb=>0, # verbosity (0)
	    prepend_thread_number_to_subroutine=>0, # send thread number (0) 
           sumhash=>undef, # if defined, returned results are added to this using Hash::RobMerge::arithmetically_add_hashes
	);



  while(1)
  {
     # send args to the queue
     $q->q(...);
    
     $q->qwait(); # don't let the queue get too large (memory leak!)
  }

  $q->end; # when you are finished, sync and end

  # subroutine to do the work
  sub my_subroutine
  {
      my $nthread=$_[0];
      my $arg = $_[1];
      
      ...
      ... do stuff with $arg ...
      ...
  }



=head2 EXPORT

None by default.



=head1 SEE ALSO

Mention other useful documentation such as the documentation of
related modules or operating system documentation (such as man pages
in UNIX), or any relevant external documentation such as RFCs or
standards.

If you have a mailing list set up for your module, mention it here.

If you have a web site set up for your module, mention it here.

=head1 AUTHOR

A. U. Thor, E<lt>izzard@(none)E<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2013 by A. U. Thor

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.19.0 or,
at your option, any later version of Perl 5 you may have available.


=cut
